use strict;
use warnings;

package App::Embra::File;
$App::Embra::File::VERSION = '0.001'; # TRIAL
# ABSTRACT: a file from your site

use File::Basename;
use File::Spec::Functions qw< canonpath >;
use Method::Signatures;
use Moo;

# mostly Dist::Zilla::File::OnDisk


use overload fallback => 1,
    '""' => method( $other, $swap ) { $self->name };


has 'name' => (
    is => 'rw',
    required => 1,
);


has 'content' => (
  is  => 'rw',
  lazy => 1,
  default => method { $self->_read_file },
);


has 'mode' => (
  is  => 'rw',
  lazy => 1,
  default => method { oct(644) },
);


has '_original_name' => (
  is  => 'ro',
  init_arg => undef,
);


has 'notes' => (
    is => 'ro',
    default => method { {} },
);


has 'ext' => (
    is => 'rw',
    lazy => 1,
    builder => 1,
    trigger => 1,
);

method _split_name {
    fileparse( $self->name, qr{ (?<= [.] ) [^.]+ $ }x );
}

method _build_ext {
    ($self->_split_name)[2];
}


method with_ext( $ext ) {
    $ext =~ s/ \A [.] //xms;
    my ($f, $d, $e) = $self->_split_name;
    return $self->name if $e eq $ext;
    return canonpath( $d . $f . $ext );
}

method _trigger_ext( $old_ext ) {
    $self->name( $self->with_ext( $self->ext ) );
}

method BUILD( $args ) {
  $self->{_original_name} = $self->name;
}

method _read_file {
  my $fname = $self->_original_name;
  open my $fh, '<', $fname or die "can't open $fname for reading: $!";

  binmode $fh, ':raw';

  my $content = do { local $/; <$fh> };
}


method update_notes( %notes ) {
    @{ $self->notes }{ keys %notes } = values %notes;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

App::Embra::File - a file from your site

=head1 VERSION

version 0.001

=head1 DESCRIPTION

This represents a file to be included in your site.

=head1 ATTRIBUTES

=head2 name

The name of the file. Change this to change where the file will appear in the site.

=head2 content

The content of the file. Change this to change the content of the file when it appears in the site. Defaults to the contents of C<_original_name>.

=head2 mode

The permissions of the file. Defaults to 0644.

=head2 _original_name

The original name of this file. This is automatically saved from the C<name> attributes used to construct the object, and can't be altered.

=head2 notes

A hash ref which stores extra values associated with the file. Transform plugins will read and write notes, and Assemble plugins will read notes.

=head2 ext

The extention of the file's C<name>. Changing this will cause the file's C<name> to be updated to match.

=head1 METHODS

=head2 with_ext

    $file->with_ext( $ext );

Returns file's name with its extension changed to <$ext>.

=head2 update_notes

    $file->update_notes( %more_notes );

Merges C<%more_notes> into the file's existing notes.

=head1 AUTHOR

Daniel Holz <dgholz@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Daniel Holz.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
