package RDF::SKOS::OeNACE;

use strict;
use warnings;

use Data::Dumper;

use base 'RDF::SKOS';

use Text::CSV_XS;
my $csv = Text::CSV_XS->new ({
    quote_char          => '"',
    sep_char            => ';',
    binary => 1
			     });

=head1 NAME

RDF::SKOS::OeNACE - SKOS - OeNACE Classification Data

=head1 SYNOPSIS

   use RDF::SKOS::OeNACE;
   my $oenace = new RDF::SKOS::OeNACE;

   # this is a subclass of RDF::SKOS
   @cs = $oenace->concepts;
   ...

=head1 DESCRIPTION

To quote from L<http://portal.wko.at/wk/format_detail.wk?AngID=1&StID=372762&DstID=17>

   Auf nationaler Ebene wird ab 2008 die NACE 2008 angewandt. Diese
   Aktivitätsklassifikation untergliedert die europäische NACE
   mittels Unterklassen noch detaillierter, sodass österreichische
   Spezifika berücksichtigt werden können. Ihr Code ist 5-stellig
   und stimmt bis zur 4. Stelle mit der NACE überein. Die
   Unterklassen werden dabei durch die mit Bindestrich abgetrennte
   letzte Ziffer des Codes dargestellt.

B<NOTE>: I<Austrian specifics> is the usual euphemism of our local
buerocracy. Austria is ... special.

This package implements an SKOS view over this data. It is a subclass L<RDF::SKOS>.

=head2 SKOS Interpretations

=over

=item

The data contains german (C<de>) and english (C<en>) preferred labels.

=item

As hidden label the alphanumeric code is used. It is also the ID.

=item

No other I<labels>, I<notes>, I<descriptions>, .... are available.

=item

There is no I<related> information.

=item

As for the parent class, there is no *Match functionality.

=back


=head2 Limitations

=over

=item

C<narrowerTransitive> does not respect transitivity. Same with C<broaderTransitive>.

=back

=head1 INTERFACE

=head2 Constructor

The constructor does not expect any parameters.

=cut

our %SKOS;
sub _initialize {
    <DATA>; # skip first
    while ($_ = <DATA>) {
	last if /----/;
	$csv->parse ($_);
	my @fs = $csv->fields ();
	$SKOS{$fs[1]} = [ @fs[2,3] ];
    }
    <DATA>; # skip first
    while ($_ = <DATA>) {
	$csv->parse ($_);
	my @fs = $csv->fields ();
	push @{ $SKOS{$fs[1]} }, $fs[3];
    }

    foreach my $k (keys %SKOS ) { #-- internal consistency check
	die "inconsistent data" unless scalar @{ $SKOS{$k} } == 3;
    }
}


sub new {
    my $class = shift;
    _initialize unless %SKOS;
    my $self = bless {}, $class;
    return $self;
}

=pod

=head2 Methods

See also L<RDF::SKOS>.

=over

=item B<concept>

Different to the superclass, this method is read-only.

=cut

sub concept {
    my $self = shift;
    my $id   = shift;
    return new RDF::SKOS::Concept ($self, $id) if $SKOS{$id};
}

sub concepts {
    my $self = shift;
    return map { new RDF::SKOS::Concept ($self, $_) } keys %SKOS;
}

=pod

=item B<schemes>

As OeNACE does not have any explicit scheme, this will return an empty list.

=item B<scheme>

As there are no scheme, we will die here.

=cut

sub scheme { die "no schemeing"; }

=pod

=item B<topConcepts>

While there are no schemes, there are top concepts in OeNACE. These
are those with a single letter ID (A, B, C, ...) and these will be
returned. Any scheme parameter will be ignored.

=cut

sub topConcepts {
    my $self = shift;
    return 
	map  { new RDF::SKOS::Concept ($self, $_) }
        grep { $_ =~ /^[A-Z]$/}
        keys %SKOS;
}

sub prefLabels {
    my $self = shift;
    my $id   = shift;
    return
	[ $SKOS{$id} [1], 'de' ],
	[ $SKOS{$id} [2], 'en' ];
}

sub altLabels      { () }

sub hiddenLabels {
    my $self = shift;
    my $id   = shift;
    return [ $id, undef ];
}

sub notes          { () }
sub scopeNotes     { () }
sub definitions    { () }
sub examples       { () }
sub historyNotes   { () }
sub editorialNotes { () }
sub changeNotes    { () }


sub narrower {
    my $self = shift;
    my $id   = shift;
    if ($id =~ /^[A-Z]$/) {
	return
	    map { bless { id => $_, skos => $self }, 'RDF::SKOS::Concept' }
	    grep { /$id..$/ }
	    keys %SKOS;
    } else {
	return
	    map { bless { id => $_, skos => $self }, 'RDF::SKOS::Concept' }
	    grep { /$id.$/ }
	    keys %SKOS;
    }
}

=pod

=item B<narrowerTransitive>

B<NOTE>: At the moment this does not honor transitivity.

=cut

sub narrowerTransitive {
    my $self = shift;
    $self->narrower (@_);
}

sub broader {
    my $self = shift;
    my $id   = shift;

    if ($id =~ /^([A-Z])..$/) {
	return bless { id => $1, skos => $self }, 'RDF::SKOS::Concept';
    } elsif ($id =~ /^[A-Z]$/) {
	return ();
    } elsif ($id =~ /([A-Z].*).$/) {
	return bless { id => $1, skos => $self }, 'RDF::SKOS::Concept';
    }
}

=pod

=item B<broaderTransitive>

B<NOTE>: At the moment this does not honor transitivity.

=cut

sub broaderTransitive {
    my $self = shift;
    $self->broader (@_);
}

=pod

=back

=head1 AUTHOR

Robert Barta, C<< <drrho at cpan.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-rdf-skos at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=RDF-SKOS>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 COPYRIGHT & LICENSE

Copyright 2009 Robert Barta, all rights reserved.

This program is free software; you can redistribute it and/or modify it under the same terms as Perl
itself.

=cut

our $VERSION = '0.01';

"against all odds";

__DATA__
"Ebene";"EDV-Code";"Code";"Titel"
1;"A";"A";"LAND- UND FORSTWIRTSCHAFT, FISCHEREI"
2;"A01";"A 01";"Landwirtschaft, Jagd und damit verbundene Ttigkeiten"
3;"A011";"A 01.1";"Anbau einjhriger Pflanzen"
4;"A0111";"A 01.11";"Anbau von Getreide (ohne Reis), Hlsenfrchten und lsaaten"
5;"A01110";"A 01.11-0";"Anbau von Getreide (ohne Reis), Hlsenfrchten und lsaaten"
4;"A0112";"A 01.12";"Anbau von Reis"
5;"A01120";"A 01.12-0";"Anbau von Reis"
4;"A0113";"A 01.13";"Anbau von Gemse und Melonen sowie Wurzeln und Knollen"
5;"A01130";"A 01.13-0";"Anbau von Gemse und Melonen sowie Wurzeln und Knollen"
4;"A0114";"A 01.14";"Anbau von Zuckerrohr"
5;"A01140";"A 01.14-0";"Anbau von Zuckerrohr"
4;"A0115";"A 01.15";"Anbau von Tabak"
5;"A01150";"A 01.15-0";"Anbau von Tabak"
4;"A0116";"A 01.16";"Anbau von Faserpflanzen"
5;"A01160";"A 01.16-0";"Anbau von Faserpflanzen"
4;"A0119";"A 01.19";"Anbau von sonstigen einjhrigen Pflanzen"
5;"A01190";"A 01.19-0";"Anbau von sonstigen einjhrigen Pflanzen"
3;"A012";"A 01.2";"Anbau mehrjhriger Pflanzen"
4;"A0121";"A 01.21";"Anbau von Wein- und Tafeltrauben"
5;"A01210";"A 01.21-0";"Anbau von Wein- und Tafeltrauben"
4;"A0122";"A 01.22";"Anbau von tropischen und subtropischen Frchten"
5;"A01220";"A 01.22-0";"Anbau von tropischen und subtropischen Frchten"
4;"A0123";"A 01.23";"Anbau von Zitrusfrchten"
5;"A01230";"A 01.23-0";"Anbau von Zitrusfrchten"
4;"A0124";"A 01.24";"Anbau von Kern- und Steinobst"
5;"A01240";"A 01.24-0";"Anbau von Kern- und Steinobst"
4;"A0125";"A 01.25";"Anbau von sonstigem Obst und Nssen"
5;"A01250";"A 01.25-0";"Anbau von sonstigem Obst und Nssen"
4;"A0126";"A 01.26";"Anbau von lhaltigen Frchten"
5;"A01260";"A 01.26-0";"Anbau von lhaltigen Frchten"
4;"A0127";"A 01.27";"Anbau von Pflanzen zur Herstellung von Getrnken"
5;"A01270";"A 01.27-0";"Anbau von Pflanzen zur Herstellung von Getrnken"
4;"A0128";"A 01.28";"Anbau von Gewrzpflanzen, Pflanzen fr aromatische, narkotische und pharmazeutische Zwecke"
5;"A01280";"A 01.28-0";"Anbau von Gewrzpflanzen, Pflanzen fr aromatische, narkotische und pharmazeutische Zwecke"
4;"A0129";"A 01.29";"Anbau sonstiger mehrjhriger Pflanzen"
5;"A01290";"A 01.29-0";"Anbau sonstiger mehrjhriger Pflanzen"
3;"A013";"A 01.3";"Betrieb von Baumschulen, sowie Anbau von Pflanzen zu Vermehrungszwecken"
4;"A0130";"A 01.30";"Betrieb von Baumschulen, sowie Anbau von Pflanzen zu Vermehrungszwecken"
5;"A01300";"A 01.30-0";"Betrieb von Baumschulen, sowie Anbau von Pflanzen zu Vermehrungszwecken"
3;"A014";"A 01.4";"Tierhaltung"
4;"A0141";"A 01.41";"Haltung von Milchkhen"
5;"A01410";"A 01.41-0";"Haltung von Milchkhen"
4;"A0142";"A 01.42";"Haltung von anderen Rindern"
5;"A01420";"A 01.42-0";"Haltung von anderen Rindern"
4;"A0143";"A 01.43";"Haltung von Pferden und Eseln"
5;"A01430";"A 01.43-0";"Haltung von Pferden und Eseln"
4;"A0144";"A 01.44";"Haltung von Kamelen"
5;"A01440";"A 01.44-0";"Haltung von Kamelen"
4;"A0145";"A 01.45";"Haltung von Schafen und Ziegen"
5;"A01450";"A 01.45-0";"Haltung von Schafen und Ziegen"
4;"A0146";"A 01.46";"Haltung von Schweinen"
5;"A01460";"A 01.46-0";"Haltung von Schweinen"
4;"A0147";"A 01.47";"Haltung von Geflgel"
5;"A01471";"A 01.47-1";"Mstereien"
5;"A01472";"A 01.47-2";"Brtereien und Legeanstalten"
4;"A0149";"A 01.49";"Sonstige Tierhaltung"
5;"A01490";"A 01.49-0";"Sonstige Tierhaltung"
3;"A015";"A 01.5";"Gemischte Landwirtschaft"
4;"A0150";"A 01.50";"Gemischte Landwirtschaft"
5;"A01500";"A 01.50-0";"Gemischte Landwirtschaft"
3;"A016";"A 01.6";"Erbringung von landwirtschaftlichen Dienstleistungen"
4;"A0161";"A 01.61";"Erbringung von landwirtschaftlichen Dienstleistungen fr den Pflanzenbau"
5;"A01610";"A 01.61-0";"Erbringung von landwirtschaftlichen Dienstleistungen fr den Pflanzenbau"
4;"A0162";"A 01.62";"Erbringung von landwirtschaftlichen Dienstleistungen fr die Tierhaltung"
5;"A01620";"A 01.62-0";"Erbringung von landwirtschaftlichen Dienstleistungen fr die Tierhaltung"
4;"A0163";"A 01.63";"Nach der Ernte anfallende Ttigkeiten in der pflanzlichen Erzeugung"
5;"A01630";"A 01.63-0";"Nach der Ernte anfallende Ttigkeiten in der pflanzlichen Erzeugung"
4;"A0164";"A 01.64";"Saatgutaufbereitung"
5;"A01640";"A 01.64-0";"Saatgutaufbereitung"
3;"A017";"A 01.7";"Jagd, Fallenstellerei und damit verbundene Ttigkeiten"
4;"A0170";"A 01.70";"Jagd, Fallenstellerei und damit verbundene Ttigkeiten"
5;"A01700";"A 01.70-0";"Jagd, Fallenstellerei und damit verbundene Ttigkeiten"
2;"A02";"A 02";"Forstwirtschaft und Holzeinschlag"
3;"A021";"A 02.1";"Forstwirtschaft"
4;"A0210";"A 02.10";"Forstwirtschaft"
5;"A02100";"A 02.10-0";"Forstwirtschaft"
3;"A022";"A 02.2";"Holzeinschlag"
4;"A0220";"A 02.20";"Holzeinschlag"
5;"A02200";"A 02.20-0";"Holzeinschlag"
3;"A023";"A 02.3";"Sammeln von wildwachsenden Produkten (ohne Holz)"
4;"A0230";"A 02.30";"Sammeln von wildwachsenden Produkten (ohne Holz)"
5;"A02300";"A 02.30-0";"Sammeln von wildwachsenden Produkten (ohne Holz)"
3;"A024";"A 02.4";"Erbringung von Dienstleistungen fr Forstwirtschaft und Holzeinschlag"
4;"A0240";"A 02.40";"Erbringung von Dienstleistungen fr Forstwirtschaft und Holzeinschlag"
5;"A02400";"A 02.40-0";"Erbringung von Dienstleistungen fr Forstwirtschaft und Holzeinschlag"
2;"A03";"A 03";"Fischerei und Aquakultur"
3;"A031";"A 03.1";"Fischerei"
4;"A0311";"A 03.11";"Meeresfischerei"
5;"A03110";"A 03.11-0";"Meeresfischerei"
4;"A0312";"A 03.12";"Swasserfischerei"
5;"A03120";"A 03.12-0";"Swasserfischerei"
3;"A032";"A 03.2";"Aquakultur"
4;"A0321";"A 03.21";"Meeresaquakultur"
5;"A03210";"A 03.21-0";"Meeresaquakultur"
4;"A0322";"A 03.22";"Swasseraquakultur"
5;"A03220";"A 03.22-0";"Swasseraquakultur"
1;"B";"B";"BERGBAU UND GEWINNUNG VON STEINEN UND ERDEN"
2;"B05";"B 05";"Kohlenbergbau"
3;"B051";"B 05.1";"Steinkohlenbergbau"
4;"B0510";"B 05.10";"Steinkohlenbergbau"
5;"B05100";"B 05.10-0";"Steinkohlenbergbau"
3;"B052";"B 05.2";"Braunkohlenbergbau"
4;"B0520";"B 05.20";"Braunkohlenbergbau"
5;"B05200";"B 05.20-0";"Braunkohlenbergbau"
2;"B06";"B 06";"Gewinnung von Erdl und Erdgas"
3;"B061";"B 06.1";"Gewinnung von Erdl"
4;"B0610";"B 06.10";"Gewinnung von Erdl"
5;"B06100";"B 06.10-0";"Gewinnung von Erdl"
3;"B062";"B 06.2";"Gewinnung von Erdgas"
4;"B0620";"B 06.20";"Gewinnung von Erdgas"
5;"B06200";"B 06.20-0";"Gewinnung von Erdgas"
2;"B07";"B 07";"Erzbergbau"
3;"B071";"B 07.1";"Eisenerzbergbau"
4;"B0710";"B 07.10";"Eisenerzbergbau"
5;"B07100";"B 07.10-0";"Eisenerzbergbau"
3;"B072";"B 07.2";"NE-Metallerzbergbau"
4;"B0721";"B 07.21";"Bergbau auf Uran- und Thoriumerze"
5;"B07210";"B 07.21-0";"Bergbau auf Uran- und Thoriumerze"
4;"B0729";"B 07.29";"Sonstiger NE-Metallerzbergbau"
5;"B07290";"B 07.29-0";"Sonstiger NE-Metallerzbergbau"
2;"B08";"B 08";"Gewinnung von Steinen und Erden, sonstiger Bergbau"
3;"B081";"B 08.1";"Gewinnung von Natursteinen, Kies, Sand, Ton und Kaolin"
4;"B0811";"B 08.11";"Gewinnung von Naturwerksteinen und Natursteinen, Kalk- und Gipsstein, Kreide und Schiefer"
5;"B08110";"B 08.11-0";"Gewinnung von Naturwerksteinen und Natursteinen, Kalk- und Gipsstein, Kreide und Schiefer"
4;"B0812";"B 08.12";"Gewinnung von Kies, Sand, Ton und Kaolin"
5;"B08120";"B 08.12-0";"Gewinnung von Kies, Sand, Ton und Kaolin"
3;"B089";"B 08.9";"Sonstiger Bergbau; Gewinnung von Steinen und Erden a.n.g."
4;"B0891";"B 08.91";"Bergbau auf chemische und Dngemittelminerale"
5;"B08910";"B 08.91-0";"Bergbau auf chemische und Dngemittelminerale"
4;"B0892";"B 08.92";"Torfgewinnung"
5;"B08920";"B 08.92-0";"Torfgewinnung"
4;"B0893";"B 08.93";"Gewinnung von Salz"
5;"B08930";"B 08.93-0";"Gewinnung von Salz"
4;"B0899";"B 08.99";"Gewinnung von Steinen und Erden a.n.g."
5;"B08990";"B 08.99-0";"Gewinnung von Steinen und Erden a.n.g."
2;"B09";"B 09";"Erbringung von Dienstleistungen fr den Bergbau und fr die Gewinnung von Steinen und Erden"
3;"B091";"B 09.1";"Erbringung von Dienstleistungen fr die Gewinnung von Erdl und Erdgas"
4;"B0910";"B 09.10";"Erbringung von Dienstleistungen fr die Gewinnung von Erdl und Erdgas"
5;"B09100";"B 09.10-0";"Erbringung von Dienstleistungen fr die Gewinnung von Erdl und Erdgas"
3;"B099";"B 09.9";"Erbringung von Dienstleistungen fr den sonstigen Bergbau und die Gewinnung von Steinen und Erden"
4;"B0990";"B 09.90";"Erbringung von Dienstleistungen fr den sonstigen Bergbau und die Gewinnung von Steinen und Erden"
5;"B09900";"B 09.90-0";"Erbringung von Dienstleistungen fr den sonstigen Bergbau und die Gewinnung von Steinen und Erden"
1;"C";"C";"HERSTELLUNG VON WAREN"
2;"C10";"C 10";"Herstellung von Nahrungs- und Futtermitteln"
3;"C101";"C 10.1";"Schlachten und Fleischverarbeitung"
4;"C1011";"C 10.11";"Schlachten (ohne Schlachten von Geflgel)"
5;"C10110";"C 10.11-0";"Schlachten (ohne Schlachten von Geflgel)"
4;"C1012";"C 10.12";"Schlachten von Geflgel"
5;"C10120";"C 10.12-0";"Schlachten von Geflgel"
4;"C1013";"C 10.13";"Fleischverarbeitung"
5;"C10130";"C 10.13-0";"Fleischverarbeitung"
3;"C102";"C 10.2";"Fischverarbeitung"
4;"C1020";"C 10.20";"Fischverarbeitung"
5;"C10200";"C 10.20-0";"Fischverarbeitung"
3;"C103";"C 10.3";"Obst- und Gemseverarbeitung"
4;"C1031";"C 10.31";"Kartoffelverarbeitung"
5;"C10310";"C 10.31-0";"Kartoffelverarbeitung"
4;"C1032";"C 10.32";"Herstellung von Frucht- und Gemsesften"
5;"C10320";"C 10.32-0";"Herstellung von Frucht- und Gemsesften"
4;"C1039";"C 10.39";"Sonstige Verarbeitung von Obst und Gemse"
5;"C10390";"C 10.39-0";"Sonstige Verarbeitung von Obst und Gemse"
3;"C104";"C 10.4";"Herstellung von pflanzlichen und tierischen len und Fetten"
4;"C1041";"C 10.41";"Herstellung von len und Fetten (ohne Margarine u. . Nahrungsfette)"
5;"C10410";"C 10.41-0";"Herstellung von len und Fetten (ohne Margarine u. . Nahrungsfette)"
4;"C1042";"C 10.42";"Herstellung von Margarine u. . Nahrungsfetten"
5;"C10420";"C 10.42-0";"Herstellung von Margarine u. . Nahrungsfetten"
3;"C105";"C 10.5";"Milchverarbeitung"
4;"C1051";"C 10.51";"Milchverarbeitung (ohne Herstellung von Speiseeis)"
5;"C10510";"C 10.51-0";"Milchverarbeitung (ohne Herstellung von Speiseeis)"
4;"C1052";"C 10.52";"Herstellung von Speiseeis"
5;"C10520";"C 10.52-0";"Herstellung von Speiseeis"
3;"C106";"C 10.6";"Mahl- und Schlmhlen, Herstellung von Strke und Strkeerzeugnissen"
4;"C1061";"C 10.61";"Mahl- und Schlmhlen"
5;"C10610";"C 10.61-0";"Mahl- und Schlmhlen"
4;"C1062";"C 10.62";"Herstellung von Strke und Strkeerzeugnissen"
5;"C10620";"C 10.62-0";"Herstellung von Strke und Strkeerzeugnissen"
3;"C107";"C 10.7";"Herstellung von Back- und Teigwaren"
4;"C1071";"C 10.71";"Herstellung von Backwaren (ohne Dauerbackwaren)"
5;"C10711";"C 10.71-1";"Herstellung von Schwarz- und Weibackwaren"
5;"C10712";"C 10.71-2";"Herstellung von Zuckerbcker- und Konditorwaren"
4;"C1072";"C 10.72";"Herstellung von Dauerbackwaren"
5;"C10720";"C 10.72-0";"Herstellung von Dauerbackwaren"
4;"C1073";"C 10.73";"Herstellung von Teigwaren"
5;"C10730";"C 10.73-0";"Herstellung von Teigwaren"
3;"C108";"C 10.8";"Herstellung von sonstigen Nahrungsmitteln"
4;"C1081";"C 10.81";"Herstellung von Zucker"
5;"C10810";"C 10.81-0";"Herstellung von Zucker"
4;"C1082";"C 10.82";"Herstellung von Swaren (ohne Dauerbackwaren)"
5;"C10820";"C 10.82-0";"Herstellung von Swaren (ohne Dauerbackwaren)"
4;"C1083";"C 10.83";"Verarbeitung von Kaffee und Tee, Herstellung von Kaffee-Ersatz"
5;"C10830";"C 10.83-0";"Verarbeitung von Kaffee und Tee, Herstellung von Kaffee-Ersatz"
4;"C1084";"C 10.84";"Herstellung von Wrzmitteln und Soen"
5;"C10840";"C 10.84-0";"Herstellung von Wrzmitteln und Soen"
4;"C1085";"C 10.85";"Herstellung von Fertiggerichten"
5;"C10850";"C 10.85-0";"Herstellung von Fertiggerichten"
4;"C1086";"C 10.86";"Herstellung von homogenisierten und ditetischen Nahrungsmitteln"
5;"C10860";"C 10.86-0";"Herstellung von homogenisierten und ditetischen Nahrungsmitteln"
4;"C1089";"C 10.89";"Herstellung von sonstigen Nahrungsmitteln a.n.g."
5;"C10890";"C 10.89-0";"Herstellung von sonstigen Nahrungsmitteln a.n.g."
3;"C109";"C 10.9";"Herstellung von Futtermitteln"
4;"C1091";"C 10.91";"Herstellung von Futtermitteln fr Nutztiere"
5;"C10910";"C 10.91-0";"Herstellung von Futtermitteln fr Nutztiere"
4;"C1092";"C 10.92";"Herstellung von Futtermitteln fr sonstige Tiere"
5;"C10920";"C 10.92-0";"Herstellung von Futtermitteln fr sonstige Tiere"
2;"C11";"C 11";"Getrnkeherstellung"
3;"C110";"C 11.0";"Getrnkeherstellung"
4;"C1101";"C 11.01";"Herstellung von Spirituosen"
5;"C11010";"C 11.01-0";"Herstellung von Spirituosen"
4;"C1102";"C 11.02";"Herstellung von Traubenwein"
5;"C11020";"C 11.02-0";"Herstellung von Traubenwein"
4;"C1103";"C 11.03";"Herstellung von Apfelwein und anderen Fruchtweinen"
5;"C11030";"C 11.03-0";"Herstellung von Apfelwein und anderen Fruchtweinen"
4;"C1104";"C 11.04";"Herstellung von Wermutwein und sonstigen aromatisierten Weinen"
5;"C11040";"C 11.04-0";"Herstellung von Wermutwein und sonstigen aromatisierten Weinen"
4;"C1105";"C 11.05";"Herstellung von Bier"
5;"C11050";"C 11.05-0";"Herstellung von Bier"
4;"C1106";"C 11.06";"Herstellung von Malz"
5;"C11060";"C 11.06-0";"Herstellung von Malz"
4;"C1107";"C 11.07";"Herstellung von Erfrischungsgetrnken; Gewinnung natrlicher Mineralwsser"
5;"C11070";"C 11.07-0";"Herstellung von Erfrischungsgetrnken; Gewinnung natrlicher Mineralwsser"
2;"C12";"C 12";"Tabakverarbeitung"
3;"C120";"C 12.0";"Tabakverarbeitung"
4;"C1200";"C 12.00";"Tabakverarbeitung"
5;"C12000";"C 12.00-0";"Tabakverarbeitung"
2;"C13";"C 13";"Herstellung von Textilien"
3;"C131";"C 13.1";"Spinnstoffaufbereitung und Spinnerei"
4;"C1310";"C 13.10";"Spinnstoffaufbereitung und Spinnerei"
5;"C13100";"C 13.10-0";"Spinnstoffaufbereitung und Spinnerei"
3;"C132";"C 13.2";"Weberei"
4;"C1320";"C 13.20";"Weberei"
5;"C13200";"C 13.20-0";"Weberei"
3;"C133";"C 13.3";"Veredlung von Textilien und Bekleidung"
4;"C1330";"C 13.30";"Veredlung von Textilien und Bekleidung"
5;"C13300";"C 13.30-0";"Veredlung von Textilien und Bekleidung"
3;"C139";"C 13.9";"Herstellung von sonstigen Textilwaren"
4;"C1391";"C 13.91";"Herstellung von gewirktem und gestricktem Stoff"
5;"C13910";"C 13.91-0";"Herstellung von gewirktem und gestricktem Stoff"
4;"C1392";"C 13.92";"Herstellung von konfektionierten Textilwaren (ohne Bekleidung)"
5;"C13920";"C 13.92-0";"Herstellung von konfektionierten Textilwaren (ohne Bekleidung)"
4;"C1393";"C 13.93";"Herstellung von Teppichen"
5;"C13930";"C 13.93-0";"Herstellung von Teppichen"
4;"C1394";"C 13.94";"Herstellung von Seilerwaren"
5;"C13940";"C 13.94-0";"Herstellung von Seilerwaren"
4;"C1395";"C 13.95";"Herstellung von Vliesstoff und Erzeugnissen daraus (ohne Bekleidung)"
5;"C13950";"C 13.95-0";"Herstellung von Vliesstoff und Erzeugnissen daraus (ohne Bekleidung)"
4;"C1396";"C 13.96";"Herstellung von technischen Textilien"
5;"C13960";"C 13.96-0";"Herstellung von technischen Textilien"
4;"C1399";"C 13.99";"Herstellung von sonstigen Textilwaren a.n.g."
5;"C13991";"C 13.99-1";"Herstellung von Stickereien"
5;"C13999";"C 13.99-9";"Herstellung von sonstigen Textilwaren a.n.g. (ohne Herstellung von Stickereien)"
2;"C14";"C 14";"Herstellung von Bekleidung"
3;"C141";"C 14.1";"Herstellung von Bekleidung (ohne Pelzbekleidung)"
4;"C1411";"C 14.11";"Herstellung von Lederbekleidung"
5;"C14110";"C 14.11-0";"Herstellung von Lederbekleidung"
4;"C1412";"C 14.12";"Herstellung von Arbeits- und Berufsbekleidung"
5;"C14120";"C 14.12-0";"Herstellung von Arbeits- und Berufsbekleidung"
4;"C1413";"C 14.13";"Herstellung von sonstiger Oberbekleidung"
5;"C14130";"C 14.13-0";"Herstellung von sonstiger Oberbekleidung"
4;"C1414";"C 14.14";"Herstellung von Wsche"
5;"C14140";"C 14.14-0";"Herstellung von Wsche"
4;"C1419";"C 14.19";"Herstellung von sonstiger Bekleidung und Bekleidungszubehr a.n.g."
5;"C14190";"C 14.19-0";"Herstellung von sonstiger Bekleidung und Bekleidungszubehr a.n.g."
3;"C142";"C 14.2";"Herstellung von Pelzwaren"
4;"C1420";"C 14.20";"Herstellung von Pelzwaren"
5;"C14200";"C 14.20-0";"Herstellung von Pelzwaren"
3;"C143";"C 14.3";"Herstellung von Bekleidung aus gewirktem und gestricktem Stoff"
4;"C1431";"C 14.31";"Herstellung von Strumpfwaren"
5;"C14310";"C 14.31-0";"Herstellung von Strumpfwaren"
4;"C1439";"C 14.39";"Herstellung von sonstiger Bekleidung aus gewirktem und gestricktem Stoff"
5;"C14390";"C 14.39-0";"Herstellung von sonstiger Bekleidung aus gewirktem und gestricktem Stoff"
2;"C15";"C 15";"Herstellung von Leder, Lederwaren und Schuhen"
3;"C151";"C 15.1";"Herstellung von Leder und Lederwaren (ohne Herstellung von Lederbekleidung)"
4;"C1511";"C 15.11";"Herstellung von Leder und Lederfaserstoff; Zurichtung und Frben von Fellen"
5;"C15110";"C 15.11-0";"Herstellung von Leder und Lederfaserstoff; Zurichtung und Frben von Fellen"
4;"C1512";"C 15.12";"Lederverarbeitung (ohne Herstellung von Lederbekleidung)"
5;"C15120";"C 15.12-0";"Lederverarbeitung (ohne Herstellung von Lederbekleidung)"
3;"C152";"C 15.2";"Herstellung von Schuhen"
4;"C1520";"C 15.20";"Herstellung von Schuhen"
5;"C15200";"C 15.20-0";"Herstellung von Schuhen"
2;"C16";"C 16";"Herstellung von Holz-, Flecht-, Korb- und Korkwaren (ohne Mbel)"
3;"C161";"C 16.1";"Sge-, Hobel- und Holzimprgnierwerke"
4;"C1610";"C 16.10";"Sge-, Hobel- und Holzimprgnierwerke"
5;"C16100";"C 16.10-0";"Sge-, Hobel- und Holzimprgnierwerke"
3;"C162";"C 16.2";"Herstellung von sonstigen Holz-, Kork-, Flecht- und Korbwaren (ohne Mbel)"
4;"C1621";"C 16.21";"Herstellung von Furnier-, Sperrholz-, Holzfaser- und Holzspanplatten"
5;"C16210";"C 16.21-0";"Herstellung von Furnier-, Sperrholz-, Holzfaser- und Holzspanplatten"
4;"C1622";"C 16.22";"Herstellung von Parketttafeln"
5;"C16220";"C 16.22-0";"Herstellung von Parketttafeln"
4;"C1623";"C 16.23";"Herstellung von sonstigen Konstruktionsteilen, Fertigbauteilen, Ausbauelementen und Fertigteilbauten aus Holz"
5;"C16231";"C 16.23-1";"Fertighaus- und Hallenerzeugung"
5;"C16239";"C 16.23-9";"Herstellung von sonstigen Konstruktionsteilen und Ausbauelementen aus Holz a.n.g."
4;"C1624";"C 16.24";"Herstellung von Verpackungsmitteln, Lagerbehltern und Ladungstrgern aus Holz"
5;"C16240";"C 16.24-0";"Herstellung von Verpackungsmitteln, Lagerbehltern und Ladungstrgern aus Holz"
4;"C1629";"C 16.29";"Herstellung von Holzwaren a.n.g., Kork-, Flecht- und Korbwaren (ohne Mbel)"
5;"C16290";"C 16.29-0";"Herstellung von Holzwaren a.n.g., Kork-, Flecht- und Korbwaren (ohne Mbel)"
2;"C17";"C 17";"Herstellung von Papier, Pappe und Waren daraus"
3;"C171";"C 17.1";"Herstellung von Holz- und Zellstoff, Papier, Karton und Pappe"
4;"C1711";"C 17.11";"Herstellung von Holz- und Zellstoff"
5;"C17110";"C 17.11-0";"Herstellung von Holz- und Zellstoff"
4;"C1712";"C 17.12";"Herstellung von Papier, Karton und Pappe"
5;"C17120";"C 17.12-0";"Herstellung von Papier, Karton und Pappe"
3;"C172";"C 17.2";"Herstellung von Waren aus Papier, Karton und Pappe"
4;"C1721";"C 17.21";"Herstellung von Wellpapier und -pappe sowie von Verpackungsmitteln aus Papier, Karton und Pappe"
5;"C17210";"C 17.21-0";"Herstellung von Wellpapier und -pappe sowie von Verpackungsmitteln aus Papier, Karton und Pappe"
4;"C1722";"C 17.22";"Herstellung von Haushalts-, Hygiene- und Toilettenartikeln aus Zellstoff, Papier und Pappe"
5;"C17220";"C 17.22-0";"Herstellung von Haushalts-, Hygiene- und Toilettenartikeln aus Zellstoff, Papier und Pappe"
4;"C1723";"C 17.23";"Herstellung von Schreibwaren und Brobedarf aus Papier, Karton und Pappe"
5;"C17230";"C 17.23-0";"Herstellung von Schreibwaren und Brobedarf aus Papier, Karton und Pappe"
4;"C1724";"C 17.24";"Herstellung von Tapeten"
5;"C17240";"C 17.24-0";"Herstellung von Tapeten"
4;"C1729";"C 17.29";"Herstellung von sonstigen Waren aus Papier, Karton und Pappe"
5;"C17290";"C 17.29-0";"Herstellung von sonstigen Waren aus Papier, Karton und Pappe"
2;"C18";"C 18";"Herstellung von Druckerzeugnissen; Vervielfltigung von bespielten Ton-, Bild- und Datentrgern"
3;"C181";"C 18.1";"Herstellung von Druckerzeugnissen"
4;"C1811";"C 18.11";"Drucken von Zeitungen"
5;"C18110";"C 18.11-0";"Drucken von Zeitungen"
4;"C1812";"C 18.12";"Drucken a.n.g."
5;"C18120";"C 18.12-0";"Drucken a.n.g."
4;"C1813";"C 18.13";"Druck- und Medienvorstufe"
5;"C18130";"C 18.13-0";"Druck- und Medienvorstufe"
4;"C1814";"C 18.14";"Binden von Druckerzeugnissen und damit verbundene Dienstleistungen"
5;"C18140";"C 18.14-0";"Binden von Druckerzeugnissen und damit verbundene Dienstleistungen"
3;"C182";"C 18.2";"Vervielfltigung von bespielten Ton-, Bild- und Datentrgern"
4;"C1820";"C 18.20";"Vervielfltigung von bespielten Ton-, Bild- und Datentrgern"
5;"C18200";"C 18.20-0";"Vervielfltigung von bespielten Ton-, Bild- und Datentrgern"
2;"C19";"C 19";"Kokerei und Minerallverarbeitung"
3;"C191";"C 19.1";"Kokerei"
4;"C1910";"C 19.10";"Kokerei"
5;"C19100";"C 19.10-0";"Kokerei"
3;"C192";"C 19.2";"Minerallverarbeitung"
4;"C1920";"C 19.20";"Minerallverarbeitung"
5;"C19200";"C 19.20-0";"Minerallverarbeitung"
2;"C20";"C 20";"Herstellung von chemischen Erzeugnissen"
3;"C201";"C 20.1";"Herstellung von chemischen Grundstoffen, Dngemitteln und Stickstoffverbindungen, Kunststoffen in Primrformen und synthetischem Kautschuk in Primrformen"
4;"C2011";"C 20.11";"Herstellung von Industriegasen"
5;"C20110";"C 20.11-0";"Herstellung von Industriegasen"
4;"C2012";"C 20.12";"Herstellung von Farbstoffen und Pigmenten"
5;"C20120";"C 20.12-0";"Herstellung von Farbstoffen und Pigmenten"
4;"C2013";"C 20.13";"Herstellung von sonstigen anorganischen Grundstoffen und Chemikalien"
5;"C20130";"C 20.13-0";"Herstellung von sonstigen anorganischen Grundstoffen und Chemikalien"
4;"C2014";"C 20.14";"Herstellung von sonstigen organischen Grundstoffen und Chemikalien"
5;"C20140";"C 20.14-0";"Herstellung von sonstigen organischen Grundstoffen und Chemikalien"
4;"C2015";"C 20.15";"Herstellung von Dngemitteln und Stickstoffverbindungen"
5;"C20150";"C 20.15-0";"Herstellung von Dngemitteln und Stickstoffverbindungen"
4;"C2016";"C 20.16";"Herstellung von Kunststoffen in Primrformen"
5;"C20160";"C 20.16-0";"Herstellung von Kunststoffen in Primrformen"
4;"C2017";"C 20.17";"Herstellung von synthetischem Kautschuk in Primrformen"
5;"C20170";"C 20.17-0";"Herstellung von synthetischem Kautschuk in Primrformen"
3;"C202";"C 20.2";"Herstellung von Schdlingsbekmpfungs-, Pflanzenschutz- und Desinfektionsmitteln"
4;"C2020";"C 20.20";"Herstellung von Schdlingsbekmpfungs-, Pflanzenschutz- und Desinfektionsmitteln"
5;"C20200";"C 20.20-0";"Herstellung von Schdlingsbekmpfungs-, Pflanzenschutz- und Desinfektionsmitteln"
3;"C203";"C 20.3";"Herstellung von Anstrichmitteln, Druckfarben und Kitten"
4;"C2030";"C 20.30";"Herstellung von Anstrichmitteln, Druckfarben und Kitten"
5;"C20300";"C 20.30-0";"Herstellung von Anstrichmitteln, Druckfarben und Kitten"
3;"C204";"C 20.4";"Herstellung von Seifen, Wasch-, Reinigungs- und Krperpflegemitteln sowie von Duftstoffen"
4;"C2041";"C 20.41";"Herstellung von Seifen, Wasch-, Reinigungs- und Poliermitteln"
5;"C20410";"C 20.41-0";"Herstellung von Seifen, Wasch-, Reinigungs- und Poliermitteln"
4;"C2042";"C 20.42";"Herstellung von Krperpflegemitteln und Duftstoffen"
5;"C20420";"C 20.42-0";"Herstellung von Krperpflegemitteln und Duftstoffen"
3;"C205";"C 20.5";"Herstellung von sonstigen chemischen Erzeugnissen"
4;"C2051";"C 20.51";"Herstellung von pyrotechnischen Erzeugnissen"
5;"C20510";"C 20.51-0";"Herstellung von pyrotechnischen Erzeugnissen"
4;"C2052";"C 20.52";"Herstellung von Klebstoffen"
5;"C20520";"C 20.52-0";"Herstellung von Klebstoffen"
4;"C2053";"C 20.53";"Herstellung von etherischen len"
5;"C20530";"C 20.53-0";"Herstellung von etherischen len"
4;"C2059";"C 20.59";"Herstellung von sonstigen chemischen Erzeugnissen a.n.g."
5;"C20590";"C 20.59-0";"Herstellung von sonstigen chemischen Erzeugnissen a.n.g."
3;"C206";"C 20.6";"Herstellung von Chemiefasern"
4;"C2060";"C 20.60";"Herstellung von Chemiefasern"
5;"C20600";"C 20.60-0";"Herstellung von Chemiefasern"
2;"C21";"C 21";"Herstellung von pharmazeutischen Erzeugnissen"
3;"C211";"C 21.1";"Herstellung von pharmazeutischen Grundstoffen"
4;"C2110";"C 21.10";"Herstellung von pharmazeutischen Grundstoffen"
5;"C21100";"C 21.10-0";"Herstellung von pharmazeutischen Grundstoffen"
3;"C212";"C 21.2";"Herstellung von pharmazeutischen Spezialitten und sonstigen pharmazeutischen Erzeugnissen"
4;"C2120";"C 21.20";"Herstellung von pharmazeutischen Spezialitten und sonstigen pharmazeutischen Erzeugnissen"
5;"C21200";"C 21.20-0";"Herstellung von pharmazeutischen Spezialitten und sonstigen pharmazeutischen Erzeugnissen"
2;"C22";"C 22";"Herstellung von Gummi- und Kunststoffwaren"
3;"C221";"C 22.1";"Herstellung von Gummiwaren"
4;"C2211";"C 22.11";"Herstellung und Runderneuerung von Bereifungen"
5;"C22110";"C 22.11-0";"Herstellung und Runderneuerung von Bereifungen"
4;"C2219";"C 22.19";"Herstellung von sonstigen Gummiwaren"
5;"C22190";"C 22.19-0";"Herstellung von sonstigen Gummiwaren"
3;"C222";"C 22.2";"Herstellung von Kunststoffwaren"
4;"C2221";"C 22.21";"Herstellung von Platten, Folien, Schluchen und Profilen aus Kunststoffen"
5;"C22210";"C 22.21-0";"Herstellung von Platten, Folien, Schluchen und Profilen aus Kunststoffen"
4;"C2222";"C 22.22";"Herstellung von Verpackungsmitteln aus Kunststoffen"
5;"C22220";"C 22.22-0";"Herstellung von Verpackungsmitteln aus Kunststoffen"
4;"C2223";"C 22.23";"Herstellung von Baubedarfsartikeln aus Kunststoffen"
5;"C22231";"C 22.23-1";"Herstellung von Tren, Fenstern, Rahmen, Rolllden und Jalousien aus Kunststoffen"
5;"C22239";"C 22.23-9";"Herstellung von sonstigen Baubedarfsartikeln aus Kunststoffen"
4;"C2229";"C 22.29";"Herstellung von sonstigen Kunststoffwaren"
5;"C22291";"C 22.29-1";"Herstellung von technischen Kunststoffteilen"
5;"C22299";"C 22.29-9";"Herstellung von sonstigen Kunststoffwaren a.n.g."
2;"C23";"C 23";"Herstellung von Glas und Glaswaren, Keramik, Verarbeitung von Steinen und Erden"
3;"C231";"C 23.1";"Herstellung von Glas und Glaswaren"
4;"C2311";"C 23.11";"Herstellung von Flachglas"
5;"C23110";"C 23.11-0";"Herstellung von Flachglas"
4;"C2312";"C 23.12";"Veredlung und Bearbeitung von Flachglas"
5;"C23120";"C 23.12-0";"Veredlung und Bearbeitung von Flachglas"
4;"C2313";"C 23.13";"Herstellung von Hohlglas"
5;"C23130";"C 23.13-0";"Herstellung von Hohlglas"
4;"C2314";"C 23.14";"Herstellung von Glasfasern und Waren daraus"
5;"C23140";"C 23.14-0";"Herstellung von Glasfasern und Waren daraus"
4;"C2319";"C 23.19";"Herstellung, Veredlung und Bearbeitung von sonstigem Glas einschlielich technischen Glaswaren"
5;"C23190";"C 23.19-0";"Herstellung, Veredlung und Bearbeitung von sonstigem Glas einschlielich technischen Glaswaren"
3;"C232";"C 23.2";"Herstellung von feuerfesten keramischen Werkstoffen und Waren"
4;"C2320";"C 23.20";"Herstellung von feuerfesten keramischen Werkstoffen und Waren"
5;"C23200";"C 23.20-0";"Herstellung von feuerfesten keramischen Werkstoffen und Waren"
3;"C233";"C 23.3";"Herstellung von keramischen Baumaterialien"
4;"C2331";"C 23.31";"Herstellung von keramischen Wand- und Bodenfliesen und -platten"
5;"C23310";"C 23.31-0";"Herstellung von keramischen Wand- und Bodenfliesen und -platten"
4;"C2332";"C 23.32";"Herstellung von Ziegeln und sonstiger Baukeramik"
5;"C23320";"C 23.32-0";"Herstellung von Ziegeln und sonstiger Baukeramik"
3;"C234";"C 23.4";"Herstellung von sonstigen Porzellan- und keramischen Erzeugnissen"
4;"C2341";"C 23.41";"Herstellung von keramischen Haushaltswaren und Ziergegenstnden"
5;"C23410";"C 23.41-0";"Herstellung von keramischen Haushaltswaren und Ziergegenstnden"
4;"C2342";"C 23.42";"Herstellung von Sanitrkeramik"
5;"C23420";"C 23.42-0";"Herstellung von Sanitrkeramik"
4;"C2343";"C 23.43";"Herstellung von Isolatoren und Isolierteilen aus Keramik"
5;"C23430";"C 23.43-0";"Herstellung von Isolatoren und Isolierteilen aus Keramik"
4;"C2344";"C 23.44";"Herstellung von keramischen Erzeugnissen fr sonstige technische Zwecke"
5;"C23440";"C 23.44-0";"Herstellung von keramischen Erzeugnissen fr sonstige technische Zwecke"
4;"C2349";"C 23.49";"Herstellung von sonstigen keramischen Erzeugnissen"
5;"C23490";"C 23.49-0";"Herstellung von sonstigen keramischen Erzeugnissen"
3;"C235";"C 23.5";"Herstellung von Zement, Kalk und gebranntem Gips"
4;"C2351";"C 23.51";"Herstellung von Zement"
5;"C23510";"C 23.51-0";"Herstellung von Zement"
4;"C2352";"C 23.52";"Herstellung von Kalk und gebranntem Gips"
5;"C23520";"C 23.52-0";"Herstellung von Kalk und gebranntem Gips"
3;"C236";"C 23.6";"Herstellung von Erzeugnissen aus Beton, Zement und Gips"
4;"C2361";"C 23.61";"Herstellung von Erzeugnissen aus Beton, Zement und Kalksandstein fr den Bau"
5;"C23610";"C 23.61-0";"Herstellung von Erzeugnissen aus Beton, Zement und Kalksandstein fr den Bau"
4;"C2362";"C 23.62";"Herstellung von Gipserzeugnissen fr den Bau"
5;"C23620";"C 23.62-0";"Herstellung von Gipserzeugnissen fr den Bau"
4;"C2363";"C 23.63";"Herstellung von Frischbeton (Transportbeton)"
5;"C23630";"C 23.63-0";"Herstellung von Frischbeton (Transportbeton)"
4;"C2364";"C 23.64";"Herstellung von Mrtel und anderem Beton (Trockenbeton)"
5;"C23640";"C 23.64-0";"Herstellung von Mrtel und anderem Beton (Trockenbeton)"
4;"C2365";"C 23.65";"Herstellung von Faserzementwaren"
5;"C23650";"C 23.65-0";"Herstellung von Faserzementwaren"
4;"C2369";"C 23.69";"Herstellung von sonstigen Erzeugnissen aus Beton, Zement und Gips a.n.g."
5;"C23690";"C 23.69-0";"Herstellung von sonstigen Erzeugnissen aus Beton, Zement und Gips a.n.g."
3;"C237";"C 23.7";"Be- und Verarbeitung von Naturwerksteinen und Natursteinen a.n.g."
4;"C2370";"C 23.70";"Be- und Verarbeitung von Naturwerksteinen und Natursteinen a.n.g."
5;"C23700";"C 23.70-0";"Be- und Verarbeitung von Naturwerksteinen und Natursteinen a.n.g."
3;"C239";"C 23.9";"Herstellung von Schleifkrpern und Schleifmitteln auf Unterlage sowie sonstigen Erzeugnissen aus nicht metallischen Mineralien a.n.g."
4;"C2391";"C 23.91";"Herstellung von Schleifkrpern und Schleifmitteln auf Unterlage"
5;"C23910";"C 23.91-0";"Herstellung von Schleifkrpern und Schleifmitteln auf Unterlage"
4;"C2399";"C 23.99";"Herstellung von sonstigen Erzeugnissen aus nicht metallischen Mineralien a.n.g."
5;"C23990";"C 23.99-0";"Herstellung von sonstigen Erzeugnissen aus nicht metallischen Mineralien a.n.g."
2;"C24";"C 24";"Metallerzeugung und -bearbeitung"
3;"C241";"C 24.1";"Erzeugung von Roheisen, Stahl und Ferrolegierungen"
4;"C2410";"C 24.10";"Erzeugung von Roheisen, Stahl und Ferrolegierungen"
5;"C24100";"C 24.10-0";"Erzeugung von Roheisen, Stahl und Ferrolegierungen"
3;"C242";"C 24.2";"Herstellung von Stahlrohren, Rohrform-, Rohrverschluss- und Rohrverbindungsstcken aus Stahl"
4;"C2420";"C 24.20";"Herstellung von Stahlrohren, Rohrform-, Rohrverschluss- und Rohrverbindungsstcken aus Stahl"
5;"C24200";"C 24.20-0";"Herstellung von Stahlrohren, Rohrform-, Rohrverschluss- und Rohrverbindungsstcken aus Stahl"
3;"C243";"C 24.3";"Sonstige erste Bearbeitung von Eisen und Stahl"
4;"C2431";"C 24.31";"Herstellung von Blankstahl"
5;"C24310";"C 24.31-0";"Herstellung von Blankstahl"
4;"C2432";"C 24.32";"Herstellung von Kaltband mit einer Breite von weniger als 600 mm"
5;"C24320";"C 24.32-0";"Herstellung von Kaltband mit einer Breite von weniger als 600 mm"
4;"C2433";"C 24.33";"Herstellung von Kaltprofilen"
5;"C24330";"C 24.33-0";"Herstellung von Kaltprofilen"
4;"C2434";"C 24.34";"Herstellung von kaltgezogenem Draht"
5;"C24340";"C 24.34-0";"Herstellung von kaltgezogenem Draht"
3;"C244";"C 24.4";"Erzeugung und erste Bearbeitung von NE-Metallen"
4;"C2441";"C 24.41";"Erzeugung und erste Bearbeitung von Edelmetallen"
5;"C24410";"C 24.41-0";"Erzeugung und erste Bearbeitung von Edelmetallen"
4;"C2442";"C 24.42";"Erzeugung und erste Bearbeitung von Aluminium"
5;"C24420";"C 24.42-0";"Erzeugung und erste Bearbeitung von Aluminium"
4;"C2443";"C 24.43";"Erzeugung und erste Bearbeitung von Blei, Zink und Zinn"
5;"C24430";"C 24.43-0";"Erzeugung und erste Bearbeitung von Blei, Zink und Zinn"
4;"C2444";"C 24.44";"Erzeugung und erste Bearbeitung von Kupfer"
5;"C24440";"C 24.44-0";"Erzeugung und erste Bearbeitung von Kupfer"
4;"C2445";"C 24.45";"Erzeugung und erste Bearbeitung von sonstigen NE-Metallen"
5;"C24450";"C 24.45-0";"Erzeugung und erste Bearbeitung von sonstigen NE-Metallen"
4;"C2446";"C 24.46";"Aufbereitung von Kernbrennstoffen"
5;"C24460";"C 24.46-0";"Aufbereitung von Kernbrennstoffen"
3;"C245";"C 24.5";"Gieereien"
4;"C2451";"C 24.51";"Eisengieereien"
5;"C24510";"C 24.51-0";"Eisengieereien"
4;"C2452";"C 24.52";"Stahlgieereien"
5;"C24520";"C 24.52-0";"Stahlgieereien"
4;"C2453";"C 24.53";"Leichtmetallgieereien"
5;"C24530";"C 24.53-0";"Leichtmetallgieereien"
4;"C2454";"C 24.54";"Buntmetallgieereien"
5;"C24540";"C 24.54-0";"Buntmetallgieereien"
2;"C25";"C 25";"Herstellung von Metallerzeugnissen"
3;"C251";"C 25.1";"Stahl- und Leichtmetallbau"
4;"C2511";"C 25.11";"Herstellung von Metallkonstruktionen"
5;"C25110";"C 25.11-0";"Herstellung von Metallkonstruktionen"
4;"C2512";"C 25.12";"Herstellung von Ausbauelementen aus Metall"
5;"C25121";"C 25.12-1";"Herstellung von Ausbauelementen aus Stahl"
5;"C25122";"C 25.12-2";"Herstellung von Ausbauelementen aus Leichtmetall"
3;"C252";"C 25.2";"Herstellung von Metalltanks und -behltern; Herstellung von Heizkrpern und -kesseln fr Zentralheizungen"
4;"C2521";"C 25.21";"Herstellung von Heizkrpern und kesseln fr Zentralheizungen"
5;"C25210";"C 25.21-0";"Herstellung von Heizkrpern und kesseln fr Zentralheizungen"
4;"C2529";"C 25.29";"Herstellung von Sammelbehltern, Tanks u. . Behltern aus Metall"
5;"C25290";"C 25.29-0";"Herstellung von Sammelbehltern, Tanks u. . Behltern aus Metall"
3;"C253";"C 25.3";"Herstellung von Dampfkesseln (ohne Zentralheizungskessel)"
4;"C2530";"C 25.30";"Herstellung von Dampfkesseln (ohne Zentralheizungskessel)"
5;"C25300";"C 25.30-0";"Herstellung von Dampfkesseln (ohne Zentralheizungskessel)"
3;"C254";"C 25.4";"Herstellung von Waffen und Munition"
4;"C2540";"C 25.40";"Herstellung von Waffen und Munition"
5;"C25400";"C 25.40-0";"Herstellung von Waffen und Munition"
3;"C255";"C 25.5";"Herstellung von Schmiede-, Press-, Zieh- und Stanzteilen, gewalzten Ringen und pulvermetallurgischen Erzeugnissen"
4;"C2550";"C 25.50";"Herstellung von Schmiede-, Press-, Zieh- und Stanzteilen, gewalzten Ringen und pulvermetallurgischen Erzeugnissen"
5;"C25500";"C 25.50-0";"Herstellung von Schmiede-, Press-, Zieh- und Stanzteilen, gewalzten Ringen und pulvermetallurgischen Erzeugnissen"
3;"C256";"C 25.6";"Oberflchenveredlung und Wrmebehandlung; Mechanik a.n.g."
4;"C2561";"C 25.61";"Oberflchenveredlung und Wrmebehandlung"
5;"C25610";"C 25.61-0";"Oberflchenveredlung und Wrmebehandlung"
4;"C2562";"C 25.62";"Mechanik a.n.g."
5;"C25620";"C 25.62-0";"Mechanik a.n.g."
3;"C257";"C 25.7";"Herstellung von Schneidwaren, Werkzeugen, Schlssern und Beschlgen aus unedlen Metallen"
4;"C2571";"C 25.71";"Herstellung von Schneidwaren und Bestecken aus unedlen Metallen"
5;"C25710";"C 25.71-0";"Herstellung von Schneidwaren und Bestecken aus unedlen Metallen"
4;"C2572";"C 25.72";"Herstellung von Schlssern und Beschlgen aus unedlen Metallen"
5;"C25721";"C 25.72-1";"Herstellung von Schlssern aus unedlen Metallen"
5;"C25722";"C 25.72-2";"Herstellung von Beschlgen aus unedlen Metallen"
4;"C2573";"C 25.73";"Herstellung von Werkzeugen"
5;"C25731";"C 25.73-1";"Herstellung von Formen und Formksten"
5;"C25739";"C 25.73-9";"Herstellung von sonstigen Werkzeugen"
3;"C259";"C 25.9";"Herstellung von sonstigen Metallwaren"
4;"C2591";"C 25.91";"Herstellung von Fssern, Trommeln, Dosen, Eimern u. . Behltern aus Metall"
5;"C25910";"C 25.91-0";"Herstellung von Fssern, Trommeln, Dosen, Eimern u. . Behltern aus Metall"
4;"C2592";"C 25.92";"Herstellung von Verpackungen und Verschlssen aus Eisen, Stahl und NE-Metall"
5;"C25920";"C 25.92-0";"Herstellung von Verpackungen und Verschlssen aus Eisen, Stahl und NE-Metall"
4;"C2593";"C 25.93";"Herstellung von Drahtwaren, Ketten und Federn"
5;"C25930";"C 25.93-0";"Herstellung von Drahtwaren, Ketten und Federn"
4;"C2594";"C 25.94";"Herstellung von Schrauben und Nieten"
5;"C25940";"C 25.94-0";"Herstellung von Schrauben und Nieten"
4;"C2599";"C 25.99";"Herstellung von sonstigen Metallwaren a.n.g."
5;"C25990";"C 25.99-0";"Herstellung von sonstigen Metallwaren a.n.g."
2;"C26";"C 26";"Herstellung von Datenverarbeitungsgerten, elektronischen und optischen Erzeugnissen"
3;"C261";"C 26.1";"Herstellung von elektronischen Bauelementen und Leiterplatten"
4;"C2611";"C 26.11";"Herstellung von elektronischen Bauelementen"
5;"C26110";"C 26.11-0";"Herstellung von elektronischen Bauelementen"
4;"C2612";"C 26.12";"Herstellung von bestckten Leiterplatten"
5;"C26120";"C 26.12-0";"Herstellung von bestckten Leiterplatten"
3;"C262";"C 26.2";"Herstellung von Datenverarbeitungsgerten und peripheren Gerten"
4;"C2620";"C 26.20";"Herstellung von Datenverarbeitungsgerten und peripheren Gerten"
5;"C26200";"C 26.20-0";"Herstellung von Datenverarbeitungsgerten und peripheren Gerten"
3;"C263";"C 26.3";"Herstellung von Gerten und Einrichtungen der Telekommunikationstechnik"
4;"C2630";"C 26.30";"Herstellung von Gerten und Einrichtungen der Telekommunikationstechnik"
5;"C26300";"C 26.30-0";"Herstellung von Gerten und Einrichtungen der Telekommunikationstechnik"
3;"C264";"C 26.4";"Herstellung von Gerten der Unterhaltungselektronik"
4;"C2640";"C 26.40";"Herstellung von Gerten der Unterhaltungselektronik"
5;"C26400";"C 26.40-0";"Herstellung von Gerten der Unterhaltungselektronik"
3;"C265";"C 26.5";"Herstellung von Mess-, Kontroll-, Navigations- u. . Instrumenten und Vorrichtungen; Herstellung von Uhren"
4;"C2651";"C 26.51";"Herstellung von Mess-, Kontroll-, Navigations- u. . Instrumenten und Vorrichtungen"
5;"C26510";"C 26.51-0";"Herstellung von Mess-, Kontroll-, Navigations- u. . Instrumenten und Vorrichtungen"
4;"C2652";"C 26.52";"Herstellung von Uhren"
5;"C26520";"C 26.52-0";"Herstellung von Uhren"
3;"C266";"C 26.6";"Herstellung von Bestrahlungs- und Elektrotherapiegerten und elektromedizinischen Gerten"
4;"C2660";"C 26.60";"Herstellung von Bestrahlungs- und Elektrotherapiegerten und elektromedizinischen Gerten"
5;"C26600";"C 26.60-0";"Herstellung von Bestrahlungs- und Elektrotherapiegerten und elektromedizinischen Gerten"
3;"C267";"C 26.7";"Herstellung von optischen und fotografischen Instrumenten und Gerten"
4;"C2670";"C 26.70";"Herstellung von optischen und fotografischen Instrumenten und Gerten"
5;"C26700";"C 26.70-0";"Herstellung von optischen und fotografischen Instrumenten und Gerten"
3;"C268";"C 26.8";"Herstellung von magnetischen und optischen Datentrgern"
4;"C2680";"C 26.80";"Herstellung von magnetischen und optischen Datentrgern"
5;"C26800";"C 26.80-0";"Herstellung von magnetischen und optischen Datentrgern"
2;"C27";"C 27";"Herstellung von elektrischen Ausrstungen"
3;"C271";"C 27.1";"Herstellung von Elektromotoren, Generatoren, Transformatoren, Elektrizittsverteilungs- und -schalteinrichtungen"
4;"C2711";"C 27.11";"Herstellung von Elektromotoren, Generatoren und Transformatoren"
5;"C27110";"C 27.11-0";"Herstellung von Elektromotoren, Generatoren und Transformatoren"
4;"C2712";"C 27.12";"Herstellung von Elektrizittsverteilungs- und -schalteinrichtungen"
5;"C27120";"C 27.12-0";"Herstellung von Elektrizittsverteilungs- und -schalteinrichtungen"
3;"C272";"C 27.2";"Herstellung von Batterien und Akkumulatoren"
4;"C2720";"C 27.20";"Herstellung von Batterien und Akkumulatoren"
5;"C27200";"C 27.20-0";"Herstellung von Batterien und Akkumulatoren"
3;"C273";"C 27.3";"Herstellung von Kabeln und elektrischem Installationsmaterial"
4;"C2731";"C 27.31";"Herstellung von Glasfaserkabeln"
5;"C27310";"C 27.31-0";"Herstellung von Glasfaserkabeln"
4;"C2732";"C 27.32";"Herstellung von sonstigen elektronischen und elektrischen Drhten und Kabeln"
5;"C27320";"C 27.32-0";"Herstellung von sonstigen elektronischen und elektrischen Drhten und Kabeln"
4;"C2733";"C 27.33";"Herstellung von elektrischem Installationsmaterial"
5;"C27330";"C 27.33-0";"Herstellung von elektrischem Installationsmaterial"
3;"C274";"C 27.4";"Herstellung von elektrischen Lampen und Leuchten"
4;"C2740";"C 27.40";"Herstellung von elektrischen Lampen und Leuchten"
5;"C27400";"C 27.40-0";"Herstellung von elektrischen Lampen und Leuchten"
3;"C275";"C 27.5";"Herstellung von Haushaltsgerten"
4;"C2751";"C 27.51";"Herstellung von elektrischen Haushaltsgerten"
5;"C27510";"C 27.51-0";"Herstellung von elektrischen Haushaltsgerten"
4;"C2752";"C 27.52";"Herstellung von nicht elektrischen Haushaltsgerten"
5;"C27520";"C 27.52-0";"Herstellung von nicht elektrischen Haushaltsgerten"
3;"C279";"C 27.9";"Herstellung von sonstigen elektrischen Ausrstungen und Gerten a.n.g."
4;"C2790";"C 27.90";"Herstellung von sonstigen elektrischen Ausrstungen und Gerten a.n.g."
5;"C27900";"C 27.90-0";"Herstellung von sonstigen elektrischen Ausrstungen und Gerten a.n.g."
2;"C28";"C 28";"Maschinenbau"
3;"C281";"C 28.1";"Herstellung von nicht wirtschaftszweigspezifischen Maschinen"
4;"C2811";"C 28.11";"Herstellung von Verbrennungsmotoren und Turbinen (ohne Motoren fr Luft- und Straenfahrzeuge)"
5;"C28110";"C 28.11-0";"Herstellung von Verbrennungsmotoren und Turbinen (ohne Motoren fr Luft- und Straenfahrzeuge)"
4;"C2812";"C 28.12";"Herstellung von hydraulischen und pneumatischen Komponenten und Systemen"
5;"C28120";"C 28.12-0";"Herstellung von hydraulischen und pneumatischen Komponenten und Systemen"
4;"C2813";"C 28.13";"Herstellung von Pumpen und Kompressoren a.n.g."
5;"C28130";"C 28.13-0";"Herstellung von Pumpen und Kompressoren a.n.g."
4;"C2814";"C 28.14";"Herstellung von Armaturen a. n. g."
5;"C28140";"C 28.14-0";"Herstellung von Armaturen a. n. g."
4;"C2815";"C 28.15";"Herstellung von Lagern, Getrieben, Zahnrdern und Antriebselementen"
5;"C28150";"C 28.15-0";"Herstellung von Lagern, Getrieben, Zahnrdern und Antriebselementen"
3;"C282";"C 28.2";"Herstellung von sonstigen nicht wirtschaftszweigspezifischen Maschinen"
4;"C2821";"C 28.21";"Herstellung von fen und Brennern"
5;"C28210";"C 28.21-0";"Herstellung von fen und Brennern"
4;"C2822";"C 28.22";"Herstellung von Hebezeugen und Frdermitteln"
5;"C28220";"C 28.22-0";"Herstellung von Hebezeugen und Frdermitteln"
4;"C2823";"C 28.23";"Herstellung von Bromaschinen (ohne Datenverarbeitungsgerte und periphere Gerte)"
5;"C28230";"C 28.23-0";"Herstellung von Bromaschinen (ohne Datenverarbeitungsgerte und periphere Gerte)"
4;"C2824";"C 28.24";"Herstellung von handgefhrten Werkzeugen mit Motorantrieb"
5;"C28240";"C 28.24-0";"Herstellung von handgefhrten Werkzeugen mit Motorantrieb"
4;"C2825";"C 28.25";"Herstellung von klte- und lufttechnischen Erzeugnissen, nicht fr den Haushalt"
5;"C28250";"C 28.25-0";"Herstellung von klte- und lufttechnischen Erzeugnissen, nicht fr den Haushalt"
4;"C2829";"C 28.29";"Herstellung von sonstigen nicht wirtschaftszweigspezifischen Maschinen a.n.g."
5;"C28290";"C 28.29-0";"Herstellung von sonstigen nicht wirtschaftszweigspezifischen Maschinen a.n.g."
3;"C283";"C 28.3";"Herstellung von land- und forstwirtschaftlichen Maschinen"
4;"C2830";"C 28.30";"Herstellung von land- und forstwirtschaftlichen Maschinen"
5;"C28300";"C 28.30-0";"Herstellung von land- und forstwirtschaftlichen Maschinen"
3;"C284";"C 28.4";"Herstellung von Werkzeugmaschinen"
4;"C2841";"C 28.41";"Herstellung von Werkzeugmaschinen fr die Metallbearbeitung"
5;"C28410";"C 28.41-0";"Herstellung von Werkzeugmaschinen fr die Metallbearbeitung"
4;"C2849";"C 28.49";"Herstellung von sonstigen Werkzeugmaschinen"
5;"C28491";"C 28.49-1";"Herstellung von Werkzeugmaschinen fr die Holzbearbeitung"
5;"C28499";"C 28.49-9";"Herstellung von sonstigen Werkzeugmaschinen a.n.g."
3;"C289";"C 28.9";"Herstellung von Maschinen fr sonstige bestimmte Wirtschaftszweige"
4;"C2891";"C 28.91";"Herstellung von Maschinen fr die Metallerzeugung, von Walzwerkseinrichtungen und Giemaschinen"
5;"C28910";"C 28.91-0";"Herstellung von Maschinen fr die Metallerzeugung, von Walzwerkseinrichtungen und Giemaschinen"
4;"C2892";"C 28.92";"Herstellung von Bergwerks-, Bau- und Baustoffmaschinen"
5;"C28920";"C 28.92-0";"Herstellung von Bergwerks-, Bau- und Baustoffmaschinen"
4;"C2893";"C 28.93";"Herstellung von Maschinen fr die Nahrungs- und Genussmittelerzeugung und die Tabakverarbeitung"
5;"C28930";"C 28.93-0";"Herstellung von Maschinen fr die Nahrungs- und Genussmittelerzeugung und die Tabakverarbeitung"
4;"C2894";"C 28.94";"Herstellung von Maschinen fr die Textil- und Bekleidungsherstellung und die Lederverarbeitung"
5;"C28940";"C 28.94-0";"Herstellung von Maschinen fr die Textil- und Bekleidungsherstellung und die Lederverarbeitung"
4;"C2895";"C 28.95";"Herstellung von Maschinen fr die Papiererzeugung und -verarbeitung"
5;"C28950";"C 28.95-0";"Herstellung von Maschinen fr die Papiererzeugung und -verarbeitung"
4;"C2896";"C 28.96";"Herstellung von Maschinen fr die Verarbeitung von Kunststoffen und Kautschuk"
5;"C28960";"C 28.96-0";"Herstellung von Maschinen fr die Verarbeitung von Kunststoffen und Kautschuk"
4;"C2899";"C 28.99";"Herstellung von Maschinen fr sonstige bestimmte Wirtschaftszweige a.n.g."
5;"C28991";"C 28.99-1";"Herstellung von Druckerei- und Buchbindereimaschinen"
5;"C28999";"C 28.99-9";"Herstellung von sonstigen Maschinen fr sonstige bestimmte Wirtschaftszweige a.n.g."
2;"C29";"C 29";"Herstellung von Kraftwagen und Kraftwagenteilen"
3;"C291";"C 29.1";"Herstellung von Kraftwagen und Kraftwagenmotoren"
4;"C2910";"C 29.10";"Herstellung von Kraftwagen und Kraftwagenmotoren"
5;"C29100";"C 29.10-0";"Herstellung von Kraftwagen und Kraftwagenmotoren"
3;"C292";"C 29.2";"Herstellung von Karosserien, Aufbauten und Anhngern"
4;"C2920";"C 29.20";"Herstellung von Karosserien, Aufbauten und Anhngern"
5;"C29200";"C 29.20-0";"Herstellung von Karosserien, Aufbauten und Anhngern"
3;"C293";"C 29.3";"Herstellung von Teilen und Zubehr fr Kraftwagen"
4;"C2931";"C 29.31";"Herstellung elektrischer und elektronischer Ausrstungsgegenstnde fr Kraftwagen"
5;"C29310";"C 29.31-0";"Herstellung elektrischer und elektronischer Ausrstungsgegenstnde fr Kraftwagen"
4;"C2932";"C 29.32";"Herstellung von sonstigen Teilen und sonstigem Zubehr fr Kraftwagen"
5;"C29320";"C 29.32-0";"Herstellung von sonstigen Teilen und sonstigem Zubehr fr Kraftwagen"
2;"C30";"C 30";"Sonstiger Fahrzeugbau"
3;"C301";"C 30.1";"Schiff- und Bootsbau"
4;"C3011";"C 30.11";"Schiffbau (ohne Boots- und Yachtbau)"
5;"C30110";"C 30.11-0";"Schiffbau (ohne Boots- und Yachtbau)"
4;"C3012";"C 30.12";"Boots- und Yachtbau"
5;"C30120";"C 30.12-0";"Boots- und Yachtbau"
3;"C302";"C 30.2";"Schienenfahrzeugbau"
4;"C3020";"C 30.20";"Schienenfahrzeugbau"
5;"C30200";"C 30.20-0";"Schienenfahrzeugbau"
3;"C303";"C 30.3";"Luft- und Raumfahrzeugbau"
4;"C3030";"C 30.30";"Luft- und Raumfahrzeugbau"
5;"C30300";"C 30.30-0";"Luft- und Raumfahrzeugbau"
3;"C304";"C 30.4";"Herstellung von militrischen Kampffahrzeugen"
4;"C3040";"C 30.40";"Herstellung von militrischen Kampffahrzeugen"
5;"C30400";"C 30.40-0";"Herstellung von militrischen Kampffahrzeugen"
3;"C309";"C 30.9";"Herstellung von Fahrzeugen a.n.g."
4;"C3091";"C 30.91";"Herstellung von Kraftrdern"
5;"C30910";"C 30.91-0";"Herstellung von Kraftrdern"
4;"C3092";"C 30.92";"Herstellung von Fahrrdern sowie von Behindertenfahrzeugen"
5;"C30920";"C 30.92-0";"Herstellung von Fahrrdern sowie von Behindertenfahrzeugen"
4;"C3099";"C 30.99";"Herstellung von sonstigen Fahrzeugen a.n.g."
5;"C30990";"C 30.99-0";"Herstellung von sonstigen Fahrzeugen a.n.g."
2;"C31";"C 31";"Herstellung von Mbeln"
3;"C310";"C 31.0";"Herstellung von Mbeln"
4;"C3101";"C 31.01";"Herstellung von Bro- und Ladenmbeln"
5;"C31011";"C 31.01-1";"Herstellung von Brombeln"
5;"C31012";"C 31.01-2";"Herstellung von Ladenmbeln"
4;"C3102";"C 31.02";"Herstellung von Kchenmbeln"
5;"C31020";"C 31.02-0";"Herstellung von Kchenmbeln"
4;"C3103";"C 31.03";"Herstellung von Matratzen"
5;"C31030";"C 31.03-0";"Herstellung von Matratzen"
4;"C3109";"C 31.09";"Herstellung von sonstigen Mbeln"
5;"C31090";"C 31.09-0";"Herstellung von sonstigen Mbeln"
2;"C32";"C 32";"Herstellung von sonstigen Waren"
3;"C321";"C 32.1";"Herstellung von Mnzen, Schmuck und hnlichen Erzeugnissen"
4;"C3211";"C 32.11";"Herstellung von Mnzen"
5;"C32110";"C 32.11-0";"Herstellung von Mnzen"
4;"C3212";"C 32.12";"Herstellung von Schmuck, Gold- und Silberschmiedewaren (ohne Fantasieschmuck)"
5;"C32120";"C 32.12-0";"Herstellung von Schmuck, Gold- und Silberschmiedewaren (ohne Fantasieschmuck)"
4;"C3213";"C 32.13";"Herstellung von Fantasieschmuck"
5;"C32130";"C 32.13-0";"Herstellung von Fantasieschmuck"
3;"C322";"C 32.2";"Herstellung von Musikinstrumenten"
4;"C3220";"C 32.20";"Herstellung von Musikinstrumenten"
5;"C32200";"C 32.20-0";"Herstellung von Musikinstrumenten"
3;"C323";"C 32.3";"Herstellung von Sportgerten"
4;"C3230";"C 32.30";"Herstellung von Sportgerten"
5;"C32300";"C 32.30-0";"Herstellung von Sportgerten"
3;"C324";"C 32.4";"Herstellung von Spielwaren"
4;"C3240";"C 32.40";"Herstellung von Spielwaren"
5;"C32400";"C 32.40-0";"Herstellung von Spielwaren"
3;"C325";"C 32.5";"Herstellung von medizinischen und zahnmedizinischen Apparaten und Materialien"
4;"C3250";"C 32.50";"Herstellung von medizinischen und zahnmedizinischen Apparaten und Materialien"
5;"C32501";"C 32.50-1";"Zahntechnische Laboratorien"
5;"C32502";"C 32.50-2";"Herstellung von Sehbehelfen und deren Teile"
5;"C32509";"C 32.50-9";"Herstellung von sonstigen medizinischen und zahnmedizinischen Apparaten und Materialien"
3;"C329";"C 32.9";"Herstellung von Erzeugnissen a.n.g."
4;"C3291";"C 32.91";"Herstellung von Besen und Brsten"
5;"C32910";"C 32.91-0";"Herstellung von Besen und Brsten"
4;"C3299";"C 32.99";"Herstellung von sonstigen Erzeugnissen a.n.g."
5;"C32990";"C 32.99-0";"Herstellung von sonstigen Erzeugnissen a.n.g."
2;"C33";"C 33";"Reparatur und Installation von Maschinen und Ausrstungen"
3;"C331";"C 33.1";"Reparatur von Metallerzeugnissen, Maschinen und Ausrstungen"
4;"C3311";"C 33.11";"Reparatur von Metallerzeugnissen"
5;"C33110";"C 33.11-0";"Reparatur von Metallerzeugnissen"
4;"C3312";"C 33.12";"Reparatur von Maschinen"
5;"C33120";"C 33.12-0";"Reparatur von Maschinen"
4;"C3313";"C 33.13";"Reparatur von elektronischen und optischen Gerten"
5;"C33130";"C 33.13-0";"Reparatur von elektronischen und optischen Gerten"
4;"C3314";"C 33.14";"Reparatur von elektrischen Ausrstungen"
5;"C33140";"C 33.14-0";"Reparatur von elektrischen Ausrstungen"
4;"C3315";"C 33.15";"Reparatur und Instandhaltung von Schiffen, Booten und Yachten"
5;"C33150";"C 33.15-0";"Reparatur und Instandhaltung von Schiffen, Booten und Yachten"
4;"C3316";"C 33.16";"Reparatur und Instandhaltung von Luft- und Raumfahrzeugen"
5;"C33160";"C 33.16-0";"Reparatur und Instandhaltung von Luft- und Raumfahrzeugen"
4;"C3317";"C 33.17";"Reparatur und Instandhaltung von Fahrzeugen a.n.g."
5;"C33170";"C 33.17-0";"Reparatur und Instandhaltung von Fahrzeugen a.n.g."
4;"C3319";"C 33.19";"Reparatur von sonstigen Ausrstungen"
5;"C33190";"C 33.19-0";"Reparatur von sonstigen Ausrstungen"
3;"C332";"C 33.2";"Installation von Maschinen und Ausrstungen a.n.g."
4;"C3320";"C 33.20";"Installation von Maschinen und Ausrstungen a.n.g."
5;"C33200";"C 33.20-0";"Installation von Maschinen und Ausrstungen a.n.g."
1;"D";"D";"ENERGIEVERSORGUNG"
2;"D35";"D 35";"Energieversorgung"
3;"D351";"D 35.1";"Elektrizittsversorgung"
4;"D3511";"D 35.11";"Elektrizittserzeugung"
5;"D35110";"D 35.11-0";"Elektrizittserzeugung"
4;"D3512";"D 35.12";"Elektrizittsbertragung"
5;"D35120";"D 35.12-0";"Elektrizittsbertragung"
4;"D3513";"D 35.13";"Elektrizittsverteilung"
5;"D35130";"D 35.13-0";"Elektrizittsverteilung"
4;"D3514";"D 35.14";"Elektrizittshandel"
5;"D35140";"D 35.14-0";"Elektrizittshandel"
3;"D352";"D 35.2";"Gasversorgung"
4;"D3521";"D 35.21";"Gaserzeugung"
5;"D35210";"D 35.21-0";"Gaserzeugung"
4;"D3522";"D 35.22";"Gasverteilung durch Rohrleitungen"
5;"D35220";"D 35.22-0";"Gasverteilung durch Rohrleitungen"
4;"D3523";"D 35.23";"Gashandel durch Rohrleitungen"
5;"D35230";"D 35.23-0";"Gashandel durch Rohrleitungen"
3;"D353";"D 35.3";"Wrme- und Klteversorgung"
4;"D3530";"D 35.30";"Wrme- und Klteversorgung"
5;"D35300";"D 35.30-0";"Wrme- und Klteversorgung"
1;"E";"E";"WASSERVERSORGUNG; ABWASSER- UND ABFALLENTSORGUNG UND BESEITIGUNG VON UMWELTVERSCHMUTZUNGEN"
2;"E36";"E 36";"Wasserversorgung"
3;"E360";"E 36.0";"Wasserversorgung"
4;"E3600";"E 36.00";"Wasserversorgung"
5;"E36000";"E 36.00-0";"Wasserversorgung"
2;"E37";"E 37";"Abwasserentsorgung"
3;"E370";"E 37.0";"Abwasserentsorgung"
4;"E3700";"E 37.00";"Abwasserentsorgung"
5;"E37000";"E 37.00-0";"Abwasserentsorgung"
2;"E38";"E 38";"Sammlung, Behandlung und Beseitigung von Abfllen; Rckgewinnung"
3;"E381";"E 38.1";"Sammlung von Abfllen"
4;"E3811";"E 38.11";"Sammlung nicht gefhrlicher Abflle"
5;"E38110";"E 38.11-0";"Sammlung nicht gefhrlicher Abflle"
4;"E3812";"E 38.12";"Sammlung gefhrlicher Abflle"
5;"E38120";"E 38.12-0";"Sammlung gefhrlicher Abflle"
3;"E382";"E 38.2";"Abfallbehandlung und -beseitigung"
4;"E3821";"E 38.21";"Behandlung und Beseitigung nicht gefhrlicher Abflle"
5;"E38211";"E 38.21-1";"Deponierung nicht gefhrlicher Abflle"
5;"E38219";"E 38.21-9";"Sonstige Behandlung und Beseitigung nicht gefhrlicher Abflle"
4;"E3822";"E 38.22";"Behandlung und Beseitigung gefhrlicher Abflle"
5;"E38220";"E 38.22-0";"Behandlung und Beseitigung gefhrlicher Abflle"
3;"E383";"E 38.3";"Rckgewinnung"
4;"E3831";"E 38.31";"Zerlegen von Schiffs- und Fahrzeugwracks und anderen Altwaren"
5;"E38310";"E 38.31-0";"Zerlegen von Schiffs- und Fahrzeugwracks und anderen Altwaren"
4;"E3832";"E 38.32";"Rckgewinnung sortierter Werkstoffe"
5;"E38321";"E 38.32-1";"Rckgewinnung von Eisen und NE-Metallen"
5;"E38329";"E 38.32-9";"Rckgewinnung sonstiger sortierter Werkstoffe"
2;"E39";"E 39";"Beseitigung von Umweltverschmutzungen und sonstige Entsorgung"
3;"E390";"E 39.0";"Beseitigung von Umweltverschmutzungen und sonstige Entsorgung"
4;"E3900";"E 39.00";"Beseitigung von Umweltverschmutzungen und sonstige Entsorgung"
5;"E39000";"E 39.00-0";"Beseitigung von Umweltverschmutzungen und sonstige Entsorgung"
1;"F";"F";"BAU"
2;"F41";"F 41";"Hochbau"
3;"F411";"F 41.1";"Erschlieung von Grundstcken; Bautrger"
4;"F4110";"F 41.10";"Erschlieung von Grundstcken; Bautrger"
5;"F41100";"F 41.10-0";"Erschlieung von Grundstcken; Bautrger"
3;"F412";"F 41.2";"Bau von Gebuden"
4;"F4120";"F 41.20";"Bau von Gebuden"
5;"F41201";"F 41.20-1";"Wohnungs- und Siedlungsbau"
5;"F41202";"F 41.20-2";"Adaptierungsarbeiten im Hochbau"
5;"F41209";"F 41.20-9";"Sonstiger Hochbau"
2;"F42";"F 42";"Tiefbau"
3;"F421";"F 42.1";"Bau von Straen und Bahnverkehrsstrecken"
4;"F4211";"F 42.11";"Bau von Straen"
5;"F42110";"F 42.11-0";"Bau von Straen"
4;"F4212";"F 42.12";"Bau von Bahnverkehrsstrecken"
5;"F42120";"F 42.12-0";"Bau von Bahnverkehrsstrecken"
4;"F4213";"F 42.13";"Brcken- und Tunnelbau"
5;"F42131";"F 42.13-1";"Brcken- und Hochstraenbau"
5;"F42132";"F 42.13-2";"Tunnelbau"
3;"F422";"F 42.2";"Leitungstiefbau und Klranlagenbau"
4;"F4221";"F 42.21";"Rohrleitungstiefbau, Brunnenbau und Klranlagenbau"
5;"F42210";"F 42.21-0";"Rohrleitungstiefbau, Brunnenbau und Klranlagenbau"
4;"F4222";"F 42.22";"Kabelnetzleitungstiefbau"
5;"F42220";"F 42.22-0";"Kabelnetzleitungstiefbau"
3;"F429";"F 42.9";"Sonstiger Tiefbau"
4;"F4291";"F 42.91";"Wasserbau"
5;"F42910";"F 42.91-0";"Wasserbau"
4;"F4299";"F 42.99";"Sonstiger Tiefbau a.n.g."
5;"F42990";"F 42.99-0";"Sonstiger Tiefbau a.n.g."
2;"F43";"F 43";"Vorbereitende Baustellenarbeiten, Bauinstallation und sonstiges Ausbaugewerbe"
3;"F431";"F 43.1";"Abbrucharbeiten und vorbereitende Baustellenarbeiten"
4;"F4311";"F 43.11";"Abbrucharbeiten"
5;"F43110";"F 43.11-0";"Abbrucharbeiten"
4;"F4312";"F 43.12";"Vorbereitende Baustellenarbeiten"
5;"F43120";"F 43.12-0";"Vorbereitende Baustellenarbeiten"
4;"F4313";"F 43.13";"Test- und Suchbohrung"
5;"F43130";"F 43.13-0";"Test- und Suchbohrung"
3;"F432";"F 43.2";"Bauinstallation"
4;"F4321";"F 43.21";"Elektroinstallation"
5;"F43210";"F 43.21-0";"Elektroinstallation"
4;"F4322";"F 43.22";"Gas-, Wasser-, Heizungs- sowie Lftungs- und Klimainstallation"
5;"F43220";"F 43.22-0";"Gas-, Wasser-, Heizungs- sowie Lftungs- und Klimainstallation"
4;"F4329";"F 43.29";"Sonstige Bauinstallation"
5;"F43290";"F 43.29-0";"Sonstige Bauinstallation"
3;"F433";"F 43.3";"Sonstiger Ausbau"
4;"F4331";"F 43.31";"Anbringen von Stuckaturen, Gipserei und Verputzerei"
5;"F43310";"F 43.31-0";"Anbringen von Stuckaturen, Gipserei und Verputzerei"
4;"F4332";"F 43.32";"Bautischlerei und -schlosserei"
5;"F43321";"F 43.32-1";"Bautischlerei"
5;"F43322";"F 43.32-2";"Bauschlosserei"
4;"F4333";"F 43.33";"Fuboden-, Fliesen- und Plattenlegerei, Tapeziererei"
5;"F43331";"F 43.33-1";"Fubodenlegerei"
5;"F43332";"F 43.33-2";"Fliesenlegerei"
5;"F43333";"F 43.33-3";"Tapeziererei"
5;"F43334";"F 43.33-4";"Ofensetzerei"
4;"F4334";"F 43.34";"Malerei und Glaserei"
5;"F43341";"F 43.34-1";"Malerei und Anstreicherei"
5;"F43342";"F 43.34-2";"Glaserei"
4;"F4339";"F 43.39";"Sonstiger Ausbau a.n.g."
5;"F43390";"F 43.39-0";"Sonstiger Ausbau a.n.g."
3;"F439";"F 43.9";"Sonstige spezialisierte Bauttigkeiten"
4;"F4391";"F 43.91";"Dachdeckerei und Zimmerei"
5;"F43911";"F 43.91-1";"Dachdeckerei"
5;"F43912";"F 43.91-2";"Zimmerei"
5;"F43913";"F 43.91-3";"Bauspenglerei"
4;"F4399";"F 43.99";"Sonstige spezialisierte Bauttigkeiten a.n.g."
5;"F43991";"F 43.99-1";"Isolierer"
5;"F43999";"F 43.99-9";"Sonstige spezialisierte Bauttigkeiten a.n.g. (ohne Isolierer)"
1;"G";"G";"HANDEL; INSTANDHALTUNG UND REPARATUR VON KRAFTFAHRZEUGEN"
2;"G45";"G 45";"Handel mit Kraftfahrzeugen; Instandhaltung und Reparatur von Kraftfahrzeugen"
3;"G451";"G 45.1";"Handel mit Kraftwagen"
4;"G4511";"G 45.11";"Handel mit Kraftwagen mit einem Gesamtgewicht von 3,5 t oder weniger"
5;"G45111";"G 45.11-1";"Grohandel mit Kraftwagen mit einem Gesamtgewicht von 3,5 t oder weniger"
5;"G45112";"G 45.11-2";"Einzelhandel mit Kraftwagen mit einem Gesamtgewicht von 3,5 t oder weniger"
4;"G4519";"G 45.19";"Handel mit Kraftwagen mit einem Gesamtgewicht von mehr als 3,5 t"
5;"G45190";"G 45.19-0";"Handel mit Kraftwagen mit einem Gesamtgewicht von mehr als 3,5 t"
3;"G452";"G 45.2";"Instandhaltung und Reparatur von Kraftwagen"
4;"G4520";"G 45.20";"Instandhaltung und Reparatur von Kraftwagen"
5;"G45200";"G 45.20-0";"Instandhaltung und Reparatur von Kraftwagen"
3;"G453";"G 45.3";"Handel mit Kraftwagenteilen und -zubehr"
4;"G4531";"G 45.31";"Grohandel mit Kraftwagenteilen und -zubehr"
5;"G45310";"G 45.31-0";"Grohandel mit Kraftwagenteilen und -zubehr"
4;"G4532";"G 45.32";"Einzelhandel mit Kraftwagenteilen und -zubehr"
5;"G45320";"G 45.32-0";"Einzelhandel mit Kraftwagenteilen und -zubehr"
3;"G454";"G 45.4";"Handel mit Kraftrdern, Kraftradteilen und -zubehr; Instandhaltung und Reparatur von Kraftrdern"
4;"G4540";"G 45.40";"Handel mit Kraftrdern, Kraftradteilen und -zubehr; Instandhaltung und Reparatur von Kraftrdern"
5;"G45400";"G 45.40-0";"Handel mit Kraftrdern, Kraftradteilen und -zubehr; Instandhaltung und Reparatur von Kraftrdern"
2;"G46";"G 46";"Grohandel (ohne Handel mit Kraftfahrzeugen und Kraftrdern)"
3;"G461";"G 46.1";"Handelsvermittlung"
4;"G4611";"G 46.11";"Handelsvermittlung von landwirtschaftlichen Grundstoffen, lebenden Tieren, textilen Rohstoffen und Halbwaren"
5;"G46110";"G 46.11-0";"Handelsvermittlung von landwirtschaftlichen Grundstoffen, lebenden Tieren, textilen Rohstoffen und Halbwaren"
4;"G4612";"G 46.12";"Handelsvermittlung von Brennstoffen, Erzen, Metallen und technischen Chemikalien"
5;"G46120";"G 46.12-0";"Handelsvermittlung von Brennstoffen, Erzen, Metallen und technischen Chemikalien"
4;"G4613";"G 46.13";"Handelsvermittlung von Holz, Baustoffen und Anstrichmitteln"
5;"G46130";"G 46.13-0";"Handelsvermittlung von Holz, Baustoffen und Anstrichmitteln"
4;"G4614";"G 46.14";"Handelsvermittlung von Maschinen, technischem Bedarf, Wasser- und Luftfahrzeugen"
5;"G46140";"G 46.14-0";"Handelsvermittlung von Maschinen, technischem Bedarf, Wasser- und Luftfahrzeugen"
4;"G4615";"G 46.15";"Handelsvermittlung von Mbeln, Einrichtungs- und Haushaltsgegenstnden, Eisen- und Metallwaren"
5;"G46150";"G 46.15-0";"Handelsvermittlung von Mbeln, Einrichtungs- und Haushaltsgegenstnden, Eisen- und Metallwaren"
4;"G4616";"G 46.16";"Handelsvermittlung von Textilien, Bekleidung, Schuhen und Lederwaren"
5;"G46160";"G 46.16-0";"Handelsvermittlung von Textilien, Bekleidung, Schuhen und Lederwaren"
4;"G4617";"G 46.17";"Handelsvermittlung von Nahrungsmitteln, Getrnken und Tabakwaren"
5;"G46170";"G 46.17-0";"Handelsvermittlung von Nahrungsmitteln, Getrnken und Tabakwaren"
4;"G4618";"G 46.18";"Handelsvermittlung von sonstigen Waren"
5;"G46180";"G 46.18-0";"Handelsvermittlung von sonstigen Waren"
4;"G4619";"G 46.19";"Handelsvermittlung von Waren ohne ausgeprgten Schwerpunkt"
5;"G46190";"G 46.19-0";"Handelsvermittlung von Waren ohne ausgeprgten Schwerpunkt"
3;"G462";"G 46.2";"Grohandel mit landwirtschaftlichen Grundstoffen und lebenden Tieren"
4;"G4621";"G 46.21";"Grohandel mit Getreide, Rohtabak, Saatgut und Futtermitteln"
5;"G46210";"G 46.21-0";"Grohandel mit Getreide, Rohtabak, Saatgut und Futtermitteln"
4;"G4622";"G 46.22";"Grohandel mit Blumen und Pflanzen"
5;"G46220";"G 46.22-0";"Grohandel mit Blumen und Pflanzen"
4;"G4623";"G 46.23";"Grohandel mit lebenden Tieren"
5;"G46230";"G 46.23-0";"Grohandel mit lebenden Tieren"
4;"G4624";"G 46.24";"Grohandel mit Huten, Fellen und Leder"
5;"G46240";"G 46.24-0";"Grohandel mit Huten, Fellen und Leder"
3;"G463";"G 46.3";"Grohandel mit Nahrungs- und Genussmitteln, Getrnken und Tabakwaren"
4;"G4631";"G 46.31";"Grohandel mit Obst, Gemse und Kartoffeln"
5;"G46310";"G 46.31-0";"Grohandel mit Obst, Gemse und Kartoffeln"
4;"G4632";"G 46.32";"Grohandel mit Fleisch und Fleischwaren"
5;"G46320";"G 46.32-0";"Grohandel mit Fleisch und Fleischwaren"
4;"G4633";"G 46.33";"Grohandel mit Milch, Milcherzeugnissen, Eiern, Speiselen und Nahrungsfetten"
5;"G46330";"G 46.33-0";"Grohandel mit Milch, Milcherzeugnissen, Eiern, Speiselen und Nahrungsfetten"
4;"G4634";"G 46.34";"Grohandel mit Getrnken"
5;"G46341";"G 46.34-1";"Grohandel mit Wein und Spirituosen"
5;"G46349";"G 46.34-9";"Grohandel mit sonstigen Getrnken"
4;"G4635";"G 46.35";"Grohandel mit Tabakwaren"
5;"G46350";"G 46.35-0";"Grohandel mit Tabakwaren"
4;"G4636";"G 46.36";"Grohandel mit Zucker, Swaren und Backwaren"
5;"G46360";"G 46.36-0";"Grohandel mit Zucker, Swaren und Backwaren"
4;"G4637";"G 46.37";"Grohandel mit Kaffee, Tee, Kakao und Gewrzen"
5;"G46370";"G 46.37-0";"Grohandel mit Kaffee, Tee, Kakao und Gewrzen"
4;"G4638";"G 46.38";"Grohandel mit sonstigen Nahrungs- und Genussmitteln"
5;"G46380";"G 46.38-0";"Grohandel mit sonstigen Nahrungs- und Genussmitteln"
4;"G4639";"G 46.39";"Grohandel mit Nahrungs- und Genussmitteln, Getrnken und Tabakwaren, ohne ausgeprgten Schwerpunkt"
5;"G46390";"G 46.39-0";"Grohandel mit Nahrungs- und Genussmitteln, Getrnken und Tabakwaren, ohne ausgeprgten Schwerpunkt"
3;"G464";"G 46.4";"Grohandel mit Gebrauchs- und Verbrauchsgtern"
4;"G4641";"G 46.41";"Grohandel mit Textilien"
5;"G46410";"G 46.41-0";"Grohandel mit Textilien"
4;"G4642";"G 46.42";"Grohandel mit Bekleidung und Schuhen"
5;"G46421";"G 46.42-1";"Grohandel mit Bekleidung"
5;"G46422";"G 46.42-2";"Grohandel mit Schuhen"
4;"G4643";"G 46.43";"Grohandel mit Foto- und optischen Erzeugnissen, elektrischen Haushaltsgerten und Gerten der Unterhaltungselektronik"
5;"G46431";"G 46.43-1";"Grohandel mit Foto- und optischen Erzeugnissen"
5;"G46432";"G 46.43-2";"Grohandel mit elektrischen Haushaltsgerten"
5;"G46433";"G 46.43-3";"Grohandel mit Gerten der Unterhaltungselektronik"
4;"G4644";"G 46.44";"Grohandel mit keramischen Erzeugnissen, Glaswaren und Reinigungsmitteln"
5;"G46440";"G 46.44-0";"Grohandel mit keramischen Erzeugnissen, Glaswaren und Reinigungsmitteln"
4;"G4645";"G 46.45";"Grohandel mit kosmetischen Erzeugnissen und Krperpflegemitteln"
5;"G46450";"G 46.45-0";"Grohandel mit kosmetischen Erzeugnissen und Krperpflegemitteln"
4;"G4646";"G 46.46";"Grohandel mit pharmazeutischen, medizinischen und orthopdischen Erzeugnissen"
5;"G46461";"G 46.46-1";"Grohandel mit pharmazeutischen Erzeugnissen"
5;"G46462";"G 46.46-2";"Grohandel mit medizinischen und orthopdischen Artikeln und Laborbedarf"
4;"G4647";"G 46.47";"Grohandel mit Mbeln, Teppichen, Lampen und Leuchten"
5;"G46470";"G 46.47-0";"Grohandel mit Mbeln, Teppichen, Lampen und Leuchten"
4;"G4648";"G 46.48";"Grohandel mit Uhren und Schmuck"
5;"G46480";"G 46.48-0";"Grohandel mit Uhren und Schmuck"
4;"G4649";"G 46.49";"Grohandel mit sonstigen Gebrauchs- und Verbrauchsgtern"
5;"G46491";"G 46.49-1";"Grohandel mit Schreibwaren und Brobedarf"
5;"G46492";"G 46.49-2";"Grohandel mit Bchern, Zeitschriften und Zeitungen"
5;"G46493";"G 46.49-3";"Grohandel mit Sportartikeln"
5;"G46494";"G 46.49-4";"Grohandel mit Spielwaren und Musikinstrumenten"
5;"G46499";"G 46.49-9";"Grohandel mit sonstigen Gebrauchs- und Verbrauchsgtern a.n.g."
3;"G465";"G 46.5";"Grohandel mit Gerten der Informations- und Kommunikationstechnik"
4;"G4651";"G 46.51";"Grohandel mit Datenverarbeitungsgerten, peripheren Gerten und Software"
5;"G46511";"G 46.51-1";"Grohandel mit Datenverarbeitungsgerten und peripheren Einheiten"
5;"G46512";"G 46.51-2";"Grohandel mit Standardsoftware"
4;"G4652";"G 46.52";"Grohandel mit elektronischen Bauteilen und Telekommunikationsgerten"
5;"G46520";"G 46.52-0";"Grohandel mit elektronischen Bauteilen und Telekommunikationsgerten"
3;"G466";"G 46.6";"Grohandel mit sonstigen Maschinen, Ausrstungen und Zubehr"
4;"G4661";"G 46.61";"Grohandel mit landwirtschaftlichen Maschinen und Gerten"
5;"G46610";"G 46.61-0";"Grohandel mit landwirtschaftlichen Maschinen und Gerten"
4;"G4662";"G 46.62";"Grohandel mit Werkzeugmaschinen"
5;"G46620";"G 46.62-0";"Grohandel mit Werkzeugmaschinen"
4;"G4663";"G 46.63";"Grohandel mit Bergwerks-, Bau- und Baustoffmaschinen"
5;"G46630";"G 46.63-0";"Grohandel mit Bergwerks-, Bau- und Baustoffmaschinen"
4;"G4664";"G 46.64";"Grohandel mit Textil-, Nh- und Strickmaschinen"
5;"G46640";"G 46.64-0";"Grohandel mit Textil-, Nh- und Strickmaschinen"
4;"G4665";"G 46.65";"Grohandel mit Brombeln"
5;"G46650";"G 46.65-0";"Grohandel mit Brombeln"
4;"G4666";"G 46.66";"Grohandel mit sonstigen Bromaschinen und -einrichtungen"
5;"G46660";"G 46.66-0";"Grohandel mit sonstigen Bromaschinen und -einrichtungen"
4;"G4669";"G 46.69";"Grohandel mit sonstigen Maschinen und Ausrstungen"
5;"G46690";"G 46.69-0";"Grohandel mit sonstigen Maschinen und Ausrstungen"
3;"G467";"G 46.7";"Sonstiger Grohandel"
4;"G4671";"G 46.71";"Grohandel mit festen Brennstoffen und Minerallerzeugnissen"
5;"G46711";"G 46.71-1";"Grohandel mit festen Brennstoffen"
5;"G46712";"G 46.71-2";"Grohandel mit Minerallerzeugnissen"
4;"G4672";"G 46.72";"Grohandel mit Erzen, Metallen und Metallhalbzeug"
5;"G46721";"G 46.72-1";"Grohandel mit Erzen, Eisen, Stahl und Halbzeug"
5;"G46722";"G 46.72-2";"Grohandel mit NE-Metallen und Halbzeug"
4;"G4673";"G 46.73";"Grohandel mit Holz, Baustoffen, Anstrichmitteln und Sanitrkeramik"
5;"G46731";"G 46.73-1";"Grohandel mit Holz"
5;"G46732";"G 46.73-2";"Grohandel mit Baustoffen und Sanitrkeramik"
5;"G46733";"G 46.73-3";"Grohandel mit Flachglas, Anstrichmitteln, Tapeten und Bodenbelgen"
4;"G4674";"G 46.74";"Grohandel mit Metall- und Kunststoffwaren fr Bauzwecke sowie Installationsbedarf fr Gas, Wasser und Heizung"
5;"G46740";"G 46.74-0";"Grohandel mit Metall- und Kunststoffwaren fr Bauzwecke sowie Installationsbedarf fr Gas, Wasser und Heizung"
4;"G4675";"G 46.75";"Grohandel mit chemischen Erzeugnissen"
5;"G46750";"G 46.75-0";"Grohandel mit chemischen Erzeugnissen"
4;"G4676";"G 46.76";"Grohandel mit sonstigen Halbwaren"
5;"G46761";"G 46.76-1";"Grohandel mit Papier und Pappe"
5;"G46769";"G 46.76-9";"Grohandel mit sonstigen Halbwaren a.n.g."
4;"G4677";"G 46.77";"Grohandel mit Altmaterialien und Reststoffen"
5;"G46770";"G 46.77-0";"Grohandel mit Altmaterialien und Reststoffen"
3;"G469";"G 46.9";"Grohandel ohne ausgeprgten Schwerpunkt"
4;"G4690";"G 46.90";"Grohandel ohne ausgeprgten Schwerpunkt"
5;"G46900";"G 46.90-0";"Grohandel ohne ausgeprgten Schwerpunkt"
2;"G47";"G 47";"Einzelhandel (ohne Handel mit Kraftfahrzeugen)"
3;"G471";"G 47.1";"Einzelhandel mit Waren verschiedener Art (in Verkaufsrumen)"
4;"G4711";"G 47.11";"Einzelhandel mit Waren verschiedener Art, Hauptrichtung Nahrungs- und Genussmittel, Getrnke und Tabakwaren"
5;"G47110";"G 47.11-0";"Einzelhandel mit Waren verschiedener Art, Hauptrichtung Nahrungs- und Genussmittel, Getrnke und Tabakwaren"
4;"G4719";"G 47.19";"Sonstiger Einzelhandel mit Waren verschiedener Art"
5;"G47191";"G 47.19-1";"Warenhuser"
5;"G47199";"G 47.19-9";"Sonstiger Einzelhandel mit Waren verschiedener Art (ohne Warenhuser)"
3;"G472";"G 47.2";"Einzelhandel mit Nahrungs- und Genussmitteln, Getrnken und Tabakwaren (in Verkaufsrumen)"
4;"G4721";"G 47.21";"Einzelhandel mit Obst, Gemse und Kartoffeln"
5;"G47210";"G 47.21-0";"Einzelhandel mit Obst, Gemse und Kartoffeln"
4;"G4722";"G 47.22";"Einzelhandel mit Fleisch und Fleischwaren"
5;"G47220";"G 47.22-0";"Einzelhandel mit Fleisch und Fleischwaren"
4;"G4723";"G 47.23";"Einzelhandel mit Fisch, Meeresfrchten und Fischerzeugnissen"
5;"G47230";"G 47.23-0";"Einzelhandel mit Fisch, Meeresfrchten und Fischerzeugnissen"
4;"G4724";"G 47.24";"Einzelhandel mit Back- und Swaren"
5;"G47240";"G 47.24-0";"Einzelhandel mit Back- und Swaren"
4;"G4725";"G 47.25";"Einzelhandel mit Getrnken"
5;"G47250";"G 47.25-0";"Einzelhandel mit Getrnken"
4;"G4726";"G 47.26";"Einzelhandel mit Tabakwaren"
5;"G47260";"G 47.26-0";"Einzelhandel mit Tabakwaren"
4;"G4729";"G 47.29";"Sonstiger Einzelhandel mit Nahrungs- und Genussmitteln"
5;"G47290";"G 47.29-0";"Sonstiger Einzelhandel mit Nahrungs- und Genussmitteln"
3;"G473";"G 47.3";"Einzelhandel mit Motorenkraftstoffen (Tankstellen)"
4;"G4730";"G 47.30";"Einzelhandel mit Motorenkraftstoffen (Tankstellen)"
5;"G47300";"G 47.30-0";"Einzelhandel mit Motorenkraftstoffen (Tankstellen)"
3;"G474";"G 47.4";"Einzelhandel mit Gerten der Informations- und Kommunikationstechnik (in Verkaufsrumen)"
4;"G4741";"G 47.41";"Einzelhandel mit Datenverarbeitungsgerten, peripheren Gerten und Software"
5;"G47410";"G 47.41-0";"Einzelhandel mit Datenverarbeitungsgerten, peripheren Gerten und Software"
4;"G4742";"G 47.42";"Einzelhandel mit Telekommunikationsgerten"
5;"G47420";"G 47.42-0";"Einzelhandel mit Telekommunikationsgerten"
4;"G4743";"G 47.43";"Einzelhandel mit Gerten der Unterhaltungselektronik"
5;"G47430";"G 47.43-0";"Einzelhandel mit Gerten der Unterhaltungselektronik"
3;"G475";"G 47.5";"Einzelhandel mit sonstigen Haushaltsgerten, Textilien, Heimwerker- und Einrichtungsbedarf (in Verkaufsrumen)"
4;"G4751";"G 47.51";"Einzelhandel mit Textilien"
5;"G47510";"G 47.51-0";"Einzelhandel mit Textilien"
4;"G4752";"G 47.52";"Einzelhandel mit Metallwaren, Anstrichmitteln, Bau- und Heimwerkerbedarf"
5;"G47520";"G 47.52-0";"Einzelhandel mit Metallwaren, Anstrichmitteln, Bau- und Heimwerkerbedarf"
4;"G4753";"G 47.53";"Einzelhandel mit Vorhngen, Teppichen, Fubodenbelgen und Tapeten"
5;"G47530";"G 47.53-0";"Einzelhandel mit Vorhngen, Teppichen, Fubodenbelgen und Tapeten"
4;"G4754";"G 47.54";"Einzelhandel mit elektrischen Haushaltsgerten"
5;"G47540";"G 47.54-0";"Einzelhandel mit elektrischen Haushaltsgerten"
4;"G4759";"G 47.59";"Einzelhandel mit Mbeln, Einrichtungsgegenstnden und sonstigem Hausrat"
5;"G47591";"G 47.59-1";"Einzelhandel mit Musikinstrumenten und Musikalien"
5;"G47592";"G 47.59-2";"Einzelhandel mit Wohnmbeln"
5;"G47599";"G 47.59-9";"Einzelhandel mit Einrichtungsgegenstnden und Hausrat a.n.g."
3;"G476";"G 47.6";"Einzelhandel mit Verlagsprodukten, Sportausrstungen und Spielwaren (in Verkaufsrumen)"
4;"G4761";"G 47.61";"Einzelhandel mit Bchern"
5;"G47610";"G 47.61-0";"Einzelhandel mit Bchern"
4;"G4762";"G 47.62";"Einzelhandel mit Zeitschriften, Zeitungen, Schreibwaren und Brobedarf"
5;"G47620";"G 47.62-0";"Einzelhandel mit Zeitschriften, Zeitungen, Schreibwaren und Brobedarf"
4;"G4763";"G 47.63";"Einzelhandel mit bespielten Ton- und Bildtrgern"
5;"G47630";"G 47.63-0";"Einzelhandel mit bespielten Ton- und Bildtrgern"
4;"G4764";"G 47.64";"Einzelhandel mit Fahrrdern, Sport- und Campingartikeln"
5;"G47640";"G 47.64-0";"Einzelhandel mit Fahrrdern, Sport- und Campingartikeln"
4;"G4765";"G 47.65";"Einzelhandel mit Spielwaren"
5;"G47650";"G 47.65-0";"Einzelhandel mit Spielwaren"
3;"G477";"G 47.7";"Einzelhandel mit sonstigen Gtern (in Verkaufsrumen)"
4;"G4771";"G 47.71";"Einzelhandel mit Bekleidung"
5;"G47710";"G 47.71-0";"Einzelhandel mit Bekleidung"
4;"G4772";"G 47.72";"Einzelhandel mit Schuhen und Lederwaren"
5;"G47721";"G 47.72-1";"Einzelhandel mit Schuhen"
5;"G47722";"G 47.72-2";"Einzelhandel mit Leder- und Taschnerwaren"
4;"G4773";"G 47.73";"Apotheken"
5;"G47730";"G 47.73-0";"Apotheken"
4;"G4774";"G 47.74";"Einzelhandel mit medizinischen und orthopdischen Artikeln"
5;"G47740";"G 47.74-0";"Einzelhandel mit medizinischen und orthopdischen Artikeln"
4;"G4775";"G 47.75";"Einzelhandel mit kosmetischen Erzeugnissen und Krperpflegemitteln"
5;"G47750";"G 47.75-0";"Einzelhandel mit kosmetischen Erzeugnissen und Krperpflegemitteln"
4;"G4776";"G 47.76";"Einzelhandel mit Blumen, Pflanzen, Smereien, Dngemitteln, zoologischem Bedarf und lebenden Tieren"
5;"G47761";"G 47.76-1";"Einzelhandel mit Blumen, Pflanzen, Smereien, Dngemitteln; Blumenbinderei"
5;"G47762";"G 47.76-2";"Einzelhandel mit zoologischem Bedarf und lebenden Tieren"
4;"G4777";"G 47.77";"Einzelhandel mit Uhren und Schmuck"
5;"G47770";"G 47.77-0";"Einzelhandel mit Uhren und Schmuck"
4;"G4778";"G 47.78";"Sonstiger Einzelhandel in Verkaufsrumen (ohne Antiquitten und Gebrauchtwaren)"
5;"G47781";"G 47.78-1";"Einzelhandel mit Fotoartikeln, optischen und feinmechanischen Artikeln"
5;"G47782";"G 47.78-2";"Einzelhandel mit Brennstoffen"
5;"G47789";"G 47.78-9";"Sonstiger Einzelhandel a.n.g. in Verkaufsrumen (ohne Antiquitten und Gebrauchtwaren)"
4;"G4779";"G 47.79";"Einzelhandel mit Antiquitten und Gebrauchtwaren"
5;"G47790";"G 47.79-0";"Einzelhandel mit Antiquitten und Gebrauchtwaren"
3;"G478";"G 47.8";"Einzelhandel an Verkaufsstnden und auf Mrkten"
4;"G4781";"G 47.81";"Einzelhandel mit Nahrungs- und Genussmitteln, Getrnken und Tabakwaren an Verkaufsstnden und auf Mrkten"
5;"G47810";"G 47.81-0";"Einzelhandel mit Nahrungs- und Genussmitteln, Getrnken und Tabakwaren an Verkaufsstnden und auf Mrkten"
4;"G4782";"G 47.82";"Einzelhandel mit Textilien, Bekleidung und Schuhen an Verkaufsstnden und auf Mrkten"
5;"G47820";"G 47.82-0";"Einzelhandel mit Textilien, Bekleidung und Schuhen an Verkaufsstnden und auf Mrkten"
4;"G4789";"G 47.89";"Einzelhandel mit sonstigen Gtern an Verkaufsstnden und auf Mrkten"
5;"G47890";"G 47.89-0";"Einzelhandel mit sonstigen Gtern an Verkaufsstnden und auf Mrkten"
3;"G479";"G 47.9";"Einzelhandel, nicht in Verkaufsrumen, an Verkaufsstnden oder auf Mrkten"
4;"G4791";"G 47.91";"Versand- und Internet-Einzelhandel"
5;"G47910";"G 47.91-0";"Versand- und Internet-Einzelhandel"
4;"G4799";"G 47.99";"Sonstiger Einzelhandel, nicht in Verkaufsrumen, an Verkaufsstnden oder auf Mrkten"
5;"G47990";"G 47.99-0";"Sonstiger Einzelhandel, nicht in Verkaufsrumen, an Verkaufsstnden oder auf Mrkten"
1;"H";"H";"VERKEHR UND LAGEREI"
2;"H49";"H 49";"Landverkehr und Transport in Rohrfernleitungen"
3;"H491";"H 49.1";"Personenbefrderung im Eisenbahnfernverkehr"
4;"H4910";"H 49.10";"Personenbefrderung im Eisenbahnfernverkehr"
5;"H49100";"H 49.10-0";"Personenbefrderung im Eisenbahnfernverkehr"
3;"H492";"H 49.2";"Gterbefrderung im Eisenbahnverkehr"
4;"H4920";"H 49.20";"Gterbefrderung im Eisenbahnverkehr"
5;"H49200";"H 49.20-0";"Gterbefrderung im Eisenbahnverkehr"
3;"H493";"H 49.3";"Sonstige Personenbefrderung im Landverkehr"
4;"H4931";"H 49.31";"Personenbefrderung im Nahverkehr zu Lande (ohne Taxis)"
5;"H49311";"H 49.31-1";"Straenbahn-, O-Bus, U-Bahn- und Schnellbahnlinienverkehr"
5;"H49312";"H 49.31-2";"Autobusliniennahverkehr"
4;"H4932";"H 49.32";"Betrieb von Taxis"
5;"H49320";"H 49.32-0";"Betrieb von Taxis"
4;"H4939";"H 49.39";"Sonstige Personenbefrderung im Landverkehr a.n.g."
5;"H49391";"H 49.39-1";"Seilbahn-, Sessel- und Schleppliftverkehr"
5;"H49399";"H 49.39-9";"Sonstige Personenbefrderung im Landverkehr a.n.g. (ohne Seilbahnwirtschaft)"
3;"H494";"H 49.4";"Gterbefrderung im Straenverkehr, Umzugstransporte"
4;"H4941";"H 49.41";"Gterbefrderung im Straenverkehr"
5;"H49410";"H 49.41-0";"Gterbefrderung im Straenverkehr"
4;"H4942";"H 49.42";"Umzugstransporte"
5;"H49420";"H 49.42-0";"Umzugstransporte"
3;"H495";"H 49.5";"Transport in Rohrfernleitungen"
4;"H4950";"H 49.50";"Transport in Rohrfernleitungen"
5;"H49500";"H 49.50-0";"Transport in Rohrfernleitungen"
2;"H50";"H 50";"Schifffahrt"
3;"H501";"H 50.1";"Personenbefrderung in der See- und Kstenschifffahrt"
4;"H5010";"H 50.10";"Personenbefrderung in der See- und Kstenschifffahrt"
5;"H50100";"H 50.10-0";"Personenbefrderung in der See- und Kstenschifffahrt"
3;"H502";"H 50.2";"Gterbefrderung in der See- und Kstenschifffahrt"
4;"H5020";"H 50.20";"Gterbefrderung in der See- und Kstenschifffahrt"
5;"H50200";"H 50.20-0";"Gterbefrderung in der See- und Kstenschifffahrt"
3;"H503";"H 50.3";"Personenbefrderung in der Binnenschifffahrt"
4;"H5030";"H 50.30";"Personenbefrderung in der Binnenschifffahrt"
5;"H50300";"H 50.30-0";"Personenbefrderung in der Binnenschifffahrt"
3;"H504";"H 50.4";"Gterbefrderung in der Binnenschifffahrt"
4;"H5040";"H 50.40";"Gterbefrderung in der Binnenschifffahrt"
5;"H50400";"H 50.40-0";"Gterbefrderung in der Binnenschifffahrt"
2;"H51";"H 51";"Luftfahrt"
3;"H511";"H 51.1";"Personenbefrderung in der Luftfahrt"
4;"H5110";"H 51.10";"Personenbefrderung in der Luftfahrt"
5;"H51100";"H 51.10-0";"Personenbefrderung in der Luftfahrt"
3;"H512";"H 51.2";"Gterbefrderung in der Luftfahrt und Raumtransport"
4;"H5121";"H 51.21";"Gterbefrderung in der Luftfahrt"
5;"H51210";"H 51.21-0";"Gterbefrderung in der Luftfahrt"
4;"H5122";"H 51.22";"Raumtransport"
5;"H51220";"H 51.22-0";"Raumtransport"
2;"H52";"H 52";"Lagerei sowie Erbringung von sonstigen Dienstleistungen fr den Verkehr"
3;"H521";"H 52.1";"Lagerei"
4;"H5210";"H 52.10";"Lagerei"
5;"H52100";"H 52.10-0";"Lagerei"
3;"H522";"H 52.2";"Erbringung von sonstigen Dienstleistungen fr den Verkehr"
4;"H5221";"H 52.21";"Erbringung von sonstigen Dienstleistungen fr den Landverkehr"
5;"H52211";"H 52.21-1";"Parkhuser und Parkgaragen"
5;"H52212";"H 52.21-2";"Mautstraen"
5;"H52219";"H 52.21-9";"Erbringung von sonstigen Dienstleistungen fr den Landverkehr a.n.g."
4;"H5222";"H 52.22";"Erbringung von sonstigen Dienstleistungen fr die Schifffahrt"
5;"H52220";"H 52.22-0";"Erbringung von sonstigen Dienstleistungen fr die Schifffahrt"
4;"H5223";"H 52.23";"Erbringung von sonstigen Dienstleistungen fr die Luftfahrt"
5;"H52230";"H 52.23-0";"Erbringung von sonstigen Dienstleistungen fr die Luftfahrt"
4;"H5224";"H 52.24";"Frachtumschlag"
5;"H52240";"H 52.24-0";"Frachtumschlag"
4;"H5229";"H 52.29";"Erbringung von sonstigen Dienstleistungen fr den Verkehr a.n.g."
5;"H52290";"H 52.29-0";"Erbringung von sonstigen Dienstleistungen fr den Verkehr a.n.g."
2;"H53";"H 53";"Post-, Kurier- und Expressdienste"
3;"H531";"H 53.1";"Postdienste von Universaldienstleistungsanbietern"
4;"H5310";"H 53.10";"Postdienste von Universaldienstleistungsanbietern"
5;"H53100";"H 53.10-0";"Postdienste von Universaldienstleistungsanbietern"
3;"H532";"H 53.2";"Sonstige Post-, Kurier- und Expressdienste"
4;"H5320";"H 53.20";"Sonstige Post-, Kurier- und Expressdienste"
5;"H53200";"H 53.20-0";"Sonstige Post-, Kurier- und Expressdienste"
1;"I";"I";"BEHERBERGUNG UND GASTRONOMIE"
2;"I55";"I 55";"Beherbergung"
3;"I551";"I 55.1";"Hotels, Gasthfe und Pensionen"
4;"I5510";"I 55.10";"Hotels, Gasthfe und Pensionen"
5;"I55101";"I 55.10-1";"Hotels"
5;"I55102";"I 55.10-2";"Gasthfe"
5;"I55103";"I 55.10-3";"Pensionen"
3;"I552";"I 55.2";"Ferienunterknfte und hnliche Beherbergungssttten"
4;"I5520";"I 55.20";"Ferienunterknfte und hnliche Beherbergungssttten"
5;"I55201";"I 55.20-1";"Jugendherbergen"
5;"I55202";"I 55.20-2";"Schutzhtten"
5;"I55203";"I 55.20-3";"Ferienhuser und Ferienwohnungen"
3;"I553";"I 55.3";"Campingpltze"
4;"I5530";"I 55.30";"Campingpltze"
5;"I55300";"I 55.30-0";"Campingpltze"
3;"I559";"I 55.9";"Sonstige Beherbergungssttten"
4;"I5590";"I 55.90";"Sonstige Beherbergungssttten"
5;"I55900";"I 55.90-0";"Sonstige Beherbergungssttten"
2;"I56";"I 56";"Gastronomie"
3;"I561";"I 56.1";"Restaurants, Gaststtten, Imbissstuben, Cafs, Eissalons u. ."
4;"I5610";"I 56.10";"Restaurants, Gaststtten, Imbissstuben, Cafs, Eissalons u. ."
5;"I56101";"I 56.10-1";"Restaurants und Gaststtten"
5;"I56102";"I 56.10-2";"Buffets und Imbissstuben"
5;"I56103";"I 56.10-3";"Cafehuser"
5;"I56104";"I 56.10-4";"Eissalons"
3;"I562";"I 56.2";"Caterer und Erbringung sonstiger Verpflegungsdienstleistungen"
4;"I5621";"I 56.21";"Event-Caterer"
5;"I56210";"I 56.21-0";"Event-Caterer"
4;"I5629";"I 56.29";"Erbringung sonstiger Verpflegungsdienstleistungen"
5;"I56290";"I 56.29-0";"Erbringung sonstiger Verpflegungsdienstleistungen"
3;"I563";"I 56.3";"Ausschank von Getrnken"
4;"I5630";"I 56.30";"Ausschank von Getrnken"
5;"I56301";"I 56.30-1";"Bars und Diskotheken"
5;"I56309";"I 56.30-9";"Sonstiger Ausschank von Getrnken"
1;"J";"J";"INFORMATION UND KOMMUNIKATION"
2;"J58";"J 58";"Verlagswesen"
3;"J581";"J 58.1";"Verlegen von Bchern und Zeitschriften; sonstiges Verlagswesen (ohne Software)"
4;"J5811";"J 58.11";"Verlegen von Bchern"
5;"J58110";"J 58.11-0";"Verlegen von Bchern"
4;"J5812";"J 58.12";"Verlegen von Adressbchern und Verzeichnissen"
5;"J58120";"J 58.12-0";"Verlegen von Adressbchern und Verzeichnissen"
4;"J5813";"J 58.13";"Verlegen von Zeitungen"
5;"J58130";"J 58.13-0";"Verlegen von Zeitungen"
4;"J5814";"J 58.14";"Verlegen von Zeitschriften"
5;"J58140";"J 58.14-0";"Verlegen von Zeitschriften"
4;"J5819";"J 58.19";"Sonstiges Verlagswesen (ohne Software)"
5;"J58190";"J 58.19-0";"Sonstiges Verlagswesen (ohne Software)"
3;"J582";"J 58.2";"Verlegen von Software"
4;"J5821";"J 58.21";"Verlegen von Computerspielen"
5;"J58210";"J 58.21-0";"Verlegen von Computerspielen"
4;"J5829";"J 58.29";"Verlegen von sonstiger Software"
5;"J58290";"J 58.29-0";"Verlegen von sonstiger Software"
2;"J59";"J 59";"Herstellung, Verleih und Vertrieb von Filmen und Fernsehprogrammen; Kinos; Tonstudios und Verlegen von Musik"
3;"J591";"J 59.1";"Herstellung von Filmen und Fernsehprogrammen, deren Verleih und Vertrieb, Kinos"
4;"J5911";"J 59.11";"Herstellung von Filmen, Videofilmen und Fernsehprogrammen"
5;"J59110";"J 59.11-0";"Herstellung von Filmen, Videofilmen und Fernsehprogrammen"
4;"J5912";"J 59.12";"Nachbearbeitung und sonstige Filmtechnik"
5;"J59120";"J 59.12-0";"Nachbearbeitung und sonstige Filmtechnik"
4;"J5913";"J 59.13";"Filmverleih und -vertrieb (ohne Videotheken)"
5;"J59130";"J 59.13-0";"Filmverleih und -vertrieb (ohne Videotheken)"
4;"J5914";"J 59.14";"Kinos"
5;"J59140";"J 59.14-0";"Kinos"
3;"J592";"J 59.2";"Tonstudios; Herstellung von Hrfunkbeitrgen; Verlegen von bespielten Tontrgern und Musikalien"
4;"J5920";"J 59.20";"Tonstudios; Herstellung von Hrfunkbeitrgen; Verlegen von bespielten Tontrgern und Musikalien"
5;"J59200";"J 59.20-0";"Tonstudios; Herstellung von Hrfunkbeitrgen; Verlegen von bespielten Tontrgern und Musikalien"
2;"J60";"J 60";"Rundfunkveranstalter"
3;"J601";"J 60.1";"Hrfunkveranstalter"
4;"J6010";"J 60.10";"Hrfunkveranstalter"
5;"J60100";"J 60.10-0";"Hrfunkveranstalter"
3;"J602";"J 60.2";"Fernsehveranstalter"
4;"J6020";"J 60.20";"Fernsehveranstalter"
5;"J60200";"J 60.20-0";"Fernsehveranstalter"
2;"J61";"J 61";"Telekommunikation"
3;"J611";"J 61.1";"Leitungsgebundene Telekommunikation"
4;"J6110";"J 61.10";"Leitungsgebundene Telekommunikation"
5;"J61100";"J 61.10-0";"Leitungsgebundene Telekommunikation"
3;"J612";"J 61.2";"Drahtlose Telekommunikation"
4;"J6120";"J 61.20";"Drahtlose Telekommunikation"
5;"J61200";"J 61.20-0";"Drahtlose Telekommunikation"
3;"J613";"J 61.3";"Satellitentelekommunikation"
4;"J6130";"J 61.30";"Satellitentelekommunikation"
5;"J61300";"J 61.30-0";"Satellitentelekommunikation"
3;"J619";"J 61.9";"Sonstige Telekommunikation"
4;"J6190";"J 61.90";"Sonstige Telekommunikation"
5;"J61900";"J 61.90-0";"Sonstige Telekommunikation"
2;"J62";"J 62";"Erbringung von Dienstleistungen der Informationstechnologie"
3;"J620";"J 62.0";"Erbringung von Dienstleistungen der Informationstechnologie"
4;"J6201";"J 62.01";"Programmierungsttigkeiten"
5;"J62010";"J 62.01-0";"Programmierungsttigkeiten"
4;"J6202";"J 62.02";"Erbringung von Beratungsleistungen auf dem Gebiet der Informationstechnologie"
5;"J62020";"J 62.02-0";"Erbringung von Beratungsleistungen auf dem Gebiet der Informationstechnologie"
4;"J6203";"J 62.03";"Betrieb von Datenverarbeitungseinrichtungen fr Dritte"
5;"J62030";"J 62.03-0";"Betrieb von Datenverarbeitungseinrichtungen fr Dritte"
4;"J6209";"J 62.09";"Erbringung von sonstigen Dienstleistungen der Informationstechnologie"
5;"J62090";"J 62.09-0";"Erbringung von sonstigen Dienstleistungen der Informationstechnologie"
2;"J63";"J 63";"Informationsdienstleistungen"
3;"J631";"J 63.1";"Datenverarbeitung, Hosting und damit verbundene Ttigkeiten; Webportale"
4;"J6311";"J 63.11";"Datenverarbeitung, Hosting und damit verbundene Ttigkeiten"
5;"J63110";"J 63.11-0";"Datenverarbeitung, Hosting und damit verbundene Ttigkeiten"
4;"J6312";"J 63.12";"Webportale"
5;"J63120";"J 63.12-0";"Webportale"
3;"J639";"J 63.9";"Erbringung von sonstigen Informationsdienstleistungen"
4;"J6391";"J 63.91";"Korrespondenz- und Nachrichtenbros"
5;"J63910";"J 63.91-0";"Korrespondenz- und Nachrichtenbros"
4;"J6399";"J 63.99";"Erbringung von sonstigen Informationsdienstleistungen a.n.g."
5;"J63990";"J 63.99-0";"Erbringung von sonstigen Informationsdienstleistungen a.n.g."
1;"K";"K";"ERBRINGUNG VON FINANZ- UND VERSICHERUNGSDIENSTLEISTUNGEN"
2;"K64";"K 64";"Erbringung von Finanzdienstleistungen"
3;"K641";"K 64.1";"Zentralbanken und Kreditinstitute"
4;"K6411";"K 64.11";"Zentralbanken"
5;"K64110";"K 64.11-0";"Zentralbanken"
4;"K6419";"K 64.19";"Kreditinstitute (ohne Spezialkreditinstitute)"
5;"K64190";"K 64.19-0";"Kreditinstitute (ohne Spezialkreditinstitute)"
3;"K642";"K 64.2";"Beteiligungsgesellschaften"
4;"K6420";"K 64.20";"Beteiligungsgesellschaften"
5;"K64200";"K 64.20-0";"Beteiligungsgesellschaften"
3;"K643";"K 64.3";"Treuhand- und sonstige Fonds und hnliche Finanzinstitutionen"
4;"K6430";"K 64.30";"Treuhand- und sonstige Fonds und hnliche Finanzinstitutionen"
5;"K64301";"K 64.30-1";"Investment- und Immobilienfonds"
5;"K64309";"K 64.30-9";"Treuhand- und sonstige Fonds a.n.g. und hnliche Finanzinstitutionen"
3;"K649";"K 64.9";"Sonstige Finanzierungsinstitutionen"
4;"K6491";"K 64.91";"Institutionen fr Finanzierungsleasing"
5;"K64910";"K 64.91-0";"Institutionen fr Finanzierungsleasing"
4;"K6492";"K 64.92";"Spezialkreditinstitute"
5;"K64920";"K 64.92-0";"Spezialkreditinstitute"
4;"K6499";"K 64.99";"Sonstige Finanzdienstleistungen a.n.g."
5;"K64990";"K 64.99-0";"Sonstige Finanzdienstleistungen a.n.g."
2;"K65";"K 65";"Versicherungen, Rckversicherungen und Pensionskassen (ohne Sozialversicherung)"
3;"K651";"K 65.1";"Versicherungen"
4;"K6511";"K 65.11";"Lebensversicherung"
5;"K65110";"K 65.11-0";"Lebensversicherung"
4;"K6512";"K 65.12";"Nichtlebensversicherungen"
5;"K65120";"K 65.12-0";"Nichtlebensversicherungen"
3;"K652";"K 65.2";"Rckversicherungen"
4;"K6520";"K 65.20";"Rckversicherungen"
5;"K65200";"K 65.20-0";"Rckversicherungen"
3;"K653";"K 65.3";"Pensionskassen und Pensionsfonds"
4;"K6530";"K 65.30";"Pensionskassen und Pensionsfonds"
5;"K65300";"K 65.30-0";"Pensionskassen und Pensionsfonds"
2;"K66";"K 66";"Mit Finanz- und Versicherungsdienstleistungen verbundene Ttigkeiten"
3;"K661";"K 66.1";"Mit Finanzdienstleistungen verbundene Ttigkeiten"
4;"K6611";"K 66.11";"Effekten- und Warenbrsen"
5;"K66110";"K 66.11-0";"Effekten- und Warenbrsen"
4;"K6612";"K 66.12";"Effekten- und Warenhandel"
5;"K66120";"K 66.12-0";"Effekten- und Warenhandel"
4;"K6619";"K 66.19";"Sonstige mit Finanzdienstleistungen verbundene Ttigkeiten"
5;"K66190";"K 66.19-0";"Sonstige mit Finanzdienstleistungen verbundene Ttigkeiten"
3;"K662";"K 66.2";"Mit Versicherungsdienstleistungen und Pensionskassen verbundene Ttigkeiten"
4;"K6621";"K 66.21";"Risiko- und Schadensbewertung"
5;"K66210";"K 66.21-0";"Risiko- und Schadensbewertung"
4;"K6622";"K 66.22";"Ttigkeit von Versicherungsmaklerinnen und Versicherungsmaklern"
5;"K66220";"K 66.22-0";"Ttigkeit von Versicherungsmaklerinnen und Versicherungsmaklern"
4;"K6629";"K 66.29";"Sonstige mit Versicherungsdienstleistungen und Pensionskassen verbundene Ttigkeiten"
5;"K66290";"K 66.29-0";"Sonstige mit Versicherungsdienstleistungen und Pensionskassen verbundene Ttigkeiten"
3;"K663";"K 66.3";"Fondsmanagement"
4;"K6630";"K 66.30";"Fondsmanagement"
5;"K66300";"K 66.30-0";"Fondsmanagement"
1;"L";"L";"GRUNDSTCKS- UND WOHNUNGSWESEN"
2;"L68";"L 68";"Grundstcks- und Wohnungswesen"
3;"L681";"L 68.1";"Kauf und Verkauf von eigenen Grundstcken, Gebuden und Wohnungen"
4;"L6810";"L 68.10";"Kauf und Verkauf von eigenen Grundstcken, Gebuden und Wohnungen"
5;"L68100";"L 68.10-0";"Kauf und Verkauf von eigenen Grundstcken, Gebuden und Wohnungen"
3;"L682";"L 68.2";"Vermietung, Verpachtung von eigenen oder geleasten Grundstcken, Gebuden und Wohnungen"
4;"L6820";"L 68.20";"Vermietung, Verpachtung von eigenen oder geleasten Grundstcken, Gebuden und Wohnungen"
5;"L68201";"L 68.20-1";"Leasing von Grundstcken, Gebuden und Wohnungen"
5;"L68209";"L 68.20-9";"Sonstige Vermietung und Verpachtung von eigenen oder geleasten Grundstcken, Gebuden und Wohnungen"
3;"L683";"L 68.3";"Vermittlung und Verwaltung von Grundstcken, Gebuden und Wohnungen fr Dritte"
4;"L6831";"L 68.31";"Vermittlung von Grundstcken, Gebuden und Wohnungen fr Dritte"
5;"L68310";"L 68.31-0";"Vermittlung von Grundstcken, Gebuden und Wohnungen fr Dritte"
4;"L6832";"L 68.32";"Verwaltung von Grundstcken, Gebuden und Wohnungen fr Dritte"
5;"L68320";"L 68.32-0";"Verwaltung von Grundstcken, Gebuden und Wohnungen fr Dritte"
1;"M";"M";"ERBRINGUNG VON FREIBERUFLICHEN, WISSENSCHAFTLICHEN UND TECHNISCHEN DIENSTLEISTUNGEN"
2;"M69";"M 69";"Rechts- und Steuerberatung, Wirtschaftsprfung"
3;"M691";"M 69.1";"Rechtsberatung"
4;"M6910";"M 69.10";"Rechtsberatung"
5;"M69100";"M 69.10-0";"Rechtsberatung"
3;"M692";"M 69.2";"Wirtschaftsprfung und Steuerberatung; Buchfhrung"
4;"M6920";"M 69.20";"Wirtschaftsprfung und Steuerberatung; Buchfhrung"
5;"M69200";"M 69.20-0";"Wirtschaftsprfung und Steuerberatung; Buchfhrung"
2;"M70";"M 70";"Verwaltung und Fhrung von Unternehmen und Betrieben; Unternehmensberatung"
3;"M701";"M 70.1";"Verwaltung und Fhrung von Unternehmen und Betrieben"
4;"M7010";"M 70.10";"Verwaltung und Fhrung von Unternehmen und Betrieben"
5;"M70100";"M 70.10-0";"Verwaltung und Fhrung von Unternehmen und Betrieben"
3;"M702";"M 70.2";"Public-Relations- und Unternehmensberatung"
4;"M7021";"M 70.21";"Public-Relations-Beratung"
5;"M70210";"M 70.21-0";"Public-Relations-Beratung"
4;"M7022";"M 70.22";"Unternehmensberatung"
5;"M70220";"M 70.22-0";"Unternehmensberatung"
2;"M71";"M 71";"Architektur- und Ingenieurbros; technische, physikalische und chemische Untersuchung"
3;"M711";"M 71.1";"Architektur- und Ingenieurbros"
4;"M7111";"M 71.11";"Architekturbros"
5;"M71110";"M 71.11-0";"Architekturbros"
4;"M7112";"M 71.12";"Ingenieurbros"
5;"M71120";"M 71.12-0";"Ingenieurbros"
3;"M712";"M 71.2";"Technische, physikalische und chemische Untersuchung"
4;"M7120";"M 71.20";"Technische, physikalische und chemische Untersuchung"
5;"M71200";"M 71.20-0";"Technische, physikalische und chemische Untersuchung"
2;"M72";"M 72";"Forschung und Entwicklung"
3;"M721";"M 72.1";"Forschung und Entwicklung im Bereich Natur-, Ingenieur-, Agrarwissenschaften und Medizin"
4;"M7211";"M 72.11";"Forschung und Entwicklung im Bereich Biotechnologie"
5;"M72110";"M 72.11-0";"Forschung und Entwicklung im Bereich Biotechnologie"
4;"M7219";"M 72.19";"Sonstige Forschung und Entwicklung im Bereich Natur-, Ingenieur-, Agrarwissenschaften und Medizin"
5;"M72190";"M 72.19-0";"Sonstige Forschung und Entwicklung im Bereich Natur-, Ingenieur-, Agrarwissenschaften und Medizin"
3;"M722";"M 72.2";"Forschung und Entwicklung im Bereich Rechts-, Wirtschafts- und Sozialwissenschaften sowie im Bereich Sprach-, Kultur- und Kunstwissenschaften"
4;"M7220";"M 72.20";"Forschung und Entwicklung im Bereich Rechts-, Wirtschafts- und Sozialwissenschaften sowie im Bereich Sprach-, Kultur- und Kunstwissenschaften"
5;"M72200";"M 72.20-0";"Forschung und Entwicklung im Bereich Rechts-, Wirtschafts- und Sozialwissenschaften sowie im Bereich Sprach-, Kultur- und Kunstwissenschaften"
2;"M73";"M 73";"Werbung und Marktforschung"
3;"M731";"M 73.1";"Werbung"
4;"M7311";"M 73.11";"Werbeagenturen"
5;"M73111";"M 73.11-1";"Werbegestaltung"
5;"M73112";"M 73.11-2";"Werbemittelverbreitung"
4;"M7312";"M 73.12";"Vermarktung und Vermittlung von Werbezeiten und Werbeflchen"
5;"M73120";"M 73.12-0";"Vermarktung und Vermittlung von Werbezeiten und Werbeflchen"
3;"M732";"M 73.2";"Markt- und Meinungsforschung"
4;"M7320";"M 73.20";"Markt- und Meinungsforschung"
5;"M73200";"M 73.20-0";"Markt- und Meinungsforschung"
2;"M74";"M 74";"Sonstige freiberufliche, wissenschaftliche und technische Ttigkeiten"
3;"M741";"M 74.1";"Ateliers fr Textil-, Schmuck-, Grafik- u. . Design"
4;"M7410";"M 74.10";"Ateliers fr Textil-, Schmuck-, Grafik- u. . Design"
5;"M74100";"M 74.10-0";"Ateliers fr Textil-, Schmuck-, Grafik- u. . Design"
3;"M742";"M 74.2";"Fotografie und Fotolabors"
4;"M7420";"M 74.20";"Fotografie und Fotolabors"
5;"M74200";"M 74.20-0";"Fotografie und Fotolabors"
3;"M743";"M 74.3";"bersetzen und Dolmetschen"
4;"M7430";"M 74.30";"bersetzen und Dolmetschen"
5;"M74300";"M 74.30-0";"bersetzen und Dolmetschen"
3;"M749";"M 74.9";"Sonstige freiberufliche, wissenschaftliche und technische Ttigkeiten a. n. g."
4;"M7490";"M 74.90";"Sonstige freiberufliche, wissenschaftliche und technische Ttigkeiten a. n. g."
5;"M74900";"M 74.90-0";"Sonstige freiberufliche, wissenschaftliche und technische Ttigkeiten a. n. g."
2;"M75";"M 75";"Veterinrwesen"
3;"M750";"M 75.0";"Veterinrwesen"
4;"M7500";"M 75.00";"Veterinrwesen"
5;"M75000";"M 75.00-0";"Veterinrwesen"
1;"N";"N";"ERBRINGUNG VON SONSTIGEN WIRTSCHAFTLICHEN DIENSTLEISTUNGEN"
2;"N77";"N 77";"Vermietung von beweglichen Sachen"
3;"N771";"N 77.1";"Vermietung von Kraftwagen"
4;"N7711";"N 77.11";"Vermietung von Kraftwagen mit einem Gesamtgewicht von 3,5 t oder weniger"
5;"N77111";"N 77.11-1";"Leasing von Kraftwagen mit einem Gesamtgewicht von 3,5 t oder weniger"
5;"N77112";"N 77.11-2";"Vermietung von Kraftwagen mit einem Gesamtgewicht von 3,5 t oder weniger"
4;"N7712";"N 77.12";"Vermietung von Kraftwagen mit einem Gesamtgewicht von mehr als 3,5 t"
5;"N77120";"N 77.12-0";"Vermietung von Kraftwagen mit einem Gesamtgewicht von mehr als 3,5 t"
3;"N772";"N 77.2";"Vermietung von Gebrauchsgtern"
4;"N7721";"N 77.21";"Vermietung von Sport- und Freizeitgerten"
5;"N77210";"N 77.21-0";"Vermietung von Sport- und Freizeitgerten"
4;"N7722";"N 77.22";"Videotheken"
5;"N77220";"N 77.22-0";"Videotheken"
4;"N7729";"N 77.29";"Vermietung von sonstigen Gebrauchsgtern"
5;"N77290";"N 77.29-0";"Vermietung von sonstigen Gebrauchsgtern"
3;"N773";"N 77.3";"Vermietung von Maschinen, Gerten und sonstigen beweglichen Sachen"
4;"N7731";"N 77.31";"Vermietung von landwirtschaftlichen Maschinen und Gerten"
5;"N77310";"N 77.31-0";"Vermietung von landwirtschaftlichen Maschinen und Gerten"
4;"N7732";"N 77.32";"Vermietung von Baumaschinen und -gerten"
5;"N77320";"N 77.32-0";"Vermietung von Baumaschinen und -gerten"
4;"N7733";"N 77.33";"Vermietung von Bromaschinen, Datenverarbeitungsgerten und -einrichtungen"
5;"N77330";"N 77.33-0";"Vermietung von Bromaschinen, Datenverarbeitungsgerten und -einrichtungen"
4;"N7734";"N 77.34";"Vermietung von Wasserfahrzeugen"
5;"N77340";"N 77.34-0";"Vermietung von Wasserfahrzeugen"
4;"N7735";"N 77.35";"Vermietung von Luftfahrzeugen"
5;"N77350";"N 77.35-0";"Vermietung von Luftfahrzeugen"
4;"N7739";"N 77.39";"Vermietung von sonstigen Maschinen, Gerten und beweglichen Sachen a. n. g."
5;"N77390";"N 77.39-0";"Vermietung von sonstigen Maschinen, Gerten und beweglichen Sachen a. n. g."
3;"N774";"N 77.4";"Leasing von nichtfinanziellen immateriellen Vermgensgegenstnden (ohne Copyrights)"
4;"N7740";"N 77.40";"Leasing von nichtfinanziellen immateriellen Vermgensgegenstnden (ohne Copyrights)"
5;"N77400";"N 77.40-0";"Leasing von nichtfinanziellen immateriellen Vermgensgegenstnden (ohne Copyrights)"
2;"N78";"N 78";"Vermittlung und berlassung von Arbeitskrften"
3;"N781";"N 78.1";"Vermittlung von Arbeitskrften"
4;"N7810";"N 78.10";"Vermittlung von Arbeitskrften"
5;"N78100";"N 78.10-0";"Vermittlung von Arbeitskrften"
3;"N782";"N 78.2";"Befristete berlassung von Arbeitskrften"
4;"N7820";"N 78.20";"Befristete berlassung von Arbeitskrften"
5;"N78200";"N 78.20-0";"Befristete berlassung von Arbeitskrften"
3;"N783";"N 78.3";"Sonstige berlassung von Arbeitskrften"
4;"N7830";"N 78.30";"Sonstige berlassung von Arbeitskrften"
5;"N78300";"N 78.30-0";"Sonstige berlassung von Arbeitskrften"
2;"N79";"N 79";"Reisebros, Reiseveranstalter und Erbringung sonstiger Reservierungsdienstleistungen"
3;"N791";"N 79.1";"Reisebros und Reiseveranstalter"
4;"N7911";"N 79.11";"Reisebros"
5;"N79110";"N 79.11-0";"Reisebros"
4;"N7912";"N 79.12";"Reiseveranstalter"
5;"N79120";"N 79.12-0";"Reiseveranstalter"
3;"N799";"N 79.9";"Erbringung sonstiger Reservierungsdienstleistungen"
4;"N7990";"N 79.90";"Erbringung sonstiger Reservierungsdienstleistungen"
5;"N79901";"N 79.90-1";"Reise- und Fremdenfhrer"
5;"N79902";"N 79.90-2";"Tourismusbros"
5;"N79909";"N 79.90-9";"Erbringung sonstiger Reservierungsdienstleistungen a.n.g."
2;"N80";"N 80";"Wach- und Sicherheitsdienste sowie Detekteien"
3;"N801";"N 80.1";"Private Wach- und Sicherheitsdienste"
4;"N8010";"N 80.10";"Private Wach- und Sicherheitsdienste"
5;"N80100";"N 80.10-0";"Private Wach- und Sicherheitsdienste"
3;"N802";"N 80.2";"Sicherheitsdienste mithilfe von berwachungs- und Alarmsystemen"
4;"N8020";"N 80.20";"Sicherheitsdienste mithilfe von berwachungs- und Alarmsystemen"
5;"N80200";"N 80.20-0";"Sicherheitsdienste mithilfe von berwachungs- und Alarmsystemen"
3;"N803";"N 80.3";"Detekteien"
4;"N8030";"N 80.30";"Detekteien"
5;"N80300";"N 80.30-0";"Detekteien"
2;"N81";"N 81";"Gebudebetreuung; Garten- und Landschaftsbau"
3;"N811";"N 81.1";"Hausmeisterdienste"
4;"N8110";"N 81.10";"Hausmeisterdienste"
5;"N81101";"N 81.10-1";"Hauswartung"
5;"N81102";"N 81.10-2";"Facility Management (ohne Hauswartung)"
3;"N812";"N 81.2";"Reinigung von Gebuden, Straen und Verkehrsmitteln"
4;"N8121";"N 81.21";"Allgemeine Gebudereinigung"
5;"N81210";"N 81.21-0";"Allgemeine Gebudereinigung"
4;"N8122";"N 81.22";"Spezielle Reinigung von Gebuden und Reinigung von Maschinen"
5;"N81221";"N 81.22-1";"Rauchfangkehrerinnen und -kehrer"
5;"N81229";"N 81.22-9";"Spezielle Reinigung von Gebuden und Maschinen a.n.g."
4;"N8129";"N 81.29";"Reinigung a. n. g."
5;"N81291";"N 81.29-1";"Schneerumung"
5;"N81299";"N 81.29-9";"Sonstige Reinigung a.n.g."
3;"N813";"N 81.3";"Garten- und Landschaftsbau sowie Erbringung von sonstigen grtnerischen Dienstleistungen"
4;"N8130";"N 81.30";"Garten- und Landschaftsbau sowie Erbringung von sonstigen grtnerischen Dienstleistungen"
5;"N81300";"N 81.30-0";"Garten- und Landschaftsbau sowie Erbringung von sonstigen grtnerischen Dienstleistungen"
2;"N82";"N 82";"Erbringung von wirtschaftlichen Dienstleistungen fr Unternehmen und Privatpersonen a. n. g."
3;"N821";"N 82.1";"Sekretariats- und Schreibdienste, Copy-Shops"
4;"N8211";"N 82.11";"Allgemeine Sekretariats- und Schreibdienste"
5;"N82110";"N 82.11-0";"Allgemeine Sekretariats- und Schreibdienste"
4;"N8219";"N 82.19";"Copy-Shops; Dokumentenvorbereitung und Erbringung sonstiger spezieller Sekretariatsdienste"
5;"N82190";"N 82.19-0";"Copy-Shops; Dokumentenvorbereitung und Erbringung sonstiger spezieller Sekretariatsdienste"
3;"N822";"N 82.2";"Call Centers"
4;"N8220";"N 82.20";"Call Centers"
5;"N82200";"N 82.20-0";"Call Centers"
3;"N823";"N 82.3";"Messe-, Ausstellungs- und Kongressveranstalter"
4;"N8230";"N 82.30";"Messe-, Ausstellungs- und Kongressveranstalter"
5;"N82300";"N 82.30-0";"Messe-, Ausstellungs- und Kongressveranstalter"
3;"N829";"N 82.9";"Erbringung sonstiger wirtschaftlicher Dienstleistungen fr Unternehmen und Privatpersonen"
4;"N8291";"N 82.91";"Inkassobros und Auskunfteien"
5;"N82910";"N 82.91-0";"Inkassobros und Auskunfteien"
4;"N8292";"N 82.92";"Abfllen und Verpacken"
5;"N82920";"N 82.92-0";"Abfllen und Verpacken"
4;"N8299";"N 82.99";"Erbringung sonstiger wirtschaftlicher Dienstleistungen fr Unternehmen und Privatpersonen a. n. g."
5;"N82990";"N 82.99-0";"Erbringung sonstiger wirtschaftlicher Dienstleistungen fr Unternehmen und Privatpersonen a. n. g."
1;"O";"O";"FFENTLICHE VERWALTUNG, VERTEIDIGUNG; SOZIALVERSICHERUNG"
2;"O84";"O 84";"ffentliche Verwaltung, Verteidigung; Sozialversicherung"
3;"O841";"O 84.1";"ffentliche Verwaltung"
4;"O8411";"O 84.11";"Allgemeine ffentliche Verwaltung"
5;"O84110";"O 84.11-0";"Allgemeine ffentliche Verwaltung"
4;"O8412";"O 84.12";"ffentliche Verwaltung auf den Gebieten Gesundheitswesen, Bildung, Kultur und Sozialwesen"
5;"O84120";"O 84.12-0";"ffentliche Verwaltung auf den Gebieten Gesundheitswesen, Bildung, Kultur und Sozialwesen"
4;"O8413";"O 84.13";"Wirtschaftsfrderung, -ordnung und -aufsicht"
5;"O84130";"O 84.13-0";"Wirtschaftsfrderung, -ordnung und -aufsicht"
3;"O842";"O 84.2";"Auswrtige Angelegenheiten, Verteidigung, Justiz, ffentliche Sicherheit und Ordnung"
4;"O8421";"O 84.21";"Auswrtige Angelegenheiten"
5;"O84210";"O 84.21-0";"Auswrtige Angelegenheiten"
4;"O8422";"O 84.22";"Verteidigung"
5;"O84220";"O 84.22-0";"Verteidigung"
4;"O8423";"O 84.23";"Justiz"
5;"O84230";"O 84.23-0";"Justiz"
4;"O8424";"O 84.24";"ffentliche Sicherheit und Ordnung"
5;"O84240";"O 84.24-0";"ffentliche Sicherheit und Ordnung"
4;"O8425";"O 84.25";"Feuerwehren"
5;"O84250";"O 84.25-0";"Feuerwehren"
3;"O843";"O 84.3";"Sozialversicherung"
4;"O8430";"O 84.30";"Sozialversicherung"
5;"O84300";"O 84.30-0";"Sozialversicherung"
1;"P";"P";"ERZIEHUNG UND UNTERRICHT"
2;"P85";"P 85";"Erziehung und Unterricht"
3;"P851";"P 85.1";"Kindergrten und Vorschulen"
4;"P8510";"P 85.10";"Kindergrten und Vorschulen"
5;"P85100";"P 85.10-0";"Kindergrten und Vorschulen"
3;"P852";"P 85.2";"Volksschulen"
4;"P8520";"P 85.20";"Volksschulen"
5;"P85200";"P 85.20-0";"Volksschulen"
3;"P853";"P 85.3";"Weiterfhrende Schulen"
4;"P8531";"P 85.31";"Allgemein bildende weiterfhrende Schulen"
5;"P85311";"P 85.31-1";"Hauptschulen und Polytechnische Lehrgnge"
5;"P85312";"P 85.31-2";"Allgemein bildende hhere Schulen"
4;"P8532";"P 85.32";"Berufsbildende weiterfhrende Schulen"
5;"P85321";"P 85.32-1";"Berufsbildende Pflichtschulen"
5;"P85322";"P 85.32-2";"Berufsbildende mittlere Schulen"
5;"P85323";"P 85.32-3";"Berufsbildende hhere Schulen"
3;"P854";"P 85.4";"Tertirer und post-sekundrer, nicht tertirer Unterricht"
4;"P8541";"P 85.41";"Post-sekundrer, nicht tertirer Unterricht"
5;"P85410";"P 85.41-0";"Post-sekundrer, nicht tertirer Unterricht"
4;"P8542";"P 85.42";"Tertirer Unterricht"
5;"P85420";"P 85.42-0";"Tertirer Unterricht"
3;"P855";"P 85.5";"Sonstiger Unterricht"
4;"P8551";"P 85.51";"Sport- und Freizeitunterricht"
5;"P85510";"P 85.51-0";"Sport- und Freizeitunterricht"
4;"P8552";"P 85.52";"Kulturunterricht"
5;"P85521";"P 85.52-1";"Tanzschulen"
5;"P85529";"P 85.52-9";"Sonstiger Kulturunterricht"
4;"P8553";"P 85.53";"Fahr- und Flugschulen"
5;"P85530";"P 85.53-0";"Fahr- und Flugschulen"
4;"P8559";"P 85.59";"Sonstiger Unterricht a. n. g."
5;"P85590";"P 85.59-0";"Sonstiger Unterricht a. n. g."
3;"P856";"P 85.6";"Erbringung von Dienstleistungen fr den Unterricht"
4;"P8560";"P 85.60";"Erbringung von Dienstleistungen fr den Unterricht"
5;"P85600";"P 85.60-0";"Erbringung von Dienstleistungen fr den Unterricht"
1;"Q";"Q";"GESUNDHEITS- UND SOZIALWESEN"
2;"Q86";"Q 86";"Gesundheitswesen"
3;"Q861";"Q 86.1";"Krankenhuser"
4;"Q8610";"Q 86.10";"Krankenhuser"
5;"Q86100";"Q 86.10-0";"Krankenhuser"
3;"Q862";"Q 86.2";"Arzt- und Zahnarztpraxen"
4;"Q8621";"Q 86.21";"Arztpraxen fr Allgemeinmedizin"
5;"Q86210";"Q 86.21-0";"Arztpraxen fr Allgemeinmedizin"
4;"Q8622";"Q 86.22";"Facharztpraxen"
5;"Q86220";"Q 86.22-0";"Facharztpraxen"
4;"Q8623";"Q 86.23";"Zahnarztpraxen"
5;"Q86230";"Q 86.23-0";"Zahnarztpraxen"
3;"Q869";"Q 86.9";"Gesundheitswesen a. n. g."
4;"Q8690";"Q 86.90";"Gesundheitswesen a. n. g."
5;"Q86901";"Q 86.90-1";"Krankentransporte und Rettungsdienste"
5;"Q86902";"Q 86.90-2";"Hauskrankenpflege"
5;"Q86903";"Q 86.90-3";"Psychotherapeutinnen und -therapeuten, Klinische und Gesundheitspsychologinnen und -psychologen"
5;"Q86909";"Q 86.90-9";"Sonstiges Gesundheitswesen a.n.g."
2;"Q87";"Q 87";"Heime (ohne Erholungs- und Ferienheime)"
3;"Q871";"Q 87.1";"Pflegeheime"
4;"Q8710";"Q 87.10";"Pflegeheime"
5;"Q87100";"Q 87.10-0";"Pflegeheime"
3;"Q872";"Q 87.2";"Stationre Einrichtungen zur psychosozialen Betreuung, Suchtbekmpfung u. ."
4;"Q8720";"Q 87.20";"Stationre Einrichtungen zur psychosozialen Betreuung, Suchtbekmpfung u. ."
5;"Q87200";"Q 87.20-0";"Stationre Einrichtungen zur psychosozialen Betreuung, Suchtbekmpfung u. ."
3;"Q873";"Q 87.3";"Altenheime; Alten- und Behindertenwohnheime"
4;"Q8730";"Q 87.30";"Altenheime; Alten- und Behindertenwohnheime"
5;"Q87300";"Q 87.30-0";"Altenheime; Alten- und Behindertenwohnheime"
3;"Q879";"Q 87.9";"Sonstige Heime (ohne Erholungs- und Ferienheime)"
4;"Q8790";"Q 87.90";"Sonstige Heime (ohne Erholungs- und Ferienheime)"
5;"Q87900";"Q 87.90-0";"Sonstige Heime (ohne Erholungs- und Ferienheime)"
2;"Q88";"Q 88";"Sozialwesen (ohne Heime)"
3;"Q881";"Q 88.1";"Soziale Betreuung lterer Menschen und Behinderter"
4;"Q8810";"Q 88.10";"Soziale Betreuung lterer Menschen und Behinderter"
5;"Q88100";"Q 88.10-0";"Soziale Betreuung lterer Menschen und Behinderter"
3;"Q889";"Q 88.9";"Sonstiges Sozialwesen (ohne Heime)"
4;"Q8891";"Q 88.91";"Tagesbetreuung von Kindern"
5;"Q88910";"Q 88.91-0";"Tagesbetreuung von Kindern"
4;"Q8899";"Q 88.99";"Sonstiges Sozialwesen a. n. g."
5;"Q88990";"Q 88.99-0";"Sonstiges Sozialwesen a. n. g."
1;"R";"R";"KUNST, UNTERHALTUNG UND ERHOLUNG"
2;"R90";"R 90";"Kreative, knstlerische und unterhaltende Ttigkeiten"
3;"R900";"R 90.0";"Kreative, knstlerische und unterhaltende Ttigkeiten"
4;"R9001";"R 90.01";"Darstellende Kunst"
5;"R90010";"R 90.01-0";"Darstellende Kunst"
4;"R9002";"R 90.02";"Erbringung von Dienstleistungen fr die darstellende Kunst"
5;"R90020";"R 90.02-0";"Erbringung von Dienstleistungen fr die darstellende Kunst"
4;"R9003";"R 90.03";"Knstlerisches und schriftstellerisches Schaffen"
5;"R90030";"R 90.03-0";"Knstlerisches und schriftstellerisches Schaffen"
4;"R9004";"R 90.04";"Betrieb von Kultur- und Unterhaltungseinrichtungen"
5;"R90040";"R 90.04-0";"Betrieb von Kultur- und Unterhaltungseinrichtungen"
2;"R91";"R 91";"Bibliotheken, Archive, Museen, botanische und zoologische Grten"
3;"R910";"R 91.0";"Bibliotheken, Archive, Museen, botanische und zoologische Grten"
4;"R9101";"R 91.01";"Bibliotheken und Archive"
5;"R91010";"R 91.01-0";"Bibliotheken und Archive"
4;"R9102";"R 91.02";"Museen"
5;"R91020";"R 91.02-0";"Museen"
4;"R9103";"R 91.03";"Betrieb von historischen Sttten und Gebuden und hnlichen Attraktionen"
5;"R91030";"R 91.03-0";"Betrieb von historischen Sttten und Gebuden und hnlichen Attraktionen"
4;"R9104";"R 91.04";"Botanische und zoologische Grten sowie Naturparks"
5;"R91040";"R 91.04-0";"Botanische und zoologische Grten sowie Naturparks"
2;"R92";"R 92";"Spiel-, Wett- und Lotteriewesen"
3;"R920";"R 92.0";"Spiel-, Wett- und Lotteriewesen"
4;"R9200";"R 92.00";"Spiel-, Wett- und Lotteriewesen"
5;"R92001";"R 92.00-1";"Wett-, Toto- und Lotteriewesen"
5;"R92002";"R 92.00-2";"Casinos"
5;"R92003";"R 92.00-3";"Betrieb von Glcksspielautomaten"
2;"R93";"R 93";"Erbringung von Dienstleistungen des Sports, der Unterhaltung und der Erholung"
3;"R931";"R 93.1";"Erbringung von Dienstleistungen des Sports"
4;"R9311";"R 93.11";"Betrieb von Sportanlagen"
5;"R93111";"R 93.11-1";"Schwimmbder und Schwimmstadien"
5;"R93119";"R 93.11-9";"Betrieb von sonstigen Sportanlagen"
4;"R9312";"R 93.12";"Sportvereine"
5;"R93120";"R 93.12-0";"Sportvereine"
4;"R9313";"R 93.13";"Fitnesszentren"
5;"R93130";"R 93.13-0";"Fitnesszentren"
4;"R9319";"R 93.19";"Erbringung von sonstigen Dienstleistungen des Sports"
5;"R93190";"R 93.19-0";"Erbringung von sonstigen Dienstleistungen des Sports"
3;"R932";"R 93.2";"Erbringung von sonstigen Dienstleistungen der Unterhaltung und der Erholung"
4;"R9321";"R 93.21";"Vergngungs- und Themenparks"
5;"R93210";"R 93.21-0";"Vergngungs- und Themenparks"
4;"R9329";"R 93.29";"Erbringung von Dienstleistungen der Unterhaltung und der Erholung a. n. g."
5;"R93290";"R 93.29-0";"Erbringung von Dienstleistungen der Unterhaltung und der Erholung a. n. g."
1;"S";"S";"ERBRINGUNG VON SONSTIGEN DIENSTLEISTUNGEN"
2;"S94";"S 94";"Interessenvertretungen sowie kirchliche und sonstige religise Vereinigungen (ohne Sozialwesen und Sport)"
3;"S941";"S 94.1";"Wirtschafts- und Arbeitgeberverbnde, Berufsorganisationen"
4;"S9411";"S 94.11";"Wirtschafts- und Arbeitgeberverbnde"
5;"S94110";"S 94.11-0";"Wirtschafts- und Arbeitgeberverbnde"
4;"S9412";"S 94.12";"Berufsorganisationen"
5;"S94120";"S 94.12-0";"Berufsorganisationen"
3;"S942";"S 94.2";"Arbeitnehmervereinigungen"
4;"S9420";"S 94.20";"Arbeitnehmervereinigungen"
5;"S94200";"S 94.20-0";"Arbeitnehmervereinigungen"
3;"S949";"S 94.9";"Kirchliche Vereinigungen; politische Parteien sowie sonstige Interessenvertretungen und Vereinigungen a. n. g."
4;"S9491";"S 94.91";"Kirchliche und sonstige religise Vereinigungen"
5;"S94910";"S 94.91-0";"Kirchliche und sonstige religise Vereinigungen"
4;"S9492";"S 94.92";"Politische Parteien und Vereinigungen"
5;"S94920";"S 94.92-0";"Politische Parteien und Vereinigungen"
4;"S9499";"S 94.99";"Sonstige Interessenvertretungen und Vereinigungen a. n. g."
5;"S94990";"S 94.99-0";"Sonstige Interessenvertretungen und Vereinigungen a. n. g."
2;"S95";"S 95";"Reparatur von Datenverarbeitungsgerten und Gebrauchsgtern"
3;"S951";"S 95.1";"Reparatur von Datenverarbeitungs- und Telekommunikationsgerten"
4;"S9511";"S 95.11";"Reparatur von Datenverarbeitungsgerten und peripheren Gerten"
5;"S95110";"S 95.11-0";"Reparatur von Datenverarbeitungsgerten und peripheren Gerten"
4;"S9512";"S 95.12";"Reparatur von Telekommunikationsgerten"
5;"S95120";"S 95.12-0";"Reparatur von Telekommunikationsgerten"
3;"S952";"S 95.2";"Reparatur von Gebrauchsgtern"
4;"S9521";"S 95.21";"Reparatur von Gerten der Unterhaltungselektronik"
5;"S95210";"S 95.21-0";"Reparatur von Gerten der Unterhaltungselektronik"
4;"S9522";"S 95.22";"Reparatur von elektrischen Haushaltsgerten und Gartengerten"
5;"S95220";"S 95.22-0";"Reparatur von elektrischen Haushaltsgerten und Gartengerten"
4;"S9523";"S 95.23";"Reparatur von Schuhen und Lederwaren"
5;"S95230";"S 95.23-0";"Reparatur von Schuhen und Lederwaren"
4;"S9524";"S 95.24";"Reparatur von Mbeln und Einrichtungsgegenstnden"
5;"S95240";"S 95.24-0";"Reparatur von Mbeln und Einrichtungsgegenstnden"
4;"S9525";"S 95.25";"Reparatur von Uhren und Schmuck"
5;"S95250";"S 95.25-0";"Reparatur von Uhren und Schmuck"
4;"S9529";"S 95.29";"Reparatur von sonstigen Gebrauchsgtern"
5;"S95290";"S 95.29-0";"Reparatur von sonstigen Gebrauchsgtern"
2;"S96";"S 96";"Erbringung von sonstigen berwiegend persnlichen Dienstleistungen"
3;"S960";"S 96.0";"Erbringung von sonstigen berwiegend persnlichen Dienstleistungen"
4;"S9601";"S 96.01";"Wscherei und chemische Reinigung"
5;"S96011";"S 96.01-1";"Wschereien"
5;"S96012";"S 96.01-2";"Chemische Reinigung und Bekleidungsfrberei"
4;"S9602";"S 96.02";"Frisr- und Kosmetiksalons"
5;"S96021";"S 96.02-1";"Frisrsalons"
5;"S96022";"S 96.02-2";"Kosmetiksalons"
5;"S96023";"S 96.02-3";"Fupflege"
4;"S9603";"S 96.03";"Bestattungswesen"
5;"S96030";"S 96.03-0";"Bestattungswesen"
4;"S9604";"S 96.04";"Saunas, Solarien, Bder u. ."
5;"S96041";"S 96.04-1";"Schlankheits- und Massagezentren"
5;"S96042";"S 96.04-2";"Heil-, Kur- und Thermalbder"
5;"S96049";"S 96.04-9";"Solarien, Saunas, Bder a.n.g"
4;"S9609";"S 96.09";"Erbringung von sonstigen Dienstleistungen a. n. g."
5;"S96090";"S 96.09-0";"Erbringung von sonstigen Dienstleistungen a. n. g."
1;"T";"T";"PRIVATE HAUSHALTE MIT HAUSPERSONAL; HERSTELLUNG VON WAREN UND ERBRINGUNG VON DIENSTLEISTUNGEN DURCH PRIVATE HAUSHALTE FR DEN EIGENBEDARF OHNE AUSGEPRGTEN SCHWERPUNKT"
2;"T97";"T 97";"Private Haushalte mit Hauspersonal"
3;"T970";"T 97.0";"Private Haushalte mit Hauspersonal"
4;"T9700";"T 97.00";"Private Haushalte mit Hauspersonal"
5;"T97000";"T 97.00-0";"Private Haushalte mit Hauspersonal"
2;"T98";"T 98";"Herstellung von Waren und Erbringung von Dienstleistungen durch private Haushalte fr den Eigenbedarf ohne ausgeprgten Schwerpunkt"
3;"T981";"T 98.1";"Herstellung von Waren durch private Haushalte fr den Eigenbedarf ohne ausgeprgten Schwerpunkt"
4;"T9810";"T 98.10";"Herstellung von Waren durch private Haushalte fr den Eigenbedarf ohne ausgeprgten Schwerpunkt"
5;"T98100";"T 98.10-0";"Herstellung von Waren durch private Haushalte fr den Eigenbedarf ohne ausgeprgten Schwerpunkt"
3;"T982";"T 98.2";"Erbringungen von Dienstleistungen durch private Haushalte fr den Eigenbedarf ohne ausgeprgten Schwerpunkt"
4;"T9820";"T 98.20";"Erbringungen von Dienstleistungen durch private Haushalte fr den Eigenbedarf ohne ausgeprgten Schwerpunkt"
5;"T98200";"T 98.20-0";"Erbringungen von Dienstleistungen durch private Haushalte fr den Eigenbedarf ohne ausgeprgten Schwerpunkt"
1;"U";"U";"EXTERRITORIALE ORGANISATIONEN UND KRPERSCHAFTEN"
2;"U99";"U 99";"Exterritoriale Organisationen und Krperschaften"
3;"U990";"U 99.0";"Exterritoriale Organisationen und Krperschaften"
4;"U9900";"U 99.00";"Exterritoriale Organisationen und Krperschaften"
5;"U99000";"U 99.00-0";"Exterritoriale Organisationen und Krperschaften"
------
"Ebene";"EDV-Code";"Code";"Titel"
1;"A";"A";"SECTION A  AGRICULTURE, FORESTRY AND FISHING"
2;"A01";"A 01";"Crop and animal production, hunting and related service activities"
3;"A011";"A 01.1";"Growing of non-perennial crops"
4;"A0111";"A 01.11";"Growing of cereals (except rice), leguminous crops and oil seeds"
5;"A01110";"A 01.11-0";"Growing of cereals (except rice), leguminous crops and oil seeds"
4;"A0112";"A 01.12";"Growing of rice"
5;"A01120";"A 01.12-0";"Growing of rice"
4;"A0113";"A 01.13";"Growing of vegetables and melons, roots and tubers"
5;"A01130";"A 01.13-0";"Growing of vegetables and melons, roots and tubers"
4;"A0114";"A 01.14";"Growing of sugar cane"
5;"A01140";"A 01.14-0";"Growing of sugar cane"
4;"A0115";"A 01.15";"Growing of tobacco"
5;"A01150";"A 01.15-0";"Growing of tobacco"
4;"A0116";"A 01.16";"Growing of fibre crops"
5;"A01160";"A 01.16-0";"Growing of fibre crops"
4;"A0119";"A 01.19";"Growing of other non-perennial crops"
5;"A01190";"A 01.19-0";"Growing of other non-perennial crops"
3;"A012";"A 01.2";"Growing of perennial crops"
4;"A0121";"A 01.21";"Growing of grapes"
5;"A01210";"A 01.21-0";"Growing of grapes"
4;"A0122";"A 01.22";"Growing of tropical and subtropical fruits"
5;"A01220";"A 01.22-0";"Growing of tropical and subtropical fruits"
4;"A0123";"A 01.23";"Growing of citrus fruits"
5;"A01230";"A 01.23-0";"Growing of citrus fruits"
4;"A0124";"A 01.24";"Growing of pome fruits and stone fruits"
5;"A01240";"A 01.24-0";"Growing of pome fruits and stone fruits"
4;"A0125";"A 01.25";"Growing of other tree and bush fruits and nuts"
5;"A01250";"A 01.25-0";"Growing of other tree and bush fruits and nuts"
4;"A0126";"A 01.26";"Growing of oleaginous fruits"
5;"A01260";"A 01.26-0";"Growing of oleaginous fruits"
4;"A0127";"A 01.27";"Growing of beverage crops"
5;"A01270";"A 01.27-0";"Growing of beverage crops"
4;"A0128";"A 01.28";"Growing of spices, aromatic, drug and pharmaceutical crops"
5;"A01280";"A 01.28-0";"Growing of spices, aromatic, drug and pharmaceutical crops"
4;"A0129";"A 01.29";"Growing of other perennial crops"
5;"A01290";"A 01.29-0";"Growing of other perennial crops"
3;"A013";"A 01.3";"Plant propagation"
4;"A0130";"A 01.30";"Plant propagation"
5;"A01300";"A 01.30-0";"Plant propagation"
3;"A014";"A 01.4";"Animal production"
4;"A0141";"A 01.41";"Raising of dairy cattle"
5;"A01410";"A 01.41-0";"Raising of dairy cattle"
4;"A0142";"A 01.42";"Raising of other cattle and buffaloes"
5;"A01420";"A 01.42-0";"Raising of other cattle and buffaloes"
4;"A0143";"A 01.43";"Raising of horses and other equines"
5;"A01430";"A 01.43-0";"Raising of horses and other equines"
4;"A0144";"A 01.44";"Raising of camels and camelids"
5;"A01440";"A 01.44-0";"Raising of camels and camelids"
4;"A0145";"A 01.45";"Raising of sheep and goats"
5;"A01450";"A 01.45-0";"Raising of sheep and goats"
4;"A0146";"A 01.46";"Raising of swine/pigs"
5;"A01460";"A 01.46-0";"Raising of swine/pigs"
4;"A0147";"A 01.47";"Raising of poultry"
5;"A01471";"A 01.47-1";"Breeding of poultry"
5;"A01472";"A 01.47-2";"Poultry hatcheries and laying farms"
4;"A0149";"A 01.49";"Raising of other animals"
5;"A01490";"A 01.49-0";"Raising of other animals"
3;"A015";"A 01.5";"Mixed farming"
4;"A0150";"A 01.50";"Mixed farming"
5;"A01500";"A 01.50-0";"Mixed farming"
3;"A016";"A 01.6";"Support activities to agriculture and post-harvest crop activities"
4;"A0161";"A 01.61";"Support activities for crop production"
5;"A01610";"A 01.61-0";"Support activities for crop production"
4;"A0162";"A 01.62";"Support activities for animal production"
5;"A01620";"A 01.62-0";"Support activities for animal production"
4;"A0163";"A 01.63";"Post-harvest crop activities"
5;"A01630";"A 01.63-0";"Post-harvest crop activities"
4;"A0164";"A 01.64";"Seed processing for propagation"
5;"A01640";"A 01.64-0";"Seed processing for propagation"
3;"A017";"A 01.7";"Hunting, trapping and related service activities"
4;"A0170";"A 01.70";"Hunting, trapping and related service activities"
5;"A01700";"A 01.70-0";"Hunting, trapping and related service activities"
2;"A02";"A 02";"Forestry and logging"
3;"A021";"A 02.1";"Silviculture and other forestry activities"
4;"A0210";"A 02.10";"Silviculture and other forestry activities"
5;"A02100";"A 02.10-0";"Silviculture and other forestry activities"
3;"A022";"A 02.2";"Logging"
4;"A0220";"A 02.20";"Logging"
5;"A02200";"A 02.20-0";"Logging"
3;"A023";"A 02.3";"Gathering of wild growing non-wood products"
4;"A0230";"A 02.30";"Gathering of wild growing non-wood products"
5;"A02300";"A 02.30-0";"Gathering of wild growing non-wood products"
3;"A024";"A 02.4";"Support services to forestry"
4;"A0240";"A 02.40";"Support services to forestry"
5;"A02400";"A 02.40-0";"Support services to forestry"
2;"A03";"A 03";"Fishing and aquaculture"
3;"A031";"A 03.1";"Fishing"
4;"A0311";"A 03.11";"Marine fishing"
5;"A03110";"A 03.11-0";"Marine fishing"
4;"A0312";"A 03.12";"Freshwater fishing"
5;"A03120";"A 03.12-0";"Freshwater fishing"
3;"A032";"A 03.2";"Aquaculture"
4;"A0321";"A 03.21";"Marine aquaculture"
5;"A03210";"A 03.21-0";"Marine aquaculture"
4;"A0322";"A 03.22";"Freshwater aquaculture"
5;"A03220";"A 03.22-0";"Freshwater aquaculture"
1;"B";"B";"SECTION B  MINING AND QUARRYING"
2;"B05";"B 05";"Mining of coal and lignite"
3;"B051";"B 05.1";"Mining of hard coal"
4;"B0510";"B 05.10";"Mining of hard coal"
5;"B05100";"B 05.10-0";"Mining of hard coal"
3;"B052";"B 05.2";"Mining of lignite"
4;"B0520";"B 05.20";"Mining of lignite"
5;"B05200";"B 05.20-0";"Mining of lignite"
2;"B06";"B 06";"Extraction of crude petroleum and natural gas"
3;"B061";"B 06.1";"Extraction of crude petroleum"
4;"B0610";"B 06.10";"Extraction of crude petroleum"
5;"B06100";"B 06.10-0";"Extraction of crude petroleum"
3;"B062";"B 06.2";"Extraction of natural gas"
4;"B0620";"B 06.20";"Extraction of natural gas"
5;"B06200";"B 06.20-0";"Extraction of natural gas"
2;"B07";"B 07";"Mining of metal ores"
3;"B071";"B 07.1";"Mining of iron ores"
4;"B0710";"B 07.10";"Mining of iron ores"
5;"B07100";"B 07.10-0";"Mining of iron ores"
3;"B072";"B 07.2";"Mining of non-ferrous metal ores"
4;"B0721";"B 07.21";"Mining of uranium and thorium ores"
5;"B07210";"B 07.21-0";"Mining of uranium and thorium ores"
4;"B0729";"B 07.29";"Mining of other non-ferrous metal ores"
5;"B07290";"B 07.29-0";"Mining of other non-ferrous metal ores"
2;"B08";"B 08";"Other mining and quarrying"
3;"B081";"B 08.1";"Quarrying of stone, sand and clay"
4;"B0811";"B 08.11";"Quarrying of ornamental and building stone, limestone, gypsum, chalk and slate"
5;"B08110";"B 08.11-0";"Quarrying of ornamental and building stone, limestone, gypsum, chalk and slate"
4;"B0812";"B 08.12";"Operation of gravel and sand pits; mining of clays and kaolin"
5;"B08120";"B 08.12-0";"Operation of gravel and sand pits; mining of clays and kaolin"
3;"B089";"B 08.9";"Mining and quarrying n.e.c."
4;"B0891";"B 08.91";"Mining of chemical and fertiliser minerals"
5;"B08910";"B 08.91-0";"Mining of chemical and fertiliser minerals"
4;"B0892";"B 08.92";"Extraction of peat"
5;"B08920";"B 08.92-0";"Extraction of peat"
4;"B0893";"B 08.93";"Extraction of salt"
5;"B08930";"B 08.93-0";"Extraction of salt"
4;"B0899";"B 08.99";"Other mining and quarrying n.e.c."
5;"B08990";"B 08.99-0";"Other mining and quarrying n.e.c."
2;"B09";"B 09";"Mining support service activities"
3;"B091";"B 09.1";"Support activities for petroleum and natural gas extraction"
4;"B0910";"B 09.10";"Support activities for petroleum and natural gas extraction"
5;"B09100";"B 09.10-0";"Support activities for petroleum and natural gas extraction"
3;"B099";"B 09.9";"Support activities for other mining and quarrying"
4;"B0990";"B 09.90";"Support activities for other mining and quarrying"
5;"B09900";"B 09.90-0";"Support activities for other mining and quarrying"
1;"C";"C";"SECTION C  MANUFACTURING"
2;"C10";"C 10";"Manufacture of food products"
3;"C101";"C 10.1";"Processing and preserving of meat and production of meat products"
4;"C1011";"C 10.11";"Processing and preserving of meat"
5;"C10110";"C 10.11-0";"Processing and preserving of meat"
4;"C1012";"C 10.12";"Processing and preserving of poultry meat"
5;"C10120";"C 10.12-0";"Processing and preserving of poultry meat"
4;"C1013";"C 10.13";"Production of meat and poultry meat products"
5;"C10130";"C 10.13-0";"Production of meat and poultry meat products"
3;"C102";"C 10.2";"Processing and preserving of fish, crustaceans and molluscs"
4;"C1020";"C 10.20";"Processing and preserving of fish, crustaceans and molluscs"
5;"C10200";"C 10.20-0";"Processing and preserving of fish, crustaceans and molluscs"
3;"C103";"C 10.3";"Processing and preserving of fruit and vegetables"
4;"C1031";"C 10.31";"Processing and preserving of potatoes"
5;"C10310";"C 10.31-0";"Processing and preserving of potatoes"
4;"C1032";"C 10.32";"Manufacture of fruit and vegetable juice"
5;"C10320";"C 10.32-0";"Manufacture of fruit and vegetable juice"
4;"C1039";"C 10.39";"Other processing and preserving of fruit and vegetables"
5;"C10390";"C 10.39-0";"Other processing and preserving of fruit and vegetables"
3;"C104";"C 10.4";"Manufacture of vegetable and animal oils and fats"
4;"C1041";"C 10.41";"Manufacture of oils and fats"
5;"C10410";"C 10.41-0";"Manufacture of oils and fats"
4;"C1042";"C 10.42";"Manufacture of margarine and similar edible fats"
5;"C10420";"C 10.42-0";"Manufacture of margarine and similar edible fats"
3;"C105";"C 10.5";"Manufacture of dairy products"
4;"C1051";"C 10.51";"Operation of dairies and cheese making"
5;"C10510";"C 10.51-0";"Operation of dairies and cheese making"
4;"C1052";"C 10.52";"Manufacture of ice cream"
5;"C10520";"C 10.52-0";"Manufacture of ice cream"
3;"C106";"C 10.6";"Manufacture of grain mill products, starches and starch products"
4;"C1061";"C 10.61";"Manufacture of grain mill products"
5;"C10610";"C 10.61-0";"Manufacture of grain mill products"
4;"C1062";"C 10.62";"Manufacture of starches and starch products"
5;"C10620";"C 10.62-0";"Manufacture of starches and starch products"
3;"C107";"C 10.7";"Manufacture of bakery and farinaceous products"
4;"C1071";"C 10.71";"Manufacture of bread; manufacture of fresh pastry goods and cakes"
5;"C10711";"C 10.71-1";"Manufacture of bread"
5;"C10712";"C 10.71-2";"Manufacture of fresh pastry goods and cakes"
4;"C1072";"C 10.72";"Manufacture of rusks and biscuits; manufacture of preserved pastry goods and cakes"
5;"C10720";"C 10.72-0";"Manufacture of rusks and biscuits; manufacture of preserved pastry goods and cakes"
4;"C1073";"C 10.73";"Manufacture of macaroni, noodles, couscous and similar farinaceous products"
5;"C10730";"C 10.73-0";"Manufacture of macaroni, noodles, couscous and similar farinaceous products"
3;"C108";"C 10.8";"Manufacture of other food products"
4;"C1081";"C 10.81";"Manufacture of sugar"
5;"C10810";"C 10.81-0";"Manufacture of sugar"
4;"C1082";"C 10.82";"Manufacture of cocoa, chocolate and sugar confectionery"
5;"C10820";"C 10.82-0";"Manufacture of cocoa, chocolate and sugar confectionery"
4;"C1083";"C 10.83";"Processing of tea and coffee"
5;"C10830";"C 10.83-0";"Processing of tea and coffee"
4;"C1084";"C 10.84";"Manufacture of condiments and seasonings"
5;"C10840";"C 10.84-0";"Manufacture of condiments and seasonings"
4;"C1085";"C 10.85";"Manufacture of prepared meals and dishes"
5;"C10850";"C 10.85-0";"Manufacture of prepared meals and dishes"
4;"C1086";"C 10.86";"Manufacture of homogenised food preparations and dietetic food"
5;"C10860";"C 10.86-0";"Manufacture of homogenised food preparations and dietetic food"
4;"C1089";"C 10.89";"Manufacture of other food products n.e.c."
5;"C10890";"C 10.89-0";"Manufacture of other food products n.e.c."
3;"C109";"C 10.9";"Manufacture of prepared animal feeds"
4;"C1091";"C 10.91";"Manufacture of prepared feeds for farm animals"
5;"C10910";"C 10.91-0";"Manufacture of prepared feeds for farm animals"
4;"C1092";"C 10.92";"Manufacture of prepared pet foods"
5;"C10920";"C 10.92-0";"Manufacture of prepared pet foods"
2;"C11";"C 11";"Manufacture of beverages"
3;"C110";"C 11.0";"Manufacture of beverages"
4;"C1101";"C 11.01";"Distilling, rectifying and blending of spirits"
5;"C11010";"C 11.01-0";"Distilling, rectifying and blending of spirits"
4;"C1102";"C 11.02";"Manufacture of wine from grape"
5;"C11020";"C 11.02-0";"Manufacture of wine from grape"
4;"C1103";"C 11.03";"Manufacture of cider and other fruit wines"
5;"C11030";"C 11.03-0";"Manufacture of cider and other fruit wines"
4;"C1104";"C 11.04";"Manufacture of other non-distilled fermented beverages"
5;"C11040";"C 11.04-0";"Manufacture of other non-distilled fermented beverages"
4;"C1105";"C 11.05";"Manufacture of beer"
5;"C11050";"C 11.05-0";"Manufacture of beer"
4;"C1106";"C 11.06";"Manufacture of malt"
5;"C11060";"C 11.06-0";"Manufacture of malt"
4;"C1107";"C 11.07";"Manufacture of soft drinks; production of mineral waters and other bottled waters"
5;"C11070";"C 11.07-0";"Manufacture of soft drinks; production of mineral waters and other bottled waters"
2;"C12";"C 12";"Manufacture of tobacco products"
3;"C120";"C 12.0";"Manufacture of tobacco products"
4;"C1200";"C 12.00";"Manufacture of tobacco products"
5;"C12000";"C 12.00-0";"Manufacture of tobacco products"
2;"C13";"C 13";"Manufacture of textiles"
3;"C131";"C 13.1";"Preparation and spinning of textile fibres"
4;"C1310";"C 13.10";"Preparation and spinning of textile fibres"
5;"C13100";"C 13.10-0";"Preparation and spinning of textile fibres"
3;"C132";"C 13.2";"Weaving of textiles"
4;"C1320";"C 13.20";"Weaving of textiles"
5;"C13200";"C 13.20-0";"Weaving of textiles"
3;"C133";"C 13.3";"Finishing of textiles"
4;"C1330";"C 13.30";"Finishing of textiles"
5;"C13300";"C 13.30-0";"Finishing of textiles"
3;"C139";"C 13.9";"Manufacture of other textiles"
4;"C1391";"C 13.91";"Manufacture of knitted and crocheted fabrics"
5;"C13910";"C 13.91-0";"Manufacture of knitted and crocheted fabrics"
4;"C1392";"C 13.92";"Manufacture of made-up textile articles, except apparel"
5;"C13920";"C 13.92-0";"Manufacture of made-up textile articles, except apparel"
4;"C1393";"C 13.93";"Manufacture of carpets and rugs"
5;"C13930";"C 13.93-0";"Manufacture of carpets and rugs;"
4;"C1394";"C 13.94";"Manufacture of cordage, rope, twine and netting;"
5;"C13940";"C 13.94-0";"Manufacture of cordage, rope, twine and netting;"
4;"C1395";"C 13.95";"Manufacture of non-wovens and articles made from non-wovens, except apparel;"
5;"C13950";"C 13.95-0";"Manufacture of non-wovens and articles made from non-wovens, except apparel;"
4;"C1396";"C 13.96";"Manufacture of other technical and industrial textiles;"
5;"C13960";"C 13.96-0";"Manufacture of other technical and industrial textiles;"
4;"C1399";"C 13.99";"Manufacture of other textiles n.e.c.;"
5;"C13991";"C 13.99-1";"Manufacture of embroideries;"
5;"C13999";"C 13.99-9";"Manufacture of other textiles n.e.c. (except manufacture of embroideries);"
2;"C14";"C 14";"Manufacture of wearing apparel;"
3;"C141";"C 14.1";"Manufacture of wearing apparel, except fur apparel;"
4;"C1411";"C 14.11";"Manufacture of leather clothes;"
5;"C14110";"C 14.11-0";"Manufacture of leather clothes;"
4;"C1412";"C 14.12";"Manufacture of workwear;"
5;"C14120";"C 14.12-0";"Manufacture of workwear;"
4;"C1413";"C 14.13";"Manufacture of other outerwear"
5;"C14130";"C 14.13-0";"Manufacture of other outerwear"
4;"C1414";"C 14.14";"Manufacture of underwear"
5;"C14140";"C 14.14-0";"Manufacture of underwear"
4;"C1419";"C 14.19";"Manufacture of other wearing apparel and accessories"
5;"C14190";"C 14.19-0";"Manufacture of other wearing apparel and accessories"
3;"C142";"C 14.2";"Manufacture of articles of fur"
4;"C1420";"C 14.20";"Manufacture of articles of fur"
5;"C14200";"C 14.20-0";"Manufacture of articles of fur"
3;"C143";"C 14.3";"Manufacture of knitted and crocheted apparel"
4;"C1431";"C 14.31";"Manufacture of knitted and crocheted hosiery"
5;"C14310";"C 14.31-0";"Manufacture of knitted and crocheted hosiery"
4;"C1439";"C 14.39";"Manufacture of other knitted and crocheted apparel"
5;"C14390";"C 14.39-0";"Manufacture of other knitted and crocheted apparel"
2;"C15";"C 15";"Manufacture of leather and related products"
3;"C151";"C 15.1";"Tanning and dressing of leather; manufacture of luggage, handbags, saddlery and harness; dressing and dyeing of fur"
4;"C1511";"C 15.11";"Tanning and dressing of leather; dressing and dyeing of fur"
5;"C15110";"C 15.11-0";"Tanning and dressing of leather; dressing and dyeing of fur"
4;"C1512";"C 15.12";"Manufacture of luggage, handbags and the like, saddlery and harness"
5;"C15120";"C 15.12-0";"Manufacture of luggage, handbags and the like, saddlery and harness"
3;"C152";"C 15.2";"Manufacture of footwear"
4;"C1520";"C 15.20";"Manufacture of footwear"
5;"C15200";"C 15.20-0";"Manufacture of footwear"
2;"C16";"C 16";"Manufacture of wood and of products of wood and cork, except furniture; manufacture of articles of straw and plaiting materials"
3;"C161";"C 16.1";"Sawmilling and planing of wood"
4;"C1610";"C 16.10";"Sawmilling and planing of wood"
5;"C16100";"C 16.10-0";"Sawmilling and planing of wood"
3;"C162";"C 16.2";"Manufacture of products of wood, cork, straw and plaiting materials"
4;"C1621";"C 16.21";"Manufacture of veneer sheets and wood-based panels"
5;"C16210";"C 16.21-0";"Manufacture of veneer sheets and wood-based panels"
4;"C1622";"C 16.22";"Manufacture of assembled parquet floors"
5;"C16220";"C 16.22-0";"Manufacture of assembled parquet floors"
4;"C1623";"C 16.23";"Manufacture of other builders' carpentry and joinery"
5;"C16231";"C 16.23-1";"Manufacture of prefabricated buildings of wood"
5;"C16239";"C 16.23-9";"Manufacture of other builders' carpentry and joinery n.e.c."
4;"C1624";"C 16.24";"Manufacture of wooden containers"
5;"C16240";"C 16.24-0";"Manufacture of wooden containers"
4;"C1629";"C 16.29";"Manufacture of other products of wood; manufacture of articles of cork, straw and plaiting materials"
5;"C16290";"C 16.29-0";"Manufacture of other products of wood; manufacture of articles of cork, straw and plaiting materials"
2;"C17";"C 17";"Manufacture of paper and paper products"
3;"C171";"C 17.1";"Manufacture of pulp, paper and paperboard"
4;"C1711";"C 17.11";"Manufacture of pulp"
5;"C17110";"C 17.11-0";"Manufacture of pulp"
4;"C1712";"C 17.12";"Manufacture of paper and paperboard"
5;"C17120";"C 17.12-0";"Manufacture of paper and paperboard"
3;"C172";"C 17.2";"Manufacture of articles of paper and paperboard"
4;"C1721";"C 17.21";"Manufacture of corrugated paper and paperboard and of containers of paper and paperboard"
5;"C17210";"C 17.21-0";"Manufacture of corrugated paper and paperboard and of containers of paper and paperboard"
4;"C1722";"C 17.22";"Manufacture of household and sanitary goods and of toilet requisites"
5;"C17220";"C 17.22-0";"Manufacture of household and sanitary goods and of toilet requisites"
4;"C1723";"C 17.23";"Manufacture of paper stationery"
5;"C17230";"C 17.23-0";"Manufacture of paper stationery"
4;"C1724";"C 17.24";"Manufacture of wallpaper"
5;"C17240";"C 17.24-0";"Manufacture of wallpaper"
4;"C1729";"C 17.29";"Manufacture of other articles of paper and paperboard"
5;"C17290";"C 17.29-0";"Manufacture of other articles of paper and paperboard"
2;"C18";"C 18";"Printing and reproduction of recorded media"
3;"C181";"C 18.1";"Printing and service activities related to printing"
4;"C1811";"C 18.11";"Printing of newspapers"
5;"C18110";"C 18.11-0";"Printing of newspapers"
4;"C1812";"C 18.12";"Other printing"
5;"C18120";"C 18.12-0";"Other printing"
4;"C1813";"C 18.13";"Pre-press and pre-media services"
5;"C18130";"C 18.13-0";"Pre-press and pre-media services"
4;"C1814";"C 18.14";"Binding and related services"
5;"C18140";"C 18.14-0";"Binding and related services"
3;"C182";"C 18.2";"Reproduction of recorded media"
4;"C1820";"C 18.20";"Reproduction of recorded media"
5;"C18200";"C 18.20-0";"Reproduction of recorded media"
2;"C19";"C 19";"Manufacture of coke and refined petroleum products"
3;"C191";"C 19.1";"Manufacture of coke oven products"
4;"C1910";"C 19.10";"Manufacture of coke oven products"
5;"C19100";"C 19.10-0";"Manufacture of coke oven products"
3;"C192";"C 19.2";"Manufacture of refined petroleum products"
4;"C1920";"C 19.20";"Manufacture of refined petroleum products"
5;"C19200";"C 19.20-0";"Manufacture of refined petroleum products"
2;"C20";"C 20";"Manufacture of chemicals and chemical products"
3;"C201";"C 20.1";"Manufacture of basic chemicals, fertilisers and nitrogen compounds, plastics and synthetic rubber in primary forms"
4;"C2011";"C 20.11";"Manufacture of industrial gases"
5;"C20110";"C 20.11-0";"Manufacture of industrial gases"
4;"C2012";"C 20.12";"Manufacture of dyes and pigments"
5;"C20120";"C 20.12-0";"Manufacture of dyes and pigments"
4;"C2013";"C 20.13";"Manufacture of other inorganic basic chemicals"
5;"C20130";"C 20.13-0";"Manufacture of other inorganic basic chemicals"
4;"C2014";"C 20.14";"Manufacture of other organic basic chemicals"
5;"C20140";"C 20.14-0";"Manufacture of other organic basic chemicals"
4;"C2015";"C 20.15";"Manufacture of fertilisers and nitrogen compounds"
5;"C20150";"C 20.15-0";"Manufacture of fertilisers and nitrogen compounds"
4;"C2016";"C 20.16";"Manufacture of plastics in primary forms"
5;"C20160";"C 20.16-0";"Manufacture of plastics in primary forms"
4;"C2017";"C 20.17";"Manufacture of synthetic rubber in primary forms"
5;"C20170";"C 20.17-0";"Manufacture of synthetic rubber in primary forms"
3;"C202";"C 20.2";"Manufacture of pesticides and other agrochemical products"
4;"C2020";"C 20.20";"Manufacture of pesticides and other agrochemical products"
5;"C20200";"C 20.20-0";"Manufacture of pesticides and other agrochemical products"
3;"C203";"C 20.3";"Manufacture of paints, varnishes and similar coatings, printing ink and mastics"
4;"C2030";"C 20.30";"Manufacture of paints, varnishes and similar coatings, printing ink and mastics"
5;"C20300";"C 20.30-0";"Manufacture of paints, varnishes and similar coatings, printing ink and mastics"
3;"C204";"C 20.4";"Manufacture of soap and detergents, cleaning and polishing preparations, perfumes and toilet preparations"
4;"C2041";"C 20.41";"Manufacture of soap and detergents, cleaning and polishing preparations"
5;"C20410";"C 20.41-0";"Manufacture of soap and detergents, cleaning and polishing preparations"
4;"C2042";"C 20.42";"Manufacture of perfumes and toilet preparations"
5;"C20420";"C 20.42-0";"Manufacture of perfumes and toilet preparations"
3;"C205";"C 20.5";"Manufacture of other chemical products"
4;"C2051";"C 20.51";"Manufacture of explosives"
5;"C20510";"C 20.51-0";"Manufacture of explosives"
4;"C2052";"C 20.52";"Manufacture of glues"
5;"C20520";"C 20.52-0";"Manufacture of glues"
4;"C2053";"C 20.53";"Manufacture of essential oils"
5;"C20530";"C 20.53-0";"Manufacture of essential oils"
4;"C2059";"C 20.59";"Manufacture of other chemical products n.e.c."
5;"C20590";"C 20.59-0";"Manufacture of other chemical products n.e.c."
3;"C206";"C 20.6";"Manufacture of man-made fibres"
4;"C2060";"C 20.60";"Manufacture of man-made fibres"
5;"C20600";"C 20.60-0";"Manufacture of man-made fibres"
2;"C21";"C 21";"Manufacture of basic pharmaceutical products and pharmaceutical preparations"
3;"C211";"C 21.1";"Manufacture of basic pharmaceutical products"
4;"C2110";"C 21.10";"Manufacture of basic pharmaceutical products"
5;"C21100";"C 21.10-0";"Manufacture of basic pharmaceutical products"
3;"C212";"C 21.2";"Manufacture of pharmaceutical preparations"
4;"C2120";"C 21.20";"Manufacture of pharmaceutical preparations"
5;"C21200";"C 21.20-0";"Manufacture of pharmaceutical preparations"
2;"C22";"C 22";"Manufacture of rubber and plastic products"
3;"C221";"C 22.1";"Manufacture of rubber products"
4;"C2211";"C 22.11";"Manufacture of rubber tyres and tubes; retreading and rebuilding of rubber tyres"
5;"C22110";"C 22.11-0";"Manufacture of rubber tyres and tubes; retreading and rebuilding of rubber tyres"
4;"C2219";"C 22.19";"Manufacture of other rubber products"
5;"C22190";"C 22.19-0";"Manufacture of other rubber products"
3;"C222";"C 22.2";"Manufacture of plastics products"
4;"C2221";"C 22.21";"Manufacture of plastic plates, sheets, tubes and profiles"
5;"C22210";"C 22.21-0";"Manufacture of plastic plates, sheets, tubes and profiles"
4;"C2222";"C 22.22";"Manufacture of plastic packing goods"
5;"C22220";"C 22.22-0";"Manufacture of plastic packing goods"
4;"C2223";"C 22.23";"Manufacture of builders ware of plastic"
5;"C22231";"C 22.23-1";"Manufacture of doors, windows, frames, shutters and blinds of plastic"
5;"C22239";"C 22.23-9";"Manufacture of other builders' ware of plastics"
4;"C2229";"C 22.29";"Manufacture of other plastic products"
5;"C22291";"C 22.29-1";"Manufacture of technical plastic parts"
5;"C22299";"C 22.29-9";"Manufacture of other plastic products n.e.c."
2;"C23";"C 23";"Manufacture of other non-metallic mineral products"
3;"C231";"C 23.1";"Manufacture of glass and glass products"
4;"C2311";"C 23.11";"Manufacture of flat glass"
5;"C23110";"C 23.11-0";"Manufacture of flat glass"
4;"C2312";"C 23.12";"Shaping and processing of flat glass"
5;"C23120";"C 23.12-0";"Shaping and processing of flat glass"
4;"C2313";"C 23.13";"Manufacture of hollow glass"
5;"C23130";"C 23.13-0";"Manufacture of hollow glass"
4;"C2314";"C 23.14";"Manufacture of glass fibres"
5;"C23140";"C 23.14-0";"Manufacture of glass fibres"
4;"C2319";"C 23.19";"Manufacture and processing of other glass, including technical glassware"
5;"C23190";"C 23.19-0";"Manufacture and processing of other glass, including technical glassware"
3;"C232";"C 23.2";"Manufacture of refractory products"
4;"C2320";"C 23.20";"Manufacture of refractory products"
5;"C23200";"C 23.20-0";"Manufacture of refractory products"
3;"C233";"C 23.3";"Manufacture of clay building materials"
4;"C2331";"C 23.31";"Manufacture of ceramic tiles and flags"
5;"C23310";"C 23.31-0";"Manufacture of ceramic tiles and flags"
4;"C2332";"C 23.32";"Manufacture of bricks, tiles and construction products, in baked clay"
5;"C23320";"C 23.32-0";"Manufacture of bricks, tiles and construction products, in baked clay"
3;"C234";"C 23.4";"Manufacture of other porcelain and ceramic products"
4;"C2341";"C 23.41";"Manufacture of ceramic household and ornamental articles"
5;"C23410";"C 23.41-0";"Manufacture of ceramic household and ornamental articles"
4;"C2342";"C 23.42";"Manufacture of ceramic sanitary fixtures"
5;"C23420";"C 23.42-0";"Manufacture of ceramic sanitary fixtures"
4;"C2343";"C 23.43";"Manufacture of ceramic insulators and insulating fittings"
5;"C23430";"C 23.43-0";"Manufacture of ceramic insulators and insulating fittings"
4;"C2344";"C 23.44";"Manufacture of other technical ceramic products"
5;"C23440";"C 23.44-0";"Manufacture of other technical ceramic products"
4;"C2349";"C 23.49";"Manufacture of other ceramic products"
5;"C23490";"C 23.49-0";"Manufacture of other ceramic products"
3;"C235";"C 23.5";"Manufacture of cement, lime and plaster"
4;"C2351";"C 23.51";"Manufacture of cement"
5;"C23510";"C 23.51-0";"Manufacture of cement"
4;"C2352";"C 23.52";"Manufacture of lime and plaster"
5;"C23520";"C 23.52-0";"Manufacture of lime and plaster"
3;"C236";"C 23.6";"Manufacture of articles of concrete, cement and plaster"
4;"C2361";"C 23.61";"Manufacture of concrete products for construction purposes"
5;"C23610";"C 23.61-0";"Manufacture of concrete products for construction purposes"
4;"C2362";"C 23.62";"Manufacture of plaster products for construction purposes"
5;"C23620";"C 23.62-0";"Manufacture of plaster products for construction purposes"
4;"C2363";"C 23.63";"Manufacture of ready-mixed concrete"
5;"C23630";"C 23.63-0";"Manufacture of ready-mixed concrete"
4;"C2364";"C 23.64";"Manufacture of mortars"
5;"C23640";"C 23.64-0";"Manufacture of mortars"
4;"C2365";"C 23.65";"Manufacture of fibre cement"
5;"C23650";"C 23.65-0";"Manufacture of fibre cement"
4;"C2369";"C 23.69";"Manufacture of other articles of concrete, plaster and cement"
5;"C23690";"C 23.69-0";"Manufacture of other articles of concrete, plaster and cement"
3;"C237";"C 23.7";"Cutting, shaping and finishing of stone"
4;"C2370";"C 23.70";"Cutting, shaping and finishing of stone"
5;"C23700";"C 23.70-0";"Cutting, shaping and finishing of stone"
3;"C239";"C 23.9";"Manufacture of abrasive products and non-metallic mineral products n.e.c."
4;"C2391";"C 23.91";"Production of abrasive products"
5;"C23910";"C 23.91-0";"Production of abrasive products"
4;"C2399";"C 23.99";"Manufacture of other non-metallic mineral products n.e.c."
5;"C23990";"C 23.99-0";"Manufacture of other non-metallic mineral products n.e.c."
2;"C24";"C 24";"Manufacture of basic metals"
3;"C241";"C 24.1";"Manufacture of basic iron and steel and of ferro-alloys"
4;"C2410";"C 24.10";"Manufacture of basic iron and steel and of ferro-alloys"
5;"C24100";"C 24.10-0";"Manufacture of basic iron and steel and of ferro-alloys"
3;"C242";"C 24.2";"Manufacture of tubes, pipes, hollow profiles and related fittings, of steel"
4;"C2420";"C 24.20";"Manufacture of tubes, pipes, hollow profiles and related fittings, of steel"
5;"C24200";"C 24.20-0";"Manufacture of tubes, pipes, hollow profiles and related fittings, of steel"
3;"C243";"C 24.3";"Manufacture of other products of first processing of steel"
4;"C2431";"C 24.31";"Cold drawing of bars"
5;"C24310";"C 24.31-0";"Cold drawing of bars"
4;"C2432";"C 24.32";"Cold rolling of narrow strip"
5;"C24320";"C 24.32-0";"Cold rolling of narrow strip"
4;"C2433";"C 24.33";"Cold forming or folding"
5;"C24330";"C 24.33-0";"Cold forming or folding"
4;"C2434";"C 24.34";"Cold drawing of wire"
5;"C24340";"C 24.34-0";"Cold drawing of wire"
3;"C244";"C 24.4";"Manufacture of basic precious and other non-ferrous metals"
4;"C2441";"C 24.41";"Precious metals production"
5;"C24410";"C 24.41-0";"Precious metals production"
4;"C2442";"C 24.42";"Aluminium production"
5;"C24420";"C 24.42-0";"Aluminium production"
4;"C2443";"C 24.43";"Lead, zinc and tin production"
5;"C24430";"C 24.43-0";"Lead, zinc and tin production"
4;"C2444";"C 24.44";"Copper production"
5;"C24440";"C 24.44-0";"Copper production"
4;"C2445";"C 24.45";"Other non-ferrous metal production"
5;"C24450";"C 24.45-0";"Other non-ferrous metal production"
4;"C2446";"C 24.46";"Processing of nuclear fuel"
5;"C24460";"C 24.46-0";"Processing of nuclear fuel"
3;"C245";"C 24.5";"Casting of metals"
4;"C2451";"C 24.51";"Casting of iron"
5;"C24510";"C 24.51-0";"Casting of iron"
4;"C2452";"C 24.52";"Casting of steel"
5;"C24520";"C 24.52-0";"Casting of steel"
4;"C2453";"C 24.53";"Casting of light metals"
5;"C24530";"C 24.53-0";"Casting of light metals"
4;"C2454";"C 24.54";"Casting of other non-ferrous metals"
5;"C24540";"C 24.54-0";"Casting of other non-ferrous metals"
2;"C25";"C 25";"Manufacture of fabricated metal products, except machinery and equipment"
3;"C251";"C 25.1";"Manufacture of structural metal products"
4;"C2511";"C 25.11";"Manufacture of metal structures and parts of structures"
5;"C25110";"C 25.11-0";"Manufacture of metal structures and parts of structures"
4;"C2512";"C 25.12";"Manufacture of doors and windows of metal"
5;"C25121";"C 25.12-1";"Manufacture of doors and windows of steel"
5;"C25122";"C 25.12-2";"Manufacture of doors and windows of light metal"
3;"C252";"C 25.2";"Manufacture of tanks, reservoirs and containers of metal"
4;"C2521";"C 25.21";"Manufacture of central heating radiators and boilers"
5;"C25210";"C 25.21-0";"Manufacture of central heating radiators and boilers"
4;"C2529";"C 25.29";"Manufacture of other tanks, reservoirs and containers of metal"
5;"C25290";"C 25.29-0";"Manufacture of other tanks, reservoirs and containers of metal"
3;"C253";"C 25.3";"Manufacture of steam generators, except central heating hot water boilers"
4;"C2530";"C 25.30";"Manufacture of steam generators, except central heating hot water boilers"
5;"C25300";"C 25.30-0";"Manufacture of steam generators, except central heating hot water boilers"
3;"C254";"C 25.4";"Manufacture of weapons and ammunition"
4;"C2540";"C 25.40";"Manufacture of weapons and ammunition"
5;"C25400";"C 25.40-0";"Manufacture of weapons and ammunition"
3;"C255";"C 25.5";"Forging, pressing, stamping and roll-forming of metal; powder metallurgy"
4;"C2550";"C 25.50";"Forging, pressing, stamping and roll-forming of metal; powder metallurgy"
5;"C25500";"C 25.50-0";"Forging, pressing, stamping and roll-forming of metal; powder metallurgy"
3;"C256";"C 25.6";"Treatment and coating of metals; machining"
4;"C2561";"C 25.61";"Treatment and coating of metals"
5;"C25610";"C 25.61-0";"Treatment and coating of metals"
4;"C2562";"C 25.62";"Machining"
5;"C25620";"C 25.62-0";"Machining"
3;"C257";"C 25.7";"Manufacture of cutlery, tools and general hardware"
4;"C2571";"C 25.71";"Manufacture of cutlery"
5;"C25710";"C 25.71-0";"Manufacture of cutlery"
4;"C2572";"C 25.72";"Manufacture of locks and hinges"
5;"C25721";"C 25.72-1";"Manufacture of locks"
5;"C25722";"C 25.72-2";"Manufacture of hinges"
4;"C2573";"C 25.73";"Manufacture of tools"
5;"C25731";"C 25.73-1";"Manufacture of moulds and moulding boxes"
5;"C25739";"C 25.73-9";"Manufacture of other tools"
3;"C259";"C 25.9";"Manufacture of other fabricated metal products"
4;"C2591";"C 25.91";"Manufacture of steel drums and similar containers"
5;"C25910";"C 25.91-0";"Manufacture of steel drums and similar containers"
4;"C2592";"C 25.92";"Manufacture of light metal packaging"
5;"C25920";"C 25.92-0";"Manufacture of light metal packaging"
4;"C2593";"C 25.93";"Manufacture of wire products, chain and springs"
5;"C25930";"C 25.93-0";"Manufacture of wire products, chain and springs"
4;"C2594";"C 25.94";"Manufacture of fasteners and screw machine products"
5;"C25940";"C 25.94-0";"Manufacture of fasteners and screw machine products"
4;"C2599";"C 25.99";"Manufacture of other fabricated metal products n.e.c."
5;"C25990";"C 25.99-0";"Manufacture of other fabricated metal products n.e.c."
2;"C26";"C 26";"Manufacture of computer, electronic and optical products"
3;"C261";"C 26.1";"Manufacture of electronic components and boards"
4;"C2611";"C 26.11";"Manufacture of electronic components"
5;"C26110";"C 26.11-0";"Manufacture of electronic components"
4;"C2612";"C 26.12";"Manufacture of loaded electronic boards"
5;"C26120";"C 26.12-0";"Manufacture of loaded electronic boards"
3;"C262";"C 26.2";"Manufacture of computers and peripheral equipment"
4;"C2620";"C 26.20";"Manufacture of computers and peripheral equipment"
5;"C26200";"C 26.20-0";"Manufacture of computers and peripheral equipment"
3;"C263";"C 26.3";"Manufacture of communication equipment"
4;"C2630";"C 26.30";"Manufacture of communication equipment"
5;"C26300";"C 26.30-0";"Manufacture of communication equipment"
3;"C264";"C 26.4";"Manufacture of consumer electronics"
4;"C2640";"C 26.40";"Manufacture of consumer electronics"
5;"C26400";"C 26.40-0";"Manufacture of consumer electronics"
3;"C265";"C 26.5";"Manufacture of instruments and appliances for measuring, testing and navigation; watches and clocks"
4;"C2651";"C 26.51";"Manufacture of instruments and appliances for measuring, testing and navigation"
5;"C26510";"C 26.51-0";"Manufacture of instruments and appliances for measuring, testing and navigation"
4;"C2652";"C 26.52";"Manufacture of watches and clocks"
5;"C26520";"C 26.52-0";"Manufacture of watches and clocks"
3;"C266";"C 26.6";"Manufacture of irradiation, electromedical and electrotherapeutic equipment"
4;"C2660";"C 26.60";"Manufacture of irradiation, electromedical and electrotherapeutic equipment"
5;"C26600";"C 26.60-0";"Manufacture of irradiation, electromedical and electrotherapeutic equipment"
3;"C267";"C 26.7";"Manufacture of optical instruments and photographic equipment"
4;"C2670";"C 26.70";"Manufacture of optical instruments and photographic equipment"
5;"C26700";"C 26.70-0";"Manufacture of optical instruments and photographic equipment"
3;"C268";"C 26.8";"Manufacture of magnetic and optical media"
4;"C2680";"C 26.80";"Manufacture of magnetic and optical media"
5;"C26800";"C 26.80-0";"Manufacture of magnetic and optical media"
2;"C27";"C 27";"Manufacture of electrical equipment"
3;"C271";"C 27.1";"Manufacture of electric motors, generators, transformers and electricity distribution and control apparatus"
4;"C2711";"C 27.11";"Manufacture of electric motors, generators and transformers"
5;"C27110";"C 27.11-0";"Manufacture of electric motors, generators and transformers"
4;"C2712";"C 27.12";"Manufacture of electricity distribution and control apparatus"
5;"C27120";"C 27.12-0";"Manufacture of electricity distribution and control apparatus"
3;"C272";"C 27.2";"Manufacture of batteries and accumulators"
4;"C2720";"C 27.20";"Manufacture of batteries and accumulators"
5;"C27200";"C 27.20-0";"Manufacture of batteries and accumulators"
3;"C273";"C 27.3";"Manufacture of wiring and wiring devices"
4;"C2731";"C 27.31";"Manufacture of fibre optic cables"
5;"C27310";"C 27.31-0";"Manufacture of fibre optic cables"
4;"C2732";"C 27.32";"Manufacture of other electronic and electric wires and cables"
5;"C27320";"C 27.32-0";"Manufacture of other electronic and electric wires and cables"
4;"C2733";"C 27.33";"Manufacture of wiring devices"
5;"C27330";"C 27.33-0";"Manufacture of wiring devices"
3;"C274";"C 27.4";"Manufacture of electric lighting equipment"
4;"C2740";"C 27.40";"Manufacture of electric lighting equipment"
5;"C27400";"C 27.40-0";"Manufacture of electric lighting equipment"
3;"C275";"C 27.5";"Manufacture of domestic appliances"
4;"C2751";"C 27.51";"Manufacture of electric domestic appliances"
5;"C27510";"C 27.51-0";"Manufacture of electric domestic appliances"
4;"C2752";"C 27.52";"Manufacture of non-electric domestic appliances"
5;"C27520";"C 27.52-0";"Manufacture of non-electric domestic appliances"
3;"C279";"C 27.9";"Manufacture of other electrical equipment"
4;"C2790";"C 27.90";"Manufacture of other electrical equipment"
5;"C27900";"C 27.90-0";"Manufacture of other electrical equipment"
2;"C28";"C 28";"Manufacture of machinery and equipment n.e.c."
3;"C281";"C 28.1";"Manufacture of general-purpose machinery"
4;"C2811";"C 28.11";"Manufacture of engines and turbines, except aircraft, vehicle and cycle engines"
5;"C28110";"C 28.11-0";"Manufacture of engines and turbines, except aircraft, vehicle and cycle engines"
4;"C2812";"C 28.12";"Manufacture of fluid power equipment"
5;"C28120";"C 28.12-0";"Manufacture of fluid power equipment"
4;"C2813";"C 28.13";"Manufacture of other pumps and compressors"
5;"C28130";"C 28.13-0";"Manufacture of other pumps and compressors"
4;"C2814";"C 28.14";"Manufacture of other taps and valves"
5;"C28140";"C 28.14-0";"Manufacture of other taps and valves"
4;"C2815";"C 28.15";"Manufacture of bearings, gears, gearing and driving elements"
5;"C28150";"C 28.15-0";"Manufacture of bearings, gears, gearing and driving elements"
3;"C282";"C 28.2";"Manufacture of other general-purpose machinery"
4;"C2821";"C 28.21";"Manufacture of ovens, furnaces and furnace burners"
5;"C28210";"C 28.21-0";"Manufacture of ovens, furnaces and furnace burners"
4;"C2822";"C 28.22";"Manufacture of lifting and handling equipment"
5;"C28220";"C 28.22-0";"Manufacture of lifting and handling equipment"
4;"C2823";"C 28.23";"Manufacture of office machinery and equipment (except computers and peripheral equipment)"
5;"C28230";"C 28.23-0";"Manufacture of office machinery and equipment (except computers and peripheral equipment)"
4;"C2824";"C 28.24";"Manufacture of power-driven hand tools"
5;"C28240";"C 28.24-0";"Manufacture of power-driven hand tools"
4;"C2825";"C 28.25";"Manufacture of non-domestic cooling and ventilation equipment"
5;"C28250";"C 28.25-0";"Manufacture of non-domestic cooling and ventilation equipment"
4;"C2829";"C 28.29";"Manufacture of other general-purpose machinery n.e.c."
5;"C28290";"C 28.29-0";"Manufacture of other general-purpose machinery n.e.c."
3;"C283";"C 28.3";"Manufacture of agricultural and forestry machinery"
4;"C2830";"C 28.30";"Manufacture of agricultural and forestry machinery"
5;"C28300";"C 28.30-0";"Manufacture of agricultural and forestry machinery"
3;"C284";"C 28.4";"Manufacture of metal forming machinery and machine tools"
4;"C2841";"C 28.41";"Manufacture of metal forming machinery"
5;"C28410";"C 28.41-0";"Manufacture of metal forming machinery"
4;"C2849";"C 28.49";"Manufacture of other machine tools"
5;"C28491";"C 28.49-1";"Manufacture of wood working machine tools"
5;"C28499";"C 28.49-9";"Manufacture of other machine tools n.e.c."
3;"C289";"C 28.9";"Manufacture of other special-purpose machinery"
4;"C2891";"C 28.91";"Manufacture of machinery for metallurgy"
5;"C28910";"C 28.91-0";"Manufacture of machinery for metallurgy"
4;"C2892";"C 28.92";"Manufacture of machinery for mining, quarrying and construction"
5;"C28920";"C 28.92-0";"Manufacture of machinery for mining, quarrying and construction"
4;"C2893";"C 28.93";"Manufacture of machinery for food, beverage and tobacco processing"
5;"C28930";"C 28.93-0";"Manufacture of machinery for food, beverage and tobacco processing"
4;"C2894";"C 28.94";"Manufacture of machinery for textile, apparel and leather production"
5;"C28940";"C 28.94-0";"Manufacture of machinery for textile, apparel and leather production"
4;"C2895";"C 28.95";"Manufacture of machinery for paper and paperboard production"
5;"C28950";"C 28.95-0";"Manufacture of machinery for paper and paperboard production"
4;"C2896";"C 28.96";"Manufacture of plastics and rubber machinery"
5;"C28960";"C 28.96-0";"Manufacture of plastics and rubber machinery"
4;"C2899";"C 28.99";"Manufacture of other special-purpose machinery n.e.c."
5;"C28991";"C 28.99-1";"Manufacture of printing and bookbinding machines"
5;"C28999";"C 28.99-9";"Manufacture of other special purpose machinery n.e.c. (except printing and bookbinding machines)"
2;"C29";"C 29";"Manufacture of motor vehicles, trailers and semi-trailers"
3;"C291";"C 29.1";"Manufacture of motor vehicles"
4;"C2910";"C 29.10";"Manufacture of motor vehicles"
5;"C29100";"C 29.10-0";"Manufacture of motor vehicles"
3;"C292";"C 29.2";"Manufacture of bodies (coachwork) for motor vehicles; manufacture of trailers and semi-trailers"
4;"C2920";"C 29.20";"Manufacture of bodies (coachwork) for motor vehicles; manufacture of trailers and semi-trailers"
5;"C29200";"C 29.20-0";"Manufacture of bodies (coachwork) for motor vehicles; manufacture of trailers and semi-trailers"
3;"C293";"C 29.3";"Manufacture of parts and accessories for motor vehicles"
4;"C2931";"C 29.31";"Manufacture of electrical and electronic equipment for motor vehicles"
5;"C29310";"C 29.31-0";"Manufacture of electrical and electronic equipment for motor vehicles"
4;"C2932";"C 29.32";"Manufacture of other parts and accessories for motor vehicles"
5;"C29320";"C 29.32-0";"Manufacture of other parts and accessories for motor vehicles"
2;"C30";"C 30";"Manufacture of other transport equipment"
3;"C301";"C 30.1";"Building of ships and boats"
4;"C3011";"C 30.11";"Building of ships and floating structures"
5;"C30110";"C 30.11-0";"Building of ships and floating structures"
4;"C3012";"C 30.12";"Building of pleasure and sporting boats"
5;"C30120";"C 30.12-0";"Building of pleasure and sporting boats"
3;"C302";"C 30.2";"Manufacture of railway locomotives and rolling stock"
4;"C3020";"C 30.20";"Manufacture of railway locomotives and rolling stock"
5;"C30200";"C 30.20-0";"Manufacture of railway locomotives and rolling stock"
3;"C303";"C 30.3";"Manufacture of air and spacecraft and related machinery"
4;"C3030";"C 30.30";"Manufacture of air and spacecraft and related machinery"
5;"C30300";"C 30.30-0";"Manufacture of air and spacecraft and related machinery"
3;"C304";"C 30.4";"Manufacture of military fighting vehicles"
4;"C3040";"C 30.40";"Manufacture of military fighting vehicles"
5;"C30400";"C 30.40-0";"Manufacture of military fighting vehicles"
3;"C309";"C 30.9";"Manufacture of transport equipment n.e.c."
4;"C3091";"C 30.91";"Manufacture of motorcycles"
5;"C30910";"C 30.91-0";"Manufacture of motorcycles"
4;"C3092";"C 30.92";"Manufacture of bicycles and invalid carriages"
5;"C30920";"C 30.92-0";"Manufacture of bicycles and invalid carriages"
4;"C3099";"C 30.99";"Manufacture of other transport equipment n.e.c."
5;"C30990";"C 30.99-0";"Manufacture of other transport equipment n.e.c."
2;"C31";"C 31";"Manufacture of furniture"
3;"C310";"C 31.0";"Manufacture of furniture"
4;"C3101";"C 31.01";"Manufacture of office and shop furniture"
5;"C31011";"C 31.01-1";"Manufacture of office furniture"
5;"C31012";"C 31.01-2";"Manufacture of shop furniture"
4;"C3102";"C 31.02";"Manufacture of kitchen furniture"
5;"C31020";"C 31.02-0";"Manufacture of kitchen furniture"
4;"C3103";"C 31.03";"Manufacture of mattresses"
5;"C31030";"C 31.03-0";"Manufacture of mattresses"
4;"C3109";"C 31.09";"Manufacture of other furniture"
5;"C31090";"C 31.09-0";"Manufacture of other furniture"
2;"C32";"C 32";"Other manufacturing"
3;"C321";"C 32.1";"Manufacture of jewellery, bijouterie and related articles"
4;"C3211";"C 32.11";"Striking of coins"
5;"C32110";"C 32.11-0";"Striking of coins"
4;"C3212";"C 32.12";"Manufacture of jewellery and related articles"
5;"C32120";"C 32.12-0";"Manufacture of jewellery and related articles"
4;"C3213";"C 32.13";"Manufacture of imitation jewellery and related articles"
5;"C32130";"C 32.13-0";"Manufacture of imitation jewellery and related articles"
3;"C322";"C 32.2";"Manufacture of musical instruments"
4;"C3220";"C 32.20";"Manufacture of musical instruments"
5;"C32200";"C 32.20-0";"Manufacture of musical instruments"
3;"C323";"C 32.3";"Manufacture of sports goods"
4;"C3230";"C 32.30";"Manufacture of sports goods"
5;"C32300";"C 32.30-0";"Manufacture of sports goods"
3;"C324";"C 32.4";"Manufacture of games and toys"
4;"C3240";"C 32.40";"Manufacture of games and toys"
5;"C32400";"C 32.40-0";"Manufacture of games and toys"
3;"C325";"C 32.5";"Manufacture of medical and dental instruments and supplies"
4;"C3250";"C 32.50";"Manufacture of medical and dental instruments and supplies"
5;"C32501";"C 32.50-1";"Dental laboratories"
5;"C32502";"C 32.50-2";"Manufacture of glasses and parts thereof"
5;"C32509";"C 32.50-9";"Manufacture of other medical and dental instruments and supplies"
3;"C329";"C 32.9";"Manufacturing n.e.c."
4;"C3291";"C 32.91";"Manufacture of brooms and brushes"
5;"C32910";"C 32.91-0";"Manufacture of brooms and brushes"
4;"C3299";"C 32.99";"Other manufacturing n.e.c."
5;"C32990";"C 32.99-0";"Other manufacturing n.e.c."
2;"C33";"C 33";"Repair and installation of machinery and equipment"
3;"C331";"C 33.1";"Repair of fabricated metal products, machinery and equipment"
4;"C3311";"C 33.11";"Repair of fabricated metal products"
5;"C33110";"C 33.11-0";"Repair of fabricated metal products"
4;"C3312";"C 33.12";"Repair of machinery"
5;"C33120";"C 33.12-0";"Repair of machinery"
4;"C3313";"C 33.13";"Repair of electronic and optical equipment"
5;"C33130";"C 33.13-0";"Repair of electronic and optical equipment"
4;"C3314";"C 33.14";"Repair of electrical equipment"
5;"C33140";"C 33.14-0";"Repair of electrical equipment"
4;"C3315";"C 33.15";"Repair and maintenance of ships and boats"
5;"C33150";"C 33.15-0";"Repair and maintenance of ships and boats"
4;"C3316";"C 33.16";"Repair and maintenance of aircraft and spacecraft"
5;"C33160";"C 33.16-0";"Repair and maintenance of aircraft and spacecraft"
4;"C3317";"C 33.17";"Repair and maintenance of other transport equipment"
5;"C33170";"C 33.17-0";"Repair and maintenance of other transport equipment"
4;"C3319";"C 33.19";"Repair of other equipment"
5;"C33190";"C 33.19-0";"Repair of other equipment"
3;"C332";"C 33.2";"Installation of industrial machinery and equipment"
4;"C3320";"C 33.20";"Installation of industrial machinery and equipment"
5;"C33200";"C 33.20-0";"Installation of industrial machinery and equipment"
1;"D";"D";"SECTION D  ELECTRICITY, GAS, STEAM AND AIR CONDITIONING SUPPLY"
2;"D35";"D 35";"Electricity, gas, steam and air conditioning supply"
3;"D351";"D 35.1";"Electric power generation, transmission and distribution"
4;"D3511";"D 35.11";"Production of electricity"
5;"D35110";"D 35.11-0";"Production of electricity"
4;"D3512";"D 35.12";"Transmission of electricity"
5;"D35120";"D 35.12-0";"Transmission of electricity"
4;"D3513";"D 35.13";"Distribution of electricity"
5;"D35130";"D 35.13-0";"Distribution of electricity"
4;"D3514";"D 35.14";"Trade of electricity"
5;"D35140";"D 35.14-0";"Trade of electricity"
3;"D352";"D 35.2";"Manufacture of gas; distribution of gaseous fuels through mains"
4;"D3521";"D 35.21";"Manufacture of gas"
5;"D35210";"D 35.21-0";"Manufacture of gas"
4;"D3522";"D 35.22";"Distribution of gaseous fuels through mains"
5;"D35220";"D 35.22-0";"Distribution of gaseous fuels through mains"
4;"D3523";"D 35.23";"Trade of gas through mains"
5;"D35230";"D 35.23-0";"Trade of gas through mains"
3;"D353";"D 35.3";"Steam and air conditioning supply"
4;"D3530";"D 35.30";"Steam and air conditioning supply"
5;"D35300";"D 35.30-0";"Steam and air conditioning supply"
1;"E";"E";"SECTION E  WATER SUPPLY; SEWERAGE, WASTE MANAGEMENT AND REMEDIATION ACTIVITIES"
2;"E36";"E 36";"Water collection, treatment and supply"
3;"E360";"E 36.0";"Water collection, treatment and supply"
4;"E3600";"E 36.00";"Water collection, treatment and supply"
5;"E36000";"E 36.00-0";"Water collection, treatment and supply"
2;"E37";"E 37";"Sewerage"
3;"E370";"E 37.0";"Sewerage"
4;"E3700";"E 37.00";"Sewerage"
5;"E37000";"E 37.00-0";"Sewerage"
2;"E38";"E 38";"Waste collection, treatment and disposal activities; materials recovery"
3;"E381";"E 38.1";"Waste collection"
4;"E3811";"E 38.11";"Collection of non-hazardous waste"
5;"E38110";"E 38.11-0";"Collection of non-hazardous waste"
4;"E3812";"E 38.12";"Collection of hazardous waste"
5;"E38120";"E 38.12-0";"Collection of hazardous waste"
3;"E382";"E 38.2";"Waste treatment and disposal"
4;"E3821";"E 38.21";"Treatment and disposal of non-hazardous waste"
5;"E38211";"E 38.21-1";"Operation of landfills for the disposal of non-hazardous waste"
5;"E38219";"E 38.21-9";"Other treatment and disposal of non-hazardous waste"
4;"E3822";"E 38.22";"Treatment and disposal of hazardous waste"
5;"E38220";"E 38.22-0";"Treatment and disposal of hazardous waste"
3;"E383";"E 38.3";"Materials recovery"
4;"E3831";"E 38.31";"Dismantling of wrecks"
5;"E38310";"E 38.31-0";"Dismantling of wrecks"
4;"E3832";"E 38.32";"Recovery of sorted materials"
5;"E38321";"E 38.32-1";"Recovery of sorted iron and non-ferrous metals"
5;"E38329";"E 38.32-9";"Recovery of other sorted materials"
2;"E39";"E 39";"Remediation activities and other waste management services"
3;"E390";"E 39.0";"Remediation activities and other waste management services"
4;"E3900";"E 39.00";"Remediation activities and other waste management services"
5;"E39000";"E 39.00-0";"Remediation activities and other waste management services"
1;"F";"F";"SECTION F  CONSTRUCTION"
2;"F41";"F 41";"Construction of buildings"
3;"F411";"F 41.1";"Development of building projects"
4;"F4110";"F 41.10";"Development of building projects"
5;"F41100";"F 41.10-0";"Development of building projects"
3;"F412";"F 41.2";"Construction of residential and non-residential buildings"
4;"F4120";"F 41.20";"Construction of residential and non-residential buildings"
5;"F41201";"F 41.20-1";"Construction of residential buildings"
5;"F41202";"F 41.20-2";"Renovation work of buildings"
5;"F41209";"F 41.20-9";"Construction of other non-residential buildings"
2;"F42";"F 42";"Civil engineering"
3;"F421";"F 42.1";"Construction of roads and railways"
4;"F4211";"F 42.11";"Construction of roads and motorways"
5;"F42110";"F 42.11-0";"Construction of roads and motorways"
4;"F4212";"F 42.12";"Construction of railways and underground railways"
5;"F42120";"F 42.12-0";"Construction of railways and underground railways"
4;"F4213";"F 42.13";"Construction of bridges and tunnels"
5;"F42131";"F 42.13-1";"Construction of bridges and elevated highways"
5;"F42132";"F 42.13-2";"Construction of tunnels"
3;"F422";"F 42.2";"Construction of utility projects"
4;"F4221";"F 42.21";"Construction of utility projects for fluids"
5;"F42210";"F 42.21-0";"Construction of utility projects for fluids"
4;"F4222";"F 42.22";"Construction of utility projects for electricity and telecommunications"
5;"F42220";"F 42.22-0";"Construction of utility projects for electricity and telecommunications"
3;"F429";"F 42.9";"Construction of other civil engineering projects"
4;"F4291";"F 42.91";"Construction of water projects"
5;"F42910";"F 42.91-0";"Construction of water projects"
4;"F4299";"F 42.99";"Construction of other civil engineering projects n.e.c."
5;"F42990";"F 42.99-0";"Construction of other civil engineering projects n.e.c."
2;"F43";"F 43";"Specialised construction activities"
3;"F431";"F 43.1";"Demolition and site preparation"
4;"F4311";"F 43.11";"Demolition"
5;"F43110";"F 43.11-0";"Demolition"
4;"F4312";"F 43.12";"Site preparation"
5;"F43120";"F 43.12-0";"Site preparation"
4;"F4313";"F 43.13";"Test drilling and boring"
5;"F43130";"F 43.13-0";"Test drilling and boring"
3;"F432";"F 43.2";"Electrical, plumbing and other construction installation activities"
4;"F4321";"F 43.21";"Electrical installation"
5;"F43210";"F 43.21-0";"Electrical installation"
4;"F4322";"F 43.22";"Plumbing, heat and air-conditioning installation"
5;"F43220";"F 43.22-0";"Plumbing, heat and air-conditioning installation"
4;"F4329";"F 43.29";"Other construction installation"
5;"F43290";"F 43.29-0";"Other construction installation"
3;"F433";"F 43.3";"Building completion and finishing"
4;"F4331";"F 43.31";"Plastering"
5;"F43310";"F 43.31-0";"Plastering"
4;"F4332";"F 43.32";"Joinery installation"
5;"F43321";"F 43.32-1";"Wood joinery installation"
5;"F43322";"F 43.32-2";"Metal joinery installation"
4;"F4333";"F 43.33";"Floor and wall covering"
5;"F43331";"F 43.33-1";"Floor covering work"
5;"F43332";"F 43.33-2";"Tile laying work"
5;"F43333";"F 43.33-3";"Paper-hanger"
5;"F43334";"F 43.33-4";"Stove-fitter"
4;"F4334";"F 43.34";"Painting and glazing"
5;"F43341";"F 43.34-1";"Painting"
5;"F43342";"F 43.34-2";"Glazing"
4;"F4339";"F 43.39";"Other building completion and finishing"
5;"F43390";"F 43.39-0";"Other building completion and finishing"
3;"F439";"F 43.9";"Other specialised construction activities"
4;"F4391";"F 43.91";"Roofing activities"
5;"F43911";"F 43.91-1";"Roof covering"
5;"F43912";"F 43.91-2";"Erection of wooden roofs and frames"
5;"F43913";"F 43.91-3";"Tinsmiths"
4;"F4399";"F 43.99";"Other specialised construction activities n.e.c."
5;"F43991";"F 43.99-1";"Waterproofing"
5;"F43999";"F 43.99-9";"Other specialised construction activities n.e.c. (except waterproofing)"
1;"G";"G";"SECTION G  WHOLESALE AND RETAIL TRADE; REPAIR OF MOTOR VEHICLES AND MOTORCYCLES"
2;"G45";"G 45";"Wholesale and retail trade and repair of motor vehicles and motorcycles"
3;"G451";"G 45.1";"Sale of motor vehicles"
4;"G4511";"G 45.11";"Sale of cars and light motor vehicles"
5;"G45111";"G 45.11-1";"Wholesale of cars and light motor vehicles"
5;"G45112";"G 45.11-2";"Retail sale of cars and light motor vehicles"
4;"G4519";"G 45.19";"Sale of other motor vehicles"
5;"G45190";"G 45.19-0";"Sale of other motor vehicles"
3;"G452";"G 45.2";"Maintenance and repair of motor vehicles"
4;"G4520";"G 45.20";"Maintenance and repair of motor vehicles"
5;"G45200";"G 45.20-0";"Maintenance and repair of motor vehicles"
3;"G453";"G 45.3";"Sale of motor vehicle parts and accessories"
4;"G4531";"G 45.31";"Wholesale trade of motor vehicle parts and accessories"
5;"G45310";"G 45.31-0";"Wholesale trade of motor vehicle parts and accessories"
4;"G4532";"G 45.32";"Retail trade of motor vehicle parts and accessories"
5;"G45320";"G 45.32-0";"Retail trade of motor vehicle parts and accessories"
3;"G454";"G 45.4";"Sale, maintenance and repair of motorcycles and related parts and accessories"
4;"G4540";"G 45.40";"Sale, maintenance and repair of motorcycles and related parts and accessories"
5;"G45400";"G 45.40-0";"Sale, maintenance and repair of motorcycles and related parts and accessories"
2;"G46";"G 46";"Wholesale trade, except of motor vehicles and motorcycles"
3;"G461";"G 46.1";"Wholesale on a fee or contract basis"
4;"G4611";"G 46.11";"Agents involved in the sale of agricultural raw materials, live animals, textile raw materials and semifinished goods"
5;"G46110";"G 46.11-0";"Agents involved in the sale of agricultural raw materials, live animals, textile raw materials and semifinished goods"
4;"G4612";"G 46.12";"Agents involved in the sale of fuels, ores, metals and industrial chemicals"
5;"G46120";"G 46.12-0";"Agents involved in the sale of fuels, ores, metals and industrial chemicals"
4;"G4613";"G 46.13";"Agents involved in the sale of timber and building materials"
5;"G46130";"G 46.13-0";"Agents involved in the sale of timber and building materials"
4;"G4614";"G 46.14";"Agents involved in the sale of machinery, industrial equipment, ships and aircraft"
5;"G46140";"G 46.14-0";"Agents involved in the sale of machinery, industrial equipment, ships and aircraft"
4;"G4615";"G 46.15";"Agents involved in the sale of furniture, household goods, hardware and ironmongery"
5;"G46150";"G 46.15-0";"Agents involved in the sale of furniture, household goods, hardware and ironmongery"
4;"G4616";"G 46.16";"Agents involved in the sale of textiles, clothing, fur, footwear and leather goods"
5;"G46160";"G 46.16-0";"Agents involved in the sale of textiles, clothing, fur, footwear and leather goods"
4;"G4617";"G 46.17";"Agents involved in the sale of food, beverages and tobacco"
5;"G46170";"G 46.17-0";"Agents involved in the sale of food, beverages and tobacco"
4;"G4618";"G 46.18";"Agents specialised in the sale of other particular products"
5;"G46180";"G 46.18-0";"Agents specialised in the sale of other particular products"
4;"G4619";"G 46.19";"Agents involved in the sale of a variety of goods"
5;"G46190";"G 46.19-0";"Agents involved in the sale of a variety of goods"
3;"G462";"G 46.2";"Wholesale of agricultural raw materials and live animals"
4;"G4621";"G 46.21";"Wholesale of grain, unmanufactured tobacco, seeds and animal feeds"
5;"G46210";"G 46.21-0";"Wholesale of grain, unmanufactured tobacco, seeds and animal feeds"
4;"G4622";"G 46.22";"Wholesale of flowers and plants"
5;"G46220";"G 46.22-0";"Wholesale of flowers and plants"
4;"G4623";"G 46.23";"Wholesale of live animals"
5;"G46230";"G 46.23-0";"Wholesale of live animals"
4;"G4624";"G 46.24";"Wholesale of hides, skins and leather"
5;"G46240";"G 46.24-0";"Wholesale of hides, skins and leather"
3;"G463";"G 46.3";"Wholesale of food, beverages and tobacco"
4;"G4631";"G 46.31";"Wholesale of fruit and vegetables"
5;"G46310";"G 46.31-0";"Wholesale of fruit and vegetables"
4;"G4632";"G 46.32";"Wholesale of meat and meat products"
5;"G46320";"G 46.32-0";"Wholesale of meat and meat products"
4;"G4633";"G 46.33";"Wholesale of dairy products, eggs and edible oils and fats"
5;"G46330";"G 46.33-0";"Wholesale of dairy products, eggs and edible oils and fats"
4;"G4634";"G 46.34";"Wholesale of beverages"
5;"G46341";"G 46.34-1";"Wholesale of wine and liquors"
5;"G46349";"G 46.34-9";"Wholesale of other beverages"
4;"G4635";"G 46.35";"Wholesale of tobacco products"
5;"G46350";"G 46.35-0";"Wholesale of tobacco products"
4;"G4636";"G 46.36";"Wholesale of sugar and chocolate and sugar confectionery"
5;"G46360";"G 46.36-0";"Wholesale of sugar and chocolate and sugar confectionery"
4;"G4637";"G 46.37";"Wholesale of coffee, tea, cocoa and spices"
5;"G46370";"G 46.37-0";"Wholesale of coffee, tea, cocoa and spices"
4;"G4638";"G 46.38";"Wholesale of other food, including fish, crustaceans and molluscs"
5;"G46380";"G 46.38-0";"Wholesale of other food, including fish, crustaceans and molluscs"
4;"G4639";"G 46.39";"Non-specialised wholesale of food, beverages and tobacco"
5;"G46390";"G 46.39-0";"Non-specialised wholesale of food, beverages and tobacco"
3;"G464";"G 46.4";"Wholesale of household goods"
4;"G4641";"G 46.41";"Wholesale of textiles"
5;"G46410";"G 46.41-0";"Wholesale of textiles"
4;"G4642";"G 46.42";"Wholesale of clothing and footwear"
5;"G46421";"G 46.42-1";"Wholesale of clothing"
5;"G46422";"G 46.42-2";"Wholesale of footwear"
4;"G4643";"G 46.43";"Wholesale of electrical household appliances"
5;"G46431";"G 46.43-1";"Wholesale of photographic and optical goods"
5;"G46432";"G 46.43-2";"Wholesale of electrical household appliances"
5;"G46433";"G 46.43-3";"Wholesale of audio and video equipment"
4;"G4644";"G 46.44";"Wholesale of china and glassware and cleaning materials"
5;"G46440";"G 46.44-0";"Wholesale of china and glassware and cleaning materials"
4;"G4645";"G 46.45";"Wholesale of perfume and cosmetics"
5;"G46450";"G 46.45-0";"Wholesale of perfume and cosmetics"
4;"G4646";"G 46.46";"Wholesale of pharmaceutical goods"
5;"G46461";"G 46.46-1";"Wholesale of pharmaceutical products"
5;"G46462";"G 46.46-2";"Wholesale of orthopaedic appliances and medical and surgical equipment"
4;"G4647";"G 46.47";"Wholesale of furniture, carpets and lighting equipment"
5;"G46470";"G 46.47-0";"Wholesale of furniture, carpets and lighting equipment"
4;"G4648";"G 46.48";"Wholesale of watches and jewellery"
5;"G46480";"G 46.48-0";"Wholesale of watches and jewellery"
4;"G4649";"G 46.49";"Wholesale of other household goods"
5;"G46491";"G 46.49-1";"Wholesale of stationery and office supplies"
5;"G46492";"G 46.49-2";"Wholesale of books, magazines and newspapers"
5;"G46493";"G 46.49-3";"Wholesale of sports goods"
5;"G46494";"G 46.49-4";"Wholesale of games and toys and musical instruments"
5;"G46499";"G 46.49-9";"Wholesale of other household goods n.e.c."
3;"G465";"G 46.5";"Wholesale of information and communication equipment"
4;"G4651";"G 46.51";"Wholesale of computers, computer peripheral equipment and software"
5;"G46511";"G 46.51-1";"Wholesale of computers and peripheral equipment"
5;"G46512";"G 46.51-2";"Wholesale of packaged software"
4;"G4652";"G 46.52";"Wholesale of electronic and telecommunications equipment and parts"
5;"G46520";"G 46.52-0";"Wholesale of electronic and telecommunications equipment and parts"
3;"G466";"G 46.6";"Wholesale of other machinery, equipment and supplies"
4;"G4661";"G 46.61";"Wholesale of agricultural machinery, equipment and supplies"
5;"G46610";"G 46.61-0";"Wholesale of agricultural machinery, equipment and supplies"
4;"G4662";"G 46.62";"Wholesale of machine tools"
5;"G46620";"G 46.62-0";"Wholesale of machine tools"
4;"G4663";"G 46.63";"Wholesale of mining, construction and civil engineering machinery"
5;"G46630";"G 46.63-0";"Wholesale of mining, construction and civil engineering machinery"
4;"G4664";"G 46.64";"Wholesale of machinery for the textile industry and of sewing and knitting machines"
5;"G46640";"G 46.64-0";"Wholesale of machinery for the textile industry and of sewing and knitting machines"
4;"G4665";"G 46.65";"Wholesale of office furniture"
5;"G46650";"G 46.65-0";"Wholesale of office furniture"
4;"G4666";"G 46.66";"Wholesale of other office machinery and equipment"
5;"G46660";"G 46.66-0";"Wholesale of other office machinery and equipment"
4;"G4669";"G 46.69";"Wholesale of other machinery and equipment"
5;"G46690";"G 46.69-0";"Wholesale of other machinery and equipment"
3;"G467";"G 46.7";"Other specialised wholesale"
4;"G4671";"G 46.71";"Wholesale of solid, liquid and gaseous fuels and related products"
5;"G46711";"G 46.71-1";"Wholesale of solid fuels"
5;"G46712";"G 46.71-2";"Wholesale of liquid and gaseous fuels"
4;"G4672";"G 46.72";"Wholesale of metals and metal ores"
5;"G46721";"G 46.72-1";"Wholesale of metal ores, ferrous metals in primary forms and semi-finished ferrous metal products"
5;"G46722";"G 46.72-2";"Wholesale of non-ferrous metals in primary forms and semi-finished non-ferrous metal products"
4;"G4673";"G 46.73";"Wholesale of wood, construction materials and sanitary equipment"
5;"G46731";"G 46.73-1";"Wholesale of wood"
5;"G46732";"G 46.73-2";"Wholesale of construction materials and sanitary equipment"
5;"G46733";"G 46.73-3";"Wholesale of flat glass, paints and varnishes, wallpaper and floor  coverings"
4;"G4674";"G 46.74";"Wholesale of hardware, plumbing and heating equipment and supplies"
5;"G46740";"G 46.74-0";"Wholesale of hardware, plumbing and heating equipment and supplies"
4;"G4675";"G 46.75";"Wholesale of chemical products"
5;"G46750";"G 46.75-0";"Wholesale of chemical products"
4;"G4676";"G 46.76";"Wholesale of other intermediate products"
5;"G46761";"G 46.76-1";"Wholesale of paper in bulk"
5;"G46769";"G 46.76-9";"Wholesale of other intermediate products n.e.c."
4;"G4677";"G 46.77";"Wholesale of waste and scrap"
5;"G46770";"G 46.77-0";"Wholesale of waste and scrap"
3;"G469";"G 46.9";"Non-specialised wholesale trade"
4;"G4690";"G 46.90";"Non-specialised wholesale trade"
5;"G46900";"G 46.90-0";"Non-specialised wholesale trade"
2;"G47";"G 47";"Retail trade, except of motor vehicles and motorcycles"
3;"G471";"G 47.1";"Retail sale in non-specialised stores"
4;"G4711";"G 47.11";"Retail sale in non-specialised stores with food, beverages or tobacco predominating"
5;"G47110";"G 47.11-0";"Retail sale in non-specialised stores with food, beverages or tobacco predominating"
4;"G4719";"G 47.19";"Other retail sale in non-specialised stores"
5;"G47191";"G 47.19-1";"Department stores"
5;"G47199";"G 47.19-9";"Other retail sale in non-specialised stores (except department stores)"
3;"G472";"G 47.2";"Retail sale of food, beverages and tobacco in specialised stores"
4;"G4721";"G 47.21";"Retail sale of fruit and vegetables in specialised stores"
5;"G47210";"G 47.21-0";"Retail sale of fruit and vegetables in specialised stores"
4;"G4722";"G 47.22";"Retail sale of meat and meat products in specialised stores"
5;"G47220";"G 47.22-0";"Retail sale of meat and meat products in specialised stores"
4;"G4723";"G 47.23";"Retail sale of fish, crustaceans and molluscs in specialised stores"
5;"G47230";"G 47.23-0";"Retail sale of fish, crustaceans and molluscs in specialised stores"
4;"G4724";"G 47.24";"Retail sale of bread, cakes, flour confectionery and sugar confectionery in specialised stores"
5;"G47240";"G 47.24-0";"Retail sale of bread, cakes, flour confectionery and sugar confectionery in specialised stores"
4;"G4725";"G 47.25";"Retail sale of beverages in specialised stores"
5;"G47250";"G 47.25-0";"Retail sale of beverages in specialised stores"
4;"G4726";"G 47.26";"Retail sale of tobacco products in specialised stores"
5;"G47260";"G 47.26-0";"Retail sale of tobacco products in specialised stores"
4;"G4729";"G 47.29";"Other retail sale of food in specialised stores"
5;"G47290";"G 47.29-0";"Other retail sale of food in specialised stores"
3;"G473";"G 47.3";"Retail sale of automotive fuel in specialised stores"
4;"G4730";"G 47.30";"Retail sale of automotive fuel in specialised stores"
5;"G47300";"G 47.30-0";"Retail sale of automotive fuel in specialised stores"
3;"G474";"G 47.4";"Retail sale of information and communication equipment in specialised stores"
4;"G4741";"G 47.41";"Retail sale of computers, peripheral units and software in specialised stores"
5;"G47410";"G 47.41-0";"Retail sale of computers, peripheral units and software in specialised stores"
4;"G4742";"G 47.42";"Retail sale of telecommunications equipment in specialised stores"
5;"G47420";"G 47.42-0";"Retail sale of telecommunications equipment in specialised stores"
4;"G4743";"G 47.43";"Retail sale of audio and video equipment in specialised stores"
5;"G47430";"G 47.43-0";"Retail sale of audio and video equipment in specialised stores"
3;"G475";"G 47.5";"Retail sale of other household equipment in specialised stores"
4;"G4751";"G 47.51";"Retail sale of textiles in specialised stores"
5;"G47510";"G 47.51-0";"Retail sale of textiles in specialised stores"
4;"G4752";"G 47.52";"Retail sale of hardware, paints and glass in specialised stores"
5;"G47520";"G 47.52-0";"Retail sale of hardware, paints and glass in specialised stores"
4;"G4753";"G 47.53";"Retail sale of carpets, rugs, wall and floor coverings in specialised stores"
5;"G47530";"G 47.53-0";"Retail sale of carpets, rugs, wall and floor coverings in specialised stores"
4;"G4754";"G 47.54";"Retail sale of electrical household appliances in specialised stores"
5;"G47540";"G 47.54-0";"Retail sale of electrical household appliances in specialised stores"
4;"G4759";"G 47.59";"Retail sale of furniture, lighting equipment and other household articles in specialised stores"
5;"G47591";"G 47.59-1";"Retail sale of musical instruments and scores in specialised stores"
5;"G47592";"G 47.59-2";"Retail sale of household furniture in specialised stores"
5;"G47599";"G 47.59-9";"Retail sale of other household articles n.e.c. in specialised stores"
3;"G476";"G 47.6";"Retail sale of cultural and recreation goods in specialised stores"
4;"G4761";"G 47.61";"Retail sale of books in specialised stores"
5;"G47610";"G 47.61-0";"Retail sale of books in specialised stores"
4;"G4762";"G 47.62";"Retail sale of newspapers and stationery in specialised stores"
5;"G47620";"G 47.62-0";"Retail sale of newspapers and stationery in specialised stores"
4;"G4763";"G 47.63";"Retail sale of music and video recordings in specialised stores"
5;"G47630";"G 47.63-0";"Retail sale of music and video recordings in specialised stores"
4;"G4764";"G 47.64";"Retail sale of sporting equipment in specialised stores"
5;"G47640";"G 47.64-0";"Retail sale of sporting equipment in specialised stores"
4;"G4765";"G 47.65";"Retail sale of games and toys in specialised stores"
5;"G47650";"G 47.65-0";"Retail sale of games and toys in specialised stores"
3;"G477";"G 47.7";"Retail sale of other goods in specialised stores"
4;"G4771";"G 47.71";"Retail sale of clothing in specialised stores"
5;"G47710";"G 47.71-0";"Retail sale of clothing in specialised stores"
4;"G4772";"G 47.72";"Retail sale of footwear and leather goods in specialised stores"
5;"G47721";"G 47.72-1";"Retail sale of footwear in specialised stores"
5;"G47722";"G 47.72-2";"Retail sale of leather goods and travel accessories in specialised stores"
4;"G4773";"G 47.73";"Dispensing chemist in specialised stores"
5;"G47730";"G 47.73-0";"Dispensing chemist in specialised stores"
4;"G4774";"G 47.74";"Retail sale of medical and orthopaedic goods in specialised stores"
5;"G47740";"G 47.74-0";"Retail sale of medical and orthopaedic goods in specialised stores"
4;"G4775";"G 47.75";"Retail sale of cosmetic and toilet articles in specialised stores"
5;"G47750";"G 47.75-0";"Retail sale of cosmetic and toilet articles in specialised stores"
4;"G4776";"G 47.76";"Retail sale of flowers, plants, seeds, fertilisers, pet animals and pet food in specialised stores"
5;"G47761";"G 47.76-1";"Retail sale of flowers, plants, seeds, fertilisers in specialised stores"
5;"G47762";"G 47.76-2";"Retail sale of pet animals and pet food in specialised stores"
4;"G4777";"G 47.77";"Retail sale of watches and jewellery in specialised stores"
5;"G47770";"G 47.77-0";"Retail sale of watches and jewellery in specialised stores"
4;"G4778";"G 47.78";"Other retail sale of new goods in specialised stores"
5;"G47781";"G 47.78-1";"Retail sale of photographic, optical and precision equipment in specialised stores"
5;"G47782";"G 47.78-2";"Retail sale of  household fuel oil, bottled gas, coal and fuel wood  in specialised stores"
5;"G47789";"G 47.78-9";"Other retail sale of new goods n.e.c. in specialised stores"
4;"G4779";"G 47.79";"Retail sale of second-hand goods in stores"
5;"G47790";"G 47.79-0";"Retail sale of second-hand goods in stores"
3;"G478";"G 47.8";"Retail sale via stalls and markets"
4;"G4781";"G 47.81";"Retail sale via stalls and markets of food, beverages and tobacco products"
5;"G47810";"G 47.81-0";"Retail sale via stalls and markets of food, beverages and tobacco products"
4;"G4782";"G 47.82";"Retail sale via stalls and markets of textiles, clothing and footwear"
5;"G47820";"G 47.82-0";"Retail sale via stalls and markets of textiles, clothing and footwear"
4;"G4789";"G 47.89";"Retail sale via stalls and markets of other goods"
5;"G47890";"G 47.89-0";"Retail sale via stalls and markets of other goods"
3;"G479";"G 47.9";"Retail trade not in stores, stalls or markets"
4;"G4791";"G 47.91";"Retail sale via mail order houses or via Internet"
5;"G47910";"G 47.91-0";"Retail sale via mail order houses or via Internet"
4;"G4799";"G 47.99";"Other retail sale not in stores, stalls or markets"
5;"G47990";"G 47.99-0";"Other retail sale not in stores, stalls or markets"
1;"H";"H";"SECTION H  TRANSPORTATION AND STORAGE"
2;"H49";"H 49";"Land transport and transport via pipelines"
3;"H491";"H 49.1";"Passenger rail transport, interurban"
4;"H4910";"H 49.10";"Passenger rail transport, interurban"
5;"H49100";"H 49.10-0";"Passenger rail transport, interurban"
3;"H492";"H 49.2";"Freight rail transport"
4;"H4920";"H 49.20";"Freight rail transport"
5;"H49200";"H 49.20-0";"Freight rail transport"
3;"H493";"H 49.3";"Other passenger land transport"
4;"H4931";"H 49.31";"Urban and suburban passenger land transport"
5;"H49311";"H 49.31-1";"Passanger land transport by tramways, trolley buses, undergrounds and suburban railways"
5;"H49312";"H 49.31-2";"Urban and suburban passanger land transport services by buses (without trolley buses)"
4;"H4932";"H 49.32";"Taxi operation"
5;"H49320";"H 49.32-0";"Taxi operation"
4;"H4939";"H 49.39";"Other passenger land transport n.e.c."
5;"H49391";"H 49.39-1";"Transport by cable railways, funiculars and ski-lifts"
5;"H49399";"H 49.39-9";"Other passenger land transport n.e.c. (except transport by cable railways, funiculars and ski-lifts )"
3;"H494";"H 49.4";"Freight transport by road and removal services"
4;"H4941";"H 49.41";"Freight transport by road"
5;"H49410";"H 49.41-0";"Freight transport by road"
4;"H4942";"H 49.42";"Removal services"
5;"H49420";"H 49.42-0";"Removal services"
3;"H495";"H 49.5";"Transport via pipeline"
4;"H4950";"H 49.50";"Transport via pipeline"
5;"H49500";"H 49.50-0";"Transport via pipeline"
2;"H50";"H 50";"Water transport"
3;"H501";"H 50.1";"Sea and coastal passenger water transport"
4;"H5010";"H 50.10";"Sea and coastal passenger water transport"
5;"H50100";"H 50.10-0";"Sea and coastal passenger water transport"
3;"H502";"H 50.2";"Sea and coastal freight water transport"
4;"H5020";"H 50.20";"Sea and coastal freight water transport"
5;"H50200";"H 50.20-0";"Sea and coastal freight water transport"
3;"H503";"H 50.3";"Inland passenger water transport"
4;"H5030";"H 50.30";"Inland passenger water transport"
5;"H50300";"H 50.30-0";"Inland passenger water transport"
3;"H504";"H 50.4";"Inland freight water transport"
4;"H5040";"H 50.40";"Inland freight water transport"
5;"H50400";"H 50.40-0";"Inland freight water transport"
2;"H51";"H 51";"Air transport"
3;"H511";"H 51.1";"Passenger air transport"
4;"H5110";"H 51.10";"Passenger air transport"
5;"H51100";"H 51.10-0";"Passenger air transport"
3;"H512";"H 51.2";"Freight air transport and space transport"
4;"H5121";"H 51.21";"Freight air transport"
5;"H51210";"H 51.21-0";"Freight air transport"
4;"H5122";"H 51.22";"Space transport"
5;"H51220";"H 51.22-0";"Space transport"
2;"H52";"H 52";"Warehousing and support activities for transportation"
3;"H521";"H 52.1";"Warehousing and storage"
4;"H5210";"H 52.10";"Warehousing and storage"
5;"H52100";"H 52.10-0";"Warehousing and storage"
3;"H522";"H 52.2";"Support activities for transportation"
4;"H5221";"H 52.21";"Service activities incidental to land transportation"
5;"H52211";"H 52.21-1";"Operation of car parks and garages"
5;"H52212";"H 52.21-2";"Operation of toll roads"
5;"H52219";"H 52.21-9";"Service activities incidental to land transportation n.e.c."
4;"H5222";"H 52.22";"Service activities incidental to water transportation"
5;"H52220";"H 52.22-0";"Service activities incidental to water transportation"
4;"H5223";"H 52.23";"Service activities incidental to air transportation"
5;"H52230";"H 52.23-0";"Service activities incidental to air transportation"
4;"H5224";"H 52.24";"Cargo handling"
5;"H52240";"H 52.24-0";"Cargo handling"
4;"H5229";"H 52.29";"Other transportation support activities"
5;"H52290";"H 52.29-0";"Other transportation support activities"
2;"H53";"H 53";"Postal and courier activities"
3;"H531";"H 53.1";"Postal activities under universal service obligation"
4;"H5310";"H 53.10";"Postal activities under universal service obligation"
5;"H53100";"H 53.10-0";"Postal activities under universal service obligation"
3;"H532";"H 53.2";"Other postal and courier activities"
4;"H5320";"H 53.20";"Other postal and courier activities"
5;"H53200";"H 53.20-0";"Other postal and courier activities"
1;"I";"I";"SECTION I  ACCOMMODATION AND FOOD SERVICE ACTIVITIES"
2;"I55";"I 55";"Accommodation"
3;"I551";"I 55.1";"Hotels and similar accommodation"
4;"I5510";"I 55.10";"Hotels and similar accommodation"
5;"I55101";"I 55.10-1";"Hotels"
5;"I55102";"I 55.10-2";"Inns (with accommodation)"
5;"I55103";"I 55.10-3";"Boarding houses"
3;"I552";"I 55.2";"Holiday and other short-stay accommodation"
4;"I5520";"I 55.20";"Holiday and other short-stay accommodation"
5;"I55201";"I 55.20-1";"Youth hostels"
5;"I55202";"I 55.20-2";"Mountain refuges"
5;"I55203";"I 55.20-3";"Cottages and holiday flats"
3;"I553";"I 55.3";"Camping grounds, recreational vehicle parks and trailer parks"
4;"I5530";"I 55.30";"Camping grounds, recreational vehicle parks and trailer parks"
5;"I55300";"I 55.30-0";"Camping grounds, recreational vehicle parks and trailer parks"
3;"I559";"I 55.9";"Other accommodation"
4;"I5590";"I 55.90";"Other accommodation"
5;"I55900";"I 55.90-0";"Other accommodation"
2;"I56";"I 56";"Food and beverage service activities"
3;"I561";"I 56.1";"Restaurants and mobile food service activities"
4;"I5610";"I 56.10";"Restaurants and mobile food service activities"
5;"I56101";"I 56.10-1";"Restaurants"
5;"I56102";"I 56.10-2";"Refreshment and take away stands"
5;"I56103";"I 56.10-3";"Coffee-houses"
5;"I56104";"I 56.10-4";"Ice cream parlours"
3;"I562";"I 56.2";"Event catering and other food service activities"
4;"I5621";"I 56.21";"Event catering activities"
5;"I56210";"I 56.21-0";"Event catering activities"
4;"I5629";"I 56.29";"Other food service activities"
5;"I56290";"I 56.29-0";"Other food service activities"
3;"I563";"I 56.3";"Beverage serving activities"
4;"I5630";"I 56.30";"Beverage serving activities"
5;"I56301";"I 56.30-1";"Bars and discotheques"
5;"I56309";"I 56.30-9";"Other beverage serving activities"
1;"J";"J";"SECTION J  INFORMATION AND COMMUNICATION"
2;"J58";"J 58";"Publishing activities"
3;"J581";"J 58.1";"Publishing of books, periodicals and other publishing activities"
4;"J5811";"J 58.11";"Book publishing"
5;"J58110";"J 58.11-0";"Book publishing"
4;"J5812";"J 58.12";"Publishing of directories and mailing lists"
5;"J58120";"J 58.12-0";"Publishing of directories and mailing lists"
4;"J5813";"J 58.13";"Publishing of newspapers"
5;"J58130";"J 58.13-0";"Publishing of newspapers"
4;"J5814";"J 58.14";"Publishing of journals and periodicals"
5;"J58140";"J 58.14-0";"Publishing of journals and periodicals"
4;"J5819";"J 58.19";"Other publishing activities"
5;"J58190";"J 58.19-0";"Other publishing activities"
3;"J582";"J 58.2";"Software publishing"
4;"J5821";"J 58.21";"Publishing of computer games"
5;"J58210";"J 58.21-0";"Publishing of computer games"
4;"J5829";"J 58.29";"Other software publishing"
5;"J58290";"J 58.29-0";"Other software publishing"
2;"J59";"J 59";"Motion picture, video and television programme production, sound recording and music publishing activities"
3;"J591";"J 59.1";"Motion picture, video and television programme activities"
4;"J5911";"J 59.11";"Motion picture, video and television programme production activities"
5;"J59110";"J 59.11-0";"Motion picture, video and television programme production activities"
4;"J5912";"J 59.12";"Motion picture, video and television programme post-production activities"
5;"J59120";"J 59.12-0";"Motion picture, video and television programme post-production activities"
4;"J5913";"J 59.13";"Motion picture, video and television programme distribution activities"
5;"J59130";"J 59.13-0";"Motion picture, video and television programme distribution activities"
4;"J5914";"J 59.14";"Motion picture projection activities"
5;"J59140";"J 59.14-0";"Motion picture projection activities"
3;"J592";"J 59.2";"Sound recording and music publishing activities"
4;"J5920";"J 59.20";"Sound recording and music publishing activities"
5;"J59200";"J 59.20-0";"Sound recording and music publishing activities"
2;"J60";"J 60";"Programming and broadcasting activities"
3;"J601";"J 60.1";"Radio broadcasting"
4;"J6010";"J 60.10";"Radio broadcasting"
5;"J60100";"J 60.10-0";"Radio broadcasting"
3;"J602";"J 60.2";"Television programming and broadcasting activities"
4;"J6020";"J 60.20";"Television programming and broadcasting activities"
5;"J60200";"J 60.20-0";"Television programming and broadcasting activities"
2;"J61";"J 61";"Telecommunications"
3;"J611";"J 61.1";"Wired telecommunications activities"
4;"J6110";"J 61.10";"Wired telecommunications activities"
5;"J61100";"J 61.10-0";"Wired telecommunications activities"
3;"J612";"J 61.2";"Wireless telecommunications activities"
4;"J6120";"J 61.20";"Wireless telecommunications activities"
5;"J61200";"J 61.20-0";"Wireless telecommunications activities"
3;"J613";"J 61.3";"Satellite telecommunications activities"
4;"J6130";"J 61.30";"Satellite telecommunications activities"
5;"J61300";"J 61.30-0";"Satellite telecommunications activities"
3;"J619";"J 61.9";"Other telecommunications activities"
4;"J6190";"J 61.90";"Other telecommunications activities"
5;"J61900";"J 61.90-0";"Other telecommunications activities"
2;"J62";"J 62";"Computer programming, consultancy and related activities"
3;"J620";"J 62.0";"Computer programming, consultancy and related activities"
4;"J6201";"J 62.01";"Computer programming activities"
5;"J62010";"J 62.01-0";"Computer programming activities"
4;"J6202";"J 62.02";"Computer consultancy activities"
5;"J62020";"J 62.02-0";"Computer consultancy activities"
4;"J6203";"J 62.03";"Computer facilities management activities"
5;"J62030";"J 62.03-0";"Computer facilities management activities"
4;"J6209";"J 62.09";"Other information technology and computer service activities"
5;"J62090";"J 62.09-0";"Other information technology and computer service activities"
2;"J63";"J 63";"Information service activities"
3;"J631";"J 63.1";"Data processing, hosting and related activities; web portals"
4;"J6311";"J 63.11";"Data processing, hosting and related activities"
5;"J63110";"J 63.11-0";"Data processing, hosting and related activities"
4;"J6312";"J 63.12";"Web portals"
5;"J63120";"J 63.12-0";"Web portals"
3;"J639";"J 63.9";"Other information service activities"
4;"J6391";"J 63.91";"News agency activities"
5;"J63910";"J 63.91-0";"News agency activities"
4;"J6399";"J 63.99";"Other information service activities n.e.c."
5;"J63990";"J 63.99-0";"Other information service activities n.e.c."
1;"K";"K";"SECTION K  FINANCIAL AND INSURANCE ACTIVITIES"
2;"K64";"K 64";"Financial service activities, except insurance and pension funding"
3;"K641";"K 64.1";"Monetary intermediation"
4;"K6411";"K 64.11";"Central banking"
5;"K64110";"K 64.11-0";"Central banking"
4;"K6419";"K 64.19";"Other monetary intermediation"
5;"K64190";"K 64.19-0";"Other monetary intermediation"
3;"K642";"K 64.2";"Activities of holding companies"
4;"K6420";"K 64.20";"Activities of holding companies"
5;"K64200";"K 64.20-0";"Activities of holding companies"
3;"K643";"K 64.3";"Trusts, funds and similar financial entities"
4;"K6430";"K 64.30";"Trusts, funds and similar financial entities"
5;"K64301";"K 64.30-1";"Investment and real estate funds"
5;"K64309";"K 64.30-9";"Other trusts, funds and similar financial entities n.e.c."
3;"K649";"K 64.9";"Other financial service activities, except insurance and pension funding"
4;"K6491";"K 64.91";"Financial leasing"
5;"K64910";"K 64.91-0";"Financial leasing"
4;"K6492";"K 64.92";"Other credit granting"
5;"K64920";"K 64.92-0";"Other credit granting"
4;"K6499";"K 64.99";"Other financial service activities, except insurance and pension funding n.e.c."
5;"K64990";"K 64.99-0";"Other financial service activities, except insurance and pension funding n.e.c."
2;"K65";"K 65";"Insurance, reinsurance and pension funding, except compulsory social security"
3;"K651";"K 65.1";"Insurance"
4;"K6511";"K 65.11";"Life insurance"
5;"K65110";"K 65.11-0";"Life insurance"
4;"K6512";"K 65.12";"Non-life insurance"
5;"K65120";"K 65.12-0";"Non-life insurance"
3;"K652";"K 65.2";"Reinsurance"
4;"K6520";"K 65.20";"Reinsurance"
5;"K65200";"K 65.20-0";"Reinsurance"
3;"K653";"K 65.3";"Pension funding"
4;"K6530";"K 65.30";"Pension funding"
5;"K65300";"K 65.30-0";"Pension funding"
2;"K66";"K 66";"Activities auxiliary to financial services and insurance activities"
3;"K661";"K 66.1";"Activities auxiliary to financial services, except insurance and pension funding"
4;"K6611";"K 66.11";"Administration of financial markets"
5;"K66110";"K 66.11-0";"Administration of financial markets"
4;"K6612";"K 66.12";"Security and commodity contracts brokerage"
5;"K66120";"K 66.12-0";"Security and commodity contracts brokerage"
4;"K6619";"K 66.19";"Other activities auxiliary to financial services, except insurance and pension funding"
5;"K66190";"K 66.19-0";"Other activities auxiliary to financial services, except insurance and pension funding"
3;"K662";"K 66.2";"Activities auxiliary to insurance and pension funding"
4;"K6621";"K 66.21";"Risk and damage evaluation"
5;"K66210";"K 66.21-0";"Risk and damage evaluation"
4;"K6622";"K 66.22";"Activities of insurance agents and brokers"
5;"K66220";"K 66.22-0";"Activities of insurance agents and brokers"
4;"K6629";"K 66.29";"Other activities auxiliary to insurance and pension funding"
5;"K66290";"K 66.29-0";"Other activities auxiliary to insurance and pension funding"
3;"K663";"K 66.3";"Fund management activities"
4;"K6630";"K 66.30";"Fund management activities"
5;"K66300";"K 66.30-0";"Fund management activities"
1;"L";"L";"SECTION L  REAL ESTATE ACTIVITIES"
2;"L68";"L 68";"Real estate activities"
3;"L681";"L 68.1";"Buying and selling of own real estate"
4;"L6810";"L 68.10";"Buying and selling of own real estate"
5;"L68100";"L 68.10-0";"Buying and selling of own real estate"
3;"L682";"L 68.2";"Renting and operating of own or leased real estate"
4;"L6820";"L 68.20";"Renting and operating of own or leased real estate"
5;"L68201";"L 68.20-1";"Leasing of real estate"
5;"L68209";"L 68.20-9";"Other renting and operating of own or leased real estate"
3;"L683";"L 68.3";"Real estate activities on a fee or contract basis"
4;"L6831";"L 68.31";"Real estate agencies"
5;"L68310";"L 68.31-0";"Real estate agencies"
4;"L6832";"L 68.32";"Management of real estate on a fee or contract basis"
5;"L68320";"L 68.32-0";"Management of real estate on a fee or contract basis"
1;"M";"M";"SECTION M  PROFESSIONAL, SCIENTIFIC AND TECHNICAL ACTIVITIES"
2;"M69";"M 69";"Legal and accounting activities"
3;"M691";"M 69.1";"Legal activities"
4;"M6910";"M 69.10";"Legal activities"
5;"M69100";"M 69.10-0";"Legal activities"
3;"M692";"M 69.2";"Accounting, bookkeeping and auditing activities; tax consultancy"
4;"M6920";"M 69.20";"Accounting, bookkeeping and auditing activities; tax consultancy"
5;"M69200";"M 69.20-0";"Accounting, bookkeeping and auditing activities; tax consultancy"
2;"M70";"M 70";"Activities of head offices; management consultancy activities"
3;"M701";"M 70.1";"Activities of head offices"
4;"M7010";"M 70.10";"Activities of head offices"
5;"M70100";"M 70.10-0";"Activities of head offices"
3;"M702";"M 70.2";"Management consultancy activities"
4;"M7021";"M 70.21";"Public relations and communication activities"
5;"M70210";"M 70.21-0";"Public relations and communication activities"
4;"M7022";"M 70.22";"Business and other management consultancy activities"
5;"M70220";"M 70.22-0";"Business and other management consultancy activities"
2;"M71";"M 71";"Architectural and engineering activities; technical testing and analysis"
3;"M711";"M 71.1";"Architectural and engineering activities and related technical consultancy"
4;"M7111";"M 71.11";"Architectural activities"
5;"M71110";"M 71.11-0";"Architectural activities"
4;"M7112";"M 71.12";"Engineering activities and related technical consultancy"
5;"M71120";"M 71.12-0";"Engineering activities and related technical consultancy"
3;"M712";"M 71.2";"Technical testing and analysis"
4;"M7120";"M 71.20";"Technical testing and analysis"
5;"M71200";"M 71.20-0";"Technical testing and analysis"
2;"M72";"M 72";"Scientific research and development"
3;"M721";"M 72.1";"Research and experimental development on natural sciences and engineering"
4;"M7211";"M 72.11";"Research and experimental development on biotechnology"
5;"M72110";"M 72.11-0";"Research and experimental development on biotechnology"
4;"M7219";"M 72.19";"Other research and experimental development on natural sciences and engineering"
5;"M72190";"M 72.19-0";"Other research and experimental development on natural sciences and engineering"
3;"M722";"M 72.2";"Research and experimental development on social sciences and humanities"
4;"M7220";"M 72.20";"Research and experimental development on social sciences and humanities"
5;"M72200";"M 72.20-0";"Research and experimental development on social sciences and humanities"
2;"M73";"M 73";"Advertising and market research"
3;"M731";"M 73.1";"Advertising"
4;"M7311";"M 73.11";"Advertising agencies"
5;"M73111";"M 73.11-1";"Creation of advertising campaigns"
5;"M73112";"M 73.11-2";"Distribution of advertising material"
4;"M7312";"M 73.12";"Media representation"
5;"M73120";"M 73.12-0";"Media representation"
3;"M732";"M 73.2";"Market research and public opinion polling"
4;"M7320";"M 73.20";"Market research and public opinion polling"
5;"M73200";"M 73.20-0";"Market research and public opinion polling"
2;"M74";"M 74";"Other professional, scientific and technical activities"
3;"M741";"M 74.1";"Specialised design activities"
4;"M7410";"M 74.10";"Specialised design activities"
5;"M74100";"M 74.10-0";"Specialised design activities"
3;"M742";"M 74.2";"Photographic activities"
4;"M7420";"M 74.20";"Photographic activities"
5;"M74200";"M 74.20-0";"Photographic activities"
3;"M743";"M 74.3";"Translation and interpretation activities"
4;"M7430";"M 74.30";"Translation and interpretation activities"
5;"M74300";"M 74.30-0";"Translation and interpretation activities"
3;"M749";"M 74.9";"Other professional, scientific and technical activities n.e.c."
4;"M7490";"M 74.90";"Other professional, scientific and technical activities n.e.c."
5;"M74900";"M 74.90-0";"Other professional, scientific and technical activities n.e.c."
2;"M75";"M 75";"Veterinary activities"
3;"M750";"M 75.0";"Veterinary activities"
4;"M7500";"M 75.00";"Veterinary activities"
5;"M75000";"M 75.00-0";"Veterinary activities"
1;"N";"N";"SECTION N  ADMINISTRATIVE AND SUPPORT SERVICE ACTIVITIES"
2;"N77";"N 77";"Rental and leasing activities"
3;"N771";"N 77.1";"Renting and leasing of motor vehicles"
4;"N7711";"N 77.11";"Renting and leasing of cars and light motor vehicles"
5;"N77111";"N 77.11-1";"Leasing of cars and light motor vehicles"
5;"N77112";"N 77.11-2";"Renting of cars and light motor vehicles"
4;"N7712";"N 77.12";"Renting and leasing of trucks"
5;"N77120";"N 77.12-0";"Renting and leasing of trucks"
3;"N772";"N 77.2";"Renting and leasing of personal and household goods"
4;"N7721";"N 77.21";"Renting and leasing of recreational and sports goods"
5;"N77210";"N 77.21-0";"Renting and leasing of recreational and sports goods"
4;"N7722";"N 77.22";"Renting of video tapes and disks"
5;"N77220";"N 77.22-0";"Renting of video tapes and disks"
4;"N7729";"N 77.29";"Renting and leasing of other personal and household goods"
5;"N77290";"N 77.29-0";"Renting and leasing of other personal and household goods"
3;"N773";"N 77.3";"Renting and leasing of other machinery, equipment and tangible goods"
4;"N7731";"N 77.31";"Renting and leasing of agricultural machinery and equipment"
5;"N77310";"N 77.31-0";"Renting and leasing of agricultural machinery and equipment"
4;"N7732";"N 77.32";"Renting and leasing of construction and civil engineering machinery and equipment"
5;"N77320";"N 77.32-0";"Renting and leasing of construction and civil engineering machinery and equipment"
4;"N7733";"N 77.33";"Renting and leasing of office machinery and equipment (including computers)"
5;"N77330";"N 77.33-0";"Renting and leasing of office machinery and equipment (including computers)"
4;"N7734";"N 77.34";"Renting and leasing of water transport equipment"
5;"N77340";"N 77.34-0";"Renting and leasing of water transport equipment"
4;"N7735";"N 77.35";"Renting and leasing of air transport equipment"
5;"N77350";"N 77.35-0";"Renting and leasing of air transport equipment"
4;"N7739";"N 77.39";"Renting and leasing of other machinery, equipment and tangible goods n.e.c."
5;"N77390";"N 77.39-0";"Renting and leasing of other machinery, equipment and tangible goods n.e.c."
3;"N774";"N 77.4";"Leasing of intellectual property and similar products, except copyrighted works"
4;"N7740";"N 77.40";"Leasing of intellectual property and similar products, except copyrighted works"
5;"N77400";"N 77.40-0";"Leasing of intellectual property and similar products, except copyrighted works"
2;"N78";"N 78";"Employment activities"
3;"N781";"N 78.1";"Activities of employment placement agencies"
4;"N7810";"N 78.10";"Activities of employment placement agencies"
5;"N78100";"N 78.10-0";"Activities of employment placement agencies"
3;"N782";"N 78.2";"Temporary employment agency activities"
4;"N7820";"N 78.20";"Temporary employment agency activities"
5;"N78200";"N 78.20-0";"Temporary employment agency activities"
3;"N783";"N 78.3";"Other human resources provision"
4;"N7830";"N 78.30";"Other human resources provision"
5;"N78300";"N 78.30-0";"Other human resources provision"
2;"N79";"N 79";"Travel agency, tour operator and other reservation service and related activities"
3;"N791";"N 79.1";"Travel agency and tour operator activities"
4;"N7911";"N 79.11";"Travel agency activities"
5;"N79110";"N 79.11-0";"Travel agency activities"
4;"N7912";"N 79.12";"Tour operator activities"
5;"N79120";"N 79.12-0";"Tour operator activities"
3;"N799";"N 79.9";"Other reservation services and related activities"
4;"N7990";"N 79.90";"Other reservation services and related activities"
5;"N79901";"N 79.90-1";"Activities of tourist guides"
5;"N79902";"N 79.90-2";"Activities of tourist offices"
5;"N79909";"N 79.90-9";"Other reservation services and related activities n.e.c."
2;"N80";"N 80";"Security and investigation activities"
3;"N801";"N 80.1";"Private security activities"
4;"N8010";"N 80.10";"Private security activities"
5;"N80100";"N 80.10-0";"Private security activities"
3;"N802";"N 80.2";"Security systems service activities"
4;"N8020";"N 80.20";"Security systems service activities"
5;"N80200";"N 80.20-0";"Security systems service activities"
3;"N803";"N 80.3";"Investigation activities"
4;"N8030";"N 80.30";"Investigation activities"
5;"N80300";"N 80.30-0";"Investigation activities"
2;"N81";"N 81";"Services to buildings and landscape activities"
3;"N811";"N 81.1";"Combined facilities support activities"
4;"N8110";"N 81.10";"Combined facilities support activities"
5;"N81101";"N 81.10-1";"Care-taking"
5;"N81102";"N 81.10-2";"Facility management (except care-taking)"
3;"N812";"N 81.2";"Cleaning activities"
4;"N8121";"N 81.21";"General cleaning of buildings"
5;"N81210";"N 81.21-0";"General cleaning of buildings"
4;"N8122";"N 81.22";"Other building and industrial cleaning activities"
5;"N81221";"N 81.22-1";"Chimney cleaning"
5;"N81229";"N 81.22-9";"Other building and industrial cleaning activities n.e.c."
4;"N8129";"N 81.29";"Other cleaning activities"
5;"N81291";"N 81.29-1";"Snow and ice removal"
5;"N81299";"N 81.29-9";"Other cleaning activities n.e.c."
3;"N813";"N 81.3";"Landscape service activities"
4;"N8130";"N 81.30";"Landscape service activities"
5;"N81300";"N 81.30-0";"Landscape service activities"
2;"N82";"N 82";"Office administrative, office support and other business support activities"
3;"N821";"N 82.1";"Office administrative and support activities"
4;"N8211";"N 82.11";"Combined office administrative service activities"
5;"N82110";"N 82.11-0";"Combined office administrative service activities"
4;"N8219";"N 82.19";"Photocopying, document preparation and other specialised office support activities"
5;"N82190";"N 82.19-0";"Photocopying, document preparation and other specialised office support activities"
3;"N822";"N 82.2";"Activities of call centres"
4;"N8220";"N 82.20";"Activities of call centres"
5;"N82200";"N 82.20-0";"Activities of call centres"
3;"N823";"N 82.3";"Organisation of conventions and trade shows"
4;"N8230";"N 82.30";"Organisation of conventions and trade shows"
5;"N82300";"N 82.30-0";"Organisation of conventions and trade shows"
3;"N829";"N 82.9";"Business support service activities n.e.c."
4;"N8291";"N 82.91";"Activities of collection agencies and credit bureaus"
5;"N82910";"N 82.91-0";"Activities of collection agencies and credit bureaus"
4;"N8292";"N 82.92";"Packaging activities"
5;"N82920";"N 82.92-0";"Packaging activities"
4;"N8299";"N 82.99";"Other business support service activities n.e.c."
5;"N82990";"N 82.99-0";"Other business support service activities n.e.c."
1;"O";"O";"SECTION O  PUBLIC ADMINISTRATION AND DEFENCE; COMPULSORY SOCIAL SECURITY"
2;"O84";"O 84";"Public administration and defence; compulsory social security"
3;"O841";"O 84.1";"Administration of the State and the economic and social policy of the community"
4;"O8411";"O 84.11";"General public administration activities"
5;"O84110";"O 84.11-0";"General public administration activities"
4;"O8412";"O 84.12";"Regulation of the activities of providing health care, education, cultural services and other socialservices, excluding social security"
5;"O84120";"O 84.12-0";"Regulation of the activities of providing health care, education, cultural services and other socialservices, excluding social security"
4;"O8413";"O 84.13";"Regulation of and contribution to more efficient operation of businesses"
5;"O84130";"O 84.13-0";"Regulation of and contribution to more efficient operation of businesses"
3;"O842";"O 84.2";"Provision of services to the community as a whole"
4;"O8421";"O 84.21";"Foreign affairs"
5;"O84210";"O 84.21-0";"Foreign affairs"
4;"O8422";"O 84.22";"Defence activities"
5;"O84220";"O 84.22-0";"Defence activities"
4;"O8423";"O 84.23";"Justice and judicial activities"
5;"O84230";"O 84.23-0";"Justice and judicial activities"
4;"O8424";"O 84.24";"Public order and safety activities"
5;"O84240";"O 84.24-0";"Public order and safety activities"
4;"O8425";"O 84.25";"Fire service activities"
5;"O84250";"O 84.25-0";"Fire service activities"
3;"O843";"O 84.3";"Compulsory social security activities"
4;"O8430";"O 84.30";"Compulsory social security activities"
5;"O84300";"O 84.30-0";"Compulsory social security activities"
1;"P";"P";"SECTION P  EDUCATION"
2;"P85";"P 85";"Education"
3;"P851";"P 85.1";"Pre-primary education"
4;"P8510";"P 85.10";"Pre-primary education"
5;"P85100";"P 85.10-0";"Pre-primary education"
3;"P852";"P 85.2";"Primary education"
4;"P8520";"P 85.20";"Primary education"
5;"P85200";"P 85.20-0";"Primary education"
3;"P853";"P 85.3";"Secondary education"
4;"P8531";"P 85.31";"General secondary education"
5;"P85311";"P 85.31-1";"Lower secondary schools including pre-vocational year"
5;"P85312";"P 85.31-2";"Higher-level general secondary schools"
4;"P8532";"P 85.32";"Technical and vocational secondary education"
5;"P85321";"P 85.32-1";"Compulsory technical and vocational schools (apprenticeship)"
5;"P85322";"P 85.32-2";"Medium-level technical and vocational schools"
5;"P85323";"P 85.32-3";"Higher-level technical and vocational schools"
3;"P854";"P 85.4";"Higher education"
4;"P8541";"P 85.41";"Post-secondary non-tertiary education"
5;"P85410";"P 85.41-0";"Post-secondary non-tertiary education"
4;"P8542";"P 85.42";"Tertiary education"
5;"P85420";"P 85.42-0";"Tertiary education"
3;"P855";"P 85.5";"Other education"
4;"P8551";"P 85.51";"Sports and recreation education"
5;"P85510";"P 85.51-0";"Sports and recreation education"
4;"P8552";"P 85.52";"Cultural education"
5;"P85521";"P 85.52-1";"Dancing schools"
5;"P85529";"P 85.52-9";"Other cultural education"
4;"P8553";"P 85.53";"Driving school activities"
5;"P85530";"P 85.53-0";"Driving school activities"
4;"P8559";"P 85.59";"Other education n.e.c."
5;"P85590";"P 85.59-0";"Other education n.e.c."
3;"P856";"P 85.6";"Educational support activities"
4;"P8560";"P 85.60";"Educational support activities"
5;"P85600";"P 85.60-0";"Educational support activities"
1;"Q";"Q";"SECTION Q  HUMAN HEALTH AND SOCIAL WORK ACTIVITIES"
2;"Q86";"Q 86";"Human health activities"
3;"Q861";"Q 86.1";"Hospital activities"
4;"Q8610";"Q 86.10";"Hospital activities"
5;"Q86100";"Q 86.10-0";"Hospital activities"
3;"Q862";"Q 86.2";"Medical and dental practice activities"
4;"Q8621";"Q 86.21";"General medical practice activities"
5;"Q86210";"Q 86.21-0";"General medical practice activities"
4;"Q8622";"Q 86.22";"Specialist medical practice activities"
5;"Q86220";"Q 86.22-0";"Specialist medical practice activities"
4;"Q8623";"Q 86.23";"Dental practice activities"
5;"Q86230";"Q 86.23-0";"Dental practice activities"
3;"Q869";"Q 86.9";"Other human health activities"
4;"Q8690";"Q 86.90";"Other human health activities"
5;"Q86901";"Q 86.90-1";"Ambulance and patients transports"
5;"Q86902";"Q 86.90-2";"Home nursing"
5;"Q86903";"Q 86.90-3";"Psychotherapists, clinical and health psychologists"
5;"Q86909";"Q 86.90-9";"Other human health activities n.e.c."
2;"Q87";"Q 87";"Residential care activities"
3;"Q871";"Q 87.1";"Residential nursing care activities"
4;"Q8710";"Q 87.10";"Residential nursing care activities"
5;"Q87100";"Q 87.10-0";"Residential nursing care activities"
3;"Q872";"Q 87.2";"Residential care activities for mental retardation, mental health and substance abuse"
4;"Q8720";"Q 87.20";"Residential care activities for mental retardation, mental health and substance abuse"
5;"Q87200";"Q 87.20-0";"Residential care activities for mental retardation, mental health and substance abuse"
3;"Q873";"Q 87.3";"Residential care activities for the elderly and disabled"
4;"Q8730";"Q 87.30";"Residential care activities for the elderly and disabled"
5;"Q87300";"Q 87.30-0";"Residential care activities for the elderly and disabled"
3;"Q879";"Q 87.9";"Other residential care activities"
4;"Q8790";"Q 87.90";"Other residential care activities"
5;"Q87900";"Q 87.90-0";"Other residential care activities"
2;"Q88";"Q 88";"Social work activities without accommodation"
3;"Q881";"Q 88.1";"Social work activities without accommodation for the elderly and disabled"
4;"Q8810";"Q 88.10";"Social work activities without accommodation for the elderly and disabled"
5;"Q88100";"Q 88.10-0";"Social work activities without accommodation for the elderly and disabled"
3;"Q889";"Q 88.9";"Other social work activities without accommodation"
4;"Q8891";"Q 88.91";"Child day-care activities"
5;"Q88910";"Q 88.91-0";"Child day-care activities"
4;"Q8899";"Q 88.99";"Other social work activities without accommodation n.e.c."
5;"Q88990";"Q 88.99-0";"Other social work activities without accommodation n.e.c."
1;"R";"R";"SECTION R  ARTS, ENTERTAINMENT AND RECREATION"
2;"R90";"R 90";"Creative, arts and entertainment activities"
3;"R900";"R 90.0";"Creative, arts and entertainment activities"
4;"R9001";"R 90.01";"Performing arts"
5;"R90010";"R 90.01-0";"Performing arts"
4;"R9002";"R 90.02";"Support activities to performing arts"
5;"R90020";"R 90.02-0";"Support activities to performing arts"
4;"R9003";"R 90.03";"Artistic creation"
5;"R90030";"R 90.03-0";"Artistic creation"
4;"R9004";"R 90.04";"Operation of arts facilities"
5;"R90040";"R 90.04-0";"Operation of arts facilities"
2;"R91";"R 91";"Libraries, archives, museums and other cultural activities"
3;"R910";"R 91.0";"Libraries, archives, museums and other cultural activities"
4;"R9101";"R 91.01";"Library and archives activities"
5;"R91010";"R 91.01-0";"Library and archives activities"
4;"R9102";"R 91.02";"Museums activities"
5;"R91020";"R 91.02-0";"Museums activities"
4;"R9103";"R 91.03";"Operation of historical sites and buildings and similar visitor attractions"
5;"R91030";"R 91.03-0";"Operation of historical sites and buildings and similar visitor attractions"
4;"R9104";"R 91.04";"Botanical and zoological gardens and nature reserves activities"
5;"R91040";"R 91.04-0";"Botanical and zoological gardens and nature reserves activities"
2;"R92";"R 92";"Gambling and betting activities"
3;"R920";"R 92.0";"Gambling and betting activities"
4;"R9200";"R 92.00";"Gambling and betting activities"
5;"R92001";"R 92.00-1";"Lottery and other betting activities"
5;"R92002";"R 92.00-2";"Casinos"
5;"R92003";"R 92.00-3";"Operation of gambling machines"
2;"R93";"R 93";"Sports activities and amusement and recreation activities"
3;"R931";"R 93.1";"Sports activities"
4;"R9311";"R 93.11";"Operation of sports facilities"
5;"R93111";"R 93.11-1";"Operation of swimming pools and stadiums"
5;"R93119";"R 93.11-9";"Operation of other sports arenas and stadiums"
4;"R9312";"R 93.12";"Activities of sport clubs"
5;"R93120";"R 93.12-0";"Activities of sport clubs"
4;"R9313";"R 93.13";"Fitness facilities"
5;"R93130";"R 93.13-0";"Fitness facilities"
4;"R9319";"R 93.19";"Other sports activities"
5;"R93190";"R 93.19-0";"Other sports activities"
3;"R932";"R 93.2";"Amusement and recreation activities"
4;"R9321";"R 93.21";"Activities of amusement parks and theme parks"
5;"R93210";"R 93.21-0";"Activities of amusement parks and theme parks"
4;"R9329";"R 93.29";"Other amusement and recreation activities"
5;"R93290";"R 93.29-0";"Other amusement and recreation activities"
1;"S";"S";"SECTION S  OTHER SERVICE ACTIVITIES"
2;"S94";"S 94";"Activities of membership organisations"
3;"S941";"S 94.1";"Activities of business, employers and professional membership organisations"
4;"S9411";"S 94.11";"Activities of business and employers membership organisations"
5;"S94110";"S 94.11-0";"Activities of business and employers membership organisations"
4;"S9412";"S 94.12";"Activities of professional membership organisations"
5;"S94120";"S 94.12-0";"Activities of professional membership organisations"
3;"S942";"S 94.2";"Activities of trade unions"
4;"S9420";"S 94.20";"Activities of trade unions"
5;"S94200";"S 94.20-0";"Activities of trade unions"
3;"S949";"S 94.9";"Activities of other membership organisations"
4;"S9491";"S 94.91";"Activities of religious organisations"
5;"S94910";"S 94.91-0";"Activities of religious organisations"
4;"S9492";"S 94.92";"Activities of political organisations"
5;"S94920";"S 94.92-0";"Activities of political organisations"
4;"S9499";"S 94.99";"Activities of other membership organisations n.e.c."
5;"S94990";"S 94.99-0";"Activities of other membership organisations n.e.c."
2;"S95";"S 95";"Repair of computers and personal and household goods"
3;"S951";"S 95.1";"Repair of computers and communication equipment"
4;"S9511";"S 95.11";"Repair of computers and peripheral equipment"
5;"S95110";"S 95.11-0";"Repair of computers and peripheral equipment"
4;"S9512";"S 95.12";"Repair of communication equipment"
5;"S95120";"S 95.12-0";"Repair of communication equipment"
3;"S952";"S 95.2";"Repair of personal and household goods"
4;"S9521";"S 95.21";"Repair of consumer electronics"
5;"S95210";"S 95.21-0";"Repair of consumer electronics"
4;"S9522";"S 95.22";"Repair of household appliances and home and garden equipment"
5;"S95220";"S 95.22-0";"Repair of household appliances and home and garden equipment"
4;"S9523";"S 95.23";"Repair of footwear and leather goods"
5;"S95230";"S 95.23-0";"Repair of footwear and leather goods"
4;"S9524";"S 95.24";"Repair of furniture and home furnishings"
5;"S95240";"S 95.24-0";"Repair of furniture and home furnishings"
4;"S9525";"S 95.25";"Repair of watches, clocks and jewellery"
5;"S95250";"S 95.25-0";"Repair of watches, clocks and jewellery"
4;"S9529";"S 95.29";"Repair of other personal and household goods"
5;"S95290";"S 95.29-0";"Repair of other personal and household goods"
2;"S96";"S 96";"Other personal service activities"
3;"S960";"S 96.0";"Other personal service activities"
4;"S9601";"S 96.01";"Washing and (dry-)cleaning of textile and fur products"
5;"S96011";"S 96.01-1";"Laundries"
5;"S96012";"S 96.01-2";"Dry-cleaning"
4;"S9602";"S 96.02";"Hairdressing and other beauty treatment"
5;"S96021";"S 96.02-1";"Hairdressing"
5;"S96022";"S 96.02-2";"Beauty treatment"
5;"S96023";"S 96.02-3";"Pedicures"
4;"S9603";"S 96.03";"Funeral and related activities"
5;"S96030";"S 96.03-0";"Funeral and related activities"
4;"S9604";"S 96.04";"Physical well-being activities"
5;"S96041";"S 96.04-1";"Operation of reducing and slendering salons and massage salons"
5;"S96042";"S 96.04-2";"Operation of spas"
5;"S96049";"S 96.04-9";"Operation of solariums, saunas and baths n.e.c."
4;"S9609";"S 96.09";"Other personal service activities n.e.c."
5;"S96090";"S 96.09-0";"Other personal service activities n.e.c."
1;"T";"T";"SECTION T  ACTIVITIES OF HOUSEHOLDS AS EMPLOYERS; UNDIFFERENTIATED GOODS- AND SERVICES-PRODUCING ACTIVITIES OF HOUSEHOLDS FOR OWN USE"
2;"T97";"T 97";"Activities of households as employers of domestic personnel"
3;"T970";"T 97.0";"Activities of households as employers of domestic personnel"
4;"T9700";"T 97.00";"Activities of households as employers of domestic personnel"
5;"T97000";"T 97.00-0";"Activities of households as employers of domestic personnel"
2;"T98";"T 98";"Undifferentiated goods- and services-producing activities of private households for own use"
3;"T981";"T 98.1";"Undifferentiated goods-producing activities of private households for own use"
4;"T9810";"T 98.10";"Undifferentiated goods-producing activities of private households for own use"
5;"T98100";"T 98.10-0";"Undifferentiated goods-producing activities of private households for own use"
3;"T982";"T 98.2";"Undifferentiated service-producing activities of private households for own use"
4;"T9820";"T 98.20";"Undifferentiated service-producing activities of private households for own use"
5;"T98200";"T 98.20-0";"Undifferentiated service-producing activities of private households for own use"
1;"U";"U";"SECTION U  ACTIVITIES OF EXTRATERRITORIAL ORGANISATIONS AND BODIES"
2;"U99";"U 99";"Activities of extraterritorial organisations and bodies"
3;"U990";"U 99.0";"Activities of extraterritorial organisations and bodies"
4;"U9900";"U 99.00";"Activities of extraterritorial organisations and bodies"
5;"U99000";"U 99.00-0";"Activities of extraterritorial organisations and bodies"
