package Business::EDI::Test;

use strict;
use warnings;

use JSON::XS;
use Exporter;

use base qw/Exporter/;
use vars qw/$VERSION @EXPORT_OK $parser/;

BEGIN {
    $VERSION = 0.02;
    @EXPORT_OK = qw/ JSONObject2Perl ordrsp_data pretty_json /;
}


sub JSONObject2Perl {
    my $obj = shift;
    if ( ref $obj eq 'HASH' ) {
        if ( defined $obj->{'__c'} ) {
            die "We somehow got a special (Evergreen) object in our data";
        }
        # is a hash w/o class marker; simply revivify innards
        for my $k (keys %$obj) {
            $obj->{$k} = JSONObject2Perl($obj->{$k}) unless ref $obj->{$k} eq 'JSON::XS::Boolean';
        }
    } elsif ( ref $obj eq 'ARRAY' ) {
        for my $i (0..scalar(@$obj) - 1) {
            $obj->[$i] = JSONObject2Perl($obj->[$i]) unless ref $obj->[$i] eq 'JSON::XS::Boolean';
        }
    } 
    # ELSE: return vivified non-class hashes, all arrays, and anything that isn't a hash or array ref
    return $obj;
}

sub pretty_json {
    @_ or die "pretty_json() missing required argument";
    unless ($parser) {
        $parser = JSON::XS->new;
        $parser->ascii(1);        # output \u escaped strings for any char with a value over 127
        $parser->allow_nonref(1); # allows non-reference values to equate to themselves (see perldoc)
    }
    return $parser->indent(1)->space_before(1)->decode(shift);
}

sub ordrsp_data {
    unless ($parser) {
        $parser = JSON::XS->new;
        $parser->ascii(1);        # output \u escaped strings for any char with a value over 127
        $parser->allow_nonref(1); # allows non-reference values to equate to themselves (see perldoc)
    }

    my $slurp = join('', <DATA>) or die __PACKAGE__ . " FAILED reading DATA handle";
    my $foo = ($parser->decode($slurp)) or warn "JSON parser failed to decode DATA";
    return JSONObject2Perl($foo);
}

1;

__DATA__

{
"body": [
  {
     "ORDRSP": [
        [
           "UNH",
           {
              "0062": "723",
              "S009": {
                 "0051": "UN",
                 "0052": "D",
                 "0065": "ORDRSP",
                 "0054": "96A"
              }
           }
        ],
        [
           "BGM",
           {
              "1004": "582822",
              "4343": "AC",
              "1225": "29",
              "C002": {
                 "1001": "231"
              }
           }
        ],
        [
           "DTM",
           {
              "C507": {
                 "2005": "137",
                 "2379": "102",
                 "2380": "20070618"
              }
           }
        ],
        [
           "SG1",
           [
              [
                 "RFF",
                 {
                    "C506": {
                       "1154": "E07158FIC",
                       "1153": "ON"
                    }
                 }
              ]
           ]
        ],
        [
           "SG3",
           [
              [
                 "NAD",
                 {
                    "C082": {
                       "3039": "8888888",
                       "3055": "31B"
                    },
                    "3035": "BY"
                 }
              ]
           ]
        ],
        [
           "SG3",
           [
              [
                 "NAD",
                 {
                    "C082": {
                       "3039": "1556150",
                       "3055": "31B"
                    },
                    "3035": "SU"
                 }
              ]
           ]
        ],
        [
           "SG3",
           [
              [
                 "NAD",
                 {
                    "C082": {
                       "3039": "8888888",
                       "3055": "91"
                    },
                    "3035": "BY"
                 }
              ]
           ]
        ],
        [
           "SG8",
           [
              [
                 "CUX",
                 {
                    "C504": [
                       {
                          "6345": "USD",
                          "6343": "9",
                          "6347": "2"
                       }
                    ]
                 }
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 1,
                    "C212": {
                       "7140": "9781576734131",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "LACY, AL THINGS NOT SEEN"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 10.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/1",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 2,
                    "C212": {
                       "7140": "9781590529966",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "LACY, AL FINAL JUSTICE"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 1
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 1
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 14.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/2",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 3,
                    "C212": {
                       "7140": "9780374502003",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "MALAMUD, B NATURAL"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 14,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/3",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 4,
                    "C212": {
                       "7140": "9780307263964",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "SCOTT, PAU RAJ QUARTET THE JEWEL IN"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 2
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 2
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 32.5,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/4",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 5,
                    "C212": {
                       "7140": "9780743219600",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "JAMES, P.  SHROUD FOR A NIGHTINGALE"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 14,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/6",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 6,
                    "C212": {
                       "7140": "9780842329217",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "LAHAYE, TI TRIBULATION FORCE THE CO"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 14.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/7",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 7,
                    "C212": {
                       "7140": "9780743470988",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "ZANE AFTERBURN A NOVEL"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 15,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/8",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 8,
                    "C212": {
                       "7140": "9780060096199",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "CABOT, MEG BOY NEXT DOOR"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 13.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/9",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 9,
                    "C212": {
                       "7140": "9780385334204",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "VONNEGUT,  BREAKFAST OF CHAMPIONS"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 14,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/10",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 10,
                    "C212": {
                       "7140": "9781593083526",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "DOSTOYEVSK BROTHERS KARAMAZOV"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 9.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/11",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 11,
                    "C212": {
                       "7140": "9780842387446",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "KINGSBURY, FORGIVEN"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 6
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 6
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 13.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/12",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 12,
                    "C212": {
                       "7140": "9780374299163",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "BERLINSKI, FIELDWORK"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 24,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/13",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 13,
                    "C212": {
                       "7140": "9780743249317",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "GREGORY, P MERIDON A NOVEL"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 16,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/14",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 14,
                    "C212": {
                       "7140": "9781400031368",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "MCCALL SMI MORALITY FOR BEAUTIFUL G"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 12.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/15",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 15,
                    "C212": {
                       "7140": "9780380794874",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "CLEAGE, PE WHAT LOOKS LIKE CRAZY ON"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 13.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/16",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 16,
                    "C212": {
                       "7140": "9780743249294",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "GREGORY, P WIDEACRE"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 16,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/17",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 17,
                    "C212": {
                       "7140": "9780060956691",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "FERBER, ED SO BIG"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 3
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "07",
                       "3055": "28",
                       "1131": "7B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 13,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/18",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 18,
                    "C212": {
                       "7140": "9780743227445",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "GREGORY, P OTHER BOLEYN GIRL"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 16,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4639/19",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "UNS",
           {
              "0081": "S"
           }
        ],
        [
           "CNT",
           {
              "C270": {
                 "6066": 18,
                 "6069": "2"
              }
           }
        ],
        [
           "UNT",
           {
              "0062": "723",
              "0074": 155
           }
        ]
     ]
  },
  {
     "ORDRSP": [
        [
           "UNH",
           {
              "0062": "724",
              "S009": {
                 "0051": "UN",
                 "0052": "D",
                 "0065": "ORDRSP",
                 "0054": "96A"
              }
           }
        ],
        [
           "BGM",
           {
              "1004": "582828",
              "4343": "AC",
              "1225": "29",
              "C002": {
                 "1001": "231"
              }
           }
        ],
        [
           "DTM",
           {
              "C507": {
                 "2005": "137",
                 "2379": "102",
                 "2380": "20070618"
              }
           }
        ],
        [
           "SG1",
           [
              [
                 "RFF",
                 {
                    "C506": {
                       "1154": "E07159ANF",
                       "1153": "ON"
                    }
                 }
              ]
           ]
        ],
        [
           "SG3",
           [
              [
                 "NAD",
                 {
                    "C082": {
                       "3039": "8888888",
                       "3055": "31B"
                    },
                    "3035": "BY"
                 }
              ]
           ]
        ],
        [
           "SG3",
           [
              [
                 "NAD",
                 {
                    "C082": {
                       "3039": "1556150",
                       "3055": "31B"
                    },
                    "3035": "SU"
                 }
              ]
           ]
        ],
        [
           "SG3",
           [
              [
                 "NAD",
                 {
                    "C082": {
                       "3039": "8888888",
                       "3055": "91"
                    },
                    "3035": "BY"
                 }
              ]
           ]
        ],
        [
           "SG8",
           [
              [
                 "CUX",
                 {
                    "C504": [
                       {
                          "6345": "USD",
                          "6343": "9",
                          "6347": "2"
                       }
                    ]
                 }
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 1,
                    "C212": {
                       "7140": "9781571313027",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "HOLM, BILL WINDOWS OF BRIMNES"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 3
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 22,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/1",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 2,
                    "C212": {
                       "7140": "9781413306439",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "REPA, BARB YOUR RIGHTS IN THE WORKP"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 4
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 29.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/2",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 3,
                    "C212": {
                       "7140": "9781413306910",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "GUERIN, LI ESSENTIAL GUIDE TO WORKP"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 2
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 2
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 39.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/3",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 4,
                    "C212": {
                       "7140": "9781413307023",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "CLIFFORD,  ESTATE PLANNING BASICS"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 5
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 21.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/4",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 5,
                    "C212": {
                       "7140": "9780778801603",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "FRIEDMAN,  BABY CARE BOOK"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 8
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 29.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/5",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 6,
                    "C212": {
                       "7140": "9781554072873",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "KING, RUSS ATLAS OF HUMAN MIGRATION"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 3
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 40,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/6",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 7,
                    "C212": {
                       "7140": "9781554073139",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "ASH, RUSSE FIREFLYS WORLD OF FACTS"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 6
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 6
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 29.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/7",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 8,
                    "C212": {
                       "7140": "9781413307122",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "WARNER, RA 101 LAW FORMS FOR PERSON"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 6
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 6
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 29.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/8",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 9,
                    "C212": {
                       "7140": "9781413306286",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "BRAY, ILON NOLOS ESSENTIAL GUIDE TO"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 10
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 10
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 24.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/9",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 10,
                    "C212": {
                       "7140": "9780749445546",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "WESTWOOD,  HOW TO WRITE A MARKETING"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 1
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 2
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "99",
                       "3055": "28",
                       "1131": "7B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 17.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/10",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 11,
                    "C212": {
                       "7140": "9780061238673",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "ROANE, SUS HOW TO WORK A ROOM YOUR "
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 14.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/11",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 12,
                    "C212": {
                       "7140": "9780873516068",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "GERMAIN, D REACHING PAST THE WIRE A"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 4
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 24.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/12",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 13,
                    "C212": {
                       "7140": "9780873515993",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "KLING, KEV DOG SAYS HOW"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 3
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 22.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/13",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 14,
                    "C212": {
                       "7140": "9780873515856",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "SHORT, SUS BUNDT CAKE BLISS DELICIO"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 5
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 16.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/14",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 15,
                    "C212": {
                       "7140": "9780873516013",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "BRADY, TIM GOPHER GOLD LEGENDARY FI"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 2
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 2
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 24.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/15",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 16,
                    "C212": {
                       "7140": "9780873515948",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "ROBERTS, K MINNESOTA 150 THE PEOPLE"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 8
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 19.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/16",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 17,
                    "C212": {
                       "7140": "9780375424953",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "MAK, GEERT IN EUROPE A JOURNEY THRO"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 2
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 2
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 35,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/17",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 18,
                    "C212": {
                       "7140": "9780312358914",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "DONAHUE, P PARENTING WITHOUT FEAR O"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 5
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 14.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/18",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 19,
                    "C212": {
                       "7140": "9781594864117",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "MURRAY, LI BABYCENTERS ESSENTIAL GU"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 8
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 15.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/19",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 20,
                    "C212": {
                       "7140": "9780762430758",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "LAPINE, MI SNEAKY CHEF SIMPLE STRAT"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 6
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 6
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 17.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4640/20",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "UNS",
           {
              "0081": "S"
           }
        ],
        [
           "CNT",
           {
              "C270": {
                 "6066": 20,
                 "6069": "2"
              }
           }
        ],
        [
           "UNT",
           {
              "0062": "724",
              "0074": 171
           }
        ]
     ]
  },
  {
     "ORDRSP": [
        [
           "UNH",
           {
              "0062": "725",
              "S009": {
                 "0051": "UN",
                 "0052": "D",
                 "0065": "ORDRSP",
                 "0054": "96A"
              }
           }
        ],
        [
           "BGM",
           {
              "1004": "582830",
              "4343": "AC",
              "1225": "29",
              "C002": {
                 "1001": "231"
              }
           }
        ],
        [
           "DTM",
           {
              "C507": {
                 "2005": "137",
                 "2379": "102",
                 "2380": "20070618"
              }
           }
        ],
        [
           "SG1",
           [
              [
                 "RFF",
                 {
                    "C506": {
                       "1154": "E07160FIC",
                       "1153": "ON"
                    }
                 }
              ]
           ]
        ],
        [
           "SG3",
           [
              [
                 "NAD",
                 {
                    "C082": {
                       "3039": "8888888",
                       "3055": "31B"
                    },
                    "3035": "BY"
                 }
              ]
           ]
        ],
        [
           "SG3",
           [
              [
                 "NAD",
                 {
                    "C082": {
                       "3039": "1556150",
                       "3055": "31B"
                    },
                    "3035": "SU"
                 }
              ]
           ]
        ],
        [
           "SG3",
           [
              [
                 "NAD",
                 {
                    "C082": {
                       "3039": "8888888",
                       "3055": "91"
                    },
                    "3035": "BY"
                 }
              ]
           ]
        ],
        [
           "SG8",
           [
              [
                 "CUX",
                 {
                    "C504": [
                       {
                          "6345": "USD",
                          "6343": "9",
                          "6347": "2"
                       }
                    ]
                 }
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 1,
                    "C212": {
                       "7140": "9781400098224",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "SHAW, REBE COUNTRY LOVERS"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 12.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/1",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 2,
                    "C212": {
                       "7140": "9780061136092",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "BLAKE, TON TEMPT ME TONIGHT"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 4
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "07",
                       "3055": "28",
                       "1131": "7B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 13.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/2",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 3,
                    "C212": {
                       "7140": "9780385339162",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "MONING, KA BLOODFEVER"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 6
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 6
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 22,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/3",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 4,
                    "C212": {
                       "7140": "9780758211859",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "MCKENNA, S EDGE OF MIDNIGHT"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 8
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 14,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/4",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 5,
                    "C212": {
                       "7140": "9780727865496",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "BALZO, SAN GROUNDS FOR MURDER"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 8
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 27.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/5",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 6,
                    "C212": {
                       "7140": "9780373772612",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "PALMER, DI HARD TO HANDLE"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 13.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/6",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 7,
                    "C212": {
                       "7140": "9780385341066",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "JONES, LLO MR PIP"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 3
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 20,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/7",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 8,
                    "C212": {
                       "7140": "9780060537326",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "JILES, PAU STORMY WEATHER"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 24.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/8",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 9,
                    "C212": {
                       "7140": "9781416546023",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "DELILLO, D FALLING MAN A NOVEL"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 26,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/9",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 10,
                    "C212": {
                       "7140": "9780758215116",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "MORRISON,  SWEETER THAN HONEY"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 3
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 24,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/10",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 11,
                    "C212": {
                       "7140": "9780756404215",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "SMITH, SHE FOX"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 2
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 2
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 25.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/11",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 12,
                    "C212": {
                       "7140": "9780375424861",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "GROSSMAN,  SOON I WILL BE INVINCIBL"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 2
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 2
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 22.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/12",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 13,
                    "C212": {
                       "7140": "9780143036746",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "LEWYCKA, M SHORT HISTORY OF TRACTOR"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 3
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 14,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/13",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 14,
                    "C212": {
                       "7140": "9780312375669",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "BANNISTER, FLAWED"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 4
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 4
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "03",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 24.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/14",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 15,
                    "C212": {
                       "7140": "9780764201103",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "ALEXANDER, REMEMBERED"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 13.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/15",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 16,
                    "C212": {
                       "7140": "9781566891943",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "TANIGUCHI, OCEAN IN THE CLOSET"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 2
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 2
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 14.95,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/16",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 17,
                    "C212": {
                       "7140": "9780736917018",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "HENKE, ROX SECRET OF US"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 13.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/17",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 18,
                    "C212": {
                       "7140": "9781590529218",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "HERMAN, KA EVER PRESENT DANGER"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 12.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/18",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "24",
                    "1082": 19,
                    "C212": {
                       "7140": "9781414311654",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "CHAPMAN, G IT HAPPENS EVERY SPRING"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 5
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 0
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "83",
                       "6060": 5
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "07",
                       "3055": "28",
                       "1131": "7B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 12.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/19",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "SG26",
           [
              [
                 "LIN",
                 {
                    "1229": "5",
                    "1082": 20,
                    "C212": {
                       "7140": "9781591453628",
                       "7143": "EN"
                    }
                 }
              ],
              [
                 "IMD",
                 {
                    "7081": "BST",
                    "7077": "F",
                    "C273": {
                       "7008": [
                          "JACKSON, N YADA YADA PRAYER GROUP G"
                       ]
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "21",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "12",
                       "6060": 8
                    }
                 }
              ],
              [
                 "QTY",
                 {
                    "C186": {
                       "6063": "85",
                       "6060": 0
                    }
                 }
              ],
              [
                 "FTX",
                 {
                    "4451": "LIN",
                    "C107": {
                       "4441": "01",
                       "3055": "28",
                       "1131": "8B"
                    }
                 }
              ],
              [
                 "SG30",
                 [
                    [
                       "PRI",
                       {
                          "C509": {
                             "5118": 13.99,
                             "5387": "SRP",
                             "5125": "AAB"
                          }
                       }
                    ]
                 ]
              ],
              [
                 "SG31",
                 [
                    [
                       "RFF",
                       {
                          "C506": {
                             "1154": "4641/20",
                             "1153": "LI"
                          }
                       }
                    ]
                 ]
              ]
           ]
        ],
        [
           "UNS",
           {
              "0081": "S"
           }
        ],
        [
           "CNT",
           {
              "C270": {
                 "6066": 20,
                 "6069": "2"
              }
           }
        ],
        [
           "UNT",
           {
              "0062": "725",
              "0074": 171
           }
        ]
     ]
  }
],
"trailer": [
  "UNZ",
  {
     "0036": 1,
     "0020": "2045"
  }
],
"recipient": "8888888",
"sender": "1556150",
"recipient_qual": "31B",
"sender_qual": "31B",
"UNA": {
  "ce_sep": ":",
  "decimal_sign": ".",
  "de_sep": "+",
  "seg_term": "'",
  "rep_sep": " ",
  "esc_char": "?"
},
"header": [
  "UNB",
  {
     "S003": {
        "0007": "31B",
        "0010": "8888888"
     },
     "S004": {
        "0017": 70618,
        "0019": 1556
     },
     "S001": {
        "0002": 3,
        "0001": "UNOC"
     },
     "S002": {
        "0004": "1556150",
        "0007": "31B"
     },
     "0020": "2045"
  }
]
}
