#!/usr/bin/perl -w
# -*- cperl -*-
# PODNAME: spel-wizard.pl
# ABSTRACT: converts spelchunks to to plain text
#


use strict;

use Getopt::Long;
use Pod::Usage;

use Config::Tiny;

use File::Spec;
use File::ShareDir;

use SpeL::Wizard;

sub dieExit;

my $version = 0.9;
my $author  = 'Walter Daems <walter.daems@uantwerpen.be>';

######################################
# Read command-line options/arguments
####################################

my $opt_help;
my $opt_man;
my $opt_verbose = 0;
my $opt_test = 0;


my $result = 
  GetOptions( "help"      => \$opt_help,
	      "man"       => \$opt_man,
	      "verbose"   => sub{++$opt_verbose},
	      "test"      => \$opt_test )
  || pod2usage( -exitval => 1,
		-output  => \*STDERR );
pod2usage( -exitval => 100, -verbose => 1 ) if ($opt_help);
pod2usage( -exitval => 101, -verbose => 2 ) if ($opt_man);

my $argument = $ARGV[0] if( @ARGV == 1 );

pod2usage( -message => "Incorrect number of arguments",
	   -exitval => 1,
	   -output  => \*STDERR ) unless( defined( $argument ) );


###################
# Read config file
#################
my $configfilename = $ENV{'SPELWIZARD_CONFIG'} ?
  $ENV{'SPELWIZARD_CONFIG'} :
  ( File::Spec->catfile( File::ShareDir::dist_dir( 'SpeL-Wizard' ) , "tts.conf") );
dieExit( 13, "Error: cannot find config file" )
  unless( -r $configfilename );
my $config = Config::Tiny->read( $configfilename )
  or dieExit( 14, $Config::Tiny::errstr . "\nin file '$configfilename'\n" );


########################
# generate opening line
######################
my $openingline = "spel-wizard.pl - v$version - $author";
say STDERR $openingline . "\n" .
  ("=" x length( $openingline ) )
  if( $opt_verbose >= 1 );

my $spelWizard = SpeL::Wizard->new( $argument, $config );

###################
# do the hard work
#################

# parse the aux file
eval {
  $spelWizard->parseAuxFile( $opt_verbose, $opt_test )
    or exit( 102 );
  1;
} or do {
  say STDERR "$@";
  exit( 15 );
};

# parse the spelidx file / generate the audio / generate the playlist
eval {
  $spelWizard->parseChunks( $opt_verbose, $opt_test );
  1;
} or do {
  say STDERR "$@";
  exit( 16 );
};

########################
# generate closing line
######################
say STDERR "Done." if( $opt_verbose >= 1 );

#########################################################

sub dieExit {
  my ( $code, $message ) = @_;
  say STDERR $message;
  exit( $code );
}

__END__

=pod

=encoding UTF-8

=head1 NAME

spel-wizard.pl - converts spelchunks to to plain text

=head1 VERSION

version 20240616.1411

=head1 SYNOPSIS

spel-wizard.pl [--help | -h] [--man|-m] [--test|-t] <jobname>

=head1 DESCRIPTION

The B<spel-wizard.pl> script reads a SpeL index (F<.spelidx>) file
together with the F<.aux> file (both generated by LaTeX) and converts
the text of your LaTeX document (the text chunks) to audio files using
a text-to-speech tool such as festival or awspolly.  As a side effect
it also produces MD5 fingerprints of these texts and an M3U audio file
you can use as the start of an audio-book.

The B<spel-wizard.pl> script is part of the B<SpeLbox> package that
contains multiple items:

=over 4

=item - a LaTeX package called F<spel.sty> (Speech-enabled LaTeX)

=item - an Orgmode exporter called I<spel>

=item - this script together with all modules from the SpeL::Wizard perl
package

=back

SpeLbox stands for Speech-enabled LaTeX by Orgmode eXport.

Starting by reading the documentation of the LaTeX package makes sense.
We recommend reading this documentation after that.
Finish with the Orgmode documentation if you still feel like it.

=head2 The F<.spelidx> file

The index file is line-based. Every line consists of multiple fields
separated by a pipe (C<|>) symbol.  Two types of lines exist:
I<reading lines>, containing information about what needs to be read
out loud, and I<service lines> that contain service information.

=head3 Reading lines

The first field indicates the element type (C<title>, C<author>,
C<date>, C<part>, C<chapter>, C<(sub)(sub)section, item>,
C<chunk>). These lines are items that require text-to-speech
conversion. The second field of these lines specifies the basename of
the (F<.tex>) file that contains the text to be read

=head3 Service lines

The service lines are described below:

=over 4

=item format: format of the audio that needs to be generated (ogg, wav
or mp3)

=item audiodir: specifies the directory in which the audio is to be generated

=item chunkdir: specifies the directory in which the F<.tex>-files
(that need to be converted tot speech) are located

=item language: specifies in wich language the subsequent 

=item macpp: specifies information about how a macro should be
preprocessed

=item envpp: specifies information about how an environment should be
preprocessed

=item macad: specifies an audiodescription for a macro

=item envad: specifies an audiodescription for an environment

=back

=head2 The configuration file

=head1 NAME

spel-wizard.pl - Incantating LaTeX into natural language

=head1 OPTIONS

=over 4

=item B<--help> | B<-h>

prints help message on standard output

=item B<--man> | B<-m>

prints manual page on standard output

=item B<--test> | B<-t>

write the results to standard output; this facility is provided for
development, debugging and testing purposes.

=item B<--verbose> | B<-v>

increase the verbosity of the script. Usually a single C<-v> is a
sensible amount of verbosity.

=back

=head1 ARGUMENT

=over 4

=item <jobname>

basename of your latex input file; this will allow C<spel-wizard.pl>
to find the F<.aux> file and the F<.spelidx> file.

=back

=head1 RETURN VALUE

=over 32

=item 0 if no errors occurred

=item 1 if a command-line syntax error occurred

=item 100 if the help message was invoked (e.g., using '-h')

=item 101 if the man page was invoked (e.g., using '-m')

=item 102 if the script was run in test mode (using '-t')

=item 13 insufficient read permissions for the config file

=item 14 parsing the config file failed

=back

=head1 BUGS

No bugs have been reported so far. If you find any, please,
send an e-mail to the author containing:

=over 32

=item - what you were trying;

=item - enough data such that I can reproduce your attempt
(F<.spelidx> file, F<.aux> file and the contents of your F<spel>
directory)

=item - what strange behavior you observed;

=item - what normal behavior you would have expected.

=back

=head1 LINKS

=over 32

=item https://metacpan.org/pod/SpeL::Wizard

=item https://ctan.org/pkg/spel

=item https://www.melpa.org/#/ox-spel

=back

=head1 AUTHOR

Walter Daems <wdaems@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2024 by Walter Daems.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=head1 CONTRIBUTOR

=for stopwords Paul Levrie

Paul Levrie

=cut
