// license:BSD-3-Clause
// copyright-holders: David Haywood, ElSemi

/* IGS 027 Encryptions

  the IGS027 chips are typically used with encrypted ROMs.

  Games with an External ARM ROM usually have that ROM encrypted
  otherwise (in the case of PGM) the 68k ROM ends up encrypted
  instead.  The encryption is controlled by a set of registers
  mapped to ARM space where the table and some additional values
  are uploaded.  We don't currently emulate this in realtime
  even for cases where the internal ROM is dumped.

*/



#include "emu.h"
#include "pgmcrypt.h"

#include "endianness.h"


// a common set of these are used, selectable in the real scheme by an uploaded bitfield
// some may be duplicates simply due to differing ROM sizes

#define IGS27_CRYPT1 \
	if ((i & 0x040480) != 0x000080) x ^= 0x0001;
#define IGS27_CRYPT1_ALT \
	if ((i & 0x040080) != 0x000080) x ^= 0x0001;
#define IGS27_CRYPT1_ALT2 \
	if ((i & 0x000480) != 0x000080) x ^= 0x0001;
#define IGS27_CRYPT2 \
	if ((i & 0x104008) == 0x104008) x ^= 0x0002;
#define IGS27_CRYPT2_ALT \
	if ((i & 0x004008) == 0x004008) x ^= 0x0002;
#define IGS27_CRYPT2_ALT2 \
	if ((i & 0x004008) == 0x004008 && (i & 0x180000) != 0x000000) x ^= 0x0002;
#define IGS27_CRYPT2_ALT3 \
	if ((i & 0x084008) == 0x084008) x ^= 0x0002;
#define IGS27_CRYPT3 \
	if ((i & 0x080030) == 0x080010) x ^= 0x0004;
#define IGS27_CRYPT3_ALT \
	if ((i & 0x000030) == 0x000010 && (i & 0x180000) != 0x080000) x ^= 0x0004;
#define IGS27_CRYPT3_ALT2 \
	if ((i & 0x000030) == 0x000010) x ^= 0x0004;
// ket - due to address starting at 0 and not 100000/2!
#define IGS27_CRYPT3_ALT3 \
	if ((i & 0x080030) == 0x000010) x ^= 0x0004;
#define IGS27_CRYPT4 \
	if ((i & 0x000242) != 0x000042) x ^= 0x0008;
#define IGS27_CRYPT4_ALT \
	if ((i & 0x000042) != 0x000042) x ^= 0x0008;
#define IGS27_CRYPT5 \
	if ((i & 0x008100) == 0x008000) x ^= 0x0010;
#define IGS27_CRYPT5_ALT \
	if ((i & 0x048100) == 0x048000) x ^= 0x0010;
#define IGS27_CRYPT6 \
	if ((i & 0x002004) != 0x000004) x ^= 0x0020;
#define IGS27_CRYPT6_ALT \
	if ((i & 0x022004) != 0x000004) x ^= 0x0020;
#define IGS27_CRYPT7 \
	if ((i & 0x011800) != 0x010000) x ^= 0x0040;
#define IGS27_CRYPT7_ALT \
	if ((i & 0x001800) != 0x000000) x ^= 0x0040;
#define IGS27_CRYPT8 \
	if ((i & 0x004820) == 0x004820) x ^= 0x0080;
#define IGS27_CRYPT8_ALT \
	if ((i & 0x000820) == 0x000820) x ^= 0x0080;

static const uint8_t kov_tab[256] = {
	0x17, 0x1c, 0xe3, 0x02, 0x62, 0x59, 0x97, 0x4a, 0x67, 0x4d, 0x1f, 0x11, 0x76, 0x64, 0xc1, 0xe1,
	0xd2, 0x41, 0x9f, 0xfd, 0xfa, 0x04, 0xfe, 0xab, 0x89, 0xeb, 0xc0, 0xf5, 0xac, 0x2b, 0x64, 0x22,
	0x90, 0x7d, 0x88, 0xc5, 0x8c, 0xe0, 0xd9, 0x70, 0x3c, 0xf4, 0x7d, 0x31, 0x1c, 0xca, 0xe2, 0xf1,
	0x31, 0x82, 0x86, 0xb1, 0x55, 0x95, 0x77, 0x01, 0x77, 0x3b, 0xab, 0xe6, 0x88, 0xef, 0x77, 0x11,
	0x56, 0x01, 0xac, 0x55, 0xf7, 0x6d, 0x9b, 0x6d, 0x92, 0x14, 0x23, 0xae, 0x4b, 0x80, 0xae, 0x6a,
	0x43, 0xcc, 0x35, 0xfe, 0xa1, 0x0d, 0xb3, 0x21, 0x4e, 0x4c, 0x99, 0x80, 0xc2, 0x3d, 0xce, 0x46,
	0x9b, 0x5d, 0x68, 0x75, 0xfe, 0x1e, 0x25, 0x41, 0x24, 0xa0, 0x79, 0xfd, 0xb5, 0x67, 0x93, 0x07,
	0x3a, 0x78, 0x24, 0x64, 0xe1, 0xa3, 0x62, 0x75, 0x38, 0x65, 0x8a, 0xbf, 0xf9, 0x7c, 0x00, 0xa0,
	0x6d, 0xdb, 0x1f, 0x80, 0x37, 0x37, 0x8e, 0x97, 0x1a, 0x45, 0x61, 0x0e, 0x10, 0x24, 0x8a, 0x27,
	0xf2, 0x44, 0x91, 0x3e, 0x62, 0x44, 0xc5, 0x55, 0xe6, 0x8e, 0x5a, 0x25, 0x8a, 0x90, 0x25, 0x74,
	0xa0, 0x95, 0x33, 0xf7, 0x51, 0xce, 0xe4, 0xa0, 0x13, 0xcf, 0x33, 0x1e, 0x59, 0x5b, 0xec, 0x42,
	0xc5, 0xb8, 0xe4, 0xc5, 0x71, 0x38, 0xc5, 0x6b, 0x8d, 0x1d, 0x84, 0xf8, 0x4e, 0x21, 0x6d, 0xdc,
	0x2c, 0xf1, 0xae, 0xad, 0x19, 0xc5, 0xed, 0x8e, 0x36, 0xb5, 0x81, 0x94, 0xfe, 0x62, 0x3a, 0xe8,
	0xc9, 0x95, 0x84, 0xbd, 0x65, 0x15, 0x16, 0x15, 0xd2, 0xe7, 0x16, 0xd7, 0x9c, 0xd3, 0xd2, 0x66,
	0xf6, 0x46, 0xe3, 0x32, 0x62, 0x51, 0x86, 0x4a, 0x67, 0xcc, 0x4d, 0xea, 0x37, 0x45, 0xd5, 0xa6,
	0x80, 0xe6, 0xba, 0xb3, 0x08, 0xd8, 0x30, 0x5b, 0x5f, 0xf2, 0x5a, 0xfb, 0x63, 0xb0, 0xa4, 0x41
};

void pgm_kov_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= kov_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t kovsh_tab[256] = {
	0xe7, 0x06, 0xa3, 0x70, 0xf2, 0x58, 0xe6, 0x59, 0xe4, 0xcf, 0xc2, 0x79, 0x1d, 0xe3, 0x71, 0x0e,
	0xb6, 0x90, 0x9a, 0x2a, 0x8c, 0x41, 0xf7, 0x82, 0x9b, 0xef, 0x99, 0x0c, 0xfa, 0x2f, 0xf1, 0xfe,
	0x8f, 0x70, 0xf4, 0xc1, 0xb5, 0x3d, 0x7c, 0x60, 0x4c, 0x09, 0xf4, 0x2e, 0x7c, 0x87, 0x63, 0x5f,
	0xce, 0x99, 0x84, 0x95, 0x06, 0x9a, 0x20, 0x23, 0x5a, 0xb9, 0x52, 0x95, 0x48, 0x2c, 0x84, 0x60,
	0x69, 0xe3, 0x93, 0x49, 0xb9, 0xd6, 0xbb, 0xd6, 0x9e, 0xdc, 0x96, 0x12, 0xfa, 0x60, 0xda, 0x5f,
	0x55, 0x5d, 0x5b, 0x20, 0x07, 0x1e, 0x97, 0x42, 0x77, 0xea, 0x1d, 0xe0, 0x70, 0xfb, 0x6a, 0x00,
	0x77, 0x9a, 0xef, 0x1b, 0xe0, 0xf9, 0x0d, 0xc1, 0x2e, 0x2f, 0xef, 0x25, 0x29, 0xe5, 0xd8, 0x2c,
	0xaf, 0x01, 0xd9, 0x6c, 0x31, 0xce, 0x5c, 0xea, 0xab, 0x1c, 0x92, 0x16, 0x61, 0xbc, 0xe4, 0x7c,
	0x5a, 0x76, 0xe9, 0x92, 0x39, 0x5b, 0x97, 0x60, 0xea, 0x57, 0x83, 0x9c, 0x92, 0x29, 0xa7, 0x12,
	0xa9, 0x71, 0x7a, 0xf9, 0x07, 0x68, 0xa7, 0x45, 0x88, 0x10, 0x81, 0x12, 0x2c, 0x67, 0x4d, 0x55,
	0x33, 0xf0, 0xfa, 0xd7, 0x1d, 0x4d, 0x0e, 0x63, 0x03, 0x34, 0x65, 0xe2, 0x76, 0x0f, 0x98, 0xa9,
	0x5f, 0x9a, 0xd3, 0xca, 0xdd, 0xc1, 0x5b, 0x3d, 0x4d, 0xf8, 0x40, 0x08, 0xdc, 0x05, 0x38, 0x00,
	0xcb, 0x24, 0x02, 0xff, 0x39, 0xe2, 0x9e, 0x04, 0x9a, 0x08, 0x63, 0xc8, 0x2b, 0x5a, 0x34, 0x06,
	0x62, 0xc1, 0xbb, 0x8a, 0xd0, 0x54, 0x4c, 0x43, 0x21, 0x4e, 0x4c, 0x99, 0x80, 0xc2, 0x3d, 0xce,
	0x2a, 0x7b, 0x09, 0x62, 0x1a, 0x91, 0x9b, 0xc3, 0x41, 0x24, 0xa0, 0xfd, 0xb5, 0x67, 0x93, 0x07,
	0xa7, 0xb8, 0x85, 0x8a, 0xa1, 0x1e, 0x4f, 0xb6, 0x75, 0x38, 0x65, 0x8a, 0xf9, 0x7c, 0x00, 0xa0
};


void pgm_kovsh_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= kovsh_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t photoy2k_tab[256] = {
	0xd9, 0x92, 0xb2, 0xbc, 0xa5, 0x88, 0xe3, 0x48, 0x7d, 0xeb, 0xc5, 0x4d, 0x31, 0xe4, 0x82, 0xbc,
	0x82, 0xcf, 0xe7, 0xf3, 0x15, 0xde, 0x8f, 0x91, 0xef, 0xc6, 0xb8, 0x81, 0x97, 0xe3, 0xdf, 0x4d,
	0x88, 0xbf, 0xe4, 0x05, 0x25, 0x73, 0x1e, 0xd0, 0xcf, 0x1e, 0xeb, 0x4d, 0x18, 0x4e, 0x6f, 0x9f,
	0x00, 0x72, 0xc3, 0x74, 0xbe, 0x02, 0x09, 0x0a, 0xb0, 0xb1, 0x8e, 0x9b, 0x08, 0xed, 0x68, 0x6d,
	0x25, 0xe8, 0x28, 0x94, 0xa6, 0x44, 0xa6, 0xfa, 0x95, 0x69, 0x72, 0xd3, 0x6d, 0xb6, 0xff, 0xf3,
	0x45, 0x4e, 0xa3, 0x60, 0xf2, 0x58, 0xe7, 0x59, 0xe4, 0x4f, 0x70, 0xd2, 0xdd, 0xc0, 0x6e, 0xf3,
	0xd7, 0xb2, 0xdc, 0x1e, 0xa8, 0x41, 0x07, 0x5d, 0x60, 0x15, 0xea, 0xcf, 0xdb, 0xc1, 0x1d, 0x4d,
	0xb7, 0x42, 0xec, 0xc4, 0xca, 0xa9, 0x40, 0x30, 0x0f, 0x3c, 0xe2, 0x81, 0xe0, 0x5c, 0x51, 0x07,
	0xb0, 0x1e, 0x4a, 0xb3, 0x64, 0x3e, 0x1c, 0x62, 0x17, 0xcd, 0xf2, 0xe4, 0x14, 0x9d, 0xa6, 0xd4,
	0x64, 0x36, 0xa5, 0xe8, 0x7e, 0x84, 0x0e, 0xb3, 0x5d, 0x79, 0x57, 0xea, 0xd7, 0xad, 0xbc, 0x9e,
	0x2d, 0x90, 0x03, 0x9e, 0x0e, 0xc6, 0x98, 0xdb, 0xe3, 0xb6, 0x9f, 0x9b, 0xf6, 0x21, 0xe6, 0x98,
	0x94, 0x77, 0xb7, 0x2b, 0xaa, 0xc9, 0xff, 0xef, 0x7a, 0xf2, 0x71, 0x4e, 0x52, 0x06, 0x85, 0x37,
	0x81, 0x8e, 0x86, 0x64, 0x39, 0x92, 0x2a, 0xca, 0xf3, 0x3e, 0x87, 0xb5, 0x0c, 0x7b, 0x42, 0x5e,
	0x04, 0xa7, 0xfb, 0xd7, 0x13, 0x7f, 0x83, 0x6a, 0x77, 0x0f, 0xa7, 0x34, 0x51, 0x88, 0x9c, 0xac,
	0x23, 0x90, 0x4d, 0x4d, 0x72, 0x4e, 0xa3, 0x26, 0x1a, 0x45, 0x61, 0x0e, 0x10, 0x24, 0x8a, 0x27,
	0x92, 0x14, 0x23, 0xae, 0x4b, 0x80, 0xae, 0x6a, 0x56, 0x01, 0xac, 0x55, 0xf7, 0x6d, 0x9b, 0x6d
};

void pgm_photoy2k_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT3
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		x ^= photoy2k_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t pstar[256] = {
	0x62, 0x59, 0x17, 0xe3, 0xe1, 0x11, 0x02, 0x97, 0x67, 0x4d, 0x4a, 0x1c, 0x1f, 0x76, 0x64, 0xc1,
	0xfa, 0x04, 0xd2, 0x9f, 0x22, 0xf5, 0xfd, 0xfe, 0x89, 0xeb, 0xab, 0x41, 0xc0, 0xac, 0x2b, 0x64,
	0xfe, 0x1e, 0x9b, 0x68, 0x07, 0xfd, 0x75, 0x25, 0x24, 0xa0, 0x41, 0x5d, 0x79, 0xb5, 0x67, 0x93,
	0xe1, 0xa3, 0x3a, 0x24, 0xa0, 0xbf, 0x64, 0x62, 0x38, 0x65, 0x75, 0x78, 0x8a, 0xf9, 0x7c, 0x00,
	0x71, 0x38, 0xc5, 0xe4, 0xdc, 0xf8, 0xc5, 0xc5, 0x8d, 0x1d, 0x6b, 0xb8, 0x84, 0x4e, 0x21, 0x6d,
	0x55, 0x95, 0x31, 0x86, 0x11, 0xe6, 0xb1, 0x77, 0x77, 0x3b, 0x01, 0x82, 0xab, 0x88, 0xef, 0x77,
	0x08, 0xd8, 0x80, 0xba, 0x41, 0xfb, 0xb3, 0x30, 0x5f, 0xf2, 0x5b, 0xe6, 0x5a, 0x63, 0xb0, 0xa4,
	0x37, 0x37, 0x6d, 0x1f, 0x27, 0x0e, 0x80, 0x8e, 0x1a, 0x45, 0x97, 0xdb, 0x61, 0x10, 0x24, 0x8a,
	0x62, 0x44, 0xf2, 0x91, 0x74, 0x25, 0x3e, 0xc5, 0xe6, 0x8e, 0x55, 0x44, 0x5a, 0x8a, 0x90, 0x25,
	0xa1, 0x0d, 0x43, 0x35, 0x46, 0x80, 0xfe, 0xb3, 0x4e, 0x4c, 0x21, 0xcc, 0x99, 0xc2, 0x3d, 0xce,
	0x19, 0xc5, 0x2c, 0xae, 0xe8, 0x94, 0xad, 0xed, 0x36, 0xb5, 0x8e, 0xf1, 0x81, 0xfe, 0x62, 0x3a,
	0x8c, 0xe0, 0x90, 0x88, 0xf1, 0x31, 0xc5, 0xd9, 0x3c, 0xf4, 0x70, 0x7d, 0x7d, 0x1c, 0xca, 0xe2,
	0x51, 0xce, 0xa0, 0x33, 0x42, 0x1e, 0xf7, 0xe4, 0x13, 0xcf, 0xa0, 0x95, 0x33, 0x59, 0x5b, 0xec,
	0xf7, 0x6d, 0x56, 0xac, 0x6a, 0xae, 0x55, 0x9b, 0x92, 0x14, 0x6d, 0x01, 0x23, 0x4b, 0x80, 0xae,
	0x65, 0x15, 0xc9, 0x84, 0x66, 0xd7, 0xbd, 0x16, 0xd2, 0xe7, 0x15, 0x95, 0x16, 0x9c, 0xd3, 0xd2,
	0x62, 0x51, 0xf6, 0xe3, 0xa6, 0xea, 0x32, 0x86, 0x67, 0xcc, 0x4a, 0x46, 0x4d, 0x37, 0x45, 0xd5
};

void pgm_pstar_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x100000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x = x ^ pstar[(i & 255)] << 8;
		src[i] = x ;
	}
}

static const uint8_t dfront_tab[256] = {
	0x51, 0xc4, 0xe3, 0x10, 0x1c, 0xad, 0x8a, 0x39, 0x8c, 0xe0, 0xa5, 0x04, 0x0f, 0xe4, 0x35, 0xc3,
	0x2d, 0x6b, 0x32, 0xe2, 0x60, 0x54, 0x63, 0x06, 0xa3, 0xf1, 0x0b, 0x5f, 0x6c, 0x5c, 0xb3, 0xec,
	0x77, 0x61, 0x69, 0xe7, 0x3c, 0xb7, 0x42, 0x72, 0x1a, 0x70, 0xb0, 0x96, 0xa4, 0x28, 0xc0, 0xfb,
	0x0a, 0x00, 0xcb, 0x15, 0x49, 0x48, 0xd3, 0x94, 0x58, 0xcf, 0x41, 0x86, 0x17, 0x71, 0xb1, 0xbd,
	0x21, 0x01, 0x37, 0x1e, 0xba, 0xeb, 0xf3, 0x59, 0xf6, 0xa7, 0x29, 0x4f, 0xb5, 0xca, 0x4c, 0x34,
	0x20, 0xa2, 0x62, 0x4b, 0x93, 0x9e, 0x47, 0x9f, 0x8d, 0x0e, 0x1b, 0xb6, 0x4d, 0x82, 0xd5, 0xf4,
	0x85, 0x79, 0x53, 0x92, 0x9b, 0xf7, 0xea, 0x44, 0x76, 0x1f, 0x22, 0x45, 0xed, 0xbe, 0x11, 0x55,
	0xaf, 0xf5, 0xf8, 0x50, 0x07, 0xe6, 0xc7, 0x5e, 0xd7, 0xde, 0xe5, 0x26, 0x2b, 0xf2, 0x6a, 0x8b,
	0xb8, 0x98, 0x89, 0xdb, 0x14, 0x5b, 0xc5, 0x78, 0xdc, 0xd0, 0x87, 0x5d, 0xc1, 0x0d, 0x95, 0x97,
	0x7e, 0xa8, 0x24, 0x3d, 0xe1, 0xd1, 0x19, 0xa6, 0x99, 0xd8, 0x83, 0x1d, 0xff, 0x30, 0x9d, 0x05,
	0xd4, 0x02, 0x27, 0x7b, 0x13, 0xb2, 0x7f, 0x40, 0x12, 0xa0, 0x68, 0x67, 0x4e, 0x3a, 0x46, 0xb9,
	0xee, 0xdf, 0x66, 0xd6, 0x8f, 0xa9, 0x0c, 0x91, 0x65, 0x18, 0x52, 0x56, 0xd9, 0x74, 0x09, 0x6e,
	0xc6, 0x73, 0xc9, 0xfc, 0x03, 0x43, 0xef, 0xaa, 0x7c, 0xbb, 0x2c, 0x90, 0xcc, 0xce, 0xe8, 0xae,
	0x2a, 0xf9, 0x57, 0x88, 0xc8, 0xe9, 0x5a, 0xdd, 0x2e, 0x7d, 0x64, 0xc2, 0x6d, 0x3e, 0xfa, 0x80,
	0x16, 0xcd, 0x6f, 0x84, 0x8e, 0x9c, 0xf0, 0xac, 0xb4, 0x9a, 0x2f, 0xbc, 0x31, 0x23, 0xfe, 0x38,
	0x08, 0x75, 0xa1, 0x33, 0xab, 0xd2, 0xda, 0x81, 0xbf, 0x7a, 0x3b, 0x3f, 0x4a, 0xfd, 0x25, 0x36
};

void pgm_dfront_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= dfront_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


void pgm_ddp2_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT2
		// NO CRYPT2
		// NO CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8_ALT

		src[i] = x;
	}
}

void pgm_mm_decrypt(running_machine &machine) // and dw2001
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		src[i] = x;
	}
}



void pgm_kov2_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		// NO CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8_ALT

		src[i] = x;
	}
}

void pgm_kov2p_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		src[i] = x;
	}
}

static const uint8_t puzzli2_tab[256] = {
	0xb7, 0x66, 0xa3, 0xc0, 0x51, 0x55, 0x6d, 0x63, 0x86, 0x60, 0x64, 0x6c, 0x67, 0x18, 0x0b, 0x05,
	0x62, 0xff, 0xe0, 0x1e, 0x30, 0x21, 0x2e, 0x40, 0x41, 0xb9, 0x60, 0x38, 0xd1, 0x24, 0x7e, 0x36,
	0x7a, 0x0b, 0x1c, 0x69, 0x4f, 0x09, 0xe1, 0x9e, 0xcf, 0xcd, 0x7c, 0x00, 0x73, 0x08, 0x77, 0x37,
	0x5f, 0x50, 0x32, 0x3e, 0xd3, 0x54, 0x77, 0x6b, 0x60, 0x60, 0x74, 0x7c, 0x55, 0x4f, 0x44, 0x5e,
	0x66, 0x5c, 0x58, 0x26, 0x35, 0x29, 0x3f, 0x35, 0x3f, 0x1c, 0x0b, 0x0d, 0x08, 0x5b, 0x59, 0x5c,
	0xa0, 0xa5, 0x87, 0x85, 0x24, 0x75, 0x5f, 0x42, 0x1b, 0xf3, 0x1a, 0x58, 0x17, 0x58, 0x71, 0x6b,
	0x69, 0x89, 0x7d, 0x3a, 0xf3, 0xc4, 0x5d, 0xa0, 0x4f, 0x27, 0x58, 0xc4, 0xa8, 0xdd, 0xa8, 0xfb,
	0xbe, 0xa4, 0xe2, 0xee, 0x07, 0x10, 0x90, 0x72, 0x99, 0x08, 0x68, 0x6d, 0x5c, 0x5c, 0x6d, 0x58,
	0x2f, 0xdc, 0x15, 0xd5, 0xd6, 0xd6, 0x3b, 0x3b, 0xf9, 0x32, 0xcc, 0xdd, 0xd4, 0xf1, 0xea, 0xed,
	0xe4, 0xf6, 0xf2, 0x91, 0xca, 0xc1, 0xed, 0xf2, 0xf6, 0xfb, 0xc0, 0xe8, 0xe3, 0xe7, 0xfa, 0xf1,
	0xf5, 0x08, 0x26, 0x2b, 0x2f, 0x34, 0x39, 0x13, 0x28, 0x07, 0x88, 0x5b, 0x8f, 0x94, 0x9b, 0x2e,
	0xf5, 0xab, 0x72, 0x76, 0x7a, 0x40, 0xb9, 0x09, 0xd8, 0x3b, 0xcd, 0x31, 0x3d, 0x42, 0xab, 0xb1,
	0xb5, 0xb9, 0x3b, 0xe3, 0x0b, 0x65, 0x18, 0xfb, 0x1f, 0x12, 0xe4, 0xe8, 0xec, 0xf2, 0xf7, 0xfc,
	0xc0, 0xe8, 0xe0, 0xe6, 0xfa, 0xf1, 0xf4, 0x0b, 0x26, 0x2b, 0x30, 0x35, 0x39, 0x13, 0x29, 0x21,
	0x0c, 0x11, 0x16, 0x1b, 0x1f, 0x64, 0x0e, 0x60, 0x05, 0x79, 0x7c, 0x37, 0x00, 0x0f, 0x4f, 0x38,
	0x1d, 0x18, 0xa2, 0xb6, 0xb2, 0xa9, 0xac, 0xab, 0xae, 0x91, 0x98, 0x8d, 0x91, 0xbb, 0xb1, 0xc0
};

void pgm_puzzli2_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x100000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= puzzli2_tab[i & 0xff] << 8;
		src[i] = x;
	}
}


static const uint8_t theglad_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x35, 0x52, 0x44, 0x31, 0x30, 0x32, 0x31, 0x32, 0x30, 0x33,
	0xc4, 0xa3, 0x46, 0x78, 0x30, 0xb3, 0x8b, 0xd5, 0x2f, 0xc4, 0x44, 0xbf, 0xdb, 0x76, 0xdb, 0xea,
	0xb4, 0xeb, 0x95, 0x4d, 0x15, 0x21, 0x99, 0xa1, 0xd7, 0x8c, 0x40, 0x1d, 0x43, 0xf3, 0x9f, 0x71,
	0x3d, 0x8c, 0x52, 0x01, 0xaf, 0x5b, 0x8b, 0x63, 0x34, 0xc8, 0x5c, 0x1b, 0x06, 0x7f, 0x41, 0x96,
	0x2a, 0x8d, 0xf1, 0x64, 0xda, 0xb8, 0x67, 0xba, 0x33, 0x1f, 0x2b, 0x28, 0x20, 0x13, 0xe6, 0x96,
	0x86, 0x34, 0x25, 0x85, 0xb0, 0xd0, 0x6d, 0x85, 0xfe, 0x78, 0x81, 0xf1, 0xca, 0xe4, 0xef, 0xf2,
	0x9b, 0x09, 0xe1, 0xb4, 0x8d, 0x79, 0x22, 0xe2, 0x00, 0xfb, 0x6f, 0x68, 0x80, 0x6a, 0x00, 0x69,
	0xf5, 0xd3, 0x57, 0x7e, 0x0c, 0xca, 0x48, 0x31, 0xe5, 0x0d, 0x4a, 0xb9, 0xfd, 0x5c, 0xfd, 0xf8,
	0x5f, 0x98, 0xfb, 0xb3, 0x07, 0x1a, 0xe3, 0x10, 0x96, 0x56, 0xa3, 0x56, 0x3d, 0xb1, 0x07, 0xe0,
	0xe3, 0x9f, 0x7f, 0x62, 0x99, 0x01, 0x35, 0x60, 0x40, 0xbe, 0x4f, 0xeb, 0x79, 0xa0, 0x82, 0x9f,
	0xcd, 0x71, 0xd8, 0xda, 0x1e, 0x56, 0xc2, 0x3e, 0x4e, 0x6b, 0x60, 0x69, 0x2d, 0x9f, 0x10, 0xf4,
	0xa9, 0xd3, 0x36, 0xaa, 0x31, 0x2e, 0x4c, 0x0a, 0x69, 0xc3, 0x2a, 0xff, 0x15, 0x67, 0x96, 0xde,
	0x3f, 0xcc, 0x0f, 0xa1, 0xac, 0xe2, 0xd6, 0x62, 0x7e, 0x6f, 0x3e, 0x1b, 0x2a, 0xed, 0x36, 0x9c,
	0x9d, 0xa4, 0x14, 0xcd, 0xaa, 0x08, 0xa4, 0x26, 0xb7, 0x55, 0x70, 0x6c, 0xa9, 0x69, 0x52, 0xae,
	0x0c, 0xe1, 0x38, 0x7f, 0x87, 0x78, 0x38, 0x75, 0x80, 0x9c, 0xd4, 0xe2, 0x0b, 0x52, 0x8f, 0xd2,
	0x19, 0x4c, 0xb0, 0x45, 0xde, 0x48, 0x55, 0xae, 0x82, 0xab, 0xbc, 0xab, 0x0c, 0x5e, 0xce, 0x07
};

void pgm_theglad_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= theglad_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t oldsplus_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x31, 0x33, 0x52, 0x44, 0x31, 0x30, 0x34, 0x30, 0x37, 0x32, 0x37,
	0xf5, 0x79, 0x6d, 0xab, 0x04, 0x22, 0x51, 0x96, 0xf2, 0x72, 0xe8, 0x3a, 0x96, 0xd2, 0x9a, 0xcc,
	0x3f, 0x47, 0x3c, 0x09, 0xf2, 0xd9, 0x72, 0x41, 0xe6, 0x44, 0x43, 0xa7, 0x3e, 0xe2, 0xfd, 0xd8,
	0x06, 0xd8, 0x4c, 0xa9, 0x70, 0x80, 0x95, 0x35, 0x50, 0x17, 0x99, 0x27, 0xd5, 0xa8, 0x47, 0x45,
	0x89, 0x38, 0xe1, 0x3d, 0x8c, 0x33, 0x53, 0xb4, 0x0d, 0x17, 0xd1, 0x8d, 0x09, 0x5f, 0xaf, 0x76,
	0x48, 0xb2, 0x85, 0xb9, 0x95, 0x4c, 0x83, 0x42, 0x3d, 0xad, 0x11, 0xec, 0xca, 0x82, 0xac, 0x10,
	0x01, 0xd0, 0xfd, 0x50, 0x19, 0x67, 0x3b, 0xa0, 0x3e, 0x86, 0xc2, 0x97, 0x46, 0xcb, 0xf4, 0xf5,
	0xb3, 0x5f, 0x50, 0x74, 0xe9, 0x5f, 0xd2, 0xd4, 0xb0, 0x8d, 0x8a, 0x21, 0xed, 0x37, 0x80, 0x47,
	0x9d, 0x68, 0xc7, 0xd9, 0x12, 0x4e, 0xdf, 0x1e, 0x72, 0xeb, 0x50, 0x5e, 0x6d, 0x00, 0x85, 0x6b,
	0x3e, 0x37, 0xe6, 0x72, 0xe5, 0x8f, 0x3a, 0x03, 0xa3, 0x0d, 0x3b, 0x5f, 0xb6, 0xa1, 0x7b, 0x02,
	0x56, 0x56, 0x77, 0x71, 0xef, 0xbe, 0xf9, 0x46, 0xa1, 0x9d, 0xb3, 0x79, 0xf6, 0xd5, 0x19, 0xf0,
	0xe2, 0x91, 0x7e, 0x4a, 0x01, 0xb6, 0x73, 0xe8, 0x0c, 0x86, 0x5d, 0x3e, 0x9c, 0x97, 0x55, 0x58,
	0x23, 0xf4, 0x45, 0xb0, 0x28, 0x91, 0x40, 0x2f, 0xc2, 0xf4, 0x21, 0x81, 0x58, 0x22, 0x68, 0x9d,
	0x97, 0xc7, 0x51, 0x95, 0xb4, 0xaa, 0x36, 0x9b, 0xe4, 0x51, 0x27, 0x55, 0x18, 0xf0, 0xc7, 0x62,
	0xfe, 0x98, 0x6a, 0x2d, 0x35, 0x9d, 0x6c, 0xf1, 0xcf, 0x48, 0xd4, 0x0d, 0x0c, 0xbe, 0x2a, 0x8a,
	0x55, 0x31, 0x96, 0xea, 0x78, 0x45, 0x3a, 0x33, 0x23, 0xc5, 0xd1, 0x3c, 0xa3, 0x86, 0x88, 0x38
};

void pgm_oldsplus_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		unsigned short x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= oldsplus_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t kovshp_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x39, 0x72, 0x64, 0x31, 0x30, 0x34, 0x30, 0x32, 0x31, 0x39,
	0xf9, 0x8c, 0xbd, 0x87, 0x16, 0x07, 0x39, 0xeb, 0x29, 0x9e, 0x17, 0xef, 0x4f, 0x64, 0x7c, 0xe0,
	0x5f, 0x73, 0x5b, 0xa1, 0x5e, 0x95, 0x0d, 0xf1, 0x40, 0x36, 0x2f, 0x00, 0xe2, 0x8a, 0xbc, 0x32,
	0x44, 0xfa, 0x6c, 0x33, 0x0b, 0xd5, 0x4c, 0x3b, 0x36, 0x34, 0x9e, 0xa3, 0x20, 0x2e, 0xf3, 0xa9,
	0xb7, 0x3e, 0x87, 0x80, 0xfb, 0xf1, 0xdd, 0x9c, 0xba, 0xd3, 0x9b, 0x3b, 0x8a, 0x9c, 0xa8, 0x37,
	0x07, 0x97, 0x84, 0x0c, 0x4e, 0x54, 0xe7, 0x25, 0xba, 0x8e, 0x9d, 0x6b, 0xde, 0x5f, 0xa1, 0x10,
	0xc3, 0xa2, 0x79, 0x99, 0x63, 0xa9, 0xd1, 0x2a, 0x65, 0x20, 0x5b, 0x16, 0x1b, 0x41, 0xe6, 0xa7,
	0xba, 0x3a, 0xbd, 0x2a, 0xd8, 0xdb, 0x43, 0x3f, 0x2b, 0x85, 0xcc, 0x5f, 0x80, 0x4f, 0xbe, 0xae,
	0xfa, 0x79, 0xe8, 0x03, 0x8d, 0x16, 0x22, 0x35, 0xbb, 0xf6, 0x26, 0xa9, 0x8d, 0xd2, 0xaf, 0x19,
	0xd4, 0xbb, 0xd0, 0xa6, 0xa1, 0xc4, 0x96, 0x21, 0x02, 0xef, 0xe1, 0x96, 0x00, 0x56, 0x80, 0x1b,
	0xd6, 0x9a, 0x8c, 0xd7, 0x73, 0x91, 0x07, 0x55, 0x32, 0x2b, 0xb5, 0x0b, 0xd8, 0xa5, 0x39, 0x26,
	0xce, 0xf2, 0x74, 0x98, 0xa1, 0x66, 0x1a, 0x64, 0xb8, 0xa5, 0x96, 0x29, 0x54, 0xcb, 0x21, 0xed,
	0xcd, 0xdd, 0x1e, 0x2c, 0x0b, 0x70, 0xb8, 0x22, 0x43, 0x98, 0xbe, 0x54, 0xf3, 0x14, 0xbe, 0x65,
	0x21, 0xb7, 0x61, 0x17, 0xcf, 0x19, 0x07, 0xa0, 0xc2, 0x7f, 0xa3, 0x30, 0x75, 0x08, 0xd8, 0xbf,
	0x58, 0x1a, 0x55, 0x1b, 0x4e, 0x0d, 0x6d, 0x32, 0x65, 0x15, 0xfb, 0x9e, 0xd8, 0x75, 0x76, 0x6f,
	0x42, 0xe2, 0x4f, 0x3c, 0x25, 0x35, 0x93, 0x6c, 0x9b, 0x56, 0xbe, 0xc1, 0x5b, 0x65, 0xde, 0x27
};

void pgm_kovshp_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		unsigned short x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= kovshp_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t killbldp_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x32, 0x34, 0x52, 0x44, 0x31, 0x30, 0x35, 0x30, 0x39, 0x30, 0x38,
	0x12, 0xa0, 0xd1, 0x9e, 0xb1, 0x8a, 0xfb, 0x1f, 0x50, 0x51, 0x4b, 0x81, 0x28, 0xda, 0x5f, 0x41,
	0x78, 0x6c, 0x7a, 0xf0, 0xcd, 0x6b, 0x69, 0x14, 0x94, 0x55, 0xb6, 0x42, 0xdf, 0xfe, 0x10, 0x79,
	0x74, 0x08, 0xfa, 0xc0, 0x1c, 0xa5, 0xb4, 0x03, 0x2a, 0x91, 0x67, 0x2b, 0x49, 0x4a, 0x94, 0x7d,
	0x8b, 0x92, 0xbe, 0x35, 0xaf, 0x28, 0x56, 0x63, 0xb3, 0xc2, 0xe8, 0x06, 0x9b, 0x4e, 0x85, 0x66,
	0x7f, 0x6b, 0x70, 0xb7, 0xdb, 0x22, 0x0c, 0xeb, 0x13, 0xe9, 0x06, 0xd7, 0x45, 0xda, 0xbe, 0x8b,
	0x54, 0x30, 0xfc, 0xeb, 0x32, 0x02, 0xd0, 0x92, 0x6d, 0x44, 0xca, 0xe8, 0xfd, 0xfb, 0x5b, 0x81,
	0x4c, 0xc0, 0x8b, 0xb9, 0x87, 0x78, 0xdd, 0x8e, 0x24, 0x52, 0x80, 0xbe, 0xb4, 0x01, 0xb7, 0x21,
	0xeb, 0x3c, 0x8a, 0x49, 0xed, 0x73, 0xae, 0x58, 0xdb, 0xd2, 0xb2, 0x21, 0x9e, 0x7c, 0x6c, 0x82,
	0xf3, 0x01, 0xa3, 0x00, 0xb7, 0x21, 0xfe, 0xa5, 0x75, 0xc4, 0x2d, 0x17, 0x2d, 0x39, 0x56, 0xf9,
	0x67, 0xae, 0xc2, 0x87, 0x79, 0xf1, 0xc8, 0x6d, 0x15, 0x66, 0xfa, 0xe8, 0x16, 0x48, 0x8f, 0x1f,
	0x8b, 0x24, 0x10, 0xc4, 0x04, 0x93, 0x47, 0xe6, 0x1d, 0x37, 0x65, 0x1a, 0x49, 0xf8, 0x72, 0xcb,
	0xe1, 0x80, 0xfa, 0xdd, 0x6d, 0xf5, 0xf6, 0x89, 0x32, 0xf6, 0xf8, 0x75, 0xfc, 0xd8, 0x9b, 0x12,
	0x2d, 0x22, 0x2a, 0x3b, 0x06, 0x46, 0x90, 0x0c, 0x35, 0xa2, 0x80, 0xff, 0xa0, 0xb7, 0xe5, 0x4d,
	0x71, 0xa9, 0x8c, 0x84, 0x62, 0xf7, 0x10, 0x65, 0x4a, 0x7b, 0x06, 0x00, 0xe8, 0xa4, 0x6a, 0x13,
	0xf0, 0xf3, 0x4a, 0x9f, 0x54, 0xb4, 0xb1, 0xcc, 0xd4, 0xff, 0xd6, 0xff, 0xc9, 0xee, 0x86, 0x39
};

void pgm_killbldp_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= killbldp_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


void pgm_svg_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x800000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		src[i] = x;
	}
}

static const unsigned char svgpcb_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x31, 0x37, 0x52, 0x44, 0x31, 0x30, 0x35, 0x30, 0x35, 0x30, 0x34,
	0x75, 0x0b, 0xf1, 0x6b, 0x6d, 0xd7, 0xa8, 0xe7, 0x0c, 0xc5, 0x28, 0x81, 0x1f, 0xcf, 0x30, 0x15,
	0xa8, 0x0d, 0xda, 0x76, 0xf8, 0x7d, 0xd6, 0xe1, 0x0a, 0x11, 0xe3, 0xa4, 0x23, 0xff, 0x8e, 0x0b,
	0xa8, 0x44, 0x2f, 0x8b, 0x3f, 0x7a, 0x21, 0x32, 0x2a, 0xdc, 0x41, 0x4e, 0xe0, 0x97, 0xa9, 0x5d,
	0xed, 0x53, 0xae, 0x35, 0x0b, 0x02, 0x18, 0x74, 0x82, 0xe8, 0xa1, 0x2a, 0xbd, 0xeb, 0xb0, 0xc6,
	0x2e, 0x1d, 0x56, 0x3e, 0x63, 0x87, 0x8a, 0x83, 0x69, 0x38, 0xa1, 0x24, 0x61, 0x8f, 0x11, 0x41,
	0x61, 0xc5, 0x67, 0xb3, 0x8e, 0xbe, 0x85, 0x79, 0x77, 0x10, 0x21, 0x66, 0xb4, 0x54, 0x7b, 0x09,
	0xbf, 0xad, 0x5e, 0xdd, 0x12, 0x97, 0x5a, 0xb2, 0x82, 0xf3, 0x40, 0x5b, 0xdb, 0x4f, 0xde, 0x99,
	0xbd, 0x7a, 0xfc, 0x48, 0xb6, 0x48, 0x97, 0xc8, 0xa1, 0xa2, 0x5c, 0xae, 0x3e, 0xd2, 0x68, 0xac,
	0x13, 0x0d, 0x3f, 0xbe, 0x82, 0x42, 0x0a, 0x97, 0x2c, 0x22, 0x16, 0x4b, 0x85, 0x70, 0x89, 0x3d,
	0xb8, 0x8b, 0x66, 0x4c, 0xbd, 0x39, 0xc4, 0x39, 0xb9, 0xb6, 0x4b, 0x5c, 0x96, 0xfc, 0xef, 0x87,
	0xe3, 0x55, 0xf1, 0x3b, 0xed, 0x1f, 0x13, 0x0a, 0x1f, 0xdf, 0x1a, 0x4c, 0x97, 0x8a, 0x8a, 0x06,
	0x0a, 0x0f, 0x9d, 0x17, 0xda, 0x28, 0x85, 0xa4, 0x75, 0x63, 0xe4, 0xc6, 0xf1, 0x6b, 0x88, 0x73,
	0xe6, 0x9c, 0x6b, 0xaa, 0x8b, 0xc7, 0xea, 0xe3, 0x13, 0x42, 0x46, 0xb6, 0x4a, 0x34, 0x59, 0xca,
	0x6c, 0x1f, 0x99, 0x01, 0x46, 0xae, 0x52, 0xe1, 0x90, 0xc1, 0x1f, 0x46, 0x89, 0xb7, 0xab, 0x46,
	0xd4, 0xfb, 0xa6, 0x65, 0x92, 0xd1, 0x0a, 0xfa, 0xc1, 0x63, 0x90, 0xe1, 0xd5, 0x07, 0x6d, 0x62
};

void pgm_svgpcb_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x800000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT; // ok?
		IGS27_CRYPT2 // ok?
		IGS27_CRYPT3
		IGS27_CRYPT4 // ok?
		IGS27_CRYPT5  // ok?
		IGS27_CRYPT6_ALT // ok?
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= svgpcb_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t py2k2_tab[256] = {
	0x74, 0xe8, 0xa8, 0x64, 0x26, 0x44, 0xa6, 0x9a, 0xa5, 0x69, 0xa2, 0xd3, 0x6d, 0xba, 0xff, 0xf3,
	0xeb, 0x6e, 0xe3, 0x70, 0x72, 0x58, 0x27, 0xd9, 0xe4, 0x9f, 0x50, 0xa2, 0xdd, 0xce, 0x6e, 0xf6,
	0x44, 0x72, 0x0c, 0x7e, 0x4d, 0x41, 0x77, 0x2d, 0x00, 0xad, 0x1a, 0x5f, 0x6b, 0xc0, 0x1d, 0x4e,
	0x4c, 0x72, 0x62, 0x3c, 0x32, 0x28, 0x43, 0xf8, 0x9d, 0x52, 0x05, 0x7e, 0xd1, 0xee, 0x82, 0x61,
	0x3b, 0x3f, 0x77, 0xf3, 0x8f, 0x7e, 0x3f, 0xf1, 0xdf, 0x8f, 0x68, 0x43, 0xd7, 0x68, 0xdf, 0x19,
	0x87, 0xff, 0x74, 0xe5, 0x3f, 0x43, 0x8e, 0x80, 0x0f, 0x7e, 0xdb, 0x32, 0xe8, 0xd1, 0x66, 0x8f,
	0xbe, 0xe2, 0x33, 0x94, 0xc8, 0x32, 0x39, 0xfa, 0xf0, 0x43, 0xde, 0x84, 0x18, 0xd0, 0x6d, 0xd5,
	0x74, 0x98, 0xf8, 0x64, 0xcf, 0x84, 0xc6, 0xea, 0x55, 0x32, 0xe2, 0x38, 0xdd, 0xea, 0xfd, 0x6c,
	0xeb, 0x6e, 0xe3, 0x70, 0xae, 0x38, 0xc7, 0xd9, 0x54, 0x84, 0x10, 0xc1, 0xfd, 0x1e, 0x6e, 0x6d,
	0x37, 0xe0, 0x03, 0x9e, 0x06, 0x36, 0x68, 0x5b, 0xe3, 0xf6, 0x7f, 0x0b, 0x56, 0x79, 0xe0, 0xa8,
	0x98, 0x77, 0xc7, 0x2b, 0xa5, 0x79, 0xff, 0x2f, 0xca, 0x15, 0x71, 0x7e, 0x02, 0xbf, 0x87, 0xb7,
	0x7a, 0x8e, 0xe6, 0x64, 0x32, 0x62, 0x2a, 0xca, 0x23, 0x72, 0x87, 0xb5, 0x0c, 0x02, 0x4b, 0xee,
	0x44, 0x72, 0x9c, 0x7e, 0x5d, 0xc1, 0xa7, 0x1d, 0x30, 0x38, 0xda, 0xc9, 0x5b, 0xd0, 0x11, 0xf9,
	0xb1, 0x72, 0x6c, 0x04, 0x31, 0xc9, 0x50, 0x60, 0x6f, 0xc1, 0xf2, 0xae, 0x00, 0xf4, 0x5d, 0x66,
	0x43, 0x0e, 0x7a, 0xc3, 0x76, 0xae, 0x3c, 0xc2, 0xb7, 0xc9, 0x52, 0xf4, 0x74, 0x51, 0xaf, 0x12,
	0x19, 0xc6, 0x75, 0xe8, 0x6c, 0x54, 0x7e, 0x63, 0xdd, 0xae, 0x07, 0x5a, 0xb7, 0x00, 0xb5, 0x5e
};

void pgm_py2k2_decrypt(running_machine &machine) // and ddpdoj/ddpdojbl
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT3
		IGS27_CRYPT3_ALT
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= py2k2_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


static const unsigned char ket_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x34, 0x52, 0x44, 0x31, 0x30, 0x32, 0x31, 0x30, 0x31, 0x35,
	0x7c, 0x49, 0x27, 0xa5, 0xff, 0xf6, 0x98, 0x2d, 0x0f, 0x3d, 0x12, 0x23, 0xe2, 0x30, 0x50, 0xcf,
	0xf1, 0x82, 0xf0, 0xce, 0x48, 0x44, 0x5b, 0xf3, 0x0d, 0xdf, 0xf8, 0x5d, 0x50, 0x53, 0x91, 0xd9,
	0x12, 0xaf, 0x05, 0x7a, 0x98, 0xd0, 0x2f, 0x76, 0xf1, 0x5d, 0x17, 0x44, 0xc5, 0x03, 0x58, 0xf4,
	0x61, 0xee, 0xd1, 0xce, 0x00, 0x88, 0x90, 0x2e, 0x5c, 0x76, 0xfb, 0x9f, 0x75, 0xcf, 0x40, 0x37,
	0xa1, 0x9f, 0x00, 0x32, 0xd5, 0x9c, 0x37, 0xd2, 0x32, 0x27, 0x6f, 0x76, 0xd3, 0x86, 0x25, 0xf9,
	0xd6, 0x60, 0x7b, 0x4e, 0xa9, 0x7a, 0x20, 0x59, 0x96, 0xb1, 0x7d, 0x10, 0x92, 0x37, 0x22, 0xd2,
	0x42, 0x12, 0x6f, 0x07, 0x4f, 0xd2, 0x87, 0xfa, 0xeb, 0x92, 0x71, 0xf3, 0xa4, 0x31, 0x91, 0x98,
	0x68, 0xd2, 0x47, 0x86, 0xda, 0x92, 0xe5, 0x2b, 0xd4, 0x89, 0xd7, 0xe7, 0x3d, 0x03, 0x0d, 0x63,
	0x0c, 0x00, 0xac, 0x31, 0x9d, 0xe9, 0xf6, 0xa5, 0x34, 0x95, 0x77, 0xf2, 0xcf, 0x7c, 0x72, 0x89,
	0x31, 0x3a, 0x8b, 0xae, 0x2b, 0x47, 0xb6, 0x5d, 0x2d, 0xf5, 0x5f, 0x5c, 0x0e, 0xab, 0xdb, 0xa1,
	0x18, 0x60, 0x0e, 0xe6, 0x58, 0x5b, 0x5e, 0x8b, 0x24, 0x29, 0xd8, 0xac, 0xed, 0xdf, 0xa2, 0x83,
	0x46, 0x91, 0xa1, 0xff, 0x35, 0x13, 0x6a, 0xa5, 0xba, 0xef, 0x6e, 0xa8, 0x9e, 0xa6, 0x62, 0x44,
	0x7e, 0x2c, 0xed, 0x60, 0x17, 0x9e, 0x96, 0x64, 0xd3, 0x46, 0xec, 0x58, 0x95, 0xd1, 0xf7, 0x3e,
	0xc2, 0xcf, 0xdf, 0xb0, 0x90, 0x6c, 0xdb, 0xbe, 0x93, 0x6d, 0x5d, 0x02, 0x85, 0x6e, 0x7c, 0x05,
	0x55, 0x5a, 0xa1, 0xd7, 0x73, 0x2b, 0x76, 0xe9, 0x5b, 0xe4, 0x0c, 0x2e, 0x60, 0xcb, 0x4b, 0x72
};

void pgm_ket_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base());

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT


		x ^= ket_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char espgal_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x37, 0x52, 0x44, 0x31, 0x30, 0x33, 0x30, 0x39, 0x30, 0x39,
	0xa7, 0xf1, 0x0a, 0xca, 0x69, 0xb2, 0xce, 0x86, 0xec, 0x3d, 0xa2, 0x5a, 0x03, 0xe9, 0xbf, 0xba,
	0xf7, 0xd5, 0xec, 0x68, 0x03, 0x90, 0x15, 0xcc, 0x0d, 0x08, 0x2d, 0x76, 0xa5, 0xb5, 0x41, 0xf1,
	0x43, 0x06, 0xdd, 0xcb, 0xbd, 0x0c, 0xa4, 0xe2, 0x08, 0x65, 0x2a, 0xf0, 0x30, 0x6b, 0x15, 0x59,
	0x99, 0x9e, 0x75, 0x35, 0x77, 0x4f, 0x60, 0x99, 0x8c, 0x8f, 0xd2, 0x2b, 0x21, 0x57, 0xc3, 0xe5,
	0x48, 0xf9, 0x8a, 0x29, 0x50, 0xc6, 0x71, 0x06, 0x89, 0x01, 0x9a, 0xc9, 0x39, 0x04, 0x12, 0xc8,
	0xdf, 0xb1, 0x33, 0x6b, 0xa7, 0x1c, 0x3f, 0x7b, 0x2d, 0x76, 0x3a, 0xaf, 0x76, 0x3d, 0x08, 0x74,
	0x2c, 0xa2, 0xc8, 0xfd, 0x1a, 0x3a, 0x6f, 0x8b, 0xe8, 0xe9, 0xa9, 0xfe, 0x17, 0x0c, 0xed, 0x9d,
	0x40, 0xe6, 0xdf, 0x22, 0x89, 0x4d, 0xea, 0x09, 0x68, 0x96, 0x1e, 0x1a, 0x9c, 0xbd, 0x47, 0x35,
	0x68, 0xd9, 0x4f, 0x5e, 0x12, 0xbf, 0xd6, 0x09, 0x9d, 0xf6, 0x0f, 0xa7, 0xc2, 0xdb, 0xde, 0x70,
	0x35, 0x15, 0x2f, 0x73, 0x16, 0x3c, 0x9a, 0xdc, 0xb5, 0xc5, 0x35, 0x86, 0x8a, 0x31, 0xb8, 0xc1,
	0x74, 0x76, 0xd7, 0x65, 0x32, 0xad, 0xdc, 0x17, 0x1f, 0xfe, 0x85, 0xda, 0x32, 0xc9, 0x1d, 0xda,
	0x36, 0x16, 0xde, 0x76, 0x45, 0x3f, 0x85, 0x8c, 0x8b, 0xdc, 0x37, 0x08, 0x39, 0xef, 0x94, 0xaf,
	0xc8, 0x51, 0x19, 0x29, 0x70, 0x5d, 0xbb, 0x4e, 0xe8, 0xdb, 0xc2, 0xb2, 0x5f, 0x2e, 0xe3, 0x73,
	0xba, 0xc2, 0xa1, 0x42, 0x10, 0xb0, 0xe5, 0xb0, 0x64, 0xb4, 0xdc, 0xbb, 0xa1, 0x51, 0x12, 0x98,
	0xdc, 0x43, 0xcc, 0xc3, 0xc5, 0x25, 0xab, 0x45, 0x6e, 0x63, 0x7e, 0x45, 0x40, 0x63, 0x67, 0xd2
};

void pgm_espgal_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base());

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT3
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= espgal_tab[i & 0xff] << 8;

		src[i] = x;
	}
}



static const uint8_t happy6in1_tab[256] = { // IGS0008RD1031215
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x38, 0x52, 0x44, 0x31, 0x30, 0x33, 0x31, 0x32, 0x31, 0x35,
	0x14, 0xd6, 0x37, 0x5c, 0x5e, 0xc3, 0xd3, 0x62, 0x96, 0x3d, 0xfb, 0x47, 0xf0, 0xcb, 0xbf, 0xb0,
	0x60, 0xa1, 0xc2, 0x3d, 0x90, 0xd0, 0x58, 0x56, 0x22, 0xac, 0xdd, 0x39, 0x27, 0x7e, 0x58, 0x44,
	0xe0, 0x6b, 0x51, 0x80, 0xb4, 0xa4, 0xf0, 0x6f, 0x71, 0xd0, 0x57, 0x18, 0xc7, 0xb6, 0x41, 0x50,
	0x02, 0x2f, 0xdb, 0x4a, 0x08, 0x4b, 0xe3, 0x62, 0x92, 0xc3, 0xff, 0x26, 0xaf, 0x9f, 0x60, 0xa5,
	0x76, 0x28, 0x97, 0xfd, 0x0b, 0x10, 0xb7, 0x1f, 0xd5, 0xe0, 0xac, 0xe6, 0xfd, 0xa3, 0xdb, 0x58,
	0x2a, 0xd1, 0xfc, 0x3b, 0x7c, 0x7e, 0x34, 0xdc, 0xc7, 0xc4, 0x76, 0x1b, 0x11, 0x6d, 0x1b, 0xbb,
	0x4e, 0xe5, 0xc0, 0xe8, 0x5a, 0x60, 0x60, 0x0a, 0x38, 0x47, 0xb3, 0xc9, 0x89, 0xe9, 0xc6, 0x61,
	0x50, 0x5f, 0xdb, 0x28, 0xe5, 0xc0, 0x83, 0x5c, 0x37, 0x86, 0xfa, 0x32, 0x46, 0x40, 0xc3, 0x1d,
	0xdf, 0x7a, 0x85, 0x5c, 0x9a, 0xea, 0x24, 0xc7, 0x12, 0xdc, 0x23, 0xda, 0x65, 0xdf, 0x39, 0x02,
	0xeb, 0xb1, 0x32, 0x28, 0x3a, 0x69, 0x09, 0x7c, 0x5a, 0xe3, 0x44, 0x83, 0x45, 0x71, 0x8f, 0x64,
	0xa3, 0xbf, 0x9c, 0x6f, 0xc4, 0x07, 0x3a, 0xee, 0xdd, 0x77, 0xb4, 0x31, 0x87, 0xdf, 0x6d, 0xd4,
	0x75, 0x9f, 0xb9, 0x53, 0x75, 0xd0, 0xfe, 0xd1, 0xaa, 0xb2, 0x0b, 0x25, 0x08, 0x56, 0xb8, 0x27,
	0x10, 0x8c, 0xbf, 0x39, 0xce, 0x0f, 0xdb, 0x18, 0x10, 0xf0, 0x1f, 0xe5, 0xe8, 0x40, 0x98, 0x6f,
	0x64, 0x02, 0x27, 0xc3, 0x8c, 0x4f, 0x98, 0xf6, 0x9d, 0xcb, 0x07, 0x31, 0x85, 0x48, 0x75, 0xff,
	0x9f, 0xba, 0xa6, 0xd3, 0xb0, 0x5b, 0x3d, 0xdd, 0x22, 0x1f, 0x1b, 0x0e, 0x7f, 0x5a, 0xf4, 0x6a
};

void pgm_happy6_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= happy6in1_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t sdwx_tab[] =
{
	0x49, 0x47, 0x53, 0x30, 0x30, 0x35, 0x35, 0x52, 0x44, 0x34, 0x30, 0x32, 0x30, 0x36, 0x32, 0x31,
	0x8a, 0xbb, 0x20, 0x67, 0x97, 0xa5, 0x20, 0x45, 0x6b, 0xc0, 0xe8, 0x0c, 0x80, 0xfb, 0x49, 0xaa,
	0x1e, 0xac, 0x29, 0xf2, 0xb9, 0x9f, 0x01, 0x4a, 0x8d, 0x5f, 0x95, 0x96, 0x78, 0xc3, 0xf6, 0x65,
	0x17, 0xbd, 0xb6, 0x5b, 0x25, 0x5f, 0x6b, 0xde, 0x10, 0x2e, 0x67, 0x05, 0xdc, 0xac, 0xb6, 0xbd,
	0x3d, 0x20, 0x58, 0x3d, 0xf0, 0xa8, 0xc0, 0xad, 0x5b, 0x82, 0x8d, 0x12, 0x65, 0x97, 0x87, 0x7d,
	0x97, 0x49, 0xdd, 0x74, 0x74, 0x7e, 0x9d, 0xa1, 0x15, 0xed, 0x75, 0xb9, 0x09, 0xa8, 0xa8, 0xb0,
	0x6b, 0xea, 0x54, 0x1b, 0x45, 0x23, 0xe2, 0xe5, 0x25, 0x42, 0xce, 0x36, 0xfe, 0x42, 0x99, 0xa0,
	0x41, 0xf8, 0x0b, 0x8c, 0x3c, 0x1b, 0xae, 0xe4, 0xb2, 0x94, 0x87, 0x02, 0xbc, 0x08, 0x17, 0xd9,
	0xe0, 0xa4, 0x93, 0x63, 0x6f, 0x28, 0x5f, 0x4a, 0x24, 0x36, 0xd1, 0xda, 0xfa, 0xdd, 0x23, 0x26,
	0x4e, 0x61, 0xb9, 0x7a, 0x36, 0x4d, 0x95, 0x01, 0x20, 0xbc, 0x18, 0xb7, 0xaf, 0xe4, 0xfb, 0x92,
	0xd2, 0xe3, 0x8e, 0xec, 0x26, 0xce, 0x2f, 0x34, 0x8f, 0xf7, 0x0d, 0xd6, 0x11, 0x7f, 0x1f, 0x68,
	0xf4, 0x1d, 0x5f, 0x16, 0x19, 0x2d, 0x4c, 0x4f, 0x96, 0xfc, 0x9f, 0xb0, 0x99, 0x53, 0x4c, 0x32,
	0x7b, 0x41, 0xbc, 0x90, 0x23, 0x2e, 0x4a, 0xfc, 0x9e, 0x1d, 0xfc, 0x02, 0xfc, 0x41, 0x83, 0xbc,
	0x6d, 0xc4, 0x75, 0x37, 0x9d, 0xd3, 0xc9, 0x26, 0x4d, 0xed, 0x93, 0xc6, 0x32, 0x6d, 0x02, 0x11,
	0x12, 0x56, 0x97, 0x26, 0x1d, 0x5f, 0xa7, 0xf8, 0x89, 0x3f, 0x14, 0x36, 0x72, 0x3b, 0x48, 0x7b,
	0xf1, 0xed, 0x72, 0xb7, 0x7a, 0x56, 0x05, 0xde, 0x7b, 0x27, 0x6d, 0xcf, 0x33, 0x4c, 0x14, 0x86
};


void sdwx_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT2
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= sdwx_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


void chessc2_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
	//  IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}


void zhongguo_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
	//  IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

void cjddz_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1_ALT;
		IGS27_CRYPT2_ALT
	//  IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

void cjddzp_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}


// IGS FOR V000CN 2004 12 01
void lhzb4_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
	//  IGS27_CRYPT2
	//  IGS27_CRYPT3
		IGS27_CRYPT4_ALT
	//  IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		src[i] ^= x;
	}
}



//////////////////////////////////////////////////////////////////////

void superkds_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		// IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

void fearless_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		// IGS27_CRYPT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		// IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}



static unsigned char pgm3in1_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x31, 0x31, 0x72, 0x64, 0x31, 0x30, 0x34, 0x30, 0x36, 0x33, 0x30,
	0xc2, 0xbe, 0xde, 0x8b, 0x80, 0x8f, 0x04, 0xbb, 0xd2, 0xd6, 0x11, 0x4a, 0x54, 0xa9, 0x8d, 0x27,
	0xf1, 0x74, 0x88, 0xc2, 0xa3, 0xdd, 0x73, 0x97, 0x6c, 0xf8, 0x41, 0xf9, 0x95, 0xac, 0xbe, 0x20,
	0x26, 0x65, 0x2f, 0x76, 0xa6, 0xfc, 0x2b, 0x1f, 0x15, 0xe3, 0xb9, 0xc7, 0x27, 0x95, 0xab, 0xaf,
	0xc2, 0x3d, 0x3a, 0xfb, 0xba, 0xa5, 0xe3, 0x35, 0xcb, 0x62, 0x81, 0x26, 0xd7, 0x3e, 0xac, 0xd7,
	0x62, 0xe6, 0x51, 0xe2, 0x4b, 0xd5, 0x92, 0xfe, 0xcd, 0x80, 0xe0, 0xca, 0xb6, 0xc3, 0x57, 0xda,
	0xe6, 0x8b, 0x59, 0x00, 0x0a, 0xc5, 0x6f, 0xdc, 0x9a, 0x87, 0x5c, 0xa6, 0x11, 0x7e, 0x83, 0x3b,
	0x6c, 0x98, 0x7b, 0x66, 0xe6, 0xf2, 0xf0, 0x71, 0xf2, 0x04, 0xbd, 0xed, 0x78, 0x0b, 0x47, 0xbd,
	0x54, 0xb6, 0x1d, 0x69, 0x0d, 0x15, 0xcd, 0xa2, 0xd2, 0xc0, 0x09, 0x11, 0xbb, 0x44, 0xfb, 0x64,
	0x3d, 0xd3, 0xe5, 0x9b, 0xee, 0x2b, 0xfd, 0x90, 0x7b, 0xc6, 0x88, 0xc5, 0xe7, 0x44, 0x35, 0x73,
	0x06, 0x17, 0xbc, 0xcf, 0x39, 0x6e, 0xb7, 0x9f, 0x6b, 0x63, 0xc1, 0xfe, 0x4c, 0x68, 0xcd, 0x6c,
	0xcd, 0xee, 0xc8, 0x19, 0xdc, 0x58, 0x72, 0x72, 0x62, 0x20, 0x7a, 0xed, 0x79, 0x48, 0xd8, 0x12,
	0xf2, 0x04, 0x70, 0xca, 0x07, 0xa6, 0xe5, 0xec, 0x5d, 0xc9, 0xba, 0x05, 0x3e, 0xc2, 0xb0, 0x68,
	0x14, 0x43, 0x5b, 0x77, 0x28, 0x51, 0x06, 0x30, 0x9d, 0x69, 0xc9, 0xfb, 0xa8, 0xee, 0xe9, 0xb2,
	0x11, 0xd7, 0x70, 0xf2, 0xef, 0x96, 0x0d, 0xa1, 0xa0, 0x4b, 0x2e, 0xc0, 0x07, 0x2a, 0x5c, 0x72,
	0x09, 0x29, 0xd6, 0x4d, 0x4a, 0xef, 0x71, 0xe1, 0x25, 0xf9, 0xb0, 0x87, 0xea, 0x0f, 0x1f, 0x6c
};

void pgm_decrypt_pgm3in1(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3_ALT
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= pgm3in1_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


// IGS MAHJONG CHINA VS107C
void slqz3_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

//IGS COPYRIGHT 1999.99.99
// V214US
void fruitpar_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

void oceanpar_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

static const uint8_t amazonia_tab[0x100] = {
	0x86, 0x00, 0xd2, 0x6b, 0x78, 0x79, 0xbc, 0x80, 0xc6, 0x8e, 0x9b, 0xd8, 0x3a, 0x52, 0x09, 0x15,
	0x05, 0x57, 0x7a, 0x1f, 0x2f, 0x08, 0x47, 0xa0, 0x44, 0x4c, 0xf7, 0xbe, 0x3d, 0x59, 0xd1, 0x18,
	0xac, 0xd5, 0xa0, 0x03, 0x2c, 0x34, 0xc6, 0x06, 0x08, 0x4e, 0x84, 0x94, 0x2b, 0x3f, 0x88, 0x08,
	0x48, 0x84, 0xb8, 0xe6, 0x4a, 0x53, 0x3e, 0x51, 0xfe, 0x3b, 0xd4, 0xc8, 0xff, 0xfd, 0x52, 0x25,
	0xe2, 0xa9, 0x75, 0xd7, 0xa3, 0xfc, 0xf1, 0x5f, 0x50, 0xfa, 0xba, 0x09, 0xf4, 0xc8, 0x90, 0xed,
	0xc6, 0xcc, 0xc9, 0x25, 0x93, 0x07, 0x63, 0x50, 0x69, 0xb2, 0x4a, 0x47, 0x85, 0x18, 0xe6, 0x1f,
	0x7f, 0xb3, 0xe8, 0x5f, 0xb4, 0x8a, 0x56, 0x83, 0xf4, 0xca, 0xd5, 0xb1, 0x6d, 0xa3, 0x37, 0xba,
	0xd7, 0x66, 0x45, 0x54, 0xe2, 0xdc, 0xce, 0x96, 0xdc, 0xe9, 0xf0, 0xb5, 0xa7, 0x62, 0xa5, 0xfd,
	0xdb, 0x2b, 0x92, 0x13, 0x37, 0x95, 0x0c, 0x69, 0x4d, 0xf7, 0x6c, 0x02, 0x6e, 0x8a, 0x95, 0x68,
	0xd4, 0x8a, 0xc3, 0xea, 0x0f, 0x8b, 0x96, 0x1b, 0xb1, 0x19, 0x5e, 0x88, 0x3d, 0x94, 0xa9, 0xb9,
	0x4e, 0x4a, 0x0a, 0x6a, 0x04, 0xd5, 0x2c, 0x0b, 0xb4, 0xb7, 0x18, 0x75, 0xcf, 0x35, 0xc3, 0xef,
	0x13, 0x71, 0xdb, 0x61, 0xf2, 0xcb, 0xd3, 0xd7, 0x3f, 0x79, 0x2d, 0x39, 0x20, 0x65, 0x07, 0x49,
	0x30, 0x47, 0xe9, 0xdd, 0xf3, 0x03, 0xce, 0x5f, 0x7f, 0x45, 0x70, 0x82, 0x69, 0x5a, 0xd9, 0x46,
	0xee, 0x52, 0x27, 0x2f, 0x63, 0x54, 0x9f, 0xc1, 0xde, 0x41, 0xf5, 0x40, 0x26, 0x8d, 0xda, 0xa6,
	0xd9, 0x5a, 0xc7, 0xe4, 0xdc, 0xd6, 0x09, 0x5e, 0x08, 0xd6, 0x0d, 0xa1, 0x12, 0xb3, 0xee, 0x67,
	0xbb, 0x65, 0x3c, 0xcd, 0x3a, 0xdf, 0x0f, 0xd3, 0xe6, 0xaa, 0x4d, 0x14, 0x28, 0xc5, 0x38, 0xc8
};

void amazonia_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= amazonia_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

//IGS FOR V201BR 2003 09 01

static const uint8_t amazoni2_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x37, 0x37, 0x52, 0x44, 0x34, 0x30, 0x32, 0x31, 0x30, 0x31, 0x36,
	0xed, 0x3f, 0x77, 0xd0, 0xc4, 0x72, 0x77, 0xe1, 0x35, 0x15, 0x10, 0xc6, 0x1b, 0xee, 0x25, 0x4a,
	0x0d, 0x06, 0x37, 0x19, 0xfa, 0xf9, 0x29, 0xb0, 0x03, 0x7e, 0x32, 0x83, 0x99, 0x32, 0x23, 0x6e,
	0xd8, 0xa9, 0xdb, 0x5f, 0x23, 0xb7, 0x08, 0x41, 0x00, 0xfb, 0x20, 0x2b, 0x36, 0xae, 0x9f, 0xc9,
	0x24, 0xaa, 0x82, 0x0b, 0x64, 0xbe, 0x03, 0x0d, 0xa2, 0x31, 0x97, 0x47, 0xb6, 0x94, 0x26, 0xf6,
	0x09, 0xcd, 0x8a, 0xca, 0x26, 0x61, 0x4a, 0xd1, 0x9f, 0x01, 0x98, 0xa3, 0x23, 0x58, 0x88, 0xcf,
	0xdd, 0x14, 0x93, 0x86, 0x0f, 0x33, 0x4c, 0x86, 0xef, 0x8f, 0x61, 0x4a, 0xc2, 0xad, 0xd5, 0x70,
	0x37, 0xc2, 0x7b, 0x6a, 0x06, 0x07, 0xb7, 0x69, 0xc9, 0x3f, 0x71, 0x87, 0x1a, 0x85, 0x5a, 0x33,
	0xee, 0x5b, 0x63, 0xe2, 0x32, 0xf0, 0x7a, 0xf4, 0xa4, 0xb2, 0x87, 0xe4, 0xf3, 0x14, 0xa7, 0xb4,
	0x1a, 0xa1, 0xa8, 0x98, 0xf9, 0x42, 0xc5, 0xe1, 0x36, 0xcc, 0xa3, 0x2d, 0x54, 0xcd, 0x8b, 0xcd,
	0x11, 0x98, 0xe9, 0x77, 0x05, 0x8e, 0x06, 0x2d, 0x78, 0xb0, 0x02, 0x6d, 0x83, 0x62, 0x15, 0x9a,
	0x6b, 0x82, 0x07, 0xaa, 0x3a, 0xa9, 0xed, 0x12, 0x9f, 0xc0, 0x25, 0xee, 0x08, 0xc7, 0x94, 0x75,
	0xfd, 0xe2, 0x1f, 0x9d, 0xc0, 0xa5, 0x68, 0x0a, 0x23, 0xa1, 0xca, 0x3d, 0xa9, 0x2f, 0x97, 0xfa,
	0xe0, 0x7c, 0x91, 0xfa, 0xfe, 0xd5, 0xa6, 0xd2, 0xbb, 0x34, 0xf1, 0x23, 0x6e, 0x0c, 0xed, 0x03,
	0x6b, 0x52, 0xfb, 0xac, 0x9c, 0xcc, 0x17, 0x64, 0x5d, 0x9e, 0xd7, 0xac, 0x9e, 0x12, 0xa5, 0xac,
	0x34, 0xa7, 0x3e, 0xde, 0x7f, 0x5d, 0x6a, 0xfa, 0x42, 0x40, 0xfd, 0x23, 0xbf, 0x33, 0x0e, 0x50
};


void amazoni2_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		// should be correct
		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
	//    IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT // second half is junk, not sure...
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= amazoni2_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

void pgm_dwpc_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		src[i] = x;
	}
}

// IGS FOR V100CN 2007 04 18 in the ROM
void mgcs3_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

// IGS0083RD4021212
void mgzz_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		int x = 0;

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

// IGS FOR V101JP 2007 06 08
void crzybugsj_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

// IGS MAHJONG S500CN
void qlgs_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

// JUN 19 14:13:22 2003 TARZAN_CHINA_V300CN
void tct2p_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

// FRI OCT 07 17:22:46 2005 LUCKY_FASTER
void xypdk_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

static const uint8_t luckycrs_tab[0x100] = {
	0x4b, 0x99, 0xa3, 0xa6, 0x69, 0xad, 0xb2, 0x1e, 0x42, 0x9a, 0x02, 0x91, 0x04, 0x90, 0x64, 0xcf,
	0x68, 0xe0, 0xa3, 0x8e, 0x1b, 0x53, 0x8d, 0xe6, 0x6f, 0xcd, 0x40, 0x09, 0x42, 0x23, 0xb7, 0x47,
	0x2f, 0x39, 0x95, 0x41, 0xf4, 0x60, 0x30, 0x6d, 0x23, 0xee, 0xe1, 0xcb, 0x0d, 0x1f, 0x8b, 0xe5,
	0xec, 0x2c, 0x69, 0x0d, 0x50, 0xaa, 0x1d, 0xd2, 0xcc, 0x25, 0xf7, 0xc4, 0xe0, 0xfd, 0x82, 0x6a,
	0x2a, 0x80, 0x55, 0x81, 0xc9, 0x48, 0x18, 0x76, 0x12, 0xd7, 0xd5, 0x26, 0x77, 0x72, 0x81, 0xd4,
	0xb3, 0x3b, 0xca, 0x6b, 0x3b, 0x92, 0x23, 0xf6, 0xe1, 0xad, 0x0e, 0x5d, 0xcb, 0x76, 0xa9, 0x62,
	0x94, 0xa5, 0x7c, 0xdc, 0xc0, 0x1e, 0x81, 0x32, 0x65, 0x8d, 0x75, 0x1b, 0x18, 0x3f, 0x5e, 0x94,
	0x16, 0x44, 0x5d, 0x21, 0xb4, 0xc3, 0xb6, 0x49, 0x08, 0x9d, 0x1d, 0x4c, 0xd9, 0x46, 0x43, 0x27,
	0xc5, 0xe0, 0xa1, 0xcb, 0xb1, 0x99, 0x84, 0x99, 0x76, 0x46, 0x5a, 0x21, 0xc9, 0x40, 0x3c, 0x1c,
	0x6b, 0x7f, 0xbb, 0xa7, 0x93, 0xf6, 0xef, 0xc2, 0x99, 0x2e, 0xbe, 0x09, 0xe3, 0x25, 0x69, 0xb1,
	0x15, 0x69, 0x5e, 0xc6, 0x74, 0x71, 0x39, 0xa3, 0x9b, 0x3c, 0x1b, 0xb2, 0x63, 0x2d, 0x30, 0x65,
	0x0c, 0x25, 0x7c, 0x75, 0xb0, 0xe1, 0xe5, 0x5a, 0xea, 0x97, 0x86, 0x0b, 0xc2, 0xcc, 0x33, 0xf8,
	0xdc, 0x79, 0x48, 0x44, 0xe1, 0x5e, 0xb7, 0x47, 0x2e, 0xa6, 0x51, 0x44, 0xbc, 0xbc, 0x54, 0x67,
	0x50, 0x6d, 0x36, 0x03, 0xe3, 0x3f, 0xb1, 0x09, 0x54, 0x10, 0x10, 0xcc, 0x4c, 0xf3, 0xb8, 0xf3,
	0x72, 0x47, 0xf9, 0xbf, 0xd1, 0x19, 0x17, 0x7f, 0x86, 0xbc, 0x94, 0x50, 0xae, 0xa7, 0xc0, 0x1a,
	0x8e, 0x8f, 0x84, 0xc8, 0x05, 0xc5, 0x6b, 0xc7, 0x2f, 0xd2, 0xf1, 0xc2, 0x5b, 0x50, 0x11, 0x9b
};

void luckycrs_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		// fairly high confidence in conditional xors with comments
		IGS27_CRYPT1_ALT    // 3900
		IGS27_CRYPT2        // 8bd0
		IGS27_CRYPT3        // 3a20
		IGS27_CRYPT4_ALT    // 8db0
		IGS27_CRYPT5_ALT    // 140d0
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT    // 3a40

		x ^= luckycrs_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t olympic5_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x34, 0x34, 0x52, 0x44, 0x34, 0x30, 0x32, 0x30, 0x35, 0x33, 0x31, // IGS0044RD4020531
	0xb6, 0x32, 0xe0, 0xf9, 0x95, 0x3c, 0xa7, 0x99, 0xd9, 0x14, 0xf0, 0x81, 0x05, 0xd5, 0xc4, 0xe8,
	0x64, 0x59, 0x78, 0x83, 0xe5, 0xca, 0xaa, 0xaf, 0x37, 0x66, 0x80, 0xe1, 0x20, 0x92, 0x1d, 0x50,
	0x23, 0x22, 0x64, 0xe1, 0x53, 0xe8, 0xf1, 0x53, 0x71, 0x7b, 0x54, 0xab, 0x4d, 0x49, 0x90, 0xe2,
	0x5b, 0xdf, 0xd3, 0xcd, 0x93, 0x78, 0x78, 0x4e, 0x8f, 0xc5, 0x3b, 0xb9, 0xe0, 0xfc, 0xbb, 0x89,
	0xb1, 0x23, 0x35, 0x44, 0x9f, 0x9e, 0x7f, 0xac, 0xd7, 0xf8, 0x44, 0x27, 0x72, 0xee, 0x7e, 0x71,
	0x0d, 0xc2, 0x39, 0x80, 0xad, 0xbe, 0x86, 0xb7, 0xd2, 0x06, 0xbd, 0x50, 0xd9, 0xa3, 0xf8, 0x03,
	0x97, 0xcd, 0xcd, 0xfc, 0x33, 0x79, 0x4b, 0xfc, 0x44, 0x23, 0x37, 0xcd, 0x2c, 0xdd, 0x88, 0xec,
	0xb4, 0x99, 0x20, 0x72, 0xe9, 0xb3, 0xcd, 0x44, 0x36, 0xc2, 0x80, 0x7c, 0xc3, 0xa0, 0xcc, 0x16,
	0x0d, 0xb8, 0xa2, 0xdf, 0xc6, 0x8f, 0x4b, 0x9b, 0xef, 0x95, 0xa7, 0x76, 0x34, 0x2e, 0xa4, 0xad,
	0x88, 0xfc, 0x01, 0x7d, 0x01, 0x70, 0x45, 0x4c, 0xf6, 0x91, 0xfa, 0x16, 0x56, 0x0b, 0x2f, 0x1b,
	0x4c, 0x7a, 0x2d, 0xc7, 0x11, 0xf9, 0x79, 0xe1, 0x11, 0xe6, 0x0a, 0xf8, 0x41, 0xf8, 0xcc, 0x0b,
	0xc0, 0x85, 0x54, 0x77, 0xac, 0x0c, 0xe6, 0x27, 0x47, 0x0a, 0xa5, 0xf6, 0x4b, 0xfb, 0x19, 0x68,
	0x8b, 0xae, 0xe6, 0x8a, 0xcb, 0xce, 0xcb, 0x27, 0xe0, 0xae, 0xd9, 0x2c, 0x0c, 0x54, 0xf6, 0x5e,
	0x94, 0xc9, 0x91, 0x3a, 0xa3, 0xa0, 0xa7, 0x2c, 0x63, 0xc5, 0xf7, 0xf5, 0x59, 0x88, 0x81, 0x56,
	0x02, 0xea, 0x45, 0x02, 0xac, 0x27, 0x3a, 0xc3, 0x96, 0x83, 0x8c, 0xeb, 0x4b, 0x5a, 0x1a, 0xfd,
};

void olympic5_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		// fairly high confidence in conditional xors with comments
		IGS27_CRYPT1 // 2df0
		IGS27_CRYPT2_ALT // ac50
		IGS27_CRYPT3
		IGS27_CRYPT4 // 2df0
		IGS27_CRYPT5 // 109f0
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8 // 3340

		x ^= olympic5_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


void crzybugs_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

void icescape_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		src[i] ^= x;
	}
}


void tripfev_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		src[i] ^= x;
	}
}


static const uint8_t wldfruit_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x31, 0x37, 0x38, 0x52, 0x44, 0x34, 0x30, 0x34, 0x30, 0x38, 0x32, 0x37, // IGS0178RD4040827
	0xf1, 0x99, 0x0f, 0xbf, 0xa4, 0x08, 0xb1, 0xfa, 0x5d, 0x08, 0x89, 0xc7, 0x3a, 0x97, 0x68, 0x2e,
	0x01, 0xd0, 0x4e, 0x14, 0xfa, 0x7f, 0x64, 0x6a, 0x0c, 0xe5, 0x37, 0x96, 0x1b, 0xc8, 0x37, 0x21,
	0xa1, 0xbf, 0x6f, 0xa3, 0x8b, 0xbc, 0x87, 0xec, 0x56, 0xda, 0xf7, 0x74, 0x6a, 0x69, 0x8d, 0x05,
	0xb6, 0x37, 0x22, 0xa7, 0x8e, 0x21, 0x98, 0xcc, 0xc1, 0xdb, 0x18, 0xbd, 0xfe, 0xfc, 0x8a, 0x44,
	0x69, 0x4d, 0x55, 0x9b, 0x79, 0x53, 0x59, 0x95, 0x15, 0x1a, 0x29, 0x0b, 0xef, 0x44, 0x8d, 0x89,
	0x20, 0x53, 0x38, 0x3a, 0x05, 0x33, 0xc6, 0x11, 0x59, 0x0b, 0xf9, 0x39, 0x93, 0x45, 0x35, 0xc0,
	0x82, 0xdc, 0x39, 0x7f, 0x27, 0xe5, 0x20, 0x4c, 0xd3, 0x61, 0x97, 0x64, 0x81, 0x41, 0x60, 0x12,
	0x76, 0xbb, 0x08, 0xa5, 0x17, 0xcc, 0xe4, 0x91, 0x0a, 0x0e, 0x52, 0xe4, 0x91, 0xbc, 0x2e, 0xec,
	0x23, 0x03, 0x93, 0x27, 0x4c, 0x8b, 0xd3, 0x6b, 0xc7, 0x46, 0xb9, 0x56, 0xd9, 0x78, 0xfe, 0xf9,
	0xf0, 0x07, 0x09, 0xc0, 0x7d, 0x05, 0xec, 0xa5, 0x0f, 0x7c, 0x9b, 0x95, 0xb0, 0x79, 0x6e, 0x22,
	0x84, 0x5b, 0xda, 0x6b, 0xa0, 0x5d, 0x6c, 0x49, 0x29, 0x62, 0x07, 0xbb, 0xae, 0x01, 0x5e, 0x94,
	0xc7, 0xd1, 0xb4, 0x63, 0xee, 0xf6, 0xd3, 0xa3, 0x9d, 0xec, 0x4b, 0x24, 0xa9, 0x93, 0xed, 0xb9,
	0xde, 0x7b, 0x87, 0x23, 0xdc, 0x72, 0xe1, 0x3e, 0x31, 0x4c, 0xf8, 0x6a, 0xb8, 0xf2, 0x7a, 0x3c,
	0x32, 0xae, 0x82, 0x66, 0x22, 0xb6, 0x94, 0xe4, 0xee, 0xf6, 0xdc, 0x69, 0x33, 0x22, 0xa3, 0x09,
	0x68, 0xfd, 0x12, 0x28, 0xb7, 0xe3, 0x2b, 0xa2, 0x18, 0x9d, 0x05, 0x3b, 0xb0, 0x66, 0x48, 0x4a
};

void wldfruit_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2 //
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT // $2fc0
		IGS27_CRYPT5_ALT // $35da0
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7 // $2b0
		IGS27_CRYPT8 // $1100

		x ^= wldfruit_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


void tripslot_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}


void extradrw_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x |= (x << 8);
		src[i] ^= x;
	}
}

void cjddzlf_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

void cjtljp_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		src[i] ^= x;
	}
}

void mxsqy_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		// this part is the same as luckycrs
		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		src[i] ^= x;
	}
}

void qiji6_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

void mjzb_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

void tswxp_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}


void jking04_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}


static const uint8_t jhg3d_tab[0x100] = { // IGS2200RD4011127
	0x49, 0x47, 0x53, 0x32, 0x32, 0x30, 0x30, 0x52, 0x44, 0x34, 0x30, 0x31, 0x31, 0x31, 0x32, 0x37,
	0x01, 0x8c, 0xa0, 0x88, 0x54, 0x47, 0xca, 0xa8, 0x5f, 0xc9, 0xae, 0x1b, 0x81, 0x07, 0xc7, 0x96,
	0x6c, 0xe1, 0x23, 0x35, 0x7c, 0x2b, 0xe2, 0x26, 0x54, 0xf7, 0xf2, 0x94, 0xbb, 0x26, 0x09, 0xd7,
	0xb5, 0x2e, 0xe4, 0xda, 0xe5, 0x2d, 0xbe, 0xcc, 0x23, 0x2f, 0x13, 0x0a, 0x59, 0xbc, 0x96, 0x4c,
	0xa3, 0xa6, 0x70, 0x0f, 0xa7, 0x11, 0xbb, 0x43, 0x31, 0xc3, 0x3f, 0x37, 0x13, 0xad, 0x6c, 0xc1,
	0x3c, 0xbb, 0x97, 0xb1, 0x18, 0xd9, 0x79, 0x78, 0x27, 0x46, 0xe5, 0x15, 0xe1, 0x1d, 0xc9, 0x41,
	0xc8, 0x20, 0x69, 0xdb, 0xcf, 0xc8, 0xd7, 0x96, 0xeb, 0x8b, 0xb5, 0xe1, 0xf7, 0x6d, 0x2e, 0x16,
	0xce, 0xca, 0x33, 0xe7, 0xa3, 0x62, 0xf3, 0x06, 0xa4, 0xa5, 0x9d, 0x14, 0xcf, 0x41, 0x5a, 0xcd,
	0x14, 0xea, 0x85, 0x71, 0xac, 0x69, 0x2d, 0x75, 0xb9, 0xe8, 0xcd, 0x6b, 0x1f, 0x7c, 0x4a, 0x2f,
	0xa2, 0xf3, 0x2e, 0x53, 0x40, 0xe0, 0x24, 0xcd, 0xd1, 0xe5, 0xb3, 0xe0, 0xdd, 0x41, 0x3f, 0x48,
	0xbe, 0x9a, 0x3d, 0xa9, 0xf7, 0x0b, 0xb6, 0x3a, 0xd3, 0x71, 0xff, 0xad, 0x41, 0xf3, 0xb7, 0x62,
	0xf0, 0xd0, 0x01, 0xcd, 0xa6, 0x6d, 0x03, 0x25, 0xe6, 0x9e, 0x9f, 0x4f, 0xc2, 0x34, 0x71, 0x0a,
	0xfe, 0xc8, 0x09, 0x5b, 0x67, 0xc7, 0x6a, 0x3b, 0x72, 0xbf, 0xc3, 0x80, 0x17, 0xe9, 0x6c, 0x09,
	0xf0, 0xf6, 0x24, 0x2e, 0x8e, 0x1e, 0x8a, 0x66, 0x1c, 0x67, 0xd9, 0x3a, 0x37, 0x33, 0xe8, 0x6b,
	0x0c, 0x0d, 0x61, 0x5f, 0xb4, 0xb5, 0x41, 0xd2, 0xcc, 0x69, 0x90, 0xba, 0x59, 0x76, 0x62, 0x7a,
	0xda, 0x00, 0x0f, 0x4c, 0xaf, 0x0e, 0xaf, 0xe8, 0xaa, 0xd9, 0xd8, 0x7a, 0xf3, 0x56, 0x9b, 0xc3
};

void jhg3d_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		// not confident that these are 100% correct
		IGS27_CRYPT1_ALT2
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= jhg3d_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


void royal5p_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

void mgfx_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = 0;

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		src[i] ^= x;
	}
}

// IGS TARZAN2 INTERNATIONAL ONLINE
// V101XB
// U23 27C4096 (E463).bin

static const uint8_t tarzan2_tab[0x100] = { // IGS0023RD4011129
	0x49, 0x47, 0x53, 0x30, 0x30, 0x32, 0x33, 0x52, 0x44, 0x34, 0x30, 0x31, 0x31, 0x31, 0x32, 0x39,
	0x57, 0xdd, 0x90, 0x83, 0x88, 0xb3, 0xeb, 0x8d, 0xc7, 0x2a, 0x18, 0x37, 0x09, 0x39, 0x5c, 0x51,
	0xb2, 0x90, 0xad, 0x3e, 0x24, 0x9c, 0xdb, 0x11, 0x86, 0xbe, 0xfa, 0xe8, 0xd6, 0xc7, 0x12, 0x89,
	0x32, 0xb7, 0x0a, 0x3c, 0x62, 0xb7, 0xd2, 0xe8, 0xaa, 0x54, 0xd0, 0x71, 0xcf, 0xfe, 0x8e, 0x51,
	0x68, 0xc0, 0xa1, 0x75, 0x67, 0x41, 0xdc, 0x5a, 0xe7, 0xfc, 0xb6, 0x68, 0x38, 0xbc, 0xf9, 0x8e,
	0x29, 0x5b, 0xad, 0x20, 0x95, 0xbb, 0x44, 0xed, 0x30, 0x05, 0x07, 0xa4, 0x93, 0x21, 0xc0, 0x69,
	0x86, 0x76, 0xa9, 0xb3, 0x8e, 0xe4, 0x95, 0x69, 0xb7, 0xfe, 0x5d, 0x3d, 0xa4, 0x8c, 0x8e, 0x48,
	0xd4, 0x41, 0x50, 0xe7, 0x36, 0xb9, 0x99, 0xd5, 0xf0, 0xb6, 0x94, 0x89, 0x6d, 0x9c, 0x4c, 0xd2,
	0xa5, 0x2a, 0x9d, 0xb1, 0xb1, 0x7a, 0x5d, 0x77, 0x8e, 0x3c, 0xc6, 0x1f, 0x31, 0x2f, 0x27, 0xee,
	0xcc, 0xe1, 0xcb, 0x49, 0x60, 0xa7, 0x2a, 0xd6, 0x83, 0xdf, 0x4f, 0xd7, 0x74, 0x65, 0x89, 0xc3,
	0x5c, 0x54, 0x55, 0x26, 0xe7, 0xfe, 0x8b, 0xbb, 0x03, 0x2e, 0xc9, 0xc7, 0xf8, 0x9d, 0x1d, 0xb8,
	0xa9, 0xb2, 0xf6, 0xff, 0x29, 0x7e, 0x4d, 0x2a, 0x81, 0xf8, 0x10, 0x46, 0xc0, 0x75, 0xcf, 0x74,
	0x44, 0x6b, 0xa9, 0xca, 0x48, 0x67, 0x79, 0x6d, 0xaf, 0x4c, 0x3e, 0xec, 0x10, 0xcd, 0xc8, 0xdf,
	0x02, 0x2d, 0xaa, 0xc0, 0xa9, 0x36, 0x5b, 0x09, 0x81, 0x78, 0xae, 0x8f, 0x6a, 0xc4, 0x76, 0x1e,
	0xf5, 0xe8, 0x72, 0x56, 0xee, 0xac, 0x7e, 0xc5, 0x2a, 0x0d, 0xfd, 0x46, 0x92, 0xb9, 0x81, 0x9a,
	0x70, 0xca, 0xbd, 0x44, 0xf9, 0xc7, 0xad, 0xa9, 0x1c, 0xd9, 0x03, 0x69, 0x8a, 0x4a, 0xd5, 0xf8
};

void tarzan2_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT2
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= tarzan2_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


// V200PR
// IGS FOR V103PR 2006 08 16
// u23 27C4002.bin

static const uint8_t unkigs027_tab[0x100] = { // IGS0277RD4070117
	0x49, 0x47, 0x53, 0x30, 0x32, 0x37, 0x37, 0x52, 0x44, 0x34, 0x30, 0x37, 0x30, 0x31, 0x31, 0x37,
	0xad, 0x30, 0x88, 0x4c, 0x0d, 0x9e, 0x6f, 0x1d, 0xb5, 0xa6, 0x53, 0x47, 0x14, 0xb0, 0xf7, 0xd5,
	0x12, 0x1e, 0x9f, 0xb9, 0xb6, 0xeb, 0x63, 0x7d, 0x19, 0x85, 0xb6, 0x47, 0x8c, 0xe8, 0x32, 0x14,
	0x06, 0xfb, 0xa6, 0x2d, 0xd0, 0x2f, 0x62, 0xee, 0x73, 0x56, 0xc2, 0xb7, 0xe2, 0xc1, 0xf8, 0x33,
	0xe4, 0x9e, 0x1f, 0xc8, 0xc5, 0x90, 0x7e, 0x5f, 0x3d, 0x8f, 0x19, 0x56, 0xa1, 0x03, 0x7e, 0xc2,
	0x48, 0x1c, 0xce, 0xe9, 0x40, 0x75, 0x09, 0xff, 0x32, 0xe9, 0x9f, 0x21, 0x93, 0xb4, 0x35, 0x90,
	0x0d, 0xce, 0xb4, 0x2f, 0x2d, 0x85, 0x98, 0x3e, 0x4d, 0x59, 0x78, 0x59, 0xc4, 0x1c, 0xd1, 0xac,
	0x4d, 0x4b, 0x17, 0x78, 0xb5, 0xa7, 0xfe, 0xcb, 0xc9, 0x17, 0x05, 0x7b, 0x7f, 0xc2, 0x44, 0x65,
	0x64, 0x68, 0x77, 0xe4, 0x45, 0x02, 0x4c, 0x94, 0x22, 0x9a, 0xea, 0x48, 0x4f, 0x6c, 0xc2, 0x49,
	0xec, 0x3e, 0x9a, 0xd2, 0x88, 0xfd, 0xd6, 0xc9, 0x11, 0x99, 0x0a, 0xbe, 0xff, 0x22, 0xbd, 0x29,
	0xc1, 0x24, 0x80, 0xe0, 0x67, 0x3f, 0x30, 0xd7, 0x92, 0x0b, 0x87, 0x1d, 0x99, 0x2b, 0xea, 0x13,
	0xfd, 0xaf, 0x6e, 0xee, 0x0f, 0xaf, 0x2b, 0x70, 0xe0, 0x26, 0xc6, 0xe2, 0x69, 0x0d, 0x39, 0x56,
	0xfc, 0xb8, 0xe6, 0x1b, 0xea, 0x75, 0xdc, 0x81, 0x76, 0x62, 0x69, 0xce, 0xfa, 0x8f, 0xe0, 0x80,
	0x58, 0x55, 0xac, 0xc6, 0xa4, 0xf6, 0x95, 0x39, 0x0f, 0x76, 0x52, 0xdf, 0x16, 0xb9, 0x50, 0x62,
	0xed, 0xde, 0xc2, 0x8d, 0x26, 0xdb, 0xe9, 0x08, 0xa6, 0x59, 0xa5, 0x54, 0xca, 0xd2, 0x3d, 0x0a,
	0xd5, 0xe9, 0x6b, 0x50, 0x9d, 0x09, 0xac, 0x9c, 0x76, 0x42, 0xc6, 0xad, 0x5f, 0x60, 0x99, 0xc6
};

void magtree_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		x ^= unkigs027_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

// V015US
static const uint8_t texashld_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x32, 0x32, 0x38, 0x52, 0x44, 0x34, 0x30, 0x35, 0x31, 0x30, 0x32, 0x36, // IGS0228RD4051026
	0x4b, 0x26, 0x03, 0xb7, 0x63, 0x4d, 0xa2, 0x94, 0xb6, 0xa1, 0xf7, 0xe7, 0xe4, 0x48, 0xa3, 0x24,
	0xd9, 0x86, 0xd7, 0xb0, 0x61, 0xb9, 0x7a, 0xff, 0xef, 0x94, 0x9d, 0xf2, 0xaa, 0x9b, 0x21, 0x70,
	0xb9, 0xdf, 0xa7, 0x43, 0xd0, 0xfc, 0xb8, 0x34, 0xfb, 0x50, 0x36, 0x6a, 0xaa, 0x7d, 0x03, 0x00,
	0xfe, 0x21, 0xef, 0xe6, 0x53, 0xd5, 0xe7, 0xfa, 0x0e, 0x64, 0x5d, 0x67, 0xa8, 0x4d, 0xf2, 0x3d,
	0xfb, 0x7b, 0x6b, 0x51, 0xcc, 0x43, 0xd2, 0x58, 0x9a, 0x9f, 0xec, 0x3f, 0xa6, 0xa9, 0xdb, 0xcd,
	0x43, 0x5b, 0x14, 0x7c, 0x60, 0x85, 0x84, 0x94, 0x53, 0x10, 0xfe, 0x89, 0xe8, 0x71, 0xe8, 0x97,
	0xaa, 0x70, 0x27, 0x9c, 0x71, 0x1b, 0x47, 0x35, 0x2c, 0x05, 0xef, 0x1c, 0xf0, 0xc4, 0x85, 0xc2,
	0x42, 0xaa, 0x1d, 0x28, 0xa1, 0xc3, 0xa7, 0x03, 0x57, 0x0f, 0x5a, 0x10, 0x82, 0x01, 0x5b, 0xb5,
	0x5d, 0x38, 0xb2, 0xd9, 0xd4, 0x7c, 0x6f, 0x05, 0x48, 0xfc, 0x19, 0xbb, 0x9f, 0xc6, 0x57, 0x17,
	0x90, 0x87, 0xe2, 0xa4, 0x2d, 0x85, 0xaa, 0x81, 0xb2, 0xda, 0x47, 0xb5, 0x8d, 0xf3, 0xa3, 0xcf,
	0xad, 0x49, 0xe6, 0xc2, 0x0f, 0x5e, 0xa2, 0xff, 0x87, 0xfa, 0x40, 0xd4, 0xcc, 0xa7, 0xab, 0x04,
	0xc7, 0x6a, 0x3b, 0xa8, 0x1d, 0xc4, 0xe4, 0x45, 0xfb, 0xea, 0x9f, 0x2f, 0x21, 0x40, 0x18, 0x1e,
	0x32, 0x1c, 0x9a, 0x0e, 0x3a, 0xb8, 0x39, 0x5b, 0x80, 0x78, 0x3e, 0x1d, 0x8f, 0x5e, 0xd7, 0xc2,
	0x7f, 0xcb, 0x00, 0xeb, 0x89, 0x78, 0xac, 0x88, 0xcb, 0xb5, 0x39, 0x36, 0x57, 0xe0, 0x13, 0xd9,
	0x83, 0x28, 0xa6, 0x75, 0x6d, 0x84, 0x8a, 0x52, 0xcc, 0xee, 0xea, 0x50, 0xff, 0xe5, 0x35, 0x88
};

void texashld_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1        // $02d00
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4        // $02ce0
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8        // $01300

		x ^= texashld_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

// V102US 2007
static const uint8_t crzybug2_tab[0x100] = {
		0x49, 0x47, 0x53, 0x30, 0x33, 0x30, 0x34, 0x52, 0x44, 0x34, 0x30, 0x37, 0x30, 0x36, 0x31, 0x31, // IGS0304RD4070611
		0x72, 0xba, 0x6a, 0x0e, 0xaf, 0x03, 0x46, 0x16, 0x8d, 0xd9, 0xd8, 0xb9, 0xe1, 0x8f, 0x35, 0xe5,
		0xea, 0x2b, 0x3f, 0x5c, 0xf0, 0xb6, 0x69, 0x9e, 0x1e, 0xc3, 0xad, 0x17, 0x9c, 0x3e, 0xca, 0x85,
		0xf0, 0x77, 0xc3, 0xad, 0x96, 0x6f, 0x94, 0x3e, 0xdf, 0x62, 0x37, 0x05, 0x07, 0x95, 0xd8, 0x53,
		0x9a, 0x61, 0x54, 0xac, 0x09, 0x80, 0xf8, 0xb1, 0x87, 0x97, 0x77, 0x4f, 0x2a, 0x04, 0x2c, 0x29,
		0x3f, 0xeb, 0x92, 0x44, 0xee, 0x7b, 0x01, 0xf2, 0x0f, 0x86, 0xab, 0x01, 0x4b, 0x41, 0xd6, 0x24,
		0x77, 0x58, 0x5c, 0xa1, 0x2d, 0x35, 0x60, 0x3c, 0xad, 0x92, 0x52, 0x63, 0xf2, 0x3d, 0x26, 0x9c,
		0x18, 0x2b, 0xd0, 0x2c, 0xee, 0xc1, 0x03, 0x0a, 0xd7, 0x5e, 0x2b, 0x03, 0xe6, 0x2c, 0xa9, 0x2e,
		0x3a, 0x28, 0x4e, 0x93, 0x96, 0x70, 0x1a, 0x17, 0x45, 0xcd, 0x36, 0xaa, 0x2d, 0x80, 0x2f, 0xb5,
		0x32, 0x52, 0x75, 0xbe, 0xce, 0xd6, 0x13, 0x5e, 0xee, 0x01, 0xb1, 0x65, 0x0f, 0xee, 0xc7, 0x4b,
		0x99, 0xea, 0x23, 0xda, 0x7c, 0xc7, 0x9e, 0x1a, 0x09, 0x5f, 0x1b, 0x7d, 0x12, 0x66, 0xc0, 0x4b,
		0x46, 0x75, 0x78, 0x51, 0xc6, 0x54, 0xa9, 0xc6, 0x0c, 0x88, 0x33, 0x7d, 0xfe, 0x1e, 0xa9, 0x51,
		0x4e, 0xb6, 0xd3, 0xce, 0x15, 0xd2, 0x64, 0x1d, 0xb0, 0x61, 0xf8, 0x32, 0xda, 0x87, 0x51, 0x38,
		0x0a, 0xaf, 0xd3, 0x3c, 0x0f, 0xd3, 0x3d, 0x1a, 0xea, 0x0b, 0xaa, 0xa5, 0xec, 0x54, 0xc7, 0x1a,
		0x11, 0xa3, 0x56, 0xc7, 0x9b, 0x2a, 0xe4, 0xf8, 0xf2, 0xea, 0xc6, 0x22, 0xbc, 0x7a, 0x5a, 0x53,
		0x38, 0x16, 0x7c, 0xd9, 0xe0, 0xeb, 0x47, 0x32, 0x3f, 0xa1, 0x0d, 0x34, 0x10, 0x2a, 0x99, 0x7d
};

void crzybug2_decrypt(running_machine &machine)
{
	memory_region *const region = machine.root_device().memregion("user1");
	auto const src = util::little_endian_cast<u16>(reinterpret_cast<u32 *>(region->base()));
	auto const rom_size = region->bytes();

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT        // $1900
		IGS27_CRYPT2_ALT        // $1b230
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= crzybug2_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}
