% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PB_refdist.R
\name{pb-refdist}
\alias{pb-refdist}
\alias{PBrefdist}
\alias{PBrefdist.merMod}
\alias{PBrefdist.lm}
\alias{PBrefdist.gls}
\title{Calculate reference distribution using parametric bootstrap}
\usage{
PBrefdist(
  largeModel,
  smallModel,
  nsim = 1000,
  seed = NULL,
  cl = NULL,
  details = 0
)

\method{PBrefdist}{lm}(
  largeModel,
  smallModel,
  nsim = 1000,
  seed = NULL,
  cl = NULL,
  details = 0
)

\method{PBrefdist}{merMod}(
  largeModel,
  smallModel,
  nsim = 1000,
  seed = NULL,
  cl = NULL,
  details = 0
)

\method{PBrefdist}{gls}(
  largeModel,
  smallModel,
  nsim = 1000,
  seed = NULL,
  cl = NULL,
  details = 0
)
}
\arguments{
\item{largeModel}{A linear mixed effects model as fitted with the
\code{lmer()} function in the \pkg{lme4} package. This model muse be
larger than \code{smallModel} (see below).}

\item{smallModel}{A linear mixed effects model as fitted with the
\code{lmer()} function in the \pkg{lme4} package. This model muse be
smaller than \code{largeModel} (see above).}

\item{nsim}{The number of simulations to form the reference distribution.}

\item{seed}{Seed for the random number generation.}

\item{cl}{Used for controlling parallel computations. See sections
'details' and 'examples' below.}

\item{details}{The amount of output produced. Mainly relevant for debugging
purposes.}
}
\value{
A numeric vector
}
\description{
Calculate reference distribution of likelihood ratio statistic
in mixed effects models using parametric bootstrap
}
\details{
The model \code{object} must be fitted with maximum likelihood
(i.e. with \code{REML=FALSE}). If the object is fitted with restricted
maximum likelihood (i.e. with \code{REML=TRUE}) then the model is
refitted with \code{REML=FALSE} before the p-values are calculated. Put
differently, the user needs not worry about this issue.

\if{html}{\out{<div class="sourceCode">}}\preformatted{The argument 'cl' (originally short for 'cluster') is used for
controlling parallel computations. 'cl' can be NULL (default),
positive integer or a list of clusters.
}\if{html}{\out{</div>}}

Special care must be taken
on Windows platforms (described below) but the general picture
is this:

\if{html}{\out{<div class="sourceCode">}}\preformatted{The recommended way of controlling cl is to specify the
component \code{pbcl} in options() with
e.g. \code{options("pbcl"=4)}.

If cl is NULL, the function will look at if the pbcl has been set
in the options list with \code{getOption("pbcl")}

If cl=N then N cores will be used in the computations. If cl is
NULL then the function will look for
}\if{html}{\out{</div>}}
}
\examples{

data(beets)
head(beets)
beet0 <- lmer(sugpct ~ block + sow + harvest + (1|block:harvest), data=beets, REML=FALSE)
beet_no.harv <- update(beet0, . ~ . -harvest)
rd <- PBrefdist(beet0, beet_no.harv, nsim=20, cl=1)
rd
\dontrun{
## Note: Many more simulations must be made in practice.

# Computations can be made in parallel using several processors:

# 1: On OSs that fork processes (that is, not on windows):
# --------------------------------------------------------

if (Sys.info()["sysname"] != "Windows"){
  N <- 2 ## Or N <- parallel::detectCores()

# N cores used in all calls to function in a session
  options("mc.cores"=N)
  rd <- PBrefdist(beet0, beet_no.harv, nsim=20)

# N cores used just in one specific call (when cl is set,
# options("mc.cores") is ignored):
  rd <- PBrefdist(beet0, beet_no.harv, nsim=20, cl=N)
}

# In fact, on Windows, the approach above also work but only when setting the
# number of cores to 1 (so there is to parallel computing)

# In all calls:
# options("mc.cores"=1)
# rd <- PBrefdist(beet0, beet_no.harv, nsim=20)
# Just once
# rd <- PBrefdist(beet0, beet_no.harv, nsim=20, cl=1)

# 2. On all platforms (also on Windows) one can do
# ------------------------------------------------
library(parallel)
N <- 2 ## Or N  <- detectCores()
clus <- makeCluster(rep("localhost", N))

# In all calls in a session
options("pb.cl"=clus)
rd <- PBrefdist(beet0, beet_no.harv, nsim=20)

# Just once:
rd <- PBrefdist(beet0, beet_no.harv, nsim=20, cl=clus)
stopCluster(clus)
}
}
\references{
Ulrich Halekoh, Søren Højsgaard (2014)., A Kenward-Roger
Approximation and Parametric Bootstrap Methods for Tests in Linear Mixed
Models - The R Package pbkrtest., Journal of Statistical Software,
58(10), 1-30., \url{https://www.jstatsoft.org/v59/i09/}
}
\seealso{
\code{\link{PBmodcomp}}, \code{\link{KRmodcomp}}
}
\author{
Søren Højsgaard \email{sorenh@math.aau.dk}
}
\concept{model_comparison}
\keyword{inference}
\keyword{models}
