/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.schema;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import java.lang.management.ManagementFactory;
import java.net.UnknownHostException;
import java.time.Duration;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BiConsumer;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import org.apache.cassandra.concurrent.ExecutorPlus;
import org.apache.cassandra.concurrent.FutureTask;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.exceptions.RequestFailureReason;
import org.apache.cassandra.gms.ApplicationState;
import org.apache.cassandra.gms.EndpointState;
import org.apache.cassandra.gms.Gossiper;
import org.apache.cassandra.gms.VersionedValue;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.net.Message;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.net.NoPayload;
import org.apache.cassandra.net.RequestCallback;
import org.apache.cassandra.net.Verb;
import org.apache.cassandra.schema.DistributedSchema;
import org.apache.cassandra.schema.Schema;
import org.apache.cassandra.schema.SchemaConstants;
import org.apache.cassandra.schema.SchemaDiagnostics;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.NoSpamLogger;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.Simulate;
import org.apache.cassandra.utils.concurrent.Future;
import org.apache.cassandra.utils.concurrent.ImmediateFuture;
import org.apache.cassandra.utils.concurrent.WaitQueue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Simulate(with={Simulate.With.MONITORS})
public class MigrationCoordinator {
    private static final Logger logger = LoggerFactory.getLogger(MigrationCoordinator.class);
    private static final NoSpamLogger noSpamLogger = NoSpamLogger.getLogger(logger, 1L, TimeUnit.MINUTES);
    private static final Future<Void> FINISHED_FUTURE = ImmediateFuture.success(null);
    private static final long PULL_BACKOFF_INTERVAL_MS = 1000L;
    private static LongSupplier getUptimeFn = () -> ManagementFactory.getRuntimeMXBean().getUptime();
    private static final int MIGRATION_DELAY_IN_MS = CassandraRelevantProperties.MIGRATION_DELAY.getInt();
    public static final int MAX_OUTSTANDING_VERSION_REQUESTS = 3;
    private static final Set<UUID> IGNORED_VERSIONS = MigrationCoordinator.getIgnoredVersions();
    private final Map<UUID, VersionInfo> versionInfo = new HashMap<UUID, VersionInfo>();
    private final Map<InetAddressAndPort, UUID> endpointVersions = new HashMap<InetAddressAndPort, UUID>();
    private final Set<InetAddressAndPort> ignoredEndpoints = MigrationCoordinator.getIgnoredEndpoints();
    private final ScheduledExecutorService periodicCheckExecutor;
    private final MessagingService messagingService;
    private final AtomicReference<ScheduledFuture<?>> periodicPullTask = new AtomicReference();
    private final int maxOutstandingVersionRequests;
    private final Gossiper gossiper;
    private final Supplier<UUID> schemaVersion;
    private final BiConsumer<InetAddressAndPort, Collection<Mutation>> schemaUpdateCallback;
    private final Set<InetAddressAndPort> lastPullFailures = new HashSet<InetAddressAndPort>();
    final ExecutorPlus executor;

    @VisibleForTesting
    public static void setUptimeFn(LongSupplier supplier) {
        getUptimeFn = supplier;
    }

    private static ImmutableSet<UUID> getIgnoredVersions() {
        String s2 = CassandraRelevantProperties.IGNORED_SCHEMA_CHECK_VERSIONS.getString();
        if (s2 == null || s2.isEmpty()) {
            return ImmutableSet.of();
        }
        ImmutableSet.Builder versions = ImmutableSet.builder();
        for (String version : s2.split(",")) {
            versions.add(UUID.fromString(version));
        }
        return versions.build();
    }

    private static Set<InetAddressAndPort> getIgnoredEndpoints() {
        HashSet<InetAddressAndPort> endpoints = new HashSet<InetAddressAndPort>();
        String s2 = CassandraRelevantProperties.IGNORED_SCHEMA_CHECK_ENDPOINTS.getString();
        if (s2 == null || s2.isEmpty()) {
            return endpoints;
        }
        for (String endpoint : s2.split(",")) {
            try {
                endpoints.add(InetAddressAndPort.getByName(endpoint));
            }
            catch (UnknownHostException e) {
                throw new RuntimeException(e);
            }
        }
        return endpoints;
    }

    MigrationCoordinator(MessagingService messagingService, ExecutorPlus executor, ScheduledExecutorService periodicCheckExecutor, int maxOutstandingVersionRequests, Gossiper gossiper, Supplier<UUID> schemaVersionSupplier, BiConsumer<InetAddressAndPort, Collection<Mutation>> schemaUpdateCallback) {
        this.messagingService = messagingService;
        this.executor = executor;
        this.periodicCheckExecutor = periodicCheckExecutor;
        this.maxOutstandingVersionRequests = maxOutstandingVersionRequests;
        this.gossiper = gossiper;
        this.schemaVersion = schemaVersionSupplier;
        this.schemaUpdateCallback = schemaUpdateCallback;
    }

    void start() {
        long interval = CassandraRelevantProperties.SCHEMA_PULL_INTERVAL_MS.getLong();
        logger.info("Starting migration coordinator and scheduling pulling schema versions every {}", (Object)Duration.ofMillis(interval));
        this.announce(this.schemaVersion.get());
        this.periodicPullTask.updateAndGet(curTask -> curTask == null ? this.periodicCheckExecutor.scheduleWithFixedDelay(this::pullUnreceivedSchemaVersions, interval, interval, TimeUnit.MILLISECONDS) : curTask);
    }

    private synchronized void pullUnreceivedSchemaVersions() {
        logger.debug("Pulling unreceived schema versions...");
        for (VersionInfo info : this.versionInfo.values()) {
            if (info.wasReceived() || info.outstandingRequests.size() > 0) {
                logger.trace("Skipping pull of schema {} because it has been already recevied, or it is being received ({})", (Object)info.version, (Object)info);
                continue;
            }
            this.maybePullSchema(info);
        }
    }

    private synchronized Future<Void> maybePullSchema(VersionInfo info) {
        if (info.endpoints.isEmpty() || info.wasReceived() || !this.shouldPullSchema(info.version)) {
            logger.trace("Not pulling schema {} because it was received, there is no endpoint to provide it, or we should not pull it ({})", (Object)info.version, (Object)info);
            return FINISHED_FUTURE;
        }
        if (info.outstandingRequests.size() >= this.maxOutstandingVersionRequests) {
            logger.trace("Not pulling schema {} because the number of outstanding requests has been exceeded ({} >= {})", new Object[]{info.version, info.outstandingRequests.size(), this.maxOutstandingVersionRequests});
            return FINISHED_FUTURE;
        }
        int isize = info.requestQueue.size();
        for (int i = 0; i < isize; ++i) {
            InetAddressAndPort endpoint = info.requestQueue.remove();
            if (!info.endpoints.contains(endpoint)) {
                logger.trace("Skipping request of schema {} from {} because the endpoint does not have that schema any longer", (Object)info.version, (Object)endpoint);
                continue;
            }
            if (this.shouldPullFromEndpoint(endpoint) && info.outstandingRequests.add(endpoint)) {
                return this.scheduleSchemaPull(endpoint, info);
            }
            logger.trace("Could not pull schema {} from {} - the request will be added back to the queue", (Object)info.version, (Object)endpoint);
            info.requestQueue.offer(endpoint);
        }
        return FINISHED_FUTURE;
    }

    synchronized Map<UUID, Set<InetAddressAndPort>> outstandingVersions() {
        HashMap<UUID, Set<InetAddressAndPort>> map = new HashMap<UUID, Set<InetAddressAndPort>>();
        for (VersionInfo info : this.versionInfo.values()) {
            if (info.wasReceived()) continue;
            map.put(info.version, ImmutableSet.copyOf(info.endpoints));
        }
        return map;
    }

    @VisibleForTesting
    VersionInfo getVersionInfoUnsafe(UUID version) {
        return this.versionInfo.get(version);
    }

    private boolean shouldPullSchema(UUID version) {
        UUID localSchemaVersion = this.schemaVersion.get();
        if (localSchemaVersion == null) {
            logger.debug("Not pulling schema {} because the local schama version is not known yet", (Object)version);
            return false;
        }
        if (localSchemaVersion.equals(version)) {
            logger.debug("Not pulling schema {} because it is the same as the local schema", (Object)version);
            return false;
        }
        return true;
    }

    private boolean shouldPullFromEndpoint(InetAddressAndPort endpoint) {
        if (endpoint.equals(FBUtilities.getBroadcastAddressAndPort())) {
            logger.trace("Not pulling schema from local endpoint");
            return false;
        }
        EndpointState state = this.gossiper.getEndpointStateForEndpoint(endpoint);
        if (state == null) {
            logger.trace("Not pulling schema from endpoint {} because its state is unknown", (Object)endpoint);
            return false;
        }
        VersionedValue releaseVersionValue = state.getApplicationState(ApplicationState.RELEASE_VERSION);
        if (releaseVersionValue == null) {
            return false;
        }
        String releaseVersion = releaseVersionValue.value;
        String ourMajorVersion = FBUtilities.getReleaseVersionMajor();
        if (!releaseVersion.startsWith(ourMajorVersion)) {
            logger.debug("Not pulling schema from {} because release version in Gossip is not major version {}, it is {}", new Object[]{endpoint, ourMajorVersion, releaseVersion});
            return false;
        }
        if (!this.messagingService.versions.knows(endpoint)) {
            logger.debug("Not pulling schema from {} because their messaging version is unknown", (Object)endpoint);
            return false;
        }
        if (this.messagingService.versions.getRaw(endpoint) != MessagingService.current_version) {
            logger.debug("Not pulling schema from {} because their schema format is incompatible", (Object)endpoint);
            return false;
        }
        if (this.gossiper.isGossipOnlyMember(endpoint)) {
            logger.debug("Not pulling schema from {} because it's a gossip only member", (Object)endpoint);
            return false;
        }
        return true;
    }

    private boolean shouldPullImmediately(InetAddressAndPort endpoint, UUID version) {
        UUID localSchemaVersion = this.schemaVersion.get();
        if (SchemaConstants.emptyVersion.equals(localSchemaVersion) || getUptimeFn.getAsLong() < (long)MIGRATION_DELAY_IN_MS) {
            logger.debug("Immediately submitting migration task for {}, schema versions: local={}, remote={}", new Object[]{endpoint, DistributedSchema.schemaVersionToString(localSchemaVersion), DistributedSchema.schemaVersionToString(version)});
            return true;
        }
        return false;
    }

    private synchronized boolean shouldApplySchemaFor(VersionInfo info) {
        if (info.wasReceived()) {
            return false;
        }
        return !Objects.equals(this.schemaVersion.get(), info.version);
    }

    synchronized Future<Void> reportEndpointVersion(InetAddressAndPort endpoint, UUID version) {
        logger.debug("Reported schema {} at endpoint {}", (Object)version, (Object)endpoint);
        if (this.ignoredEndpoints.contains(endpoint) || IGNORED_VERSIONS.contains(version)) {
            this.endpointVersions.remove(endpoint);
            this.removeEndpointFromVersion(endpoint, null);
            logger.debug("Discarding endpoint {} or schema {} because either endpoint or schema version were marked as ignored", (Object)endpoint, (Object)version);
            return FINISHED_FUTURE;
        }
        UUID current = this.endpointVersions.put(endpoint, version);
        if (current != null && current.equals(version)) {
            logger.trace("Skipping report of schema {} from {} because we already know that", (Object)version, (Object)endpoint);
            return FINISHED_FUTURE;
        }
        VersionInfo info = this.versionInfo.computeIfAbsent(version, VersionInfo::new);
        if (Objects.equals(this.schemaVersion.get(), version)) {
            info.markReceived();
            logger.trace("Schema {} from {} has been marked as recevied because it is equal the local schema", (Object)version, (Object)endpoint);
        } else {
            info.requestQueue.addFirst(endpoint);
        }
        info.endpoints.add(endpoint);
        logger.trace("Added endpoint {} to schema {}: {}", new Object[]{endpoint, info.version, info});
        this.removeEndpointFromVersion(endpoint, current);
        return this.maybePullSchema(info);
    }

    private synchronized void removeEndpointFromVersion(InetAddressAndPort endpoint, UUID version) {
        if (version == null) {
            return;
        }
        VersionInfo info = this.versionInfo.get(version);
        if (info == null) {
            return;
        }
        info.endpoints.remove(endpoint);
        logger.trace("Removed endpoint {} from schema {}: {}", new Object[]{endpoint, version, info});
        if (info.endpoints.isEmpty()) {
            info.waitQueue.signalAll();
            this.versionInfo.remove(version);
            logger.trace("Removed schema info: {}", (Object)info);
        }
    }

    private void clearVersionsInfo() {
        Iterator<Map.Entry<UUID, VersionInfo>> it = this.versionInfo.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<UUID, VersionInfo> entry = it.next();
            it.remove();
            entry.getValue().waitQueue.signal();
        }
    }

    private void reportCurrentSchemaVersionOnEndpoint(InetAddressAndPort endpoint) {
        if (FBUtilities.getBroadcastAddressAndPort().equals(endpoint)) {
            this.reportEndpointVersion(endpoint, this.schemaVersion.get());
        } else {
            UUID v;
            EndpointState state = this.gossiper.getEndpointStateForEndpoint(endpoint);
            if (state != null && (v = state.getSchemaVersion()) != null) {
                this.reportEndpointVersion(endpoint, v);
            }
        }
    }

    synchronized void reset() {
        logger.info("Resetting migration coordinator...");
        this.endpointVersions.clear();
        this.clearVersionsInfo();
        this.gossiper.getLiveMembers().forEach(this::reportCurrentSchemaVersionOnEndpoint);
    }

    synchronized void removeAndIgnoreEndpoint(InetAddressAndPort endpoint) {
        logger.debug("Removing and ignoring endpoint {}", (Object)endpoint);
        Preconditions.checkArgument(endpoint != null);
        this.ignoredEndpoints.add(endpoint);
        ImmutableSet<UUID> versions = ImmutableSet.copyOf(this.versionInfo.keySet());
        for (UUID version : versions) {
            this.removeEndpointFromVersion(endpoint, version);
        }
    }

    private Future<Void> scheduleSchemaPull(InetAddressAndPort endpoint, VersionInfo info) {
        FutureTask<Void> task = new FutureTask<Void>(() -> this.pullSchema(endpoint, new Callback(endpoint, info)));
        if (this.shouldPullImmediately(endpoint, info.version)) {
            if (this.lastPullFailures.contains(endpoint)) {
                logger.debug("Pulling {} immediately from {} with backoff interval = {}", new Object[]{info, endpoint, 1000L});
                ScheduledExecutors.nonPeriodicTasks.schedule(() -> this.submitToMigrationIfNotShutdown(task), 1000L, TimeUnit.MILLISECONDS);
            } else {
                logger.debug("Pulling {} immediately from {}", (Object)info, (Object)endpoint);
                this.submitToMigrationIfNotShutdown(task);
            }
        } else {
            logger.debug("Postponing pull of {} from {} for {}ms", new Object[]{info, endpoint, MIGRATION_DELAY_IN_MS});
            ScheduledExecutors.nonPeriodicTasks.schedule(() -> this.submitToMigrationIfNotShutdown(task), (long)MIGRATION_DELAY_IN_MS, TimeUnit.MILLISECONDS);
        }
        return task;
    }

    void announce(UUID schemaVersion) {
        if (this.gossiper.isEnabled()) {
            this.gossiper.addLocalApplicationState(ApplicationState.SCHEMA, StorageService.instance.valueFactory.schema(schemaVersion));
        }
        SchemaDiagnostics.versionAnnounced(Schema.instance);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Future<?> submitToMigrationIfNotShutdown(Runnable task) {
        boolean skipped = false;
        try {
            if (this.executor.isShutdown() || this.executor.isTerminated()) {
                skipped = true;
                ImmediateFuture<Object> immediateFuture = ImmediateFuture.success(null);
                return immediateFuture;
            }
            java.util.concurrent.Future future = this.executor.submit(task);
            return future;
        }
        catch (RejectedExecutionException ex) {
            skipped = true;
            ImmediateFuture<Object> immediateFuture = ImmediateFuture.success(null);
            return immediateFuture;
        }
        finally {
            if (skipped) {
                logger.info("Skipped scheduled pulling schema from other nodes: the MIGRATION executor service has been shutdown.");
            }
        }
    }

    private void pullSchema(InetAddressAndPort endpoint, RequestCallback<Collection<Mutation>> callback) {
        if (!this.gossiper.isAlive(endpoint)) {
            noSpamLogger.warn("Can't send schema pull request: node {} is down.", endpoint);
            callback.onFailure(endpoint, RequestFailureReason.UNKNOWN);
            return;
        }
        if (!this.shouldPullFromEndpoint(endpoint)) {
            logger.info("Skipped sending a migration request: node {} has a higher major version now.", (Object)endpoint);
            callback.onFailure(endpoint, RequestFailureReason.UNKNOWN);
            return;
        }
        logger.debug("Requesting schema from {}", (Object)endpoint);
        this.sendMigrationMessage(endpoint, callback);
    }

    private void sendMigrationMessage(InetAddressAndPort endpoint, RequestCallback<Collection<Mutation>> callback) {
        Message<NoPayload> message = Message.out(Verb.SCHEMA_PULL_REQ, NoPayload.noPayload);
        logger.info("Sending schema pull request to {}", (Object)endpoint);
        this.messagingService.sendWithCallback(message, endpoint, callback);
    }

    private synchronized Future<Void> pullComplete(InetAddressAndPort endpoint, VersionInfo info, boolean wasSuccessful) {
        if (wasSuccessful) {
            info.markReceived();
            this.lastPullFailures.remove(endpoint);
        } else {
            this.lastPullFailures.add(endpoint);
        }
        info.outstandingRequests.remove(endpoint);
        info.requestQueue.add(endpoint);
        return this.maybePullSchema(info);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    boolean awaitSchemaRequests(long waitMillis) {
        if (!FBUtilities.getBroadcastAddressAndPort().equals(InetAddressAndPort.getLoopbackAddress())) {
            Gossiper.waitToSettle();
        }
        if (this.versionInfo.isEmpty()) {
            logger.debug("Nothing in versionInfo - so no schemas to wait for");
        }
        ArrayList<WaitQueue.Signal> signalList = null;
        try {
            MigrationCoordinator migrationCoordinator = this;
            synchronized (migrationCoordinator) {
                signalList = new ArrayList<WaitQueue.Signal>(this.versionInfo.size());
                for (VersionInfo version : this.versionInfo.values()) {
                    if (version.wasReceived()) continue;
                    signalList.add(version.register());
                }
                if (signalList.isEmpty()) {
                    boolean bl = true;
                    return bl;
                }
            }
            long deadline = Clock.Global.nanoTime() + TimeUnit.MILLISECONDS.toNanos(waitMillis);
            boolean bl = signalList.stream().allMatch(signal -> signal.awaitUntilUninterruptibly(deadline));
            return bl;
        }
        finally {
            if (signalList != null) {
                signalList.forEach(WaitQueue.Signal::cancel);
            }
        }
    }

    Pair<Set<InetAddressAndPort>, Set<InetAddressAndPort>> pushSchemaMutations(Collection<Mutation> schemaMutations) {
        logger.debug("Pushing schema mutations: {}", schemaMutations);
        HashSet<InetAddressAndPort> schemaDestinationEndpoints = new HashSet<InetAddressAndPort>();
        HashSet<InetAddressAndPort> schemaEndpointsIgnored = new HashSet<InetAddressAndPort>();
        Message<Collection<Mutation>> message = Message.out(Verb.SCHEMA_PUSH_REQ, schemaMutations);
        for (InetAddressAndPort endpoint : this.gossiper.getLiveMembers()) {
            if (this.shouldPushSchemaTo(endpoint)) {
                logger.debug("Pushing schema mutations to {}: {}", (Object)endpoint, schemaMutations);
                this.messagingService.send(message, endpoint);
                schemaDestinationEndpoints.add(endpoint);
                continue;
            }
            schemaEndpointsIgnored.add(endpoint);
        }
        return Pair.create(schemaDestinationEndpoints, schemaEndpointsIgnored);
    }

    private boolean shouldPushSchemaTo(InetAddressAndPort endpoint) {
        return !endpoint.equals(FBUtilities.getBroadcastAddressAndPort()) && this.messagingService.versions.knows(endpoint) && this.messagingService.versions.getRaw(endpoint) == MessagingService.current_version;
    }

    private class Callback
    implements RequestCallback<Collection<Mutation>> {
        final InetAddressAndPort endpoint;
        final VersionInfo info;

        public Callback(InetAddressAndPort endpoint, VersionInfo info) {
            this.endpoint = endpoint;
            this.info = info;
        }

        @Override
        public void onFailure(InetAddressAndPort from, RequestFailureReason failureReason) {
            this.fail();
        }

        Future<Void> fail() {
            return MigrationCoordinator.this.pullComplete(this.endpoint, this.info, false);
        }

        @Override
        public void onResponse(Message<Collection<Mutation>> message) {
            this.response((Collection)message.payload);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        Future<Void> response(Collection<Mutation> mutations) {
            VersionInfo versionInfo = this.info;
            synchronized (versionInfo) {
                if (MigrationCoordinator.this.shouldApplySchemaFor(this.info)) {
                    try {
                        MigrationCoordinator.this.schemaUpdateCallback.accept(this.endpoint, mutations);
                    }
                    catch (Exception e) {
                        logger.error(String.format("Unable to merge schema from %s", this.endpoint), (Throwable)e);
                        return this.fail();
                    }
                }
                return MigrationCoordinator.this.pullComplete(this.endpoint, this.info, true);
            }
        }

        public boolean isLatencyForSnitch() {
            return false;
        }
    }

    static class VersionInfo {
        final UUID version;
        final Set<InetAddressAndPort> endpoints = Sets.newConcurrentHashSet();
        final Set<InetAddressAndPort> outstandingRequests = Sets.newConcurrentHashSet();
        final Deque<InetAddressAndPort> requestQueue = new ArrayDeque<InetAddressAndPort>();
        private final WaitQueue waitQueue = WaitQueue.newWaitQueue();
        volatile boolean receivedSchema;

        VersionInfo(UUID version) {
            this.version = version;
        }

        WaitQueue.Signal register() {
            return this.waitQueue.register();
        }

        void markReceived() {
            if (this.receivedSchema) {
                return;
            }
            this.receivedSchema = true;
            this.waitQueue.signalAll();
        }

        boolean wasReceived() {
            return this.receivedSchema;
        }

        public String toString() {
            return "VersionInfo{version=" + this.version + ", outstandingRequests=" + this.outstandingRequests + ", requestQueue=" + this.requestQueue + ", waitQueue.waiting=" + this.waitQueue.getWaiting() + ", receivedSchema=" + this.receivedSchema + "}";
        }
    }
}

