/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.tools;

import com.google.common.collect.ImmutableSet;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.function.ToLongFunction;
import java.util.stream.Collectors;
import org.apache.cassandra.config.DataStorageSpec;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.LivenessInfo;
import org.apache.cassandra.db.rows.Cell;
import org.apache.cassandra.db.rows.ColumnData;
import org.apache.cassandra.db.rows.ComplexColumnData;
import org.apache.cassandra.db.rows.RangeTombstoneMarker;
import org.apache.cassandra.db.rows.Row;
import org.apache.cassandra.db.rows.Unfiltered;
import org.apache.cassandra.db.rows.UnfilteredRowIterator;
import org.apache.cassandra.dht.Bounds;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.ISSTableScanner;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.sstable.format.big.BigFormat;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.schema.TableId;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.schema.TableMetadataRef;
import org.apache.cassandra.tools.Util;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.EstimatedHistogram;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Pair;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

public class SSTablePartitions {
    private static final String KEY_OPTION = "k";
    private static final String EXCLUDE_KEY_OPTION = "x";
    private static final String RECURSIVE_OPTION = "r";
    private static final String SNAPSHOTS_OPTION = "s";
    private static final String BACKUPS_OPTION = "b";
    private static final String PARTITIONS_ONLY_OPTION = "y";
    private static final String SIZE_THRESHOLD_OPTION = "t";
    private static final String TOMBSTONE_THRESHOLD_OPTION = "o";
    private static final String CELL_THRESHOLD_OPTION = "c";
    private static final String ROW_THRESHOLD_OPTION = "w";
    private static final String CSV_OPTION = "m";
    private static final String CURRENT_TIMESTAMP_OPTION = "u";
    private static final Options options = new Options();
    private static final TableId EMPTY_TABLE_ID = TableId.fromUUID(new UUID(0L, 0L));

    public static void main(String[] args) throws ConfigurationException, IOException {
        CommandLine cmd;
        PosixParser parser = new PosixParser();
        try {
            cmd = parser.parse(options, args);
        }
        catch (ParseException e) {
            System.err.println(e.getMessage());
            SSTablePartitions.printUsage();
            System.exit(1);
            return;
        }
        if (cmd.getArgs().length == 0) {
            System.err.println("You must supply at least one sstable or directory");
            SSTablePartitions.printUsage();
            System.exit(1);
        }
        int ec = SSTablePartitions.processArguments(cmd);
        System.exit(ec);
    }

    private static void printUsage() {
        String usage = String.format("sstablepartitions <options> <sstable files or directories>%n", new Object[0]);
        String header = "Print partition statistics of one or more sstables.";
        new HelpFormatter().printHelp(usage, header, options, "");
    }

    private static int processArguments(CommandLine cmd) throws IOException {
        String[] keys = cmd.getOptionValues(KEY_OPTION);
        Set<String> excludes = cmd.getOptionValues(EXCLUDE_KEY_OPTION) == null ? Collections.emptySet() : ImmutableSet.copyOf(cmd.getOptionValues(EXCLUDE_KEY_OPTION));
        boolean scanRecursive = cmd.hasOption(RECURSIVE_OPTION);
        boolean withSnapshots = cmd.hasOption(SNAPSHOTS_OPTION);
        boolean withBackups = cmd.hasOption(BACKUPS_OPTION);
        boolean csv = cmd.hasOption(CSV_OPTION);
        boolean partitionsOnly = cmd.hasOption(PARTITIONS_ONLY_OPTION);
        long sizeThreshold = Long.MAX_VALUE;
        int cellCountThreshold = Integer.MAX_VALUE;
        int rowCountThreshold = Integer.MAX_VALUE;
        int tombstoneCountThreshold = Integer.MAX_VALUE;
        long currentTime = Clock.Global.currentTimeMillis() / 1000L;
        try {
            if (cmd.hasOption(SIZE_THRESHOLD_OPTION)) {
                String threshold = cmd.getOptionValue(SIZE_THRESHOLD_OPTION);
                long l = sizeThreshold = NumberUtils.isParsable((String)threshold) ? Long.parseLong(threshold) : new DataStorageSpec.LongBytesBound(threshold).toBytes();
            }
            if (cmd.hasOption(CELL_THRESHOLD_OPTION)) {
                cellCountThreshold = Integer.parseInt(cmd.getOptionValue(CELL_THRESHOLD_OPTION));
            }
            if (cmd.hasOption(ROW_THRESHOLD_OPTION)) {
                rowCountThreshold = Integer.parseInt(cmd.getOptionValue(ROW_THRESHOLD_OPTION));
            }
            if (cmd.hasOption(TOMBSTONE_THRESHOLD_OPTION)) {
                tombstoneCountThreshold = Integer.parseInt(cmd.getOptionValue(TOMBSTONE_THRESHOLD_OPTION));
            }
            if (cmd.hasOption(CURRENT_TIMESTAMP_OPTION)) {
                currentTime = Integer.parseInt(cmd.getOptionValue(CURRENT_TIMESTAMP_OPTION));
            }
        }
        catch (NumberFormatException e) {
            System.err.printf("Invalid threshold argument: %s%n", e.getMessage());
            return 1;
        }
        if (sizeThreshold < 0L || cellCountThreshold < 0 || tombstoneCountThreshold < 0 || currentTime < 0L) {
            System.err.println("Negative values are not allowed");
            return 1;
        }
        ArrayList<File> directories = new ArrayList<File>();
        ArrayList<ExtendedDescriptor> descriptors = new ArrayList<ExtendedDescriptor>();
        if (!SSTablePartitions.argumentsToFiles(cmd.getArgs(), descriptors, directories)) {
            return 1;
        }
        for (File directory : directories) {
            SSTablePartitions.processDirectory(scanRecursive, withSnapshots, withBackups, directory, descriptors);
        }
        if (csv) {
            System.out.println("key,keyBinary,live,offset,size,rowCount,cellCount,tombstoneCount,rowTombstoneCount,rangeTombstoneCount,complexTombstoneCount,cellTombstoneCount,rowTtlExpired,cellTtlExpired,directory,keyspace,table,index,snapshot,backup,generation,format,version");
        }
        Collections.sort(descriptors);
        for (ExtendedDescriptor desc : descriptors) {
            SSTablePartitions.processSSTable(keys, excludes, desc, sizeThreshold, cellCountThreshold, rowCountThreshold, tombstoneCountThreshold, partitionsOnly, csv, currentTime);
        }
        return 0;
    }

    private static void processDirectory(boolean scanRecursive, boolean withSnapshots, boolean withBackups, File dir, List<ExtendedDescriptor> descriptors) {
        File[] files = dir.tryList();
        if (files == null) {
            return;
        }
        for (File file : files) {
            if (file.isFile()) {
                try {
                    if (Descriptor.componentFromFile(file) != BigFormat.Components.DATA) continue;
                    ExtendedDescriptor desc = ExtendedDescriptor.guessFromFile(file);
                    if (desc.snapshot != null && !withSnapshots || desc.backup != null && !withBackups) continue;
                    descriptors.add(desc);
                }
                catch (IllegalArgumentException illegalArgumentException) {
                    // empty catch block
                }
            }
            if (!scanRecursive || !file.isDirectory()) continue;
            SSTablePartitions.processDirectory(true, withSnapshots, withBackups, file, descriptors);
        }
    }

    private static boolean argumentsToFiles(String[] args, List<ExtendedDescriptor> descriptors, List<File> directories) {
        boolean err = false;
        for (String arg : args) {
            File file = new File(arg);
            if (!file.exists()) {
                System.err.printf("Argument '%s' does not resolve to a file or directory%n", arg);
                err = true;
            }
            if (!file.isReadable()) {
                System.err.printf("Argument '%s' is not a readable file or directory (check permissions)%n", arg);
                err = true;
                continue;
            }
            if (file.isFile()) {
                try {
                    descriptors.add(ExtendedDescriptor.guessFromFile(file));
                }
                catch (IllegalArgumentException e) {
                    System.err.printf("Argument '%s' is not an sstable%n", arg);
                    err = true;
                }
            }
            if (!file.isDirectory()) continue;
            directories.add(file);
        }
        return !err;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static void processSSTable(String[] keys, Set<String> excludedKeys, ExtendedDescriptor desc, long sizeThreshold, int cellCountThreshold, int rowCountThreshold, int tombstoneCountThreshold, boolean partitionsOnly, boolean csv, long currentTime) throws IOException {
        TableMetadata metadata = Util.metadataFromSSTable(desc.descriptor);
        SSTableReader sstable = SSTableReader.openNoValidation(null, desc.descriptor, TableMetadataRef.forOfflineTools(metadata));
        if (!csv) {
            System.out.printf("%nProcessing %s (%s uncompressed, %s on disk)%n", desc, SSTablePartitions.prettyPrintMemory(sstable.uncompressedLength()), SSTablePartitions.prettyPrintMemory(sstable.onDiskLength()));
        }
        ArrayList<PartitionStats> matches = new ArrayList<PartitionStats>();
        SSTableStats sstableStats = new SSTableStats();
        try (ISSTableScanner scanner = SSTablePartitions.buildScanner(sstable, metadata, keys, excludedKeys);){
            while (scanner.hasNext()) {
                long startOfPartition = scanner.getCurrentPosition();
                UnfilteredRowIterator partition = (UnfilteredRowIterator)scanner.next();
                try {
                    ByteBuffer key = partition.partitionKey().getKey();
                    boolean isExcluded = excludedKeys.contains(metadata.partitionKeyType.getString(key));
                    PartitionStats partitionStats = new PartitionStats(key, startOfPartition, partition.partitionLevelDeletion().isLive());
                    while (partition.hasNext()) {
                        Unfiltered unfiltered = (Unfiltered)partition.next();
                        if (partitionsOnly || isExcluded) continue;
                        partitionStats.addUnfiltered(desc, currentTime, unfiltered);
                    }
                    partitionStats.endOfPartition(scanner.getCurrentPosition());
                    if (isExcluded) continue;
                    sstableStats.addPartition(partitionStats);
                    if (partitionStats.size < sizeThreshold && partitionStats.rowCount < rowCountThreshold && partitionStats.cellCount < cellCountThreshold && partitionStats.tombstoneCount() < tombstoneCountThreshold) continue;
                    matches.add(partitionStats);
                    if (csv) {
                        partitionStats.printPartitionInfoCSV(metadata, desc);
                        continue;
                    }
                    partitionStats.printPartitionInfo(metadata, partitionsOnly);
                }
                finally {
                    if (partition == null) continue;
                    partition.close();
                }
            }
        }
        catch (RuntimeException e) {
            System.err.printf("Failure processing sstable %s: %s%n", desc.descriptor, e);
        }
        finally {
            sstable.selfRef().release();
        }
        if (!csv) {
            SSTablePartitions.printSummary(metadata, desc, sstableStats, matches, partitionsOnly);
        }
    }

    private static String prettyPrintMemory(long bytes) {
        return FBUtilities.prettyPrintMemory(bytes, " ");
    }

    private static ISSTableScanner buildScanner(SSTableReader sstable, TableMetadata metadata, String[] keys, Set<String> excludedKeys) {
        if (keys != null && keys.length > 0) {
            try {
                return sstable.getScanner(Arrays.stream(keys).filter(key -> !excludedKeys.contains(key)).map(metadata.partitionKeyType::fromString).map(k -> sstable.getPartitioner().decorateKey((ByteBuffer)k)).sorted().map(DecoratedKey::getToken).map(token -> new Bounds<Token.KeyBound>(token.minKeyBound(), token.maxKeyBound())).collect(Collectors.toList()).iterator());
            }
            catch (RuntimeException e) {
                System.err.printf("Cannot use one or more partition keys in %s for the partition key type ('%s') of the underlying table: %s%n", Arrays.toString(keys), metadata.partitionKeyType.asCQL3Type(), e);
            }
        }
        return sstable.getScanner();
    }

    private static void printSummary(TableMetadata metadata, ExtendedDescriptor desc, SSTableStats stats, List<PartitionStats> matches, boolean partitionsOnly) {
        String format;
        if (!matches.isEmpty()) {
            System.out.printf("Summary of %s:%n  File: %s%n  %d partitions match%n  Keys:", desc, desc.descriptor.fileFor(BigFormat.Components.DATA), matches.size());
            for (PartitionStats match : matches) {
                System.out.print(" " + SSTablePartitions.maybeEscapeKeyForSummary(metadata, match.key));
            }
            System.out.println();
        }
        if (partitionsOnly) {
            System.out.printf("         %20s%n", "Partition size");
            format = "  %-5s  %20s%n";
        } else {
            System.out.printf("         %20s %20s %20s %20s%n", "Partition size", "Row count", "Cell count", "Tombstone count");
            format = "  %-5s  %20s %20d %20d %20d%n";
        }
        SSTablePartitions.printPercentile(partitionsOnly, stats, format, "~p50", h2 -> h2.percentile(0.5));
        SSTablePartitions.printPercentile(partitionsOnly, stats, format, "~p75", h2 -> h2.percentile(0.75));
        SSTablePartitions.printPercentile(partitionsOnly, stats, format, "~p90", h2 -> h2.percentile(0.9));
        SSTablePartitions.printPercentile(partitionsOnly, stats, format, "~p95", h2 -> h2.percentile(0.95));
        SSTablePartitions.printPercentile(partitionsOnly, stats, format, "~p99", h2 -> h2.percentile(0.99));
        SSTablePartitions.printPercentile(partitionsOnly, stats, format, "~p999", h2 -> h2.percentile(0.999));
        if (partitionsOnly) {
            System.out.printf(format, "min", SSTablePartitions.prettyPrintMemory(stats.minSize));
            System.out.printf(format, "max", SSTablePartitions.prettyPrintMemory(stats.maxSize));
        } else {
            System.out.printf(format, "min", SSTablePartitions.prettyPrintMemory(stats.minSize), stats.minRowCount, stats.minCellCount, stats.minTombstoneCount);
            System.out.printf(format, "max", SSTablePartitions.prettyPrintMemory(stats.maxSize), stats.maxRowCount, stats.maxCellCount, stats.maxTombstoneCount);
        }
        System.out.printf("  count  %20d%n", stats.partitionSizeHistogram.count());
    }

    private static void printPercentile(boolean partitionsOnly, SSTableStats stats, String format, String header, ToLongFunction<EstimatedHistogram> value) {
        if (partitionsOnly) {
            System.out.printf(format, header, SSTablePartitions.prettyPrintMemory(value.applyAsLong(stats.partitionSizeHistogram)));
        } else {
            System.out.printf(format, header, SSTablePartitions.prettyPrintMemory(value.applyAsLong(stats.partitionSizeHistogram)), value.applyAsLong(stats.rowCountHistogram), value.applyAsLong(stats.cellCountHistogram), value.applyAsLong(stats.tombstoneCountHistogram));
        }
    }

    private static String maybeEscapeKeyForSummary(TableMetadata metadata, ByteBuffer key) {
        String s2 = metadata.partitionKeyType.getString(key);
        if (s2.indexOf(32) == -1) {
            return s2;
        }
        return "\"" + StringUtils.replace((String)s2, (String)"\"", (String)"\"\"") + "\"";
    }

    private static String notNull(String s2) {
        return s2 != null ? s2 : "";
    }

    private static TableId notNull(TableId s2) {
        return s2 != null ? s2 : EMPTY_TABLE_ID;
    }

    static {
        DatabaseDescriptor.clientInitialization();
        Option optKey = new Option(KEY_OPTION, "key", true, "Partition keys to include");
        optKey.setArgs(-2);
        options.addOption(optKey);
        Option excludeKey = new Option(EXCLUDE_KEY_OPTION, "exclude-key", true, "Excluded partition key(s) from partition detailed row/cell/tombstone information (irrelevant, if --partitions-only is given)");
        excludeKey.setArgs(-2);
        options.addOption(excludeKey);
        Option thresholdKey = new Option(SIZE_THRESHOLD_OPTION, "min-size", true, "partition size threshold, expressed as either the number of bytes or a size with unit of the form 10KiB, 20MiB, 30GiB, etc.");
        options.addOption(thresholdKey);
        Option tombstoneKey = new Option(TOMBSTONE_THRESHOLD_OPTION, "min-tombstones", true, "partition tombstone count threshold");
        options.addOption(tombstoneKey);
        Option cellKey = new Option(CELL_THRESHOLD_OPTION, "min-cells", true, "partition cell count threshold");
        options.addOption(cellKey);
        Option rowKey = new Option(ROW_THRESHOLD_OPTION, "min-rows", true, "partition row count threshold");
        options.addOption(rowKey);
        Option currentTimestampKey = new Option(CURRENT_TIMESTAMP_OPTION, "current-timestamp", true, "timestamp (seconds since epoch, unit time) for TTL expired calculation");
        options.addOption(currentTimestampKey);
        Option recursiveKey = new Option(RECURSIVE_OPTION, "recursive", false, "scan for sstables recursively");
        options.addOption(recursiveKey);
        Option snapshotsKey = new Option(SNAPSHOTS_OPTION, "snapshots", false, "include snapshots present in data directories (recursive scans)");
        options.addOption(snapshotsKey);
        Option backupsKey = new Option(BACKUPS_OPTION, "backups", false, "include backups present in data directories (recursive scans)");
        options.addOption(backupsKey);
        Option partitionsOnlyKey = new Option(PARTITIONS_ONLY_OPTION, "partitions-only", false, "Do not process per-partition detailed row/cell/tombstone information, only brief information");
        options.addOption(partitionsOnlyKey);
        Option csvKey = new Option(CSV_OPTION, "csv", false, "CSV output (machine readable)");
        options.addOption(csvKey);
    }

    static final class ExtendedDescriptor
    implements Comparable<ExtendedDescriptor> {
        final String keyspace;
        final String table;
        final String index;
        final String snapshot;
        final String backup;
        final TableId tableId;
        final Descriptor descriptor;

        ExtendedDescriptor(String keyspace, String table, TableId tableId, String index, String snapshot, String backup, Descriptor descriptor) {
            this.keyspace = keyspace;
            this.table = table;
            this.tableId = tableId;
            this.index = index;
            this.snapshot = snapshot;
            this.backup = backup;
            this.descriptor = descriptor;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            if (this.backup != null) {
                sb.append("Backup:").append(this.backup).append(' ');
            }
            if (this.snapshot != null) {
                sb.append("Snapshot:").append(this.snapshot).append(' ');
            }
            if (this.keyspace != null) {
                sb.append(this.keyspace).append('.');
            }
            if (this.table != null) {
                sb.append(this.table);
            }
            if (this.index != null) {
                sb.append('.').append(this.index);
            }
            if (this.tableId != null) {
                sb.append('-').append(this.tableId.toHexString());
            }
            return sb.append(" #").append(this.descriptor.id).append(" (").append(this.descriptor.version.format.name()).append('-').append(this.descriptor.version.version).append(')').toString();
        }

        static ExtendedDescriptor guessFromFile(File fArg) {
            Descriptor desc = Descriptor.fromFile(fArg);
            String snapshot = null;
            String backup = null;
            String index = null;
            File parent = fArg.parent();
            File grandparent = parent.parent();
            if (parent.name().length() > 1 && parent.name().startsWith(".") && parent.name().charAt(1) != '.') {
                index = parent.name().substring(1);
                parent = parent.parent();
                grandparent = parent.parent();
            }
            if (parent.name().equals("backups")) {
                backup = parent.name();
                parent = parent.parent();
                grandparent = parent.parent();
            }
            if (grandparent.name().equals("snapshots")) {
                snapshot = parent.name();
                parent = grandparent.parent();
                grandparent = parent.parent();
            }
            try {
                Pair<String, TableId> tableNameAndId = TableId.tableNameAndIdFromFilename(parent.name());
                if (tableNameAndId != null) {
                    return new ExtendedDescriptor(grandparent.name(), (String)tableNameAndId.left, (TableId)tableNameAndId.right, index, snapshot, backup, desc);
                }
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
            return new ExtendedDescriptor(null, null, null, index, snapshot, backup, desc);
        }

        @Override
        public int compareTo(ExtendedDescriptor o) {
            int c = this.descriptor.directory.toString().compareTo(o.descriptor.directory.toString());
            if (c != 0) {
                return c;
            }
            c = SSTablePartitions.notNull(this.keyspace).compareTo(SSTablePartitions.notNull(o.keyspace));
            if (c != 0) {
                return c;
            }
            c = SSTablePartitions.notNull(this.table).compareTo(SSTablePartitions.notNull(o.table));
            if (c != 0) {
                return c;
            }
            c = SSTablePartitions.notNull(this.tableId).toString().compareTo(SSTablePartitions.notNull(o.tableId).toString());
            if (c != 0) {
                return c;
            }
            c = SSTablePartitions.notNull(this.index).compareTo(SSTablePartitions.notNull(o.index));
            if (c != 0) {
                return c;
            }
            c = SSTablePartitions.notNull(this.snapshot).compareTo(SSTablePartitions.notNull(o.snapshot));
            if (c != 0) {
                return c;
            }
            c = SSTablePartitions.notNull(this.backup).compareTo(SSTablePartitions.notNull(o.backup));
            if (c != 0) {
                return c;
            }
            c = SSTablePartitions.notNull(this.descriptor.id.toString()).compareTo(SSTablePartitions.notNull(o.descriptor.id.toString()));
            if (c != 0) {
                return c;
            }
            return Integer.compare(System.identityHashCode(this), System.identityHashCode(o));
        }
    }

    static final class PartitionStats {
        final ByteBuffer key;
        final long offset;
        final boolean live;
        long size = -1L;
        int rowCount = 0;
        int cellCount = 0;
        int rowTombstoneCount = 0;
        int rangeTombstoneCount = 0;
        int complexTombstoneCount = 0;
        int cellTombstoneCount = 0;
        int rowTtlExpired = 0;
        int cellTtlExpired = 0;

        PartitionStats(ByteBuffer key, long offset, boolean live) {
            this.key = key;
            this.offset = offset;
            this.live = live;
        }

        void endOfPartition(long position) {
            this.size = position - this.offset;
        }

        int tombstoneCount() {
            return this.rowTombstoneCount + this.rangeTombstoneCount + this.complexTombstoneCount + this.cellTombstoneCount + this.rowTtlExpired + this.cellTtlExpired;
        }

        void addUnfiltered(ExtendedDescriptor desc, long currentTime, Unfiltered unfiltered) {
            if (unfiltered instanceof Row) {
                LivenessInfo liveInfo;
                Row row = (Row)unfiltered;
                ++this.rowCount;
                if (!row.deletion().isLive()) {
                    ++this.rowTombstoneCount;
                }
                if (!(liveInfo = row.primaryKeyLivenessInfo()).isEmpty() && liveInfo.isExpiring() && liveInfo.localExpirationTime() < currentTime) {
                    ++this.rowTtlExpired;
                }
                for (ColumnData cd2 : row) {
                    if (cd2.column().isSimple()) {
                        this.addCell((int)currentTime, liveInfo, (Cell)cd2);
                        continue;
                    }
                    ComplexColumnData complexData = (ComplexColumnData)cd2;
                    if (!complexData.complexDeletion().isLive()) {
                        ++this.complexTombstoneCount;
                    }
                    for (Cell<?> cell : complexData) {
                        this.addCell((int)currentTime, liveInfo, cell);
                    }
                }
            } else if (unfiltered instanceof RangeTombstoneMarker) {
                ++this.rangeTombstoneCount;
            } else {
                throw new UnsupportedOperationException("Unknown kind " + unfiltered.kind() + " in sstable " + desc.descriptor);
            }
        }

        private void addCell(int currentTime, LivenessInfo liveInfo, Cell<?> cell) {
            ++this.cellCount;
            if (cell.isTombstone()) {
                ++this.cellTombstoneCount;
            }
            if (cell.isExpiring() && (liveInfo.isEmpty() || cell.ttl() != liveInfo.ttl()) && !cell.isLive(currentTime)) {
                ++this.cellTtlExpired;
            }
        }

        void printPartitionInfo(TableMetadata metadata, boolean partitionsOnly) {
            String key = metadata.partitionKeyType.getString(this.key);
            if (partitionsOnly) {
                System.out.printf("  Partition: '%s' (%s) %s, size: %s%n", key, ByteBufferUtil.bytesToHex(this.key), this.live ? "live" : "not live", SSTablePartitions.prettyPrintMemory(this.size));
            } else {
                System.out.printf("  Partition: '%s' (%s) %s, size: %s, rows: %d, cells: %d, tombstones: %d (row:%d, range:%d, complex:%d, cell:%d, row-TTLd:%d, cell-TTLd:%d)%n", key, ByteBufferUtil.bytesToHex(this.key), this.live ? "live" : "not live", SSTablePartitions.prettyPrintMemory(this.size), this.rowCount, this.cellCount, this.tombstoneCount(), this.rowTombstoneCount, this.rangeTombstoneCount, this.complexTombstoneCount, this.cellTombstoneCount, this.rowTtlExpired, this.cellTtlExpired);
            }
        }

        void printPartitionInfoCSV(TableMetadata metadata, ExtendedDescriptor desc) {
            System.out.printf("\"%s\",%s,%s,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%s,%s,%s,%s,%s,%s,%s,%s,%s%n", SSTablePartitions.maybeEscapeKeyForSummary(metadata, this.key), ByteBufferUtil.bytesToHex(this.key), this.live ? "true" : "false", this.offset, this.size, this.rowCount, this.cellCount, this.tombstoneCount(), this.rowTombstoneCount, this.rangeTombstoneCount, this.complexTombstoneCount, this.cellTombstoneCount, this.rowTtlExpired, this.cellTtlExpired, desc.descriptor.fileFor(BigFormat.Components.DATA), SSTablePartitions.notNull(desc.keyspace), SSTablePartitions.notNull(desc.table), SSTablePartitions.notNull(desc.index), SSTablePartitions.notNull(desc.snapshot), SSTablePartitions.notNull(desc.backup), desc.descriptor.id, desc.descriptor.version.format.name(), desc.descriptor.version.version);
        }
    }

    static final class SSTableStats {
        EstimatedHistogram partitionSizeHistogram = new EstimatedHistogram(155, true);
        EstimatedHistogram rowCountHistogram = new EstimatedHistogram(118, true);
        EstimatedHistogram cellCountHistogram = new EstimatedHistogram(118, true);
        EstimatedHistogram tombstoneCountHistogram = new EstimatedHistogram(118, true);
        long minSize = 0L;
        long maxSize = 0L;
        int minRowCount = 0;
        int maxRowCount = 0;
        int minCellCount = 0;
        int maxCellCount = 0;
        int minTombstoneCount = 0;
        int maxTombstoneCount = 0;

        SSTableStats() {
        }

        void addPartition(PartitionStats stats) {
            this.partitionSizeHistogram.add(stats.size);
            this.rowCountHistogram.add(stats.rowCount);
            this.cellCountHistogram.add(stats.cellCount);
            this.tombstoneCountHistogram.add(stats.tombstoneCount());
            if (this.minSize == 0L || stats.size < this.minSize) {
                this.minSize = stats.size;
            }
            if (stats.size > this.maxSize) {
                this.maxSize = stats.size;
            }
            if (this.minRowCount == 0 || stats.rowCount < this.minRowCount) {
                this.minRowCount = stats.rowCount;
            }
            if (stats.rowCount > this.maxRowCount) {
                this.maxRowCount = stats.rowCount;
            }
            if (this.minCellCount == 0 || stats.cellCount < this.minCellCount) {
                this.minCellCount = stats.cellCount;
            }
            if (stats.cellCount > this.maxCellCount) {
                this.maxCellCount = stats.cellCount;
            }
            if (this.minTombstoneCount == 0 || stats.tombstoneCount() < this.minTombstoneCount) {
                this.minTombstoneCount = stats.tombstoneCount();
            }
            if (stats.tombstoneCount() > this.maxTombstoneCount) {
                this.maxTombstoneCount = stats.tombstoneCount();
            }
        }
    }
}

