// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

use crate::ensure;
use crate::error::Error;
use crate::meta::FieldType;
use crate::resolver::context::ReadContext;
use crate::serializer::collection::{DECL_ELEMENT_TYPE, HAS_NULL, IS_SAME_TYPE};
use crate::serializer::util;
use crate::serializer::Serializer;
use crate::types;
use crate::types::RefFlag;
use crate::util::ENABLE_FORY_DEBUG_OUTPUT;
use chrono::{NaiveDate, NaiveDateTime};
use std::rc::Rc;
use std::time::Duration;

#[allow(unreachable_code)]
pub fn skip_field_value(
    context: &mut ReadContext,
    field_type: &FieldType,
    read_ref_flag: bool,
) -> Result<(), Error> {
    skip_value(context, field_type, read_ref_flag, true, &None)
}

const UNKNOWN_FIELD_TYPE: FieldType = FieldType {
    type_id: types::UNKNOWN,
    nullable: true,
    generics: vec![],
};

pub fn skip_any_value(context: &mut ReadContext, read_ref_flag: bool) -> Result<(), Error> {
    // Handle ref flag first if needed
    if read_ref_flag {
        let ref_flag = context.reader.read_i8()?;
        if ref_flag == (RefFlag::Null as i8) {
            return Ok(());
        }
        if ref_flag == (RefFlag::Ref as i8) {
            // Reference to already-seen object, skip the reference index
            let _ref_index = context.reader.read_varuint32()?;
            return Ok(());
        }
        // RefValue (0) or NotNullValue (-1) means we need to read the actual object
    }

    // Read type_id first
    let type_id = context.reader.read_varuint32()?;
    let internal_id = type_id & 0xff;

    // For struct-like types, also read meta_index to get type_info
    // This is critical for polymorphic collections where elements are struct types.
    let (field_type, type_info_opt) = match internal_id {
        types::LIST | types::SET => (
            FieldType {
                type_id,
                nullable: true,
                generics: vec![UNKNOWN_FIELD_TYPE],
            },
            None,
        ),
        types::MAP => (
            FieldType {
                type_id,
                nullable: true,
                generics: vec![UNKNOWN_FIELD_TYPE, UNKNOWN_FIELD_TYPE],
            },
            None,
        ),
        types::COMPATIBLE_STRUCT | types::NAMED_COMPATIBLE_STRUCT => {
            // For compatible struct types, read meta_index to get type_info
            let meta_index = context.reader.read_varuint32()? as usize;
            let type_info = context.get_type_info_by_index(meta_index)?.clone();
            (
                FieldType {
                    type_id,
                    nullable: true,
                    generics: vec![],
                },
                Some(type_info),
            )
        }
        types::STRUCT | types::NAMED_STRUCT => {
            // For non-compatible struct types with share_meta enabled, read meta_index
            if context.is_share_meta() {
                let meta_index = context.reader.read_varuint32()? as usize;
                let type_info = context.get_type_info_by_index(meta_index)?.clone();
                (
                    FieldType {
                        type_id,
                        nullable: true,
                        generics: vec![],
                    },
                    Some(type_info),
                )
            } else {
                // Without share_meta, read namespace and type_name
                let namespace = context.read_meta_string()?.to_owned();
                let type_name = context.read_meta_string()?.to_owned();
                let rc_namespace = Rc::from(namespace);
                let rc_type_name = Rc::from(type_name);
                let type_info = context
                    .get_type_resolver()
                    .get_type_info_by_meta_string_name(rc_namespace, rc_type_name)
                    .ok_or_else(|| crate::Error::type_error("Name harness not found"))?;
                (
                    FieldType {
                        type_id,
                        nullable: true,
                        generics: vec![],
                    },
                    Some(type_info),
                )
            }
        }
        _ => (
            FieldType {
                type_id,
                nullable: true,
                generics: vec![],
            },
            None,
        ),
    };
    // Don't read ref flag again in skip_value since we already handled it.
    // Pass type_info so skip_struct doesn't try to read type_id/meta_index again.
    skip_value(context, &field_type, false, false, &type_info_opt)
}

fn skip_collection(context: &mut ReadContext, field_type: &FieldType) -> Result<(), Error> {
    let length = context.reader.read_varuint32()? as usize;
    if length == 0 {
        return Ok(());
    }
    let header = context.reader.read_u8()?;
    let has_null = (header & HAS_NULL) != 0;
    let is_same_type = (header & IS_SAME_TYPE) != 0;
    let skip_ref_flag = is_same_type && !has_null;
    let is_declared = (header & DECL_ELEMENT_TYPE) != 0;
    let default_elem_type = field_type.generics.first().unwrap();
    let (type_info, elem_field_type);
    let elem_type = if is_same_type && !is_declared {
        let type_info_rc = context.read_any_typeinfo()?;
        elem_field_type = FieldType {
            type_id: type_info_rc.get_type_id(),
            nullable: has_null,
            generics: vec![],
        };
        type_info = Some(type_info_rc);
        &elem_field_type
    } else {
        type_info = None;
        default_elem_type
    };
    context.inc_depth()?;
    for _ in 0..length {
        skip_value(context, elem_type, !skip_ref_flag, false, &type_info)?;
    }
    context.dec_depth();
    Ok(())
}

fn skip_map(context: &mut ReadContext, field_type: &FieldType) -> Result<(), Error> {
    let length = context.reader.read_varuint32()?;
    if length == 0 {
        return Ok(());
    }
    let mut len_counter = 0;
    let default_key_type = field_type.generics.first().unwrap();
    let default_value_type = field_type.generics.get(1).unwrap();
    loop {
        if len_counter == length {
            break;
        }
        let header = context.reader.read_u8()?;
        if header & crate::serializer::map::KEY_NULL != 0
            && header & crate::serializer::map::VALUE_NULL != 0
        {
            len_counter += 1;
            continue;
        }
        if header & crate::serializer::map::KEY_NULL != 0 {
            // Read value type info if not declared
            let value_declared = (header & crate::serializer::map::DECL_VALUE_TYPE) != 0;
            let (value_type_info, value_field_type);
            let value_type = if !value_declared {
                let type_info = context.read_any_typeinfo()?;
                value_field_type = FieldType {
                    type_id: type_info.get_type_id(),
                    nullable: true,
                    generics: vec![],
                };
                value_type_info = Some(type_info);
                &value_field_type
            } else {
                value_type_info = None;
                default_value_type
            };
            context.inc_depth()?;
            skip_value(context, value_type, false, false, &value_type_info)?;
            context.dec_depth();
            len_counter += 1;
            continue;
        }
        if header & crate::serializer::map::VALUE_NULL != 0 {
            // Read key type info if not declared
            let key_declared = (header & crate::serializer::map::DECL_KEY_TYPE) != 0;
            let (key_type_info, key_field_type);
            let key_type = if !key_declared {
                let type_info = context.read_any_typeinfo()?;
                key_field_type = FieldType {
                    type_id: type_info.get_type_id(),
                    nullable: true,
                    generics: vec![],
                };
                key_type_info = Some(type_info);
                &key_field_type
            } else {
                key_type_info = None;
                default_key_type
            };
            context.inc_depth()?;
            skip_value(context, key_type, false, false, &key_type_info)?;
            context.dec_depth();
            len_counter += 1;
            continue;
        }
        // Both key and value are non-null
        let chunk_size = context.reader.read_u8()?;
        let key_declared = (header & crate::serializer::map::DECL_KEY_TYPE) != 0;
        let value_declared = (header & crate::serializer::map::DECL_VALUE_TYPE) != 0;

        // Read key type info if not declared
        let (key_type_info, key_field_type);
        let key_type = if !key_declared {
            let type_info = context.read_any_typeinfo()?;
            key_field_type = FieldType {
                type_id: type_info.get_type_id(),
                nullable: true,
                generics: vec![],
            };
            key_type_info = Some(type_info);
            &key_field_type
        } else {
            key_type_info = None;
            default_key_type
        };

        // Read value type info if not declared
        let (value_type_info, value_field_type);
        let value_type = if !value_declared {
            let type_info = context.read_any_typeinfo()?;
            value_field_type = FieldType {
                type_id: type_info.get_type_id(),
                nullable: true,
                generics: vec![],
            };
            value_type_info = Some(type_info);
            &value_field_type
        } else {
            value_type_info = None;
            default_value_type
        };

        context.inc_depth()?;
        for _ in 0..chunk_size {
            skip_value(context, key_type, false, false, &key_type_info)?;
            skip_value(context, value_type, false, false, &value_type_info)?;
        }
        context.dec_depth();
        len_counter += chunk_size as u32;
    }
    Ok(())
}

fn skip_struct(
    context: &mut ReadContext,
    type_id_num: u32,
    type_info: &Option<Rc<crate::TypeInfo>>,
) -> Result<(), Error> {
    let type_info_value = if type_info.is_none() {
        let remote_type_id = context.reader.read_varuint32()?;
        ensure!(
            type_id_num == remote_type_id,
            Error::type_mismatch(type_id_num, remote_type_id)
        );
        let meta_index = context.reader.read_varuint32()?;
        context.get_type_info_by_index(meta_index as usize)?
    } else {
        type_info.as_ref().unwrap()
    };
    let type_meta = type_info_value.get_type_meta();
    if ENABLE_FORY_DEBUG_OUTPUT {
        eprintln!(
            "[skip_struct] type_name: {:?}, num_fields: {}",
            type_meta.get_type_name(),
            type_meta.get_field_infos().len()
        );
    }
    let field_infos = type_meta.get_field_infos().to_vec();
    context.inc_depth()?;
    for field_info in field_infos.iter() {
        if ENABLE_FORY_DEBUG_OUTPUT {
            eprintln!(
                "[skip_struct] field: {:?}, type_id: {}, internal_id: {}",
                field_info.field_name,
                field_info.field_type.type_id,
                field_info.field_type.type_id & 0xff
            );
        }
        let read_ref_flag = util::field_need_write_ref_into(
            field_info.field_type.type_id,
            field_info.field_type.nullable,
        );
        skip_value(context, &field_info.field_type, read_ref_flag, true, &None)?;
    }
    context.dec_depth();
    Ok(())
}

fn skip_ext(
    context: &mut ReadContext,
    type_id_num: u32,
    type_info: &Option<Rc<crate::TypeInfo>>,
) -> Result<(), Error> {
    let type_info_value = if type_info.is_none() {
        let remote_type_id = context.reader.read_varuint32()?;
        ensure!(
            type_id_num == remote_type_id,
            Error::type_mismatch(type_id_num, remote_type_id)
        );
        let meta_index = context.reader.read_varuint32()?;
        context.get_type_info_by_index(meta_index as usize)?
    } else {
        type_info.as_ref().unwrap()
    };
    let type_resolver = context.get_type_resolver();
    let type_meta = type_info_value.get_type_meta();
    type_resolver
        .get_ext_name_harness(type_meta.get_namespace(), type_meta.get_type_name())?
        .get_read_data_fn()(context)?;
    Ok(())
}

fn skip_user_struct(context: &mut ReadContext, _type_id_num: u32) -> Result<(), Error> {
    let remote_type_id = context.reader.read_varuint32()?;
    let meta_index = context.reader.read_varuint32()?;
    let type_info = context.get_type_info_by_index(meta_index as usize)?;
    let type_meta = type_info.get_type_meta();
    ensure!(
        type_meta.get_type_id() == remote_type_id,
        Error::type_mismatch(type_meta.get_type_id(), remote_type_id)
    );
    let field_infos = type_meta.get_field_infos().to_vec();
    context.inc_depth()?;
    for field_info in field_infos.iter() {
        let read_ref_flag = util::field_need_write_ref_into(
            field_info.field_type.type_id,
            field_info.field_type.nullable,
        );
        skip_value(context, &field_info.field_type, read_ref_flag, true, &None)?;
    }
    context.dec_depth();
    Ok(())
}

fn skip_user_ext(context: &mut ReadContext, type_id_num: u32) -> Result<(), Error> {
    let remote_type_id = context.reader.read_varuint32()?;
    ensure!(
        type_id_num == remote_type_id,
        Error::type_mismatch(type_id_num, remote_type_id)
    );
    context.inc_depth()?;
    let type_resolver = context.get_type_resolver();
    type_resolver
        .get_ext_harness(type_id_num)?
        .get_read_data_fn()(context)?;
    context.dec_depth();
    Ok(())
}

// call when is_field && is_compatible_mode
#[allow(unreachable_code)]
fn skip_value(
    context: &mut ReadContext,
    field_type: &FieldType,
    read_ref_flag: bool,
    _is_field: bool,
    type_info: &Option<Rc<crate::TypeInfo>>,
) -> Result<(), Error> {
    if read_ref_flag {
        let ref_flag = context.reader.read_i8()?;
        if ref_flag == (RefFlag::Null as i8) {
            return Ok(());
        }
        if ref_flag == (RefFlag::Ref as i8) {
            // Reference to already-seen object, skip the reference index
            let _ref_index = context.reader.read_varuint32()?;
            return Ok(());
        }
        // RefValue (0) or NotNullValue (-1) means we need to read the actual object
    }
    let type_id_num = field_type.type_id;

    // Check if it's a user-defined type (high bits set, meaning type_id > 255)
    if type_id_num > 255 {
        let internal_id = type_id_num & 0xff;
        // Handle struct-like types including UNKNOWN (0) which is used for polymorphic types
        if internal_id == types::COMPATIBLE_STRUCT
            || internal_id == types::STRUCT
            || internal_id == types::NAMED_STRUCT
            || internal_id == types::NAMED_COMPATIBLE_STRUCT
            || internal_id == types::UNKNOWN
        {
            // If type_info is provided (from skip_any_value), use skip_struct directly
            // which won't try to re-read type_id/meta_index. Otherwise use skip_user_struct.
            if type_info.is_some() {
                return skip_struct(context, type_id_num, type_info);
            }
            return skip_user_struct(context, type_id_num);
        } else if internal_id == types::ENUM || internal_id == types::NAMED_ENUM {
            let _ordinal = context.reader.read_varuint32()?;
            return Ok(());
        } else if internal_id == types::EXT || internal_id == types::NAMED_EXT {
            return skip_user_ext(context, type_id_num);
        } else {
            return Err(Error::type_error(format!(
                "Unknown type id: {} (internal_id: {}, type_info provided: {})",
                type_id_num,
                internal_id,
                type_info.is_some()
            )));
        }
    }

    // Match on built-in types
    match type_id_num {
        // Basic types
        types::BOOL => {
            <bool as Serializer>::fory_read_data(context)?;
        }
        types::INT8 => {
            <i8 as Serializer>::fory_read_data(context)?;
        }
        types::INT16 => {
            <i16 as Serializer>::fory_read_data(context)?;
        }
        types::INT32 => {
            <i32 as Serializer>::fory_read_data(context)?;
        }
        types::INT64 => {
            <i64 as Serializer>::fory_read_data(context)?;
        }
        types::FLOAT32 => {
            <f32 as Serializer>::fory_read_data(context)?;
        }
        types::FLOAT64 => {
            <f64 as Serializer>::fory_read_data(context)?;
        }
        types::STRING => {
            <String as Serializer>::fory_read_data(context)?;
        }
        types::LOCAL_DATE => {
            <NaiveDate as Serializer>::fory_read_data(context)?;
        }
        types::TIMESTAMP => {
            <NaiveDateTime as Serializer>::fory_read_data(context)?;
        }
        types::DURATION => {
            <Duration as Serializer>::fory_read_data(context)?;
        }
        types::BINARY => {
            <Vec<u8> as Serializer>::fory_read_data(context)?;
        }
        types::BOOL_ARRAY => {
            <Vec<bool> as Serializer>::fory_read_data(context)?;
        }
        types::INT8_ARRAY => {
            <Vec<i8> as Serializer>::fory_read_data(context)?;
        }
        types::INT16_ARRAY => {
            <Vec<i16> as Serializer>::fory_read_data(context)?;
        }
        types::INT32_ARRAY => {
            <Vec<i32> as Serializer>::fory_read_data(context)?;
        }
        types::INT64_ARRAY => {
            <Vec<i64> as Serializer>::fory_read_data(context)?;
        }
        types::FLOAT32_ARRAY => {
            <Vec<f32> as Serializer>::fory_read_data(context)?;
        }
        types::FLOAT64_ARRAY => {
            <Vec<f64> as Serializer>::fory_read_data(context)?;
        }
        types::U8 => {
            <u8 as Serializer>::fory_read_data(context)?;
        }
        types::U16 => {
            <u16 as Serializer>::fory_read_data(context)?;
        }
        types::U32 => {
            <u32 as Serializer>::fory_read_data(context)?;
        }
        types::U64 => {
            <u64 as Serializer>::fory_read_data(context)?;
        }
        types::USIZE => {
            <usize as Serializer>::fory_read_data(context)?;
        }
        types::U128 => {
            <u128 as Serializer>::fory_read_data(context)?;
        }
        types::U16_ARRAY => {
            <Vec<u16> as Serializer>::fory_read_data(context)?;
        }
        types::U32_ARRAY => {
            <Vec<u32> as Serializer>::fory_read_data(context)?;
        }
        types::U64_ARRAY => {
            <Vec<u64> as Serializer>::fory_read_data(context)?;
        }
        types::USIZE_ARRAY => {
            <Vec<usize> as Serializer>::fory_read_data(context)?;
        }
        types::U128_ARRAY => {
            <Vec<u128> as Serializer>::fory_read_data(context)?;
        }

        // Container types
        types::LIST | types::SET => {
            return skip_collection(context, field_type);
        }
        types::MAP => {
            return skip_map(context, field_type);
        }

        // Named types
        types::NAMED_ENUM => {
            let _ordinal = context.reader.read_varuint32()?;
        }
        types::NAMED_COMPATIBLE_STRUCT => {
            return skip_struct(context, type_id_num, type_info);
        }
        types::NAMED_EXT => {
            return skip_ext(context, type_id_num, type_info);
        }
        types::UNKNOWN => {
            // UNKNOWN (0) is used for polymorphic types in cross-language serialization
            return skip_any_value(context, false);
        }

        _ => {
            return Err(Error::type_error(format!(
                "Unimplemented type id: {}",
                type_id_num
            )));
        }
    }
    Ok(())
}

/// Skip enum variant data in compatible mode based on variant type.
///
/// # Arguments
/// * `context` - The read context
/// * `variant_type` - The variant type encoded in lower 2 bits:
///   - 0b0 = Unit variant (no data to skip)
///   - 0b1 = Unnamed variant (tuple data)
///   - 0b10 = Named variant (struct-like data)
/// * `type_info` - Optional type info for named variants (must be provided for 0b10)
pub fn skip_enum_variant(
    context: &mut ReadContext,
    variant_type: u32,
    type_info: &Option<Rc<crate::TypeInfo>>,
) -> Result<(), Error> {
    match variant_type {
        0b0 => {
            // Unit variant, no data to skip
            Ok(())
        }
        0b1 => {
            // Unnamed variant, skip tuple data (which is serialized as a collection)
            // Tuple uses collection format but doesn't write type info, so skip directly
            let field_type = FieldType {
                type_id: types::LIST,
                nullable: false,
                generics: vec![UNKNOWN_FIELD_TYPE],
            };
            skip_collection(context, &field_type)
        }
        0b10 => {
            // Named variant, skip struct-like data using skip_struct
            // For named variants, we need the type_info which should have been read already
            if type_info.is_some() {
                let type_id = type_info.as_ref().unwrap().get_type_id();
                skip_struct(context, type_id, type_info)
            } else {
                // If no type_info provided, read it from the stream
                let meta_index = context.reader.read_varuint32()?;
                let type_info_rc = context.get_type_info_by_index(meta_index as usize)?.clone();
                let type_id = type_info_rc.get_type_id();
                let type_info_opt = Some(type_info_rc);
                skip_struct(context, type_id, &type_info_opt)
            }
        }
        _ => {
            // Invalid variant type
            Err(Error::type_error(format!(
                "Invalid enum variant type: {}",
                variant_type
            )))
        }
    }
}
