/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.obs;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.LocalDirAllocator;
import org.apache.hadoop.fs.obs.OBSFileSystem;
import org.apache.hadoop.util.DirectBufferPool;
import org.apache.hadoop.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class OBSDataBlocks {
    private static final Logger LOG = LoggerFactory.getLogger(OBSDataBlocks.class);

    private OBSDataBlocks() {
    }

    static void validateWriteArgs(byte[] b, int off, int len) {
        Preconditions.checkNotNull((Object)b);
        if (off < 0 || off > b.length || len < 0 || off + len > b.length || off + len < 0) {
            throw new IndexOutOfBoundsException("write (b[" + b.length + "], " + off + ", " + len + ')');
        }
    }

    static BlockFactory createFactory(OBSFileSystem owner, String name) {
        switch (name) {
            case "array": {
                return new ByteArrayBlockFactory(owner);
            }
            case "disk": {
                return new DiskBlockFactory(owner);
            }
            case "bytebuffer": {
                return new ByteBufferBlockFactory(owner);
            }
        }
        throw new IllegalArgumentException("Unsupported block buffer \"" + name + '\"');
    }

    static class DiskBlock
    extends DataBlock {
        private final File bufferFile;
        private final int limit;
        private final AtomicBoolean closed = new AtomicBoolean(false);
        private int bytesWritten;
        private BufferedOutputStream out;

        DiskBlock(File destBufferFile, int limitSize, long index) throws FileNotFoundException {
            super(index);
            this.limit = limitSize;
            this.bufferFile = destBufferFile;
            this.out = new BufferedOutputStream(new FileOutputStream(destBufferFile));
        }

        @Override
        int dataSize() {
            return this.bytesWritten;
        }

        @Override
        boolean hasCapacity(long bytes) {
            return (long)this.dataSize() + bytes <= (long)this.limit;
        }

        @Override
        int remainingCapacity() {
            return this.limit - this.bytesWritten;
        }

        @Override
        int write(byte[] b, int offset, int len) throws IOException {
            super.write(b, offset, len);
            int written = Math.min(this.remainingCapacity(), len);
            this.out.write(b, offset, written);
            this.bytesWritten += written;
            return written;
        }

        @Override
        File startUpload() throws IOException {
            super.startUpload();
            try {
                this.out.flush();
            }
            finally {
                this.out.close();
                this.out = null;
            }
            return this.bufferFile;
        }

        @Override
        protected void innerClose() {
            DataBlock.DestState state = this.getState();
            LOG.debug("Closing {}", (Object)this);
            switch (state) {
                case Writing: {
                    if (!this.bufferFile.exists()) break;
                    LOG.debug("Block[{}]: Deleting buffer file as upload did not start", (Object)this.getIndex());
                    this.closeBlock();
                    break;
                }
                case Upload: {
                    LOG.debug("Block[{}]: Buffer file {} exists close upload stream", (Object)this.getIndex(), (Object)this.bufferFile);
                    break;
                }
                case Closed: {
                    this.closeBlock();
                    break;
                }
            }
        }

        @Override
        void flush() throws IOException {
            super.flush();
            this.out.flush();
        }

        public String toString() {
            return "FileBlock{index=" + this.getIndex() + ", destFile=" + this.bufferFile + ", state=" + (Object)((Object)this.getState()) + ", dataSize=" + this.dataSize() + ", limit=" + this.limit + '}';
        }

        void closeBlock() {
            LOG.debug("block[{}]: closeBlock()", (Object)this.getIndex());
            if (!this.closed.getAndSet(true)) {
                if (!this.bufferFile.delete() && this.bufferFile.exists()) {
                    LOG.warn("delete({}) returned false", (Object)this.bufferFile.getAbsoluteFile());
                }
            } else {
                LOG.debug("block[{}]: skipping re-entrant closeBlock()", (Object)this.getIndex());
            }
        }
    }

    static class DiskBlockFactory
    extends BlockFactory {
        private static LocalDirAllocator directoryAllocator;

        DiskBlockFactory(OBSFileSystem owner) {
            super(owner);
        }

        @Override
        DataBlock create(long index, int limit) throws IOException {
            File destFile = DiskBlockFactory.createTmpFileForWrite(String.format("obs-block-%04d-", index), limit, this.getOwner().getConf());
            return new DiskBlock(destFile, limit, index);
        }

        static synchronized File createTmpFileForWrite(String pathStr, long size, Configuration conf) throws IOException {
            if (directoryAllocator == null) {
                String bufferDir = conf.get("fs.obs.buffer.dir") != null ? "fs.obs.buffer.dir" : "hadoop.tmp.dir";
                directoryAllocator = new LocalDirAllocator(bufferDir);
            }
            return directoryAllocator.createTmpFileForWrite(pathStr, size, conf);
        }
    }

    static class ByteBufferBlock
    extends DataBlock {
        private final int bufferSize;
        private ByteBuffer blockBuffer;
        private Integer dataSize;
        private ByteBufferInputStream inputStream;

        ByteBufferBlock(long index, int initBufferSize) {
            super(index);
            this.bufferSize = initBufferSize;
            this.blockBuffer = ByteBufferBlockFactory.requestBuffer(initBufferSize);
        }

        @Override
        int dataSize() {
            return this.dataSize != null ? this.dataSize.intValue() : this.bufferCapacityUsed();
        }

        @Override
        InputStream startUpload() throws IOException {
            super.startUpload();
            this.dataSize = this.bufferCapacityUsed();
            this.blockBuffer.limit(this.blockBuffer.position());
            this.blockBuffer.position(0);
            this.inputStream = new ByteBufferInputStream(this.dataSize, this.blockBuffer);
            return this.inputStream;
        }

        @Override
        public boolean hasCapacity(long bytes) {
            return bytes <= (long)this.remainingCapacity();
        }

        @Override
        public int remainingCapacity() {
            return this.blockBuffer != null ? this.blockBuffer.remaining() : 0;
        }

        private int bufferCapacityUsed() {
            return this.blockBuffer.capacity() - this.blockBuffer.remaining();
        }

        @Override
        int write(byte[] b, int offset, int len) throws IOException {
            super.write(b, offset, len);
            int written = Math.min(this.remainingCapacity(), len);
            this.blockBuffer.put(b, offset, written);
            return written;
        }

        @Override
        protected void innerClose() {
            if (this.blockBuffer != null) {
                ByteBufferBlockFactory.releaseBuffer(this.blockBuffer);
                this.blockBuffer = null;
            }
            if (this.inputStream != null) {
                this.inputStream.close();
                this.inputStream = null;
            }
        }

        public String toString() {
            return "ByteBufferBlock{index=" + this.getIndex() + ", state=" + (Object)((Object)this.getState()) + ", dataSize=" + this.dataSize() + ", limit=" + this.bufferSize + ", remainingCapacity=" + this.remainingCapacity() + '}';
        }

        class ByteBufferInputStream
        extends InputStream {
            private final int size;
            private ByteBuffer byteBuffer;

            ByteBufferInputStream(int streamSize, ByteBuffer streamByteBuffer) {
                LOG.debug("Creating ByteBufferInputStream of size {}", (Object)streamSize);
                this.size = streamSize;
                this.byteBuffer = streamByteBuffer;
            }

            @Override
            public synchronized void close() {
                LOG.debug("ByteBufferInputStream.close() for {}", (Object)ByteBufferBlock.super.toString());
                this.byteBuffer = null;
            }

            private void verifyOpen() throws IOException {
                if (this.byteBuffer == null) {
                    throw new IOException("Stream is closed!");
                }
            }

            @Override
            public synchronized int read() {
                if (this.available() > 0) {
                    return this.byteBuffer.get() & 0xFF;
                }
                return -1;
            }

            @Override
            public synchronized long skip(long offset) throws IOException {
                this.verifyOpen();
                long newPos = (long)this.position() + offset;
                if (newPos < 0L) {
                    throw new EOFException("Cannot seek to a negative offset");
                }
                if (newPos > (long)this.size) {
                    throw new EOFException("Attempted to seek or read past the end of the file");
                }
                this.byteBuffer.position((int)newPos);
                return newPos;
            }

            @Override
            public synchronized int available() {
                Preconditions.checkState((this.byteBuffer != null ? 1 : 0) != 0, (Object)"Stream is closed!");
                return this.byteBuffer.remaining();
            }

            public synchronized int position() {
                return this.byteBuffer.position();
            }

            public synchronized boolean hasRemaining() {
                return this.byteBuffer.hasRemaining();
            }

            @Override
            public synchronized void mark(int readlimit) {
                LOG.debug("mark at {}", (Object)this.position());
                this.byteBuffer.mark();
            }

            @Override
            public synchronized void reset() {
                LOG.debug("reset");
                this.byteBuffer.reset();
            }

            @Override
            public boolean markSupported() {
                return true;
            }

            @Override
            public synchronized int read(byte[] b, int offset, int length) throws IOException {
                Preconditions.checkArgument((length >= 0 ? 1 : 0) != 0, (Object)"length is negative");
                Preconditions.checkArgument((b != null ? 1 : 0) != 0, (Object)"Null buffer");
                if (b.length - offset < length) {
                    throw new IndexOutOfBoundsException("Requested more bytes than destination buffer size: request length =" + length + ", with offset =" + offset + "; buffer capacity =" + (b.length - offset));
                }
                this.verifyOpen();
                if (!this.hasRemaining()) {
                    return -1;
                }
                int toRead = Math.min(length, this.available());
                this.byteBuffer.get(b, offset, toRead);
                return toRead;
            }

            public String toString() {
                StringBuilder sb = new StringBuilder("ByteBufferInputStream{");
                sb.append("size=").append(this.size);
                ByteBuffer buf = this.byteBuffer;
                if (buf != null) {
                    sb.append(", available=").append(buf.remaining());
                }
                sb.append(", ").append(ByteBufferBlock.super.toString());
                sb.append('}');
                return sb.toString();
            }
        }
    }

    static class ByteBufferBlockFactory
    extends BlockFactory {
        private static final DirectBufferPool BUFFER_POOL = new DirectBufferPool();
        private static final AtomicInteger BUFFERS_OUTSTANDING = new AtomicInteger(0);

        ByteBufferBlockFactory(OBSFileSystem owner) {
            super(owner);
        }

        @Override
        ByteBufferBlock create(long index, int limit) {
            return new ByteBufferBlock(index, limit);
        }

        public static ByteBuffer requestBuffer(int limit) {
            LOG.debug("Requesting buffer of size {}", (Object)limit);
            BUFFERS_OUTSTANDING.incrementAndGet();
            return BUFFER_POOL.getBuffer(limit);
        }

        public static void releaseBuffer(ByteBuffer buffer) {
            LOG.debug("Releasing buffer");
            BUFFER_POOL.returnBuffer(buffer);
            BUFFERS_OUTSTANDING.decrementAndGet();
        }

        public int getOutstandingBufferCount() {
            return BUFFERS_OUTSTANDING.get();
        }

        public String toString() {
            return "ByteBufferBlockFactory{buffersOutstanding=" + BUFFERS_OUTSTANDING + '}';
        }
    }

    static class ByteArrayBlock
    extends DataBlock {
        private final int limit;
        private OBSByteArrayOutputStream buffer;
        private Integer dataSize;
        private int firstBlockSize;
        private ByteArrayInputStream inputStream = null;

        ByteArrayBlock(long index, int limitBlockSize, int blockSize) {
            super(index);
            this.limit = limitBlockSize;
            this.buffer = new OBSByteArrayOutputStream(blockSize);
            this.firstBlockSize = blockSize;
        }

        @VisibleForTesting
        public int firstBlockSize() {
            return this.firstBlockSize;
        }

        @Override
        int dataSize() {
            return this.dataSize != null ? this.dataSize.intValue() : this.buffer.size();
        }

        @Override
        InputStream startUpload() throws IOException {
            super.startUpload();
            this.dataSize = this.buffer.size();
            this.inputStream = this.buffer.getInputStream();
            return this.inputStream;
        }

        @Override
        boolean hasCapacity(long bytes) {
            return (long)this.dataSize() + bytes <= (long)this.limit;
        }

        @Override
        int remainingCapacity() {
            return this.limit - this.dataSize();
        }

        @Override
        int write(byte[] b, int offset, int len) throws IOException {
            super.write(b, offset, len);
            int written = Math.min(this.remainingCapacity(), len);
            this.buffer.write(b, offset, written);
            return written;
        }

        @Override
        protected void innerClose() throws IOException {
            if (this.buffer != null) {
                this.buffer.close();
                this.buffer = null;
            }
            if (this.inputStream != null) {
                this.inputStream.close();
                this.inputStream = null;
            }
        }

        public String toString() {
            return "ByteArrayBlock{index=" + this.getIndex() + ", state=" + (Object)((Object)this.getState()) + ", limit=" + this.limit + ", dataSize=" + this.dataSize + '}';
        }
    }

    static class OBSByteArrayOutputStream
    extends ByteArrayOutputStream {
        OBSByteArrayOutputStream(int size) {
            super(size);
        }

        ByteArrayInputStream getInputStream() {
            ByteArrayInputStream bin = new ByteArrayInputStream(this.buf, 0, this.count);
            this.reset();
            this.buf = null;
            return bin;
        }
    }

    static class ByteArrayBlockFactory
    extends BlockFactory {
        ByteArrayBlockFactory(OBSFileSystem owner) {
            super(owner);
        }

        @Override
        DataBlock create(long index, int limit) {
            int firstBlockSize = ((BlockFactory)this).owner.getConf().getInt("fs.obs.fast.upload.array.first.buffer", 0x100000);
            return new ByteArrayBlock(0L, limit, firstBlockSize);
        }
    }

    static abstract class DataBlock
    implements Closeable {
        private final long index;
        private volatile DestState state = DestState.Writing;

        protected DataBlock(long dataIndex) {
            this.index = dataIndex;
        }

        protected final synchronized void enterState(DestState current, DestState next) throws IllegalStateException {
            this.verifyState(current);
            LOG.debug("{}: entering state {}", (Object)this, (Object)next);
            this.state = next;
        }

        protected final void verifyState(DestState expected) throws IllegalStateException {
            if (expected != null && this.state != expected) {
                throw new IllegalStateException("Expected stream state " + (Object)((Object)expected) + " -but actual state is " + (Object)((Object)this.state) + " in " + this);
            }
        }

        protected final DestState getState() {
            return this.state;
        }

        protected long getIndex() {
            return this.index;
        }

        abstract int dataSize();

        abstract boolean hasCapacity(long var1);

        boolean hasData() {
            return this.dataSize() > 0;
        }

        abstract int remainingCapacity();

        int write(byte[] buffer, int offset, int length) throws IOException {
            this.verifyState(DestState.Writing);
            Preconditions.checkArgument((buffer != null ? 1 : 0) != 0, (Object)"Null buffer");
            Preconditions.checkArgument((length >= 0 ? 1 : 0) != 0, (Object)"length is negative");
            Preconditions.checkArgument((offset >= 0 ? 1 : 0) != 0, (Object)"offset is negative");
            Preconditions.checkArgument((buffer.length - offset >= length ? 1 : 0) != 0, (Object)"buffer shorter than amount of data to write");
            return 0;
        }

        void flush() throws IOException {
            this.verifyState(DestState.Writing);
        }

        Object startUpload() throws IOException {
            LOG.debug("Start datablock[{}] upload", (Object)this.index);
            this.enterState(DestState.Writing, DestState.Upload);
            return null;
        }

        protected synchronized boolean enterClosedState() {
            if (!this.state.equals((Object)DestState.Closed)) {
                this.enterState(null, DestState.Closed);
                return true;
            }
            return false;
        }

        @Override
        public void close() throws IOException {
            if (this.enterClosedState()) {
                LOG.debug("Closed {}", (Object)this);
                this.innerClose();
            }
        }

        protected abstract void innerClose() throws IOException;

        static enum DestState {
            Writing,
            Upload,
            Closed;

        }
    }

    static abstract class BlockFactory {
        private final OBSFileSystem owner;

        protected BlockFactory(OBSFileSystem obsFileSystem) {
            this.owner = obsFileSystem;
        }

        abstract DataBlock create(long var1, int var3) throws IOException;

        protected OBSFileSystem getOwner() {
            return this.owner;
        }
    }
}

