/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */


import type { CommandResponse } from '../../client/client.type.js';
import { type Id } from '../identifier.utils.js';
import { wrapCommand } from '../command.utils.js';
import { COMMAND_CODE } from '../command.code.js';
import {
  serializeTargetGroup, deserializeConsumerGroup,
  type ConsumerGroup
} from './group.utils.js';

/**
 * Parameters for the get consumer group command.
 */
export type GetGroup = {
  /** Stream identifier (ID or name) */
  streamId: Id,
  /** Topic identifier (ID or name) */
  topicId: Id,
  /** Consumer group identifier (ID or name) */
  groupId: Id
};

/**
 * Get consumer group command definition.
 * Retrieves information about a specific consumer group.
 */
export const GET_GROUP = {
  code: COMMAND_CODE.GetGroup,

  serialize: ({streamId, topicId, groupId}: GetGroup) => {
    return serializeTargetGroup(streamId, topicId, groupId);
  },

  deserialize: (r: CommandResponse) => {
    if(r.status === 0 && r.length === 0)
      return null;
    return deserializeConsumerGroup(r.data).data;
  }
};


/**
 * Executable get consumer group command function.
 */
export const getGroup = wrapCommand<GetGroup, ConsumerGroup | null>(GET_GROUP);
