/*
 * Decompiled with CFR 0.152.
 */
package io.trino.spi.type;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import io.trino.spi.type.TimeZoneNotSupportedException;
import java.io.IOException;
import java.io.InputStream;
import java.time.ZoneId;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;

public final class TimeZoneKey {
    public static final TimeZoneKey UTC_KEY = new TimeZoneKey("UTC", 0);
    public static final short MAX_TIME_ZONE_KEY;
    private static final Map<String, TimeZoneKey> ZONE_ID_TO_KEY;
    private static final Set<TimeZoneKey> ZONE_KEYS;
    private static final TimeZoneKey[] TIME_ZONE_KEYS;
    private static final short OFFSET_TIME_ZONE_MIN = -840;
    private static final short OFFSET_TIME_ZONE_MAX = 840;
    private static final TimeZoneKey[] OFFSET_TIME_ZONE_KEYS;
    private static final Set<String> UTC_EQUIVALENTS;
    private static final List<String> UTC_OFFSET_PREFIXES;
    private final String id;
    private final short key;

    public static Set<TimeZoneKey> getTimeZoneKeys() {
        return ZONE_KEYS;
    }

    @JsonCreator
    public static TimeZoneKey getTimeZoneKey(short timeZoneKey) {
        TimeZoneKey.checkArgument(timeZoneKey < TIME_ZONE_KEYS.length && TIME_ZONE_KEYS[timeZoneKey] != null, "Invalid time zone key %s", timeZoneKey);
        return TIME_ZONE_KEYS[timeZoneKey];
    }

    public static TimeZoneKey getTimeZoneKey(String zoneId) {
        Objects.requireNonNull(zoneId, "zoneId is null");
        TimeZoneKey.checkArgument(!zoneId.isEmpty(), "Zone id is an empty string", new Object[0]);
        TimeZoneKey zoneKey = ZONE_ID_TO_KEY.get(zoneId);
        if (zoneKey == null) {
            zoneKey = ZONE_ID_TO_KEY.get(TimeZoneKey.normalizeZoneId(zoneId));
        }
        if (zoneKey == null) {
            throw new TimeZoneNotSupportedException(zoneId);
        }
        return zoneKey;
    }

    public static TimeZoneKey getTimeZoneKeyForOffset(long offsetMinutes) {
        if (offsetMinutes == 0L) {
            return UTC_KEY;
        }
        if (offsetMinutes < -840L || offsetMinutes > 840L) {
            throw new TrinoException((ErrorCodeSupplier)StandardErrorCode.INVALID_FUNCTION_ARGUMENT, "Invalid offset minutes " + offsetMinutes);
        }
        TimeZoneKey timeZoneKey = OFFSET_TIME_ZONE_KEYS[(int)offsetMinutes - -840];
        if (timeZoneKey == null) {
            throw new TimeZoneNotSupportedException(TimeZoneKey.zoneIdForOffset(offsetMinutes));
        }
        return timeZoneKey;
    }

    TimeZoneKey(String id, short key) {
        this.id = Objects.requireNonNull(id, "id is null");
        if (key < 0) {
            throw new IllegalArgumentException("key is negative");
        }
        this.key = key;
    }

    public String getId() {
        return this.id;
    }

    public ZoneId getZoneId() {
        return ZoneId.of(this.id);
    }

    @JsonValue
    public short getKey() {
        return this.key;
    }

    public int hashCode() {
        return Objects.hash(this.id, this.key);
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || this.getClass() != obj.getClass()) {
            return false;
        }
        TimeZoneKey other = (TimeZoneKey)obj;
        return Objects.equals(this.id, other.id) && Objects.equals(this.key, other.key);
    }

    public String toString() {
        return this.id;
    }

    public static boolean isUtcZoneId(String zoneId) {
        return TimeZoneKey.normalizeZoneId(zoneId).equals("UTC");
    }

    private static String normalizeZoneId(String zoneId) {
        if (TimeZoneKey.isUtcEquivalentName(zoneId)) {
            return "UTC";
        }
        for (String prefix : UTC_OFFSET_PREFIXES) {
            int hour;
            if (!zoneId.startsWith(prefix)) continue;
            int offset = prefix.length();
            if (offset == zoneId.length() || zoneId.charAt(offset) != '+' && zoneId.charAt(offset) != '-') {
                return zoneId;
            }
            int colon = zoneId.indexOf(58, offset);
            int minute = 0;
            try {
                if (colon != -1) {
                    hour = Integer.parseInt(zoneId, offset, colon, 10);
                    minute = Integer.parseInt(zoneId, colon + 1, zoneId.length(), 10);
                } else {
                    if (zoneId.length() - offset > 3) break;
                    hour = Integer.parseInt(zoneId, offset, zoneId.length(), 10);
                }
            }
            catch (NumberFormatException e) {
                return zoneId;
            }
            if (hour == 0 && minute == 0) {
                return "UTC";
            }
            if (prefix.equals("Etc/GMT")) {
                hour = -hour;
            }
            return TimeZoneKey.formatZoneOffset(hour, minute);
        }
        try {
            zoneId = ZoneId.of(zoneId).getId();
        }
        catch (Exception e) {
            return zoneId;
        }
        if (TimeZoneKey.isUtcEquivalentName(zoneId)) {
            return "UTC";
        }
        return zoneId;
    }

    private static String formatZoneOffset(int hour, int minute) {
        StringBuilder builder = new StringBuilder();
        builder.append(hour >= 0 ? (char)'+' : '-');
        hour = Math.abs(hour);
        if (hour < 10) {
            builder.append('0');
        }
        builder.append(hour);
        builder.append(':');
        if (minute < 10) {
            builder.append('0');
        }
        builder.append(minute);
        return builder.toString();
    }

    private static boolean isUtcEquivalentName(String zoneId) {
        return UTC_EQUIVALENTS.contains(zoneId);
    }

    private static String zoneIdForOffset(long offset) {
        return String.format("%s%02d:%02d", offset < 0L ? "-" : "+", Math.abs(offset / 60L), Math.abs(offset % 60L));
    }

    private static void checkArgument(boolean check, String message, Object ... args) {
        if (!check) {
            throw new IllegalArgumentException(String.format(message, args));
        }
    }

    static {
        OFFSET_TIME_ZONE_KEYS = new TimeZoneKey[1681];
        UTC_EQUIVALENTS = Set.of("GMT", "GMT0", "GMT+0", "GMT-0", "Etc/GMT", "Etc/GMT0", "Etc/GMT+0", "Etc/GMT-0", "UT", "UT+0", "UT-0", "Etc/UT", "Etc/UT+0", "Etc/UT-0", "UTC", "UTC+0", "UTC-0", "Etc/UTC", "Etc/UTC+0", "Etc/UTC-0", "+0000", "+00:00", "-0000", "-00:00", "Z", "Zulu", "UCT", "Greenwich", "Universal", "Etc/Universal", "Etc/UCT");
        UTC_OFFSET_PREFIXES = List.of("Etc/GMT", "Etc/UTC", "Etc/UT", "GMT", "UTC", "UT");
        try (InputStream in = TimeZoneKey.class.getResourceAsStream("zone-index.properties");){
            Properties data = new Properties(){

                @Override
                public synchronized Object put(Object key, Object value) {
                    Object existingEntry = super.put(key, value);
                    if (existingEntry != null) {
                        throw new AssertionError((Object)("Zone file has duplicate entries for " + key));
                    }
                    return null;
                }
            };
            data.load(in);
            if (data.containsKey("0")) {
                throw new AssertionError((Object)"Zone file should not contain a mapping for key 0");
            }
            TreeMap<String, TimeZoneKey> zoneIdToKey = new TreeMap<String, TimeZoneKey>();
            zoneIdToKey.put(UTC_KEY.getId(), UTC_KEY);
            short maxZoneKey = 0;
            for (Map.Entry<Object, Object> entry : data.entrySet()) {
                short zoneKey = Short.valueOf(((String)entry.getKey()).trim());
                String zoneId = ((String)entry.getValue()).trim();
                maxZoneKey = (short)Math.max(maxZoneKey, zoneKey);
                zoneIdToKey.put(zoneId, new TimeZoneKey(zoneId, zoneKey));
            }
            MAX_TIME_ZONE_KEY = maxZoneKey;
            ZONE_ID_TO_KEY = Collections.unmodifiableMap(new LinkedHashMap(zoneIdToKey));
            ZONE_KEYS = Collections.unmodifiableSet(new LinkedHashSet(zoneIdToKey.values()));
            TIME_ZONE_KEYS = new TimeZoneKey[maxZoneKey + 1];
            Iterator<Map.Entry<Object, Object>> iterator = zoneIdToKey.values().iterator();
            while (iterator.hasNext()) {
                TimeZoneKey timeZoneKey;
                TimeZoneKey.TIME_ZONE_KEYS[timeZoneKey.getKey()] = timeZoneKey = (TimeZoneKey)((Object)iterator.next());
            }
            for (int offset = -840; offset <= 840; offset = (int)((short)(offset + 1))) {
                TimeZoneKey zoneKey;
                if (offset == 0) continue;
                String zoneId = TimeZoneKey.zoneIdForOffset(offset);
                TimeZoneKey.OFFSET_TIME_ZONE_KEYS[offset - -840] = zoneKey = ZONE_ID_TO_KEY.get(zoneId);
            }
        }
        catch (IOException e) {
            throw new AssertionError("Error loading time zone index file", e);
        }
    }
}

