/*
 * Copyright (C) 2018 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#if ENABLE(WEB_RTC) && USE(LIBWEBRTC)

#include "ExceptionCode.h"
#include "LibWebRTCMacros.h"
#include "LibWebRTCUtils.h"

WTF_IGNORE_WARNINGS_IN_THIRD_PARTY_CODE_BEGIN

// See Bug 274508: Disable thread-safety-reference-return warnings in libwebrtc
IGNORE_CLANG_WARNINGS_BEGIN("thread-safety-reference-return")
IGNORE_CLANG_WARNINGS_BEGIN("nullability-completeness")
#include <webrtc/api/peer_connection_interface.h>
IGNORE_CLANG_WARNINGS_END
IGNORE_CLANG_WARNINGS_END

WTF_IGNORE_WARNINGS_IN_THIRD_PARTY_CODE_END

namespace WebCore {

template<typename Endpoint>
class CreateSessionDescriptionObserver final : public webrtc::CreateSessionDescriptionObserver {
public:
    explicit CreateSessionDescriptionObserver(Endpoint &endpoint)
        : m_endpoint(endpoint)
    {
    }

    void OnSuccess(webrtc::SessionDescriptionInterface* sessionDescription) final { m_endpoint.createSessionDescriptionSucceeded(std::unique_ptr<webrtc::SessionDescriptionInterface>(sessionDescription)); }
    void OnFailure(webrtc::RTCError error) final { m_endpoint.createSessionDescriptionFailed(toExceptionCode(error.type()), error.message()); }

    void AddRef() const { m_endpoint.AddRef(); }
    webrtc::RefCountReleaseStatus Release() const { return m_endpoint.Release(); }

private:
    Endpoint& m_endpoint;
};

template<typename Endpoint>
class SetLocalSessionDescriptionObserver final : public webrtc::SetLocalDescriptionObserverInterface {
public:
    explicit SetLocalSessionDescriptionObserver(Endpoint &endpoint)
        : m_endpoint(endpoint)
    {
    }

    void AddRef() const { m_endpoint.AddRef(); }
    webrtc::RefCountReleaseStatus Release() const { return m_endpoint.Release(); }

private:
    void OnSetLocalDescriptionComplete(webrtc::RTCError error) final
    {
        if (!error.ok()) {
            m_endpoint.setLocalSessionDescriptionFailed(toExceptionCode(error.type()), error.message());
            return;
        }
        m_endpoint.setLocalSessionDescriptionSucceeded();
    }

    Endpoint& m_endpoint;
};

template<typename Endpoint>
class SetRemoteSessionDescriptionObserver final : public webrtc::SetRemoteDescriptionObserverInterface {
public:
    explicit SetRemoteSessionDescriptionObserver(Endpoint &endpoint)
        : m_endpoint(endpoint)
    {
    }

    void AddRef() const { m_endpoint.AddRef(); }
    webrtc::RefCountReleaseStatus Release() const { return m_endpoint.Release(); }

private:
    void OnSetRemoteDescriptionComplete(webrtc::RTCError error) final
    {
        if (!error.ok()) {
            m_endpoint.setRemoteSessionDescriptionFailed(toExceptionCode(error.type()), error.message());
            return;
        }
        m_endpoint.setRemoteSessionDescriptionSucceeded();
    }

    Endpoint& m_endpoint;
};

} // namespace WebCore

#endif // ENABLE(WEB_RTC) && USE(LIBWEBRTC)
