!--------------------------------------------------------------------------------------------------!
!   CP2K: A general program to perform molecular dynamics simulations                              !
!   Copyright 2000-2025 CP2K developers group <https://cp2k.org>                                   !
!                                                                                                  !
!   SPDX-License-Identifier: GPL-2.0-or-later                                                      !
!--------------------------------------------------------------------------------------------------!

! **************************************************************************************************
!> \brief EEQ data from different sources
!> \author JGH
! **************************************************************************************************
MODULE eeq_data
   USE kinds,                           ONLY: dp
#include "./base/base_uses.f90"

   IMPLICIT NONE

   PRIVATE

   CHARACTER(len=*), PARAMETER, PRIVATE :: moduleN = 'eeq_data'

   ! ==========================================================================
   ! 2019-2020 Sebastian Ehlert : EEQ gfn0-xTB parameters
   INTEGER, PARAMETER                                    :: maxElem = 86
   ! COULOMB DATA
   !> Electronegativity for the EEQ model
   REAL(KIND=dp), PARAMETER :: eeqChi(1:maxElem) = [&
      & 1.2500000_dp, 1.2912463_dp, 0.8540050_dp, 1.1723939_dp, 1.1094487_dp, &
      & 1.3860275_dp, 1.5341534_dp, 1.5378836_dp, 1.5890750_dp, 1.2893646_dp, &
      & 0.7891208_dp, 0.9983021_dp, 0.9620847_dp, 1.0441134_dp, 1.4789559_dp, &
      & 1.3926377_dp, 1.4749100_dp, 1.2250415_dp, 0.8162292_dp, 1.1252036_dp, &
      & 0.9641451_dp, 0.8810155_dp, 0.9741986_dp, 1.1029038_dp, 1.0076949_dp, &
      & 0.7744353_dp, 0.7554040_dp, 1.0182630_dp, 1.0316167_dp, 1.6317474_dp, &
      & 1.1186739_dp, 1.0345958_dp, 1.3090772_dp, 1.4119283_dp, 1.4500674_dp, &
      & 1.1746889_dp, 0.6686200_dp, 1.0744648_dp, 0.9107813_dp, 0.7876056_dp, &
      & 1.0039889_dp, 0.9225265_dp, 0.9035515_dp, 1.0332301_dp, 1.0293975_dp, &
      & 1.0549549_dp, 1.2356867_dp, 1.2793315_dp, 1.1145650_dp, 1.1214927_dp, &
      & 1.2123167_dp, 1.4003158_dp, 1.4255511_dp, 1.1640198_dp, 0.4685133_dp, &
      & 1.0687873_dp, 0.9335398_dp, 1.0573550_dp, 1.0532043_dp, 1.0490537_dp, &
      & 1.0449031_dp, 1.0407524_dp, 1.0366018_dp, 1.0324512_dp, 1.0283005_dp, &
      & 1.0241499_dp, 1.0199992_dp, 1.0158486_dp, 1.0116980_dp, 1.0075473_dp, &
      & 1.0033967_dp, 0.8612827_dp, 1.0422031_dp, 0.7633168_dp, 0.6019707_dp, &
      & 0.7499393_dp, 0.9511744_dp, 0.9357472_dp, 1.3555382_dp, 1.2006726_dp, &
      & 1.2092025_dp, 1.1736669_dp, 1.1936584_dp, 1.3045488_dp, 1.1964604_dp, &
      & 1.2653792_dp]
   !> Chemical hardness for the EEQ model
   REAL(KIND=dp), PARAMETER :: eeqGam(1:maxElem) = [&
      &-0.3023159_dp, 0.7743046_dp, 0.5303164_dp, 0.2176474_dp, 0.1956176_dp, &
      & 0.0308461_dp, 0.0559522_dp, 0.0581228_dp, 0.1574017_dp, 0.6825784_dp, &
      & 0.3922376_dp, 0.5581866_dp, 0.3017510_dp, 0.1039137_dp, 0.2124917_dp, &
      & 0.0580720_dp, 0.2537467_dp, 0.5780354_dp, 0.3920658_dp, -0.0024897_dp, &
      &-0.0061520_dp, 0.1663252_dp, 0.1051751_dp, 0.0009900_dp, 0.0976543_dp, &
      & 0.0612028_dp, 0.0561526_dp, 0.0899774_dp, 0.1313171_dp, 0.5728071_dp, &
      & 0.1741615_dp, 0.2671888_dp, 0.2351989_dp, 0.0718104_dp, 0.3458143_dp, &
      & 0.8203265_dp, 0.4287770_dp, 0.2667067_dp, 0.0873658_dp, 0.0599431_dp, &
      & 0.1581972_dp, 0.1716374_dp, 0.2721649_dp, 0.2817608_dp, 0.1391572_dp, &
      & 0.1175925_dp, 0.2316104_dp, 0.2256303_dp, 0.1230459_dp, 0.0141941_dp, &
      & 0.0188612_dp, 0.0230207_dp, 0.3644113_dp, 0.1668461_dp, 0.5167533_dp, &
      & 0.1979578_dp, 0.0345176_dp, 0.0240233_dp, 0.0246333_dp, 0.0252433_dp, &
      & 0.0258532_dp, 0.0264632_dp, 0.0270732_dp, 0.0276832_dp, 0.0282931_dp, &
      & 0.0289031_dp, 0.0295131_dp, 0.0301230_dp, 0.0307330_dp, 0.0313430_dp, &
      & 0.0319529_dp, 0.0262881_dp, 0.1715396_dp, 0.1803633_dp, 0.3631824_dp, &
      & 0.3010980_dp, 0.1100299_dp, 0.0277514_dp, 0.0554975_dp, 0.7723231_dp, &
      & 0.1287718_dp, 0.1034598_dp, 0.0114935_dp, 0.0160842_dp, 0.3369611_dp, &
      & 0.1844179_dp]

   !> Coordination number dependence of the EN in the EEQ model
   REAL(KIND=dp), PARAMETER :: eeqkCN(1:maxElem) = [&
      & 0.0248762_dp, 0.1342276_dp, 0.0103048_dp, -0.0352374_dp, -0.0980031_dp, &
      & 0.0643920_dp, 0.1053273_dp, 0.1394809_dp, 0.1276675_dp, -0.1081936_dp, &
      &-0.0008132_dp, -0.0279860_dp, -0.0521436_dp, -0.0257206_dp, 0.1651461_dp, &
      & 0.0914418_dp, 0.1213634_dp, -0.0636298_dp, -0.0045838_dp, 0.0007509_dp, &
      &-0.0307730_dp, -0.0286150_dp, -0.0341465_dp, -0.0419655_dp, -0.0088536_dp, &
      &-0.1001069_dp, -0.1190502_dp, -0.0726233_dp, -0.0219233_dp, 0.0641913_dp, &
      &-0.0103130_dp, 0.0262628_dp, 0.0222202_dp, 0.0709954_dp, 0.0422244_dp, &
      &-0.0308245_dp, 0.0086249_dp, -0.0237146_dp, -0.0721798_dp, -0.0848810_dp, &
      &-0.0402828_dp, -0.0372396_dp, -0.0027043_dp, 0.0525839_dp, 0.0051192_dp, &
      & 0.0188401_dp, 0.0103998_dp, 0.0000549_dp, 0.0087717_dp, -0.0237228_dp, &
      & 0.0169656_dp, 0.0924186_dp, 0.0352884_dp, -0.0091444_dp, 0.0192916_dp, &
      &-0.0154483_dp, -0.0736833_dp, -0.0064191_dp, -0.0093012_dp, -0.0121833_dp, &
      &-0.0150654_dp, -0.0179475_dp, -0.0208296_dp, -0.0237117_dp, -0.0265938_dp, &
      &-0.0294759_dp, -0.0323580_dp, -0.0352400_dp, -0.0381221_dp, -0.0410042_dp, &
      &-0.0438863_dp, -0.0894776_dp, -0.0333583_dp, -0.0154963_dp, -0.0121092_dp, &
      &-0.0744239_dp, 0.0050138_dp, -0.0153757_dp, -0.0029221_dp, 0.0239125_dp, &
      & 0.0183012_dp, -0.0238011_dp, -0.0268025_dp, 0.0136505_dp, -0.0132199_dp, &
      &-0.0439890_dp]
   !> Charge width in the EEQ model
   REAL(KIND=dp), PARAMETER :: eeqAlp(1:maxElem) = [&
      & 0.7490227_dp, 0.4196569_dp, 1.4256190_dp, 2.0698743_dp, 1.7358798_dp, &
      & 1.8288757_dp, 1.9346081_dp, 1.6974795_dp, 0.8169179_dp, 0.6138441_dp, &
      & 1.7294046_dp, 1.7925036_dp, 1.2156739_dp, 1.5314457_dp, 1.3730859_dp, &
      & 1.7936326_dp, 2.4255996_dp, 1.5891656_dp, 2.1829647_dp, 1.4177623_dp, &
      & 1.5181399_dp, 1.9919805_dp, 1.7171675_dp, 2.0655063_dp, 1.3318009_dp, &
      & 1.3660068_dp, 1.5694128_dp, 1.2762644_dp, 1.0039549_dp, 0.7338863_dp, &
      & 3.2596250_dp, 1.7530299_dp, 1.5281792_dp, 2.1837813_dp, 2.1642027_dp, &
      & 2.7280594_dp, 0.7838049_dp, 1.4274742_dp, 1.8023947_dp, 1.6093288_dp, &
      & 1.3834349_dp, 1.1740977_dp, 1.5768259_dp, 1.3205263_dp, 1.4259466_dp, &
      & 1.1499748_dp, 0.7013009_dp, 1.2374416_dp, 1.3799991_dp, 1.8528424_dp, &
      & 1.8497568_dp, 2.0159294_dp, 1.2903708_dp, 2.0199161_dp, 0.9530522_dp, &
      & 1.5015025_dp, 2.1917012_dp, 1.9134370_dp, 1.9897910_dp, 2.0661450_dp, &
      & 2.1424991_dp, 2.2188531_dp, 2.2952071_dp, 2.3715611_dp, 2.4479151_dp, &
      & 2.5242691_dp, 2.6006231_dp, 2.6769771_dp, 2.7533312_dp, 2.8296852_dp, &
      & 2.9060392_dp, 1.6423047_dp, 1.3567622_dp, 1.8966648_dp, 0.8253100_dp, &
      & 0.7412219_dp, 1.0350883_dp, 0.9692278_dp, 1.0048087_dp, 2.3138674_dp, &
      & 2.8055966_dp, 3.0968677_dp, 1.6597596_dp, 3.2191575_dp, 1.5388148_dp, &
      & 2.1222013_dp]

!  ! covalent radii (taken from Pyykko and Atsumi, Chem. Eur. J. 15, 2009, 188-197)
!  ! values for metals decreased by 10 %
!  REAL(KIND=dp), PARAMETER :: rcov(1:maxElem) = [&
!     & 0.32_dp, 0.46_dp, 1.20_dp, 0.94_dp, 0.77_dp, 0.75_dp, 0.71_dp, 0.63_dp, &
!     & 0.64_dp, 0.67_dp, 1.40_dp, 1.25_dp, 1.13_dp, 1.04_dp, 1.10_dp, 1.02_dp, &
!     & 0.99_dp, 0.96_dp, 1.76_dp, 1.54_dp, 1.33_dp, 1.22_dp, 1.21_dp, 1.10_dp, &
!     & 1.07_dp, 1.04_dp, 1.00_dp, 0.99_dp, 1.01_dp, 1.09_dp, 1.12_dp, 1.09_dp, &
!     & 1.15_dp, 1.10_dp, 1.14_dp, 1.17_dp, 1.89_dp, 1.67_dp, 1.47_dp, 1.39_dp, &
!     & 1.32_dp, 1.24_dp, 1.15_dp, 1.13_dp, 1.13_dp, 1.08_dp, 1.15_dp, 1.23_dp, &
!     & 1.28_dp, 1.26_dp, 1.26_dp, 1.23_dp, 1.32_dp, 1.31_dp, 2.09_dp, 1.76_dp, &
!     & 1.62_dp, 1.47_dp, 1.58_dp, 1.57_dp, 1.56_dp, 1.55_dp, 1.51_dp, 1.52_dp, &
!     & 1.51_dp, 1.50_dp, 1.49_dp, 1.49_dp, 1.48_dp, 1.53_dp, 1.46_dp, 1.37_dp, &
!     & 1.31_dp, 1.23_dp, 1.18_dp, 1.16_dp, 1.11_dp, 1.12_dp, 1.13_dp, 1.32_dp, &
!     & 1.30_dp, 1.30_dp, 1.36_dp, 1.31_dp, 1.38_dp, 1.42_dp]
   ! ==========================================================================
   !> Electronegativity equilibration charge model published in
   !>
   !> E. Caldeweyher, S. Ehlert, A. Hansen, H. Neugebauer, S. Spicher, C. Bannwarth
   !> and S. Grimme, *J. Chem. Phys.*, **2019**, 150, 154122.
   !> DOI: [10.1063/1.5090222](https://dx.doi.org/10.1063/1.5090222)
   !
   !> Maximum atomic number allowed in EEQ calculations
   INTEGER, PARAMETER :: max_elem = 103

   !> Element-specific electronegativity for the electronegativity equilibration charges.
   REAL(dp), PARAMETER :: eeq_chi(max_elem) = [&
      & 1.23695041_dp, 1.26590957_dp, 0.54341808_dp, 0.99666991_dp, 1.26691604_dp, &
      & 1.40028282_dp, 1.55819364_dp, 1.56866440_dp, 1.57540015_dp, 1.15056627_dp, &
      & 0.55936220_dp, 0.72373742_dp, 1.12910844_dp, 1.12306840_dp, 1.52672442_dp, &
      & 1.40768172_dp, 1.48154584_dp, 1.31062963_dp, 0.40374140_dp, 0.75442607_dp, &
      & 0.76482096_dp, 0.98457281_dp, 0.96702598_dp, 1.05266584_dp, 0.93274875_dp, &
      & 1.04025281_dp, 0.92738624_dp, 1.07419210_dp, 1.07900668_dp, 1.04712861_dp, &
      & 1.15018618_dp, 1.15388455_dp, 1.36313743_dp, 1.36485106_dp, 1.39801837_dp, &
      & 1.18695346_dp, 0.36273870_dp, 0.58797255_dp, 0.71961946_dp, 0.96158233_dp, &
      & 0.89585296_dp, 0.81360499_dp, 1.00794665_dp, 0.92613682_dp, 1.09152285_dp, &
      & 1.14907070_dp, 1.13508911_dp, 1.08853785_dp, 1.11005982_dp, 1.12452195_dp, &
      & 1.21642129_dp, 1.36507125_dp, 1.40340000_dp, 1.16653482_dp, 0.34125098_dp, &
      & 0.58884173_dp, 0.68441115_dp, 0.56999999_dp, 0.56999999_dp, 0.56999999_dp, &
      & 0.56999999_dp, 0.56999999_dp, 0.56999999_dp, 0.56999999_dp, 0.56999999_dp, &
      & 0.56999999_dp, 0.56999999_dp, 0.56999999_dp, 0.56999999_dp, 0.56999999_dp, &
      & 0.56999999_dp, 0.87936784_dp, 1.02761808_dp, 0.93297476_dp, 1.10172128_dp, &
      & 0.97350071_dp, 1.16695666_dp, 1.23997927_dp, 1.18464453_dp, 1.14191734_dp, &
      & 1.12334192_dp, 1.01485321_dp, 1.12950808_dp, 1.30804834_dp, 1.33689961_dp, &
      & 1.27465977_dp, 1.06598299_dp, 0.68184178_dp, 1.04581665_dp, 1.09888688_dp, &
      & 1.07206461_dp, 1.09821942_dp, 1.10900303_dp, 1.01039812_dp, 1.00095966_dp, &
      & 1.11003303_dp, 1.16831853_dp, 1.00887482_dp, 1.05928842_dp, 1.07672363_dp, &
      & 1.11308426_dp, 1.14340090_dp, 1.13714110_dp]

   !> Element-specific chemical hardnesses for the electronegativity equilibration charges.
   REAL(dp), PARAMETER :: eeq_eta(max_elem) = [&
      &-0.35015861_dp, 1.04121227_dp, 0.09281243_dp, 0.09412380_dp, 0.26629137_dp, &
      & 0.19408787_dp, 0.05317918_dp, 0.03151644_dp, 0.32275132_dp, 1.30996037_dp, &
      & 0.24206510_dp, 0.04147733_dp, 0.11634126_dp, 0.13155266_dp, 0.15350650_dp, &
      & 0.15250997_dp, 0.17523529_dp, 0.28774450_dp, 0.42937314_dp, 0.01896455_dp, &
      & 0.07179178_dp, -0.01121381_dp, -0.03093370_dp, 0.02716319_dp, -0.01843812_dp, &
      &-0.15270393_dp, -0.09192645_dp, -0.13418723_dp, -0.09861139_dp, 0.18338109_dp, &
      & 0.08299615_dp, 0.11370033_dp, 0.19005278_dp, 0.10980677_dp, 0.12327841_dp, &
      & 0.25345554_dp, 0.58615231_dp, 0.16093861_dp, 0.04548530_dp, -0.02478645_dp, &
      & 0.01909943_dp, 0.01402541_dp, -0.03595279_dp, 0.01137752_dp, -0.03697213_dp, &
      & 0.08009416_dp, 0.02274892_dp, 0.12801822_dp, -0.02078702_dp, 0.05284319_dp, &
      & 0.07581190_dp, 0.09663758_dp, 0.09547417_dp, 0.07803344_dp, 0.64913257_dp, &
      & 0.15348654_dp, 0.05054344_dp, 0.11000000_dp, 0.11000000_dp, 0.11000000_dp, &
      & 0.11000000_dp, 0.11000000_dp, 0.11000000_dp, 0.11000000_dp, 0.11000000_dp, &
      & 0.11000000_dp, 0.11000000_dp, 0.11000000_dp, 0.11000000_dp, 0.11000000_dp, &
      & 0.11000000_dp, -0.02786741_dp, 0.01057858_dp, -0.03892226_dp, -0.04574364_dp, &
      &-0.03874080_dp, -0.03782372_dp, -0.07046855_dp, 0.09546597_dp, 0.21953269_dp, &
      & 0.02522348_dp, 0.15263050_dp, 0.08042611_dp, 0.01878626_dp, 0.08715453_dp, &
      & 0.10500484_dp, 0.10034731_dp, 0.15801991_dp, -0.00071039_dp, -0.00170887_dp, &
      &-0.00133327_dp, -0.00104386_dp, -0.00094936_dp, -0.00111390_dp, -0.00125257_dp, &
      &-0.00095936_dp, -0.00102814_dp, -0.00104450_dp, -0.00112666_dp, -0.00101529_dp, &
      &-0.00059592_dp, -0.00012585_dp, -0.00140896_dp]
   !> Element-specific CN scaling constant for the electronegativity equilibration charges.
   REAL(dp), PARAMETER :: eeq_kcn(max_elem) = [&
      & 0.04916110_dp, 0.10937243_dp, -0.12349591_dp, -0.02665108_dp, -0.02631658_dp, &
      & 0.06005196_dp, 0.09279548_dp, 0.11689703_dp, 0.15704746_dp, 0.07987901_dp, &
      &-0.10002962_dp, -0.07712863_dp, -0.02170561_dp, -0.04964052_dp, 0.14250599_dp, &
      & 0.07126660_dp, 0.13682750_dp, 0.14877121_dp, -0.10219289_dp, -0.08979338_dp, &
      &-0.08273597_dp, -0.01754829_dp, -0.02765460_dp, -0.02558926_dp, -0.08010286_dp, &
      &-0.04163215_dp, -0.09369631_dp, -0.03774117_dp, -0.05759708_dp, 0.02431998_dp, &
      &-0.01056270_dp, -0.02692862_dp, 0.07657769_dp, 0.06561608_dp, 0.08006749_dp, &
      & 0.14139200_dp, -0.05351029_dp, -0.06701705_dp, -0.07377246_dp, -0.02927768_dp, &
      &-0.03867291_dp, -0.06929825_dp, -0.04485293_dp, -0.04800824_dp, -0.01484022_dp, &
      & 0.07917502_dp, 0.06619243_dp, 0.02434095_dp, -0.01505548_dp, -0.03030768_dp, &
      & 0.01418235_dp, 0.08953411_dp, 0.08967527_dp, 0.07277771_dp, -0.02129476_dp, &
      &-0.06188828_dp, -0.06568203_dp, -0.11000000_dp, -0.11000000_dp, -0.11000000_dp, &
      &-0.11000000_dp, -0.11000000_dp, -0.11000000_dp, -0.11000000_dp, -0.11000000_dp, &
      &-0.11000000_dp, -0.11000000_dp, -0.11000000_dp, -0.11000000_dp, -0.11000000_dp, &
      &-0.11000000_dp, -0.03585873_dp, -0.03132400_dp, -0.05902379_dp, -0.02827592_dp, &
      &-0.07606260_dp, -0.02123839_dp, 0.03814822_dp, 0.02146834_dp, 0.01580538_dp, &
      &-0.00894298_dp, -0.05864876_dp, -0.01817842_dp, 0.07721851_dp, 0.07936083_dp, &
      & 0.05849285_dp, 0.00013506_dp, -0.00020631_dp, 0.00473118_dp, 0.01590519_dp, &
      & 0.00369763_dp, 0.00417543_dp, 0.00706682_dp, 0.00488679_dp, 0.00505103_dp, &
      & 0.00710682_dp, 0.00463050_dp, 0.00387799_dp, 0.00296795_dp, 0.00400648_dp, &
      & 0.00548481_dp, 0.01350400_dp, 0.00675380_dp]
   !> Element-specific charge widths for the electronegativity equilibration charges.
   REAL(dp), PARAMETER :: eeq_rad(max_elem) = [&
      & 0.55159092_dp, 0.66205886_dp, 0.90529132_dp, 1.51710827_dp, 2.86070364_dp, &
      & 1.88862966_dp, 1.32250290_dp, 1.23166285_dp, 1.77503721_dp, 1.11955204_dp, &
      & 1.28263182_dp, 1.22344336_dp, 1.70936266_dp, 1.54075036_dp, 1.38200579_dp, &
      & 2.18849322_dp, 1.36779065_dp, 1.27039703_dp, 1.64466502_dp, 1.58859404_dp, &
      & 1.65357953_dp, 1.50021521_dp, 1.30104175_dp, 1.46301827_dp, 1.32928147_dp, &
      & 1.02766713_dp, 1.02291377_dp, 0.94343886_dp, 1.14881311_dp, 1.47080755_dp, &
      & 1.76901636_dp, 1.98724061_dp, 2.41244711_dp, 2.26739524_dp, 2.95378999_dp, &
      & 1.20807752_dp, 1.65941046_dp, 1.62733880_dp, 1.61344972_dp, 1.63220728_dp, &
      & 1.60899928_dp, 1.43501286_dp, 1.54559205_dp, 1.32663678_dp, 1.37644152_dp, &
      & 1.36051851_dp, 1.23395526_dp, 1.65734544_dp, 1.53895240_dp, 1.97542736_dp, &
      & 1.97636542_dp, 2.05432381_dp, 3.80138135_dp, 1.43893803_dp, 1.75505957_dp, &
      & 1.59815118_dp, 1.76401732_dp, 1.63999999_dp, 1.63999999_dp, 1.63999999_dp, &
      & 1.63999999_dp, 1.63999999_dp, 1.63999999_dp, 1.63999999_dp, 1.63999999_dp, &
      & 1.63999999_dp, 1.63999999_dp, 1.63999999_dp, 1.63999999_dp, 1.63999999_dp, &
      & 1.63999999_dp, 1.47055223_dp, 1.81127084_dp, 1.40189963_dp, 1.54015481_dp, &
      & 1.33721475_dp, 1.57165422_dp, 1.04815857_dp, 1.78342098_dp, 2.79106396_dp, &
      & 1.78160840_dp, 2.47588882_dp, 2.37670734_dp, 1.76613217_dp, 2.66172302_dp, &
      & 2.82773085_dp, 1.04059593_dp, 0.60550051_dp, 1.22262145_dp, 1.28736399_dp, &
      & 1.44431317_dp, 1.29032833_dp, 1.41009404_dp, 1.25501213_dp, 1.15181468_dp, &
      & 1.42010424_dp, 1.43955530_dp, 1.28565237_dp, 1.35017463_dp, 1.33011749_dp, &
      & 1.30745135_dp, 1.26526071_dp, 1.34071499_dp]

   PUBLIC :: get_eeq_data

CONTAINS

! **************************************************************************************************
!> \brief ...
!> \param za ...
!> \param model ...
!> \param chi ...
!> \param eta ...
!> \param kcn ...
!> \param rad ...
! **************************************************************************************************
   SUBROUTINE get_eeq_data(za, model, chi, eta, kcn, rad)

      INTEGER, INTENT(IN)                                :: za, model
      REAL(kind=dp), INTENT(OUT), OPTIONAL               :: chi, eta, kcn, rad

      IF (model == 1) THEN
         CPASSERT(za <= maxElem)
         IF (PRESENT(chi)) chi = eeqChi(za)
         IF (PRESENT(eta)) eta = eeqGam(za)
         IF (PRESENT(kcn)) kcn = eeqkCN(za)
         IF (PRESENT(rad)) rad = eeqAlp(za)
      ELSEIF (model == 2) THEN
         CPASSERT(za <= max_elem)
         IF (PRESENT(chi)) chi = eeq_chi(za)
         IF (PRESENT(eta)) eta = eeq_eta(za)
         IF (PRESENT(kcn)) kcn = eeq_kcn(za)
         IF (PRESENT(rad)) rad = eeq_rad(za)
      ELSE
         CPABORT("get_eeq_data: unknown model")
      END IF

   END SUBROUTINE get_eeq_data

END MODULE eeq_data
