// RUN: mlir-opt -split-input-file -test-liveness-analysis %s 2>&1 | FileCheck %s

// Positive test: Type (1.a) "is an operand of an op with memory effects"
// zero is live because it is stored in memory.
// CHECK-LABEL: test_tag: zero:
// CHECK-NEXT:  result #0: live
func.func @test_1_type_1.a(%arg0: memref<i32>) {
  %c0_i32 = arith.constant {tag = "zero"} 0 : i32
  memref.store %c0_i32, %arg0[] : memref<i32>
  return
}

// -----

// Positive test: Type (1.b) "is a non-forwarded branch operand and a block
// where its op could take the control has an op with memory effects"
// %arg2 is live because it can make the control go into a block with a memory
// effecting op.
// CHECK-LABEL: test_tag: br:
// CHECK-NEXT:  operand #0: live
// CHECK-NEXT:  operand #1: live
// CHECK-NEXT:  operand #2: live
func.func @test_2_RegionBranchOpInterface_type_1.b(%arg0: memref<i32>, %arg1: memref<i32>, %arg2: i1) {
  %c0_i32 = arith.constant 0 : i32
  cf.cond_br %arg2, ^bb1(%c0_i32 : i32), ^bb2(%c0_i32 : i32) {tag = "br"}
^bb1(%0 : i32):
  memref.store %0, %arg0[] : memref<i32>
  cf.br ^bb3
^bb2(%1 : i32):
  memref.store %1, %arg1[] : memref<i32>
  cf.br ^bb3
^bb3:
  return
}

// -----

// Positive test: Type (1.b) "is a non-forwarded branch operand and a block
// where its op could take the control has an op with memory effects"
// %arg0 is live because it can make the control go into a block with a memory
// effecting op.
// CHECK-LABEL: test_tag: flag:
// CHECK-NEXT:  operand #0: live
func.func @test_3_BranchOpInterface_type_1.b(%arg0: i32, %arg1: memref<i32>, %arg2: memref<i32>) {
  %c0_i32 = arith.constant 0 : i32
  cf.switch %arg0 : i32, [
    default: ^bb1,
    42: ^bb2
  ] {tag = "flag"}
^bb1:
  memref.store %c0_i32, %arg1[] : memref<i32>
  cf.br ^bb3
^bb2:
  memref.store %c0_i32, %arg2[] : memref<i32>
  cf.br ^bb3
^bb3:
  return
}

// -----

// Positive test: Type(1.c) "is a non-forwarded branch operand and its branch
// op could result in different result"
// CHECK-LABEL: test_tag: cond_br:
// CHECK-NEXT:  operand #0: live
// CHECK-NEXT:  operand #1: live
// CHECK-NEXT:  operand #2: live
func.func @test_branch_result_in_different_result_1.c(%arg0 : tensor<f32>, %arg1 : tensor<f32>, %arg2 : i1) -> tensor<f32> {
  cf.cond_br %arg2, ^bb1(%arg0 : tensor<f32>), ^bb2(%arg1 : tensor<f32>) {tag = "cond_br"}
^bb1(%0 : tensor<f32>):
  cf.br ^bb3(%0 : tensor<f32>)
^bb2(%1 : tensor<f32>):
  cf.br ^bb3(%1 : tensor<f32>)
^bb3(%2 : tensor<f32>):
  return %2 : tensor<f32>
}

// -----

// Positive test: Type(1.c) "is a non-forwarded branch operand and its branch
// op could result in different result"
// CHECK-LABEL: test_tag: region_branch:
// CHECK-NEXT:  operand #0: live
func.func @test_region_branch_result_in_different_result_1.c(%arg0 : tensor<f32>, %arg1 : tensor<f32>, %arg2 : i1) -> tensor<f32> {
  %0 = scf.if %arg2 -> tensor<f32> {
    scf.yield %arg0 : tensor<f32>
  } else {
    scf.yield %arg1 : tensor<f32>
  } {tag="region_branch"}
  return %0 : tensor<f32>
}

// -----

func.func private @private(%arg0 : i32, %arg1 : i32) {
  func.return
}

// Positive test: Type (1.d) "is a non-forwarded call operand"
// CHECK-LABEL: test_tag: call
// CHECK-LABEL:  operand #0: not live
// CHECK-LABEL:  operand #1: not live
// CHECK-LABEL:  operand #2: live
func.func @test_4_type_1.d(%arg0: i32, %arg1: i32, %device: i32, %m0: memref<i32>) {
  test.call_on_device @private(%arg0, %arg1), %device {tag = "call"} : (i32, i32, i32) -> ()
  return
}

// -----

// Positive test: Type (2) "is returned by a public function"
// zero is live because it is returned by a public function.
// CHECK-LABEL: test_tag: zero:
// CHECK-NEXT:  result #0: live
func.func @test_5_type_2() -> (f32){
  %0 = arith.constant {tag = "zero"} 0.0 : f32
  return %0 : f32
}

// -----

// Positive test: Type (3) "is used to compute a value of type (1) or (2)"
// %arg1 is live because the scf.while has a live result and %arg1 is a
// non-forwarded branch operand.
// %arg2 is live because it is forwarded to the live result of the scf.while
// op.
// %arg5 is live because it is forwarded to %arg8 which is live.
// %arg8 is live because it is forwarded to %arg4 which is live as it writes
// to memory.
// Negative test:
// %arg3 is not live even though %arg1, %arg2, and %arg5 are live because it
// is neither a non-forwarded branch operand nor a forwarded operand that
// forwards to a live value. It actually is a forwarded operand that forwards
// to non-live values %0#1 and %arg7.
// CHECK-LABEL: test_tag: condition:
// CHECK-NEXT:  operand #0: live
// CHECK-NEXT:  operand #1: live
// CHECK-NEXT:  operand #2: not live
// CHECK-NEXT:  operand #3: live
// CHECK-LABEL: test_tag: add:
// CHECK-NEXT:  operand #0: live
func.func @test_6_RegionBranchTerminatorOpInterface_type_3(%arg0: memref<i32>, %arg1: i1) -> (i32) {
  %c0_i32 = arith.constant 0 : i32
  %c1_i32 = arith.constant 1 : i32
  %c2_i32 = arith.constant 2 : i32
  %0:3 = scf.while (%arg2 = %c0_i32, %arg3 = %c1_i32, %arg4 = %c2_i32, %arg5 = %c2_i32) : (i32, i32, i32, i32) -> (i32, i32, i32) {
    memref.store %arg4, %arg0[] : memref<i32>
    scf.condition(%arg1) {tag = "condition"} %arg2, %arg3, %arg5 : i32, i32, i32
  } do {
  ^bb0(%arg6: i32, %arg7: i32, %arg8: i32):
    %1 = arith.addi %arg8, %arg8 {tag = "add"} : i32
    %c3_i32 = arith.constant 3 : i32
    scf.yield %arg6, %arg7, %arg8, %c3_i32 : i32, i32, i32, i32
  }
  return %0#0 : i32
}

// -----

func.func private @private0(%0 : i32) -> i32 {
  %1 = arith.addi %0, %0 {tag = "in_private0"} : i32
  func.return %1 : i32
}

// Positive test: Type (3) "is used to compute a value of type (1) or (2)"
// zero, ten, and one are live because they are used to decide the number of
// times the `for` loop executes, which in turn decides the value stored in
// memory.
// in_private0 and x are also live because they decide the value stored in
// memory.
// Negative test:
// y is not live even though the non-forwarded branch operand and x are live.
// CHECK-LABEL: test_tag: in_private0:
// CHECK-NEXT:  operand #0: live
// CHECK-NEXT:  operand #1: live
// CHECK-NEXT:  result #0: live
// CHECK-LABEL: test_tag: zero:
// CHECK-NEXT:  result #0: live
// CHECK-LABEL: test_tag: ten:
// CHECK-NEXT:  result #0: live
// CHECK-LABEL: test_tag: one:
// CHECK-NEXT:  result #0: live
// CHECK-LABEL: test_tag: x:
// CHECK-NEXT:  result #0: live
// CHECK-LABEL: test_tag: y:
// CHECK-NEXT:  result #0: not live
// CHECK-LABEL: test_tag: for:
// CHECK-NEXT:  operand #0: live
// CHECK-NEXT:  operand #1: live
// CHECK-NEXT:  operand #2: live
// CHECK-NEXT:  operand #3: live
// CHECK-NEXT:  operand #4: not live
// CHECK-NEXT:  result #0: live
// CHECK-NEXT:  result #1: not live
// CHECK-NEXT:  region: #0:
// CHECK-NEXT:  argument: #0: live
// CHECK-NEXT:  argument: #1: not live
// CHECK-NEXT:  argument: #2: not live
func.func @test_7_type_3(%arg0: memref<i32>) {
  %c0 = arith.constant {tag = "zero"} 0 : index
  %c10 = arith.constant {tag = "ten"} 10 : index
  %c1 = arith.constant {tag = "one"} 1 : index
  %x = arith.constant {tag = "x"} 0 : i32
  %y = arith.constant {tag = "y"} 1 : i32
  %0:2 = scf.for %arg1 = %c0 to %c10 step %c1 iter_args(%arg2 = %x, %arg3 = %y) -> (i32, i32) {
    %1 = arith.addi %x, %x : i32
    %2 = func.call @private0(%1) : (i32) -> i32
    scf.yield %2, %arg3 : i32, i32
  } {tag = "for"}
  memref.store %0#0, %arg0[] : memref<i32>
  return
}

// -----

func.func private @private1(%0 : i32) -> i32 {
  %1 = func.call @private2(%0) : (i32) -> i32
  %2 = arith.muli %0, %1 {tag = "in_private1"} : i32
  func.return %2 : i32
}

func.func private @private2(%0 : i32) -> i32 {
  %cond = arith.index_cast %0 {tag = "in_private2"} : i32 to index
  %1 = scf.index_switch %cond -> i32
  case 1 {
    %ten = arith.constant 10 : i32
    scf.yield %ten : i32
  }
  case 2 {
    %twenty = arith.constant 20 : i32
    scf.yield %twenty : i32
  }
  default {
    %thirty = arith.constant 30 : i32
    scf.yield %thirty : i32
  }
  func.return %1 : i32
}

// Positive test: Type (3) "is used to compute a value of type (1) or (2)"
// in_private1, in_private2, and final are live because they are used to compute
// the value returned by this public function.
// CHECK-LABEL: test_tag: in_private1:
// CHECK-NEXT:  operand #0: live
// CHECK-NEXT:  operand #1: live
// CHECK-NEXT:  result #0: live
// CHECK-LABEL: test_tag: in_private2:
// CHECK-NEXT:  operand #0: live
// CHECK-NEXT:  result #0: live
// CHECK-LABEL: test_tag: final:
// CHECK-NEXT:  operand #0: live
// CHECK-NEXT:  operand #1: live
// CHECK-NEXT:  result #0: live
func.func @test_8_type_3(%arg: i32) -> (i32) {
  %0 = func.call @private1(%arg) : (i32) -> i32
  %final = arith.muli %0, %arg {tag = "final"} : i32
  return %final : i32
}

// -----

// Negative test: None of the types (1), (2), or (3)
// zero is not live because it has no effect outside the program: it doesn't
// affect the memory or the program output.
// CHECK-LABEL: test_tag: zero:
// CHECK-NEXT:  result #0: not live
// CHECK-LABEL: test_tag: one:
// CHECK-NEXT:  result #0: live
func.func @test_9_negative() -> (f32){
  %0 = arith.constant {tag = "zero"} 0.0 : f32
  %1 = arith.constant {tag = "one"} 1.0 : f32
  return %1 : f32
}

// -----

// Negative test: None of the types (1), (2), or (3)
// %1 is not live because it has no effect outside the program: it doesn't
// affect the memory or the program output. Even though it is returned by the
// function `@private_1`, it is never used by the caller.
// Note that this test clearly shows how this liveness analysis utility differs
// from the existing liveness utility present at
// llvm-project/mlir/include/mlir/Analysis/Liveness.h. The latter marks %1 as
// live as it exists the block of function `@private_1`, simply because it is
// computed inside and returned by the block, irrespective of whether or not it
// is used by the caller.
// CHECK-LABEL: test_tag: one:
// CHECK:  result #0: not live
func.func private @private_1() -> (i32, i32) {
  %0 = arith.constant 0 : i32
  %1 = arith.addi %0, %0 {tag = "one"} : i32
  return %0, %1 : i32, i32
}
func.func @test_10_negative() -> (i32) {
  %0:2 = func.call @private_1() : () -> (i32, i32)
  return %0#0 : i32
}

// -----

// Test that we correctly set a liveness value for operations in dead block.
// These won't be visited by the dataflow framework so the analysis need to
// explicitly manage them.
// CHECK-LABEL: test_tag: dead_block_cmpi:
// CHECK-NEXT: operand #0: not live
// CHECK-NEXT: operand #1: not live
// CHECK-NEXT: result #0: not live
func.func @dead_block() {
  %false = arith.constant false
  %zero = arith.constant 0 : i64
  cf.cond_br %false, ^bb1, ^bb4
  ^bb1:
    %3 = arith.cmpi eq, %zero, %zero  {tag = "dead_block_cmpi"} : i64
    cf.br ^bb1
  ^bb4:
    return
}

// -----

// CHECK-LABEL: test_tag: for:
// CHECK-NEXT: region: #0:
// CHECK-NEXT:   argument: #0: live
func.func @affine_loop_no_use_iv_has_side_effect_op() {
  %c1 = arith.constant 1 : index
  %alloc = memref.alloc() : memref<10xindex>
  affine.for %arg0 = 0 to 79 {
    memref.store %c1, %alloc[%c1] : memref<10xindex>
  } {tag = "for"}
  return
}

// -----

// CHECK-LABEL: test_tag: for:
// CHECK-NEXT: region: #0:
// CHECK-NEXT:   argument: #0: not live
func.func @affine_loop_no_use_iv() {
  affine.for %arg0 = 0 to 79 {
  } {tag = "for"}
  return
}

// -----

// CHECK-LABEL: test_tag: forall:
// CHECK-NEXT: operand #0: live
// CHECK-NEXT: region: #0:
// CHECK-NEXT:   argument: #0: live

func.func @forall_no_use_iv_has_side_effect_op(%idx1: index, %idx2: index) {
  scf.parallel (%i) = (%idx1) to (%idx2) step (%idx2) {
    %r = memref.alloca() : memref<10xf32>
    %cst = arith.constant 0.0 : f32
    scf.forall (%e2) in (%idx2) {
      memref.store %cst, %r[%idx2] : memref<10xf32>
    } {tag = "forall"}
  }
  return
}

// -----

// CHECK-LABEL: test_tag: for:
// CHECK-NEXT:   operand #0: not live
// CHECK-NEXT:   operand #1: not live
// CHECK-NEXT:   operand #2: not live
// CHECK-NEXT:   operand #3: not live

func.func @test_for_loop_read_only(%arg0: memref<10xindex>) {
  %c0 = arith.constant 0 : index
  %c10 = arith.constant 10 : index
  %c1 = arith.constant 1 : index
  %0 = scf.for %iv = %c0 to %c10 step %c1 iter_args(%idx = %c0) -> (index) {
    %loaded = memref.load %arg0[%idx] : memref<10xindex>
    scf.yield %loaded : index
  } {tag = "for"}
  return
}
