#' Override window order and frame
#'
#' These allow you to override the `PARTITION BY` and `ORDER BY` clauses
#' of window functions generated by grouped mutates.
#'
#' @inheritParams arrange.tbl_lazy
#' @param ... Variables to order by
#' @export
#' @examples
#' library(dplyr, warn.conflicts = FALSE)
#'
#' db <- memdb_frame(g = rep(1:2, each = 5), y = runif(10), z = 1:10)
#' db %>%
#'   window_order(y) %>%
#'   mutate(z = cumsum(y)) %>%
#'   show_query()
#'
#' db %>%
#'   group_by(g) %>%
#'   window_frame(-3, 0) %>%
#'   window_order(z) %>%
#'   mutate(z = sum(y)) %>%
#'   show_query()
window_order <- function(.data, ...) {
  if (!is_tbl_lazy(.data)) {
    msg <- "{.arg .data} must be a {.cls tbl_lazy}, not a {.cls {class(.data)}}."
    if (is.data.frame(.data)) {
      msg <- c(msg, i = "Did you mean to use {.fn arrange} instead?")
    }
    cli_abort(msg)
  }
  dots <- partial_eval_dots(.data, ..., .named = FALSE)
  names(dots) <- NULL

  .data$lazy_query <- add_order(.data, dots)
  .data
}

# We want to preserve this ordering (for window functions) without
# imposing an additional arrange, so we have a special op_order
add_order <- function(.data, dots) {
  .data$lazy_query$order_vars <- dots
  .data$lazy_query
}


# Frame -------------------------------------------------------------------

#' @export
#' @rdname window_order
#' @param from,to Bounds of the frame.
window_frame <- function(.data, from = -Inf, to = Inf) {
  if (!is_tbl_lazy(.data)) {
    cli_abort(
      "{.arg .data} must be a {.cls tbl_lazy}, not a {.cls {class(.data)}}."
    )
  }

  stopifnot(is.numeric(from), length(from) == 1)
  stopifnot(is.numeric(to), length(to) == 1)

  .data$lazy_query$frame <- list(range = c(from, to))
  .data
}
