// re2v $INPUT -o $OUTPUT

import arrays
import os
import strings

const bufsize = 4096
const tag_none = -1

struct State {
    file     os.File
mut:
    yyinput  []u8
    yycursor int
    yymarker int
    yylimit  int
    token    int
    // Intermediate tag variables must be part of the lexer state passed to YYFILL.
    // They don't correspond to tags and should be autogenerated by re2c.
    /*!stags:re2c format = "\t@@ int\n"; */
    eof      bool
}

struct SemVer {
    major int
    minor int
    patch int
}

fn s2n(s []u8) int { // convert pre-parsed string to number
    mut n := 0
    for c in s { n = n * 10 + int(c - 48) }
    return n
}

fn fill(mut st &State) int {
    if st.eof { return -1 } // unexpected EOF

    // Error: lexeme too long. In real life can reallocate a larger buffer.
    if st.token < 1 { return -2 }

    // Shift buffer contents (discard everything up to the current token).
    copy(mut &st.yyinput, st.yyinput[st.token..st.yylimit])
    st.yycursor -= st.token
    st.yymarker -= st.token
    st.yylimit -= st.token
    // Tag variables need to be shifted like other input positions. The check
    // for -1 is only needed if some tags are nested inside of alternative or
    // repetition, so that they can have -1 value.
    /*!stags:re2c format = "\tif st.@@ != -1 { st.@@ -= st.token }\n"; */
    st.token = 0

    // Fill free space at the end of buffer with new data from file.
    pos := st.file.tell() or { 0 }
    if n := st.file.read_bytes_into(u64(pos), mut st.yyinput[st.yylimit..bufsize]) {
        st.yylimit += n
    }
    st.yyinput[st.yylimit] = 0 // append sentinel symbol

    // If read less than expected, this is the end of input.
    st.eof = st.yylimit < bufsize

    return 0
}

fn parse(mut st &State) ?[]SemVer {
    // Final tag variables available in semantic action.
    /*!svars:re2c format = "mut @@ := tag_none\n"; */

    mut vers := []SemVer{}
loop:
    st.token = st.yycursor
    /*!re2c
        re2c:api = record;
        re2c:yyrecord = st;
        re2c:YYFILL = "fill(mut st) == 0";
        re2c:tags = 1;
        re2c:eof = 0;

        num = [0-9]+;

        num @t1 "." @t2 num @t3 ("." @t4 num)? [\n] {
            ver := SemVer {
                major: s2n(st.yyinput[st.token..t1]),
                minor: s2n(st.yyinput[t2..t3]),
                patch: if t4 == -1 { 0 } else { s2n(st.yyinput[t4..st.yycursor - 1]) }
            }
            vers = arrays.concat(vers, ver)
            unsafe { goto loop }
        }
        $ { return vers }
        * { return none }
    */
}

fn main() {
    fname := "input"
    content := "1.22.333\n";

    // Prepare input file: a few times the size of the buffer, containing
    // strings with zeroes and escaped quotes.
    mut fw := os.create(fname)!
    fw.write_string(strings.repeat_string(content, bufsize))!
    fw.close()

    // Prepare lexer state: all offsets are at the end of buffer.
    mut fr := os.open(fname)!
    mut st := &State{
        file:      fr,
        // Sentinel at `yylimit` offset is set to zero, which triggers YYFILL.
        yyinput:  []u8{len: bufsize + 1},
        yycursor: bufsize,
        yymarker: bufsize,
        yylimit:  bufsize,
        token:    bufsize,
        eof:      false,
    }

    // Run the lexer.
    expect := []SemVer{len: bufsize, init: SemVer{1, 22, 333}}
    result := parse(mut st) or { panic("parse failed") }
    if result != expect { panic("error") }

    // Cleanup: remove input file.
    fr.close()
    os.rm(fname)!
}
