use ui;
use core:io;
use core:geometry;
use graphics;

/**
 * Graphics object that outputs to a PDF file.
 *
 * Call 'nextPage' to start drawing on a new page.
 *
 * Note: Some PDF readers (for example Evince and GIMP) apparently do not support tiling patterns at
 * all. These just display as a semi-transparent average of the color of the pattern rather than the
 * pattern itself. This seems to be an issue with the PDF reader itself rather than with our output,
 * since the problem is not solved by letting Ghostscript re-compile the PDF file.
 */
class PdfGraphics extends Graphics {
	// Create, specifying dimensions. 'physical' is the page's physical dimension in millimeters,
	// and 'virtual' is the virtual dimensions in what corresponds to pixels on other output
	// devices.
	init(Size physical, Size virtual) {
		File pdf;
		Page p(pdf, Rect(Point(), toPt(physical)));

		// Compute the virtual to physical transform:

		// Flip the Y-axis.
		var tfm = scale(Size(1, -1));

		// Move the pixels back onto the screen.
		tfm *= translate(Size(0, virtual.h));

		// Convert to physical coordinates.
		tfm *= scale(Size(physical.w / virtual.w, physical.h / virtual.h));

		// Convert to points.
		tfm *= scale(2.83464567);

		init() {
			pdf = pdf;
			res = pdf.root.resources;
			current = p;
			physical = physical;
			virtual = virtual;
			physicalTfm = tfm;
		}


		preparePage();
	}

	// Save the finished file to a stream.
	void save(OStream to) {
		finishPage();
		pdf.write(to);

		// Open a new page, so that we behave in a sane manner if someone continues outputting data after saving.
		nextPage();
	}

	// Save the finished file to a file.
	void save(Url to) {
		var s = to.write();
		save(s);
		s.close();
	}

	// Start drawing to a new page.
	void nextPage() {
		finishPage();

		current = Page(pdf, Rect(Point(), toPt(physical)));
		pageFinished = false;
		preparePage();
	}

	// Push the current state on the state stack.
	void push() {
		// We have two states on the PDF stack for each state on our stack to make opacity and clip
		// path work together with the semantics of the 'transform' function.
		output << "q q\n";
		stack.push(State(stack.last));
	}

	// Push the current state with a modified opacity applied to the new state.
	void push(Float opacity) {
		createForm(opacity);
		output << "q\n";
	}

	// Push the current state with a clipping rectangle applied to the new state.
	void push(Rect clip) {
		// Save the old state, then inject the clipping rect and save the state once
		// more. Otherwise, we would lose the clipping rect when 'transform' is called.
		output << "q ";
		put(clip.p0, "");
		put(clip.size, "re W n q\n");

		stack.push(State(stack.last));
	}

	// Push, set a clip and opacity.
	void push(Rect clip, Float opacity) {
		createForm(opacity);

		put(clip.p0, "");
		put(clip.size, "re W n q\n");
	}

	// Pop the previous state.
	Bool pop() {
		if (stack.count <= 1)
			return false;

		State s = stack.last;
		stack.pop();

		if (form = s.form) {
			// Pop the last element of the stack in the XObject.
			form.data << "Q\n";

			// Draw the XObject. They have a bounding box of the entire page, so we reset the
			// transform accordingly.
			output << "q " << res.alpha(s.formAlpha) << " gs\n";
			put(stack.last.tfm.inverted, "cm\n");
			output << form.name << " Do Q\n";
		} else {
			// Pop the regular stack.
			output << "Q Q\n";
		}

		true;
	}

	// Set the transform (in relation to the previous state).
	assign transform(Transform tfm) {
		// Since we're expected to *set* the state, first restore the previous state by popping the
		// state stack and pushing again to keep a copy of the original.
		output << "Q q\n";
		put(tfm, "cm\n");

		// If nothing is saved, use the identity transform.
		if (stack.count > 1)
			stack.last.tfm = tfm * stack[stack.count - 2].tfm;
		else
			stack.last.tfm = tfm;
	}

	// Set the line width (in relation to the previous state).
	assign lineWidth(Float width) {
		// TODO: This does not remember the line width in relation to previous states.
		output << width << " w\n";
	}

	// Draw a line.
	void line(Point a, Point b, Brush brush) {
		putColor(brush, true);
		put(a, "m ");
		put(b, "l S\n");
	}

	// Draw a rectangle.
	void draw(Rect r, Brush brush) {
		putColor(brush, true);
		put(r.p0, "");
		put(r.size, "re S\n");
	}

	// Draw a rounded rectangle.
	void draw(Rect r, Size edges, Brush brush) {
		putColor(brush, true);
		putRounded(r, edges, "S\n");
	}

	// Draw a path.
	void draw(Path p, Brush brush) {
		putColor(brush, true);
		put(p, "S\n");
	}

	// Draw an oval.
	void oval(Rect r, Brush brush) {
		putColor(brush, true);
		putOval(r, "S\n");
	}

	// Fill the entire page.
	void fill(Brush brush) {
		output << "q ";
		Transform old = stack.last.tfm;
		put(old.inverted, "cm\n");
		// Note: This seems to be a bit off for some reason...
		stack.last.tfm = Transform();

		fill(Rect(Point(), virtual), brush);
		stack.last.tfm = old;
		output << "Q\n";
	}

	// Fill a rectangle.
	void fill(Rect r, Brush brush) {
		putColor(brush, false);
		put(r.p0, "");
		put(r.size, "re f\n");
	}

	// Fill a rounded rectangle.
	void fill(Rect r, Size edges, Brush brush) {
		putColor(brush, false);
		putRounded(r, edges, "f\n");
	}

	// Draw a path.
	void fill(Path p, Brush brush) {
		putColor(brush, false);
		put(p, "f\n");
	}

	// Fill an oval.
	void fillOval(Rect r, Brush brush) {
		putColor(brush, false);
		putOval(r, "f\n");
	}

	// Draw a bitmap.
	void draw(Bitmap bitmap, Rect rect, Float opacity) {
		output << "q " << res.alpha(opacity) << " gs ";
		// Make the picture right side up and at the origin.
		var tfm =  scale(Size(1, -1)) * translate(Point(0, 1));

		// Move it from the unit square to its proper location.
		tfm *= scale(rect.size) * translate(rect.p0);
		put(tfm, "cm\n");
		output << res.bitmap(pdf, bitmap) << " Do Q\n";
	}

	// Draw part of a bitmap.
	void draw(Bitmap bitmap, Rect src, Rect dest, Float opacity) {
		output << "q " << res.alpha(opacity) << " gs ";
		// Add a new clip region.
		put(dest.p0, "");
		put(dest.size, "re W n\n");

		// Make the picture right side up and at the origin.
		var tfm = scale(Size(1, -1)) * translate(Point(0, 1));

		// Scale it so that the part we want is in the unit square.
		Size sz = bitmap.size;
		tfm *= translate(Point(-src.p0.x / sz.w, -src.p0.y / sz.h));
		tfm *= scale(Size(sz.w / src.size.w, sz.h / src.size.h));

		// Move it to its proper location.
		tfm *= scale(dest.size) * translate(dest.p0);
		put(tfm, "cm\n");

		output << res.bitmap(pdf, bitmap) << " Do Q\n";
	}

	// Draw text.
	void text(Str text, Font font, Brush brush, Rect rect) {
		Text t(text, font);

		// TODO: How do we handle clip rectangles etc? DX doesn't clip or line-wrap when using this
		// mode. Maybe Pango does?
		// t.layoutBorder = rect.size;

		draw(t, brush, rect.p0);
	}

	// Draw text.
	void draw(Text text, Brush brush, Point origin) {
		// TODO: We probably want to have some fancy text effects down the line, such as stroking the outline etc.
		putColor(brush, false);

		Font font = text.font;
		Str f = res.font(font);
		output << "BT\n";
		// Set font.
		output << f << " " << font.pxHeight << " Tf\n";
		// Transform the coordinate system so the text is not upside down. Also translate so that
		// the top-left of the layout box is at 0, 0.
		output << "1 0 0 -1 " << origin.x << " " << origin.y << " Tm\n";

		Float lastStart = 0;
		for (line in text.lineInfo) {
			output << "0 " << (lastStart - line.baseline) << " Td\n";
			lastStart = line.baseline;

			// TODO: We should handle unicode somehow. It seems like we have to generate our own
			// encoding for that to work properly... At the moment, we blindly hope that ANSI
			// encoding is sufficient (which in general, it isn't).
			output << "<";
			for (c in line.text) {
				output << hex(toANSI(c));
			}
			output << ">Tj\n";
		}

		output << "ET\n";
	}

private:
	// Output to.
	File pdf;

	// Resources in the pdf.
	Resources res;

	// Current page.
	Page current;

	// Current page finished?
	Bool pageFinished;

	// Dimensions, physical and virtual.
	Size physical;
	Size virtual;

	// Transformation from the virtual coordinate space into the physical coordinate space.
	Transform physicalTfm;

	/**
	 * Description of an entry in the state stack.
	 */
	class State {
		// Current transform relative to the virtual coordinate space.
		Transform tfm;

		// The output stream for this layer;
		StrBuf output;

		// The Form XObject represented by this state (if any).
		FormXObject? form;

		// Alpha to be applied to the form, if any.
		Float formAlpha;

		// Create.
		init(StrBuf output) {
			init() { output = output; }
		}

		// Create, using a form.
		init(FormXObject form, Float alpha) {
			init() {
				output = form.data;
				form = form;
				formAlpha = alpha;
			}
		}
	}

	// The state stack.
	State[] stack;

	// Current stream.
	StrBuf output() {
		stack.last.output;
	}

	// Get the transform from the current virtual coordinate system to the physical coordinate
	// system of whatever we're writing to (either the page or a Form XObject).
	Transform currentPhysicalTfm() {
		if (stack.last.form)
			Transform();
		else
			physicalTfm;
	}

	// Prepare a new page.
	void preparePage() {
		pdf.root.pages << current;

		// Reset our state regarding the state stack.
		stack = State[];
		stack << State(current.content.data);

		// Line join style, line end style and line width.
		output << "1 j 1 J 1 w\n";

		// "convert" the coordinate system to virtual units.
		put(physicalTfm, "cm\n");

		// Push this state to the stack, so that we can restore it later, eg. when we need to apply new transforms.
		output << "q\n";
	}

	// Output a point to the current page.
	void put(Point pt, Str after) {
		output << pt.x << " " << pt.y << " " << after;
	}

	// Output a size to the current page.
	void put(Size sz, Str after) {
		output << sz.w << " " << sz.h << " " << after;
	}

	// Output a color.
	void put(Color c, Str after) {
		output << c.r << " " << c.g << " " << c.b << " " << after;
	}

	// Output a matrix.
	void put(Transform tfm, Str after) {
		putTransform(output, tfm);
		output << " " << after;
	}

	// Output a path.
	void put(Path p, Str after) {
		Point last;
		for (e in p.data) {
			if (e.t == PathPointType:tClose) {
				output << "h ";
			} else if (e.t == PathPointType:tStart) {
				put(e.p0, "m ");
				last = e.p0;
			} else if (e.t == PathPointType:tLine) {
				put(e.p0, "l ");
				last = e.p0;
			} else if (e.t == PathPointType:tBezier2) {
				put(last + (2.0/3.0)*(e.p0 - last), "");
				put(e.p1 + (2.0/3.0)*(e.p0 - e.p1), "");
				put(e.p1, "c ");
				last = e.p1;
			} else if (e.t == PathPointType:tBezier3) {
				put(e.p0, "");
				put(e.p1, "");
				put(e.p2, "c ");
				last = e.p2;
			}
		}
		output << after;
	}

	// Output a quarter of a circle as a quadratic bezier curve. Uses 'before' to move to the first
	// point if present.
	void putArc(Point from, Str move, Point middle, Point to) {
		Float ratio = 0.5522847498307935;
		Point d1 = from + (middle - from)*ratio;
		Point d2 = to   + (middle -   to)*ratio;

		if (move.any)
			put(from, move);

		put(d1, "");
		put(d2, "");
		put(to, "c ");
	}

	// Output a rounded rectangle.
	void putRounded(Rect r, Size edges, Str after) {
		Point p0 = r.p0;
		Point p1(r.p1.x, r.p0.y);
		Point p2 = r.p1;
		Point p3(r.p0.x, r.p1.y);

		putArc(p0 + Size(0, edges.h), "m ", p0, p0 + Size(edges.w, 0));
		putArc(p1 + Size(-edges.w, 0), "l ", p1, p1 + Size(0, edges.h));
		putArc(p2 + Size(0, -edges.h), "l ", p2, p2 + Size(-edges.w, 0));
		putArc(p3 + Size(edges.w, 0), "l ", p3, p3 + Size(0, -edges.h));
		output << "h " << after;
	}

	// Output an oval.
	void putOval(Rect r, Str after) {
		Point c = r.center;

		putArc(Point(r.p0.x, c.y), "m ", r.p0, Point(c.x, r.p0.y));
		putArc(Point(c.x, r.p0.y), "", Point(r.p1.x, r.p0.y), Point(r.p1.x, c.y));
		putArc(Point(r.p1.x, c.y), "", r.p1, Point(c.x, r.p1.y));
		putArc(Point(c.x, r.p1.y), "", Point(r.p0.x, r.p1.y), Point(r.p0.x, c.y));

		output << "h " << after;
	}

	// Output color information. 'stroke' indicates whether to set for stroke operations or fill operations.
	void putColor(Brush b, Bool stroke) {
		if (b as SolidBrush) {
			putSolid(b, stroke);
		} else if (b as BitmapBrush) {
			putBitmap(b, stroke);
		} else if (b as LinearGradient) {
			putLinearGradient(b, stroke);
		} else if (b as RadialGradient) {
			putRadialGradient(b, stroke);
		} else {
			print("Unsupported brush found!");
		}
	}

	// Output a solid color.
	void putSolid(SolidBrush b, Bool stroke) {
		output << res.alpha(b.opacity * b.color.a) << " gs ";
		if (stroke)
			put(b.color, "RG\n");
		else
			put(b.color, "rg\n");
	}

	// Output a bitmap brush.
	void putBitmap(BitmapBrush b, Bool stroke) {
		output << res.alpha(1.0) << " gs ";

		var pattern = res.pattern(BitmapPattern(pdf, b, stack.last.tfm * currentPhysicalTfm));
		if (stroke) {
			output << "/Pattern CS " << pattern << " SCN ";
		} else {
			output << "/Pattern cs " << pattern << " scn ";
		}
	}

	// Output a linear gradient.
	void putLinearGradient(LinearGradient g, Bool stroke) {
		putShading(AxialShading(pdf, g, stack.last.tfm * currentPhysicalTfm, current.mediaBox), stroke);
	}

	// Output a radial gradient.
	void putRadialGradient(RadialGradient g, Bool stroke) {
		putShading(RadialShading(pdf, g, stack.last.tfm * currentPhysicalTfm, current.mediaBox), stroke);
	}

	// Output a shading object of some sort.
	void putShading(ShadingPattern shading, Bool stroke) {
		if (alpha = shading.alpha) {
			output << res.softMask(alpha) << " gs ";
		} else {
			output << res.alpha(1.0) << " gs ";
		}

		var pattern = res.pattern(shading);
		if (stroke) {
			output << "/Pattern CS " << pattern << " SCN ";
		} else {
			output << "/Pattern cs " << pattern << " scn ";
		}
	}

	// Create a form.
	FormXObject createForm(Float opacity) {
		var oldTfm = stack.last.tfm;

		FormXObject form(pdf, Rect(Point(), virtual));
		stack.push(State(form, opacity));

		// The new XObject will be painted in an identity-mapped coordinate system, so we need to
		// re-apply the previous transform inside the form.
		put(oldTfm, "cm\n");

		form;
	}

	// Finish the current page before starting to draw a new one, or before saving.
	void finishPage() {
		if (pageFinished)
			return;

		reset();
		pageFinished = true;

		// Pop the final state off the PDF state stack, this make readers extra happy!
		output << "Q\n";
	}
}
