/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.io.erasurecode.rawcoder;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.apache.hadoop.io.erasurecode.ErasureCoderOptions;
import org.apache.hadoop.io.erasurecode.rawcoder.DummyRawErasureCoderFactory;
import org.apache.hadoop.io.erasurecode.rawcoder.NativeRSRawErasureCoderFactory;
import org.apache.hadoop.io.erasurecode.rawcoder.RSLegacyRawErasureCoderFactory;
import org.apache.hadoop.io.erasurecode.rawcoder.RSRawErasureCoderFactory;
import org.apache.hadoop.io.erasurecode.rawcoder.RawErasureCoderFactory;
import org.apache.hadoop.io.erasurecode.rawcoder.RawErasureDecoder;
import org.apache.hadoop.io.erasurecode.rawcoder.RawErasureEncoder;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.StopWatch;

public final class RawErasureCoderBenchmark {
    private static final int TARGET_BUFFER_SIZE_MB = 126;
    private static final int MAX_CHUNK_SIZE = 126 / BenchData.NUM_DATA_UNITS * 1024;
    private static final List<RawErasureCoderFactory> CODER_MAKERS = Collections.unmodifiableList(Arrays.asList(new DummyRawErasureCoderFactory(), new RSLegacyRawErasureCoderFactory(), new RSRawErasureCoderFactory(), new NativeRSRawErasureCoderFactory()));

    private RawErasureCoderBenchmark() {
    }

    private static void printAvailableCoders() {
        StringBuilder sb = new StringBuilder("Available coders with coderIndex:\n");
        for (CODER coder : CODER.values()) {
            sb.append(coder.ordinal()).append(":").append((Object)coder).append("\n");
        }
        System.out.println(sb.toString());
    }

    private static void usage(String message) {
        if (message != null) {
            System.out.println(message);
        }
        System.out.println("Usage: RawErasureCoderBenchmark <encode/decode> <coderIndex> [numThreads] [dataSize-in-MB] [chunkSize-in-KB]");
        RawErasureCoderBenchmark.printAvailableCoders();
        System.exit(1);
    }

    public static void main(String[] args) throws Exception {
        String opType = null;
        int coderIndex = 0;
        int dataSizeMB = 10240;
        int chunkSizeKB = 1024;
        int numThreads = 1;
        if (args.length > 1) {
            opType = args[0];
            if (!"encode".equals(opType) && !"decode".equals(opType)) {
                RawErasureCoderBenchmark.usage("Invalid type: should be either 'encode' or 'decode'");
            }
            try {
                coderIndex = Integer.parseInt(args[1]);
                if (coderIndex < 0 || coderIndex >= CODER.values().length) {
                    RawErasureCoderBenchmark.usage("Invalid coder index, should be [0-" + (CODER.values().length - 1) + "]");
                }
            }
            catch (NumberFormatException e) {
                RawErasureCoderBenchmark.usage("Malformed coder index, " + e.getMessage());
            }
        } else {
            RawErasureCoderBenchmark.usage(null);
        }
        if (args.length > 2) {
            try {
                numThreads = Integer.parseInt(args[2]);
                if (numThreads <= 0) {
                    RawErasureCoderBenchmark.usage("Invalid number of threads.");
                }
            }
            catch (NumberFormatException e) {
                RawErasureCoderBenchmark.usage("Malformed number of threads, " + e.getMessage());
            }
        }
        if (args.length > 3) {
            try {
                dataSizeMB = Integer.parseInt(args[3]);
                if (dataSizeMB <= 0) {
                    RawErasureCoderBenchmark.usage("Invalid data size.");
                }
            }
            catch (NumberFormatException e) {
                RawErasureCoderBenchmark.usage("Malformed data size, " + e.getMessage());
            }
        }
        if (args.length > 4) {
            try {
                chunkSizeKB = Integer.parseInt(args[4]);
                if (chunkSizeKB <= 0) {
                    RawErasureCoderBenchmark.usage("Chunk size should be positive.");
                }
                if (chunkSizeKB > MAX_CHUNK_SIZE) {
                    RawErasureCoderBenchmark.usage("Chunk size should be no larger than " + MAX_CHUNK_SIZE);
                }
            }
            catch (NumberFormatException e) {
                RawErasureCoderBenchmark.usage("Malformed chunk size, " + e.getMessage());
            }
        }
        RawErasureCoderBenchmark.performBench(opType, CODER.values()[coderIndex], numThreads, dataSizeMB, chunkSizeKB);
    }

    public static void performBench(String opType, CODER coder, int numThreads, int dataSizeMB, int chunkSizeKB) throws Exception {
        ByteBuffer testData;
        BenchData.configure(dataSizeMB, chunkSizeKB);
        RawErasureEncoder encoder = null;
        RawErasureDecoder decoder = null;
        boolean isEncode = opType.equals("encode");
        if (isEncode) {
            encoder = RawErasureCoderBenchmark.getRawEncoder(coder.ordinal());
            testData = RawErasureCoderBenchmark.genTestData(encoder.preferDirectBuffer(), BenchData.bufferSizeKB);
        } else {
            decoder = RawErasureCoderBenchmark.getRawDecoder(coder.ordinal());
            testData = RawErasureCoderBenchmark.genTestData(decoder.preferDirectBuffer(), BenchData.bufferSizeKB);
        }
        ExecutorService executor = Executors.newFixedThreadPool(numThreads);
        ArrayList<Future<Long>> futures = new ArrayList<Future<Long>>(numThreads);
        StopWatch sw = new StopWatch().start();
        for (int i = 0; i < numThreads; ++i) {
            futures.add(executor.submit(new BenchmarkCallable(isEncode, encoder, decoder, testData.duplicate())));
        }
        ArrayList<Long> durations = new ArrayList<Long>(numThreads);
        try {
            for (Future future : futures) {
                durations.add((Long)future.get());
            }
            long duration = sw.now(TimeUnit.MILLISECONDS);
            double totalDataSize = (double)(BenchData.totalDataSizeKB * (long)numThreads) / 1024.0;
            DecimalFormat df = new DecimalFormat("#.##");
            System.out.println((Object)((Object)coder) + " " + opType + " " + df.format(totalDataSize) + "MB data, with chunk size " + BenchData.chunkSize / 1024 + "KB");
            System.out.println("Total time: " + df.format((double)duration / 1000.0) + " s.");
            System.out.println("Total throughput: " + df.format(totalDataSize / (double)duration * 1000.0) + " MB/s");
            RawErasureCoderBenchmark.printThreadStatistics(durations, df);
        }
        catch (Exception e) {
            System.out.println("Error waiting for thread to finish.");
            e.printStackTrace();
            throw e;
        }
        finally {
            executor.shutdown();
            if (encoder != null) {
                encoder.release();
            }
            if (decoder != null) {
                decoder.release();
            }
        }
    }

    private static RawErasureEncoder getRawEncoder(int index) throws IOException {
        RawErasureEncoder encoder = CODER_MAKERS.get(index).createEncoder(BenchData.OPTIONS);
        boolean isDirect = encoder.preferDirectBuffer();
        encoder.encode(RawErasureCoderBenchmark.getBufferForInit(BenchData.NUM_DATA_UNITS, 1, isDirect), RawErasureCoderBenchmark.getBufferForInit(BenchData.NUM_PARITY_UNITS, 1, isDirect));
        return encoder;
    }

    private static RawErasureDecoder getRawDecoder(int index) throws IOException {
        RawErasureDecoder decoder = CODER_MAKERS.get(index).createDecoder(BenchData.OPTIONS);
        boolean isDirect = decoder.preferDirectBuffer();
        ByteBuffer[] inputs = RawErasureCoderBenchmark.getBufferForInit(BenchData.NUM_ALL_UNITS, 1, isDirect);
        for (int erasedIndex : BenchData.ERASED_INDEXES) {
            inputs[erasedIndex] = null;
        }
        decoder.decode(inputs, BenchData.ERASED_INDEXES, RawErasureCoderBenchmark.getBufferForInit(BenchData.ERASED_INDEXES.length, 1, isDirect));
        return decoder;
    }

    private static ByteBuffer[] getBufferForInit(int numBuf, int bufCap, boolean isDirect) {
        ByteBuffer[] buffers = new ByteBuffer[numBuf];
        for (int i = 0; i < buffers.length; ++i) {
            buffers[i] = isDirect ? ByteBuffer.allocateDirect(bufCap) : ByteBuffer.allocate(bufCap);
        }
        return buffers;
    }

    private static void printThreadStatistics(List<Long> durations, DecimalFormat df) {
        Collections.sort(durations);
        System.out.println("Threads statistics: ");
        Double min = (double)durations.get(0).longValue() / 1000.0;
        Double max = (double)durations.get(durations.size() - 1).longValue() / 1000.0;
        Long sum = 0L;
        for (Long duration : durations) {
            sum = sum + duration;
        }
        Double avg = sum.doubleValue() / (double)durations.size() / 1000.0;
        Double percentile = (double)durations.get((int)Math.ceil((double)durations.size() * 0.9) - 1).longValue() / 1000.0;
        System.out.println(durations.size() + " threads in total.");
        System.out.println("Min: " + df.format(min) + " s, Max: " + df.format(max) + " s, Avg: " + df.format(avg) + " s, 90th Percentile: " + df.format(percentile) + " s.");
    }

    private static ByteBuffer genTestData(boolean useDirectBuffer, int sizeKB) {
        Random random = new Random();
        int bufferSize = sizeKB * 1024;
        byte[] tmp = new byte[bufferSize];
        random.nextBytes(tmp);
        ByteBuffer data = useDirectBuffer ? ByteBuffer.allocateDirect(bufferSize) : ByteBuffer.allocate(bufferSize);
        data.put(tmp);
        data.flip();
        return data;
    }

    static {
        Preconditions.checkArgument((CODER_MAKERS.size() == CODER.values().length ? 1 : 0) != 0);
    }

    private static class BenchmarkCallable
    implements Callable<Long> {
        private final boolean isEncode;
        private final RawErasureEncoder encoder;
        private final RawErasureDecoder decoder;
        private final BenchData benchData;
        private final ByteBuffer testData;

        public BenchmarkCallable(boolean isEncode, RawErasureEncoder encoder, RawErasureDecoder decoder, ByteBuffer testData) {
            if (isEncode) {
                Preconditions.checkArgument((encoder != null ? 1 : 0) != 0);
                this.encoder = encoder;
                this.decoder = null;
                this.benchData = new BenchData(encoder.preferDirectBuffer());
            } else {
                Preconditions.checkArgument((decoder != null ? 1 : 0) != 0);
                this.decoder = decoder;
                this.encoder = null;
                this.benchData = new BenchData(decoder.preferDirectBuffer());
            }
            this.isEncode = isEncode;
            this.testData = testData;
        }

        @Override
        public Long call() throws Exception {
            long rounds = BenchData.totalDataSizeKB / (long)BenchData.bufferSizeKB;
            StopWatch sw = new StopWatch().start();
            for (long i = 0L; i < rounds; ++i) {
                while (this.testData.remaining() > 0) {
                    for (ByteBuffer output : this.benchData.outputs) {
                        output.clear();
                    }
                    for (int j = 0; j < this.benchData.inputs.length; ++j) {
                        ((BenchData)this.benchData).inputs[j] = this.testData.duplicate();
                        this.benchData.inputs[j].limit(this.testData.position() + BenchData.chunkSize);
                        ((BenchData)this.benchData).inputs[j] = this.benchData.inputs[j].slice();
                        this.testData.position(this.testData.position() + BenchData.chunkSize);
                    }
                    if (this.isEncode) {
                        this.benchData.encode(this.encoder);
                        continue;
                    }
                    this.benchData.prepareDecInput();
                    this.benchData.decode(this.decoder);
                }
                this.testData.clear();
            }
            return sw.now(TimeUnit.MILLISECONDS);
        }
    }

    private static class BenchData {
        public static final ErasureCoderOptions OPTIONS = new ErasureCoderOptions(6, 3);
        public static final int NUM_DATA_UNITS = OPTIONS.getNumDataUnits();
        public static final int NUM_PARITY_UNITS = OPTIONS.getNumParityUnits();
        public static final int NUM_ALL_UNITS = OPTIONS.getNumAllUnits();
        private static int chunkSize;
        private static long totalDataSizeKB;
        private static int bufferSizeKB;
        private static final int[] ERASED_INDEXES;
        private final ByteBuffer[] inputs = new ByteBuffer[NUM_DATA_UNITS];
        private ByteBuffer[] outputs = new ByteBuffer[NUM_PARITY_UNITS];
        private ByteBuffer[] decodeInputs = new ByteBuffer[NUM_ALL_UNITS];

        public static void configure(int dataSizeMB, int chunkSizeKB) {
            chunkSize = chunkSizeKB * 1024;
            int round = (int)Math.round(129024.0 / (double)NUM_DATA_UNITS / (double)chunkSizeKB);
            Preconditions.checkArgument((round > 0 ? 1 : 0) != 0);
            bufferSizeKB = NUM_DATA_UNITS * chunkSizeKB * round;
            System.out.println("Using " + bufferSizeKB / 1024 + "MB buffer.");
            round = (int)Math.round((double)dataSizeMB * 1024.0 / (double)bufferSizeKB);
            if (round == 0) {
                round = 1;
            }
            totalDataSizeKB = round * bufferSizeKB;
        }

        public BenchData(boolean useDirectBuffer) {
            for (int i = 0; i < this.outputs.length; ++i) {
                this.outputs[i] = useDirectBuffer ? ByteBuffer.allocateDirect(chunkSize) : ByteBuffer.allocate(chunkSize);
            }
        }

        public void prepareDecInput() {
            System.arraycopy(this.inputs, 0, this.decodeInputs, 0, NUM_DATA_UNITS);
        }

        public void encode(RawErasureEncoder encoder) throws IOException {
            encoder.encode(this.inputs, this.outputs);
        }

        public void decode(RawErasureDecoder decoder) throws IOException {
            decoder.decode(this.decodeInputs, ERASED_INDEXES, this.outputs);
        }

        static {
            ERASED_INDEXES = new int[]{6, 7, 8};
        }
    }

    static enum CODER {
        DUMMY_CODER("Dummy coder"),
        LEGACY_RS_CODER("Legacy Reed-Solomon Java coder"),
        RS_CODER("Reed-Solomon Java coder"),
        ISAL_CODER("ISA-L coder");

        private final String name;

        private CODER(String name) {
            this.name = name;
        }

        public String toString() {
            return this.name;
        }
    }
}

