/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hudi.common.table.timeline;

import java.util.function.Consumer;

/**
 * Time generator for global monotonically increasing time generation.
 *
 * <p>Basically there are two ways to make the time generation monotonically increasing, inline with TrueTime semantics.
 *
 * i). lock provider to guard the time generation with mutex, along with a wait for an estimated max allowed clock skew on distributed hosts;
 * ii). Globally synchronized time generation service, e.g. Google Spanner Time Service, the service itself can ensure the monotonicity.
 * Currently only AWS supplies an SDK for the TrueTime generation.
 */
public interface TimeGenerator {

  /**
   * Generates a globally monotonically increasing timestamp. The implementation must ensure that
   * a new generated timestamp T is guaranteed to be greater than
   * any timestamp generated by the preceding calls.
   *
   * @param skipLocking If this is triggered by another parent transaction, locking can be skipped.
   * @return Current TrueTime as milliseconds.
   */
  long generateTime(boolean skipLocking);

  /**
   * Passes an auto generated timestamp to the given function {@code func}. The implementations
   * need to ensure timestamp generation and executing func are atomic.
   *
   * @param skipLocking If this is triggered by another parent transaction, locking can be skipped.
   * @param func   A consumer that takes a monotonically increasing timestamp.
   */
  void consumeTime(boolean skipLocking, Consumer<Long> func);
}
