/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 *   SPDX-License-Identifier: Apache-2.0
 */

package org.apache.jena.sparql.algebra.op;

import org.apache.jena.sparql.algebra.*;
import org.apache.jena.sparql.algebra.table.TableUnit;
import org.apache.jena.sparql.sse.Tags;
import org.apache.jena.sparql.util.NodeIsomorphismMap;

public class OpTable extends Op0
{
    /** Prefer {@link OpLib#unit} for a unit {@code Op}. Use this function specifically for a unit table. */
    public static OpTable unit()
    { return new OpTable(TableFactory.createUnit()); }

    /** Prefer {@link OpLib#empty} for a zero {@code Op}. Use this function specifically for a zero table. */
    public static OpTable empty()
    { return new OpTable(TableFactory.createEmpty()); }

    public static OpTable create(Table table) {
        return new OpTable(table);
    }

    private Table table;

    private OpTable(Table table) { this.table = table; }

    public boolean isJoinIdentity()
    { return TableUnit.isTableUnit(table); }

    public Table getTable()
    { return table; }

    @Override
    public String getName() { return Tags.tagTable; }

    @Override
    public void visit(OpVisitor opVisitor)
    { opVisitor.visit(this); }

    @Override
    public Op apply(Transform transform)
    { return transform.transform(this); }

    @Override
    public Op0 copy()
    { return new OpTable(table); }

    @Override
    public int hashCode()
    { return table.hashCode(); }

    @Override
    public boolean equalTo(Op other, NodeIsomorphismMap labelMap) {
        if ( !(other instanceof OpTable) )
            return false;
        OpTable opTable = (OpTable)other;
        return table.equals(opTable.table);
    }
}
