/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.gems.tags;

import java.util.Locale;

import javax.portlet.PortletConfig;
import javax.portlet.PortletRequest;
import javax.servlet.jsp.JspException;
import javax.servlet.jsp.JspTagException;
import javax.servlet.jsp.PageContext;
import javax.servlet.jsp.jstl.core.Config;
import javax.servlet.jsp.jstl.fmt.LocalizationContext;
import javax.servlet.jsp.tagext.TagSupport;

/**
 * SetPortletBundleTag
 * 
 * @version $Id: SetPortletBundleTag.java 930022 2010-04-01 15:52:40Z woonsan $
 */
public class SetPortletBundleTag extends TagSupport
{
    private static final long serialVersionUID = 1L;

    // *********************************************************************
    // Protected state
    protected String basename; // 'basename' attribute
    
    // *********************************************************************
    // Private state
    private int scope; // 'scope' attribute
    
    private String var; // 'var' attribute

    // *********************************************************************
    // Constructor and initialization
    public SetPortletBundleTag()
    {
        super();
        init();
    }

    private void init()
    {
        basename = null;
        scope = PageContext.PAGE_SCOPE;
    }

    // *********************************************************************
    // Tag attributes known at translation time
    public void setVar(String var)
    {
        this.var = var;
    }

    public void setScope(String scope)
    {
        this.scope = getScope(scope);
    }

    // *********************************************************************
    // Accessor methods
    // for tag attribute
    public void setBasename(String basename) throws JspTagException
    {
        this.basename = basename;
    }

    // *********************************************************************
    // Tag logic
    public int doEndTag() throws JspException
    {
        LocalizationContext locCtxt = getLocalizationContext(pageContext, basename);
        
        if (var != null)
        {
            pageContext.setAttribute(var, locCtxt, scope);
        }
        else
        {
            Config.set(pageContext, Config.FMT_LOCALIZATION_CONTEXT, locCtxt, scope);
        }
        
        return EVAL_PAGE;
    }

    // Releases any resources we may have (or inherit)
    public void release()
    {
        init();
    }

    public static LocalizationContext getLocalizationContext(PageContext pc, String basename)
    {
        PortletConfig portletConfig = (PortletConfig) pc.getRequest().getAttribute("javax.portlet.config");
        
        if (portletConfig == null)
        {
            return new LocalizationContext();
        }
        else
        {
            PortletRequest portletRequest = (PortletRequest) pc.getRequest().getAttribute("javax.portlet.request");
            return new LocalizationContext(portletConfig.getResourceBundle(portletRequest != null ? portletRequest.getLocale() : Locale.getDefault()));
        }
    }

    private static int getScope(String scope)
    {
        if ("request".equalsIgnoreCase(scope))
        {
            return PageContext.REQUEST_SCOPE;
        }
        else if ("session".equalsIgnoreCase(scope))
        {
            return PageContext.SESSION_SCOPE;
        }
        else if ("application".equalsIgnoreCase(scope))
        {
            return PageContext.APPLICATION_SCOPE;
        }
        
        return PageContext.PAGE_SCOPE;
    }
}
