/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy;

import java.net.URI;
import java.util.Collection;


/**
 * Reverse Proxy mapping registry.
 * <p>
 * All the mapping information should be retrieved from this registry
 * during a reverse proxy processing.
 * </p>
 */
public interface ProxyMappingRegistry
{

    /**
     * Adds a reverse proxy mapping to the registry.
     * @param proxyMapping
     * @return
     */
    public boolean addProxyMapping(ProxyMapping proxyMapping);

    /**
     * Adds all the reverse proxy mappings in <code>proxyMappings</code> to the registry.
     * @param proxyMappings
     * @return
     */
    public boolean addAllProxyMappings(Collection<ProxyMapping> proxyMappings);

    /**
     * Removes the specified <code>proxyMapping</code> from the registry if exists.
     * If not exists, it does nothing.
     * @param proxyMapping
     * @return
     */
    public boolean removeProxyMapping(ProxyMapping proxyMapping);

    /**
     * Returns all the reverse proxy mappings in a collection.
     * If there's no reverse proxy mapping, it should return an empty collection.
     * @return
     */
    public Collection<ProxyMapping> getProxyMappings();

    /**
     * Finds and returns a reverse proxy mapping for the specific <code>localPath</code>
     * which represents the path info of the request from the client or caller.
     * If there's no reverse proxy mapping for the local request path, then it returns null.
     * @param localPath
     * @return
     */
    public ProxyMapping findProxyMappingByLocalPath(String localPath);

    /**
     * Finds and returns a reverse proxy mapping for the specific <code>remoteURI</code>
     * which represents the remote target URI at the background.
     * If there's no reverse proxy mapping for the remote target URI, then it returns null.
     * @param remoteURI
     * @return
     */
    public ProxyMapping findProxyMappingByRemoteURI(URI remoteURI);
}
