package org.apache.jetspeed.portletcontainer;

// jetspeed
import org.apache.jetspeed.portlet.*;
import org.apache.jetspeed.portlet.event.MessageEvent;
import org.apache.jetspeed.portlet.event.MessageListener;
import org.apache.jetspeed.portlet.service.*;

import org.apache.jetspeed.portletcontainer.om.applicationregistry.ApplicationEntry;
import org.apache.jetspeed.portletcontainer.util.ThreadAttributesManager;
import org.apache.jetspeed.portletcontainer.filter.PortletContextFilter;
import org.apache.jetspeed.portletcontainer.services.portletserviceregistry.PortletServiceRegistryAccess;
import org.apache.jetspeed.portletcontainer.event.MessageTriggerEvent;
import org.apache.jetspeed.portletcontainer.event.EventQueueManager;

import org.apache.jetspeed.util.*;

// turbine
import org.apache.turbine.util.Log;
import org.apache.turbine.services.resources.TurbineResources;

// java
import java.io.InputStream;
import java.io.IOException;
import java.io.File;
import java.util.*;
import java.net.URL;
import java.net.MalformedURLException;
import javax.servlet.*;
import javax.servlet.http.*;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import com.ibm.wps.portletcontainer.ClientEntryImpl;

public class PortletContextImpl implements PortletContext
{
    private ApplicationEntry applicationEntry = null;
    private ServletContext servletContext = null;
    private ClassLoader classloader = null;

    private PortletContextFilter contextFilter = null;

    public PortletContextImpl( ApplicationEntry applicationEntry,
                               ServletContext context )
    {
        this.applicationEntry = applicationEntry;
        this.servletContext = context;

        initFilter();
    }

    public String getInitParameter (String name)
    {
        return servletContext.getInitParameter(name);
    }

    public Enumeration getInitParameterNames ()
    {
        return servletContext.getInitParameterNames();
    }

    public void setAttribute (String name, Object value)
    {
        servletContext.setAttribute(
            PortletNamespaceMapper.encode(
                applicationEntry.getAid(),name), value);
    }

    public Object getAttribute (String name)
    {
        return servletContext.getAttribute(
            PortletNamespaceMapper.encode(
                applicationEntry.getAid(),name));
    }

    public Enumeration getAttributeNames ()
    {
        Vector context_attributes = new Vector();
        Enumeration attributes = servletContext.getAttributeNames();
        while (attributes.hasMoreElements())
        {
            String element = (String)attributes.nextElement();
            if (PortletNamespaceMapper.isInNamespace(
                applicationEntry.getAid(), element))
            {
                context_attributes.add(
                    PortletNamespaceMapper.decode(
                        applicationEntry.getAid(), element));
            }
        }
        return context_attributes.elements();
    }

    public void removeAttribute (String name)
    {
        servletContext.removeAttribute(
            PortletNamespaceMapper.encode(
                applicationEntry.getAid(),name));
    }

    public void include (String path,
                         PortletRequest request,
                         PortletResponse response) throws PortletException,
    IOException
    {
        // check for multi-language
        int lastSlash = path.lastIndexOf('/')+1;
        String root = path.substring(0,lastSlash);
        String fileName = path.substring(lastSlash);

        String newpath = com.ibm.wps.services.finder.Finder.find (
            servletContext,
            root,
            ((PortletRequestImpl)request).getProvider().getFrameworkClient(),
            request.getLocale(),
            null,
            fileName);
        
        if (newpath!=null)
        {

            HttpServletRequest servletRequest = ((PortletRequestImpl)request).getProxiedHttpServletRequest();
            HttpServletResponse servletResponse = ((PortletResponseImpl)response).getProxiedHttpServletResponse();

            servletRequest.setAttribute(Constants.PORTLET_REQUEST,request);
            servletRequest.setAttribute(Constants.PORTLET_RESPONSE,response);
            servletRequest.setAttribute(Constants.PORTLET_CONFIG,((PortletRequestImpl)request).getPortletConfig());

            RequestDispatcher dispatcher = servletContext.getRequestDispatcher(newpath);
            try
            {
                // try to flush any data before dispatching request
                servletResponse.flushBuffer();

                // call the servlet/JSP
                dispatcher.include(request, response);

            }
            catch (ServletException e)
            {
            	Throwable t = e.getRootCause();
            	
                String message = "PortletContextImpl.include: Could not include the following URL: "
                                 + path + " : " + e.getMessage();
                                 
				// defect #4663: jsp runtime error logging
            	// if we have a root cause, we throw this instead
            	if (t != null)
            	{
	                Log.error( message, t );
	                throw new PortletException ( t );
            	}
	            else
	            {	
	                Log.error( message, e );
	                throw new PortletException ( e );
	            }
            }
            catch (NullPointerException e)
            {
                String message = "PortletContextImpl.include: Could not include/find the following URL: "
                                 + path;
                Log.error( message );
                throw new PortletException ( message );
            }
            finally
            {
                servletRequest.removeAttribute(Constants.PORTLET_REQUEST);
                servletRequest.removeAttribute(Constants.PORTLET_RESPONSE);
                servletRequest.removeAttribute(Constants.PORTLET_CONFIG);
            }
        }
        else
        {
            String message = "PortletContextImpl.include: Could not include/find the following URL: "
                             + path;
            Log.error( message );
        }
    }

    public InputStream getResourceAsStream (String path)
    {
        // We cannot use getResourceAsStream here, because WebSphere has a bug here
//          return servletContext.getResourceAsStream(path);
        String realpath = servletContext.getRealPath(path);
        try
        {
            return new java.io.FileInputStream(realpath);
        }
        catch (java.io.FileNotFoundException e)
        {
            String message = "PortletContextImpl.getResourceAsStream: Could not find the following file: "
                             + path + "\nRealpath: "+realpath;
            Log.error( message );
            return null;
        }
    }

    public InputStream getResourceAsStream (String path, Client client, Locale locale)
    {
        PortletRequestImpl request = 
        (PortletRequestImpl)ThreadAttributesManager.getAttribute(
            org.apache.jetspeed.portlet.spi.Constants.PARAM_PORTLETREQUEST);
        if (request!=null)
        {
            // check for multi-language
            int lastSlash = path.lastIndexOf('/')+1;
            String root = path.substring(0,lastSlash);
            String fileName = path.substring(lastSlash);
            String newpath = com.ibm.wps.services.finder.Finder.find (
                servletContext,
                root,
                ((PortletRequestImpl)request).getProvider().getFrameworkClient(),
                locale,
                null,
                fileName);

            if (newpath!=null)
            {
                // We cannot use getResourceAsStream here, because WebSphere has a bug here
                //          return servletContext.getResourceAsStream(path);
                String realpath = servletContext.getRealPath(newpath);
                try
                {
                    return new java.io.FileInputStream(realpath);
                }
                catch (java.io.FileNotFoundException e)
                {
                    String message = "PortletContextImpl.getResourceAsStream: Could not find the following file: "
                                     + path + "\nRealpath: "+realpath;
                    Log.error( message );
                    return null;
                }
            }
            else
            {
                String message = "PortletContextImpl.getResourceAsStream: Could not find the following file: "
                                 + path;
                Log.error( message );
                return null;
            }
        }
        else
        {
            String message = "PortletContextImpl.getResourceAsStream: Internal portlet container error while processing file "
                             + path;
            Log.error( message );
            return null;
        }
    }

    public String getText (String bundleName, String key, Locale locale)
    {
        if (classloader!=null)
        {
            ResourceBundle bundle = ResourceBundle.getBundle(bundleName,locale,classloader);
            if (bundle==null)
            {
                String message = "PortletContextImpl.getText: Could not find the resource bundle: "
                                 + bundleName;
                Log.error( message );
                return "";
            }
            return(String)bundle.getString(key);
        }
        else {
            String message = "PortletContextImpl.getText: Could not get classloader of context.";
            Log.error( message );
            return "";
        }
    }

    public void send (String aPortletName, PortletMessage aMessage, PortletRequest request)
    throws AccessDeniedException
    {
        this.send(aPortletName, aMessage);
    }

    public void send (String aPortletName, PortletMessage aMessage)
    throws AccessDeniedException
    {
        PortletRequestImpl request = 
        (PortletRequestImpl)ThreadAttributesManager.getAttribute(
            org.apache.jetspeed.portlet.spi.Constants.PARAM_PORTLETREQUEST);
        if (request!=null)
        {
            if (!request.isEventBasedRequest())
                throw new AccessDeniedException();

            HttpServletRequest servletRequest = request.getProxiedHttpServletRequest();

            // enqueue the message event
            // at this point, only a message trigger is sent.
            // later in the message queue, this message trigger is split
            // into several message events to the real portlets
            MessageTriggerEvent messageevent = 
                new MessageTriggerEvent(request.getPortletInstanceEntry(),
                                        aPortletName,
                                        aMessage);
            EventQueueManager.addEvent(servletRequest, messageevent);
        }
        else
        {
            throw new AccessDeniedException();
        }
    }

    public void send (String aPortletName, DefaultPortletMessage aMessage)
    throws AccessDeniedException
    {
        this.send(aPortletName,(PortletMessage)aMessage);
    }

    public PortletService getService(Class service) throws PortletServiceUnavailableException,
    PortletServiceNotFoundException
    {
        return PortletServiceRegistryAccess.getPortletService(service);
    }

    public int getMajorVersion ()
    {
        return 1;
    }

    public int getMinorVersion ()
    {
        return 1;
    }

    public String getContainerInfo ()
    {
        return "IBM WebSphere Portal Server/4.1";
    }

    public PortletLog getLog ()
    {
        return PortletLogImpl.getInstance(); // tbd
    }

    // servlet methods, which cannot be changed by the filter, because they are deprecated

    public Enumeration getServlets()
    {
        // deprecated
        return new Vector().elements();
    }

    public Enumeration getServletNames()
    {
        // deprecated
        return new Vector().elements();
    }

    public Servlet getServlet(String name) throws ServletException
    {
        // deprecated
        return null;
    }

    public void log(Exception exception, String msg)
    {
        // deprecated
        getLog().error(msg, exception);
    }

    // servlet methods, which can be changed by the filter

    public ServletContext getContext(String uripath)
    {
        return contextFilter.getContext(uripath);
    }

    public RequestDispatcher getRequestDispatcher(String path)
    {
        return contextFilter.getRequestDispatcher(path);
    }

    public String getMimeType(String file)
    {
        return contextFilter.getMimeType(file);
    }

    public URL getResource(String path) throws MalformedURLException
    {
        return contextFilter.getResource(path);
    }

    public RequestDispatcher getNamedDispatcher(String name)
    {
        return contextFilter.getNamedDispatcher(name);
    }

    public void log(String msg)
    {
        contextFilter.log(msg);
    }

    public void log(String message, Throwable throwable)
    {
        contextFilter.log(message, throwable);
    }

    public String getRealPath(String path)
    {
        return contextFilter.getRealPath(path);
    }

    public String getServerInfo()
    {
        return contextFilter.getServerInfo();
    }

    // additional methods

    public void setProxiedServletContext(ServletContext context)
    {
        this.servletContext = context;
        contextFilter.setServletContext(servletContext);
    }

    public ClassLoader getServletClassLoader()
    {
        return this.classloader;
    }

    public void setServletClassLoader(ClassLoader classloader)
    {
        this.classloader = classloader;
    }

    private void initFilter()
    {
        String filterClassName = TurbineResources.getString("portletcontainer.filter.context");
        if (filterClassName==null) {
            contextFilter = new PortletContextFilter(servletContext,
                                                     this);
        }
        else {
            Class[] parameterClasses = {ServletContext.class, PortletContext.class};
            Constructor constructor = null;
            try {
                constructor = Class.forName(filterClassName).getConstructor(parameterClasses);
            }
            catch (NoSuchMethodException e) {
                Log.error("Invalid PortletContextFilter defined in resources key portletcontainer.filter.context.",e);
                throw new IllegalArgumentException("Invalid PortletContextFilter defined in resources key portletcontainer.filter.context.");
            }
            catch (ClassNotFoundException e) {
                Log.error("Invalid PortletContextFilter defined in resources key portletcontainer.filter.context.",e);
                throw new IllegalArgumentException("Invalid PortletContextFilter defined in resources key portletcontainer.filter.context.");
            }
            Object[] parameters = {servletContext, this };
            try {
                Object filter = constructor.newInstance(parameters);
                if (!(filter instanceof PortletContextFilter)) {
                    Log.error("PortletContextFilter defined in resources key portletcontainer.filter.context does not derive from PortletContextFilter.");
                    throw new IllegalArgumentException("PortletContextFilter defined in resources key portletcontainer.filter.context does not derive from PortletContextFilter.");
                }
                contextFilter = (PortletContextFilter)filter;
            }
            catch (InstantiationException e) {
                Log.error("PortletContextFilter defined in resources key portletcontainer.filter.context could not be instantiated.",e);
                throw new IllegalArgumentException("PortletContextFilter defined in resources key portletcontainer.filter.context could not be instantiated.");
            }
            catch (IllegalAccessException e) {
                Log.error("PortletContextFilter defined in resources key portletcontainer.filter.context could not be instantiated.",e);
                throw new IllegalArgumentException("PortletContextFilter defined in resources key portletcontainer.filter.context could not be instantiated.");
            }
            catch (InvocationTargetException e) {
                Log.error("PortletContextFilter defined in resources key portletcontainer.filter.context could not be instantiated.",e);
                throw new IllegalArgumentException("PortletContextFilter defined in resources key portletcontainer.filter.context could not be instantiated.");
            }
        }
    }

}
