package org.apache.jetspeed.portletcontainer;

// jetspeed
import org.apache.jetspeed.portlet.*;
import org.apache.jetspeed.portlet.event.ActionListener;
import org.apache.jetspeed.portletcontainer.information.*;
import org.apache.jetspeed.portletcontainer.invoker.*;
import org.apache.jetspeed.portletcontainer.filter.*;
import org.apache.jetspeed.portletcontainer.util.*;
import org.apache.jetspeed.portletcontainer.om.portletinstanceregistry.PortletInstanceEntry;

// turbine
import org.apache.turbine.util.Log;
import org.apache.turbine.services.resources.TurbineResources;

// java
import java.io.PrintWriter;
import java.io.IOException;
import java.util.*;
import javax.servlet.*;
import javax.servlet.http.*;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import com.ibm.wps.portletcontainer.information.*;

/**
 * This is the implementation of the PortletResponse interface.
 * <P>
 * The following header fields of the HTTP 1.1 (RFC 2616) are <b>not allowed</b> 
 * to be set:
 * <P>
 * <B>6.2 Response Header Fields</B><BR>
 * The response-header fields allow the server to pass additional information about
 * the response which cannot be placed in the Status- Line. These header fields give
 * information about the server and about further access to the resource identified
 * by the Request-URI.
 * <P>
 * <UL>
 * <LI>Accept-Ranges (Section 14.5)</LI>
 * <LI>Location (Section 14.30)</LI>
 * <LI>Proxy-Authenticate (Section 14.33)</LI>
 * <LI>Server (Section 14.38)</LI>
 * <LI>Vary (Section 14.44)</LI>
 * <LI>WWW-Authenticate (Section 14.47)</LI>
 * </UL>
 * <P>
 * <B>7.1 Entity Header Fields</B><BR>
 * Entity-header fields define metainformation about the entity-body or, if no body
 * is present, about the resource identified by the request. Some of this
 * metainformation is OPTIONAL; some might be REQUIRED by portions of this specification.
 * <P>
 * <UL>
 * <LI>Allow (Section 14.7)</LI>
 * <LI>Content-Encoding (Section 14.11)</LI>
 * <LI>Content-Language (Section 14.12)</LI>
 * <LI>Content-Length (Section 14.13)</LI>
 * <LI>Content-Location (Section 14.14)</LI>
 * <LI>Content-MD5 (Section 14.15)</LI>
 * <LI>Content-Range (Section 14.16)</LI>
 * <LI>Content-Type (Section 14.17)</LI>
 * <LI>Expires (Section 14.21)</LI>
 * <LI>Last-Modified (Section 14.29)</LI>
 * </UL>
 * <P>
 * <B>4.2 Message Headers</B><BR>
 * HTTP header fields, which include general-header (section 4.5), request-header
 * (section 5.3), response-header (section 6.2), and entity-header (section 7.1)
 * fields, follow the same generic format as that given in Section 3.1 of RFC 822 [9].
 * Each header field consists of a name followed by a colon (":") and the field value.
 * Field names are <b>case-insensitive</b>. The field value MAY be preceded by any amount
 * of LWS, though a single SP is preferred. Header fields can be extended over
 * multiple lines by preceding each extra line with at least one SP or HT.
 * Applications ought to follow "common form", where one is known or indicated, when
 * generating HTTP constructs, since there might exist some implementations that
 * fail to accept anything.
 */
public class PortletResponseImpl extends com.ibm.servlet.engine.webapp.HttpServletResponseProxy implements PortletResponse
{
    /**
     * the PortletInstanceEntry associated with this Response
     */
    private PortletInstanceEntry portletInstance = null;
    /**
     * the corresponding PortletRequest
     */
    private PortletRequestImpl portletRequest = null;
    /**
     * the proxied ServletResponse
     */
    private HttpServletResponse servletResponse = null;
    /*
     * This servlet response is needed to set cookies and headers. Otherwise
     * they are filtered in the IncludedResponse, because WAS assumes that they are not allowed
     */
    private HttpServletResponse wpsWebAppServletResponse = null;

    /**
     * the servlet response filter to be used for all servlet
     * methods not refelected in the PortletRequest interface
     */
    private PortletResponseFilter responseFilter = null;
    /**
     * contains all headers that are not allowed to be set or changed
     */
    private static HashSet notAllowedHeaders = new HashSet();

    /**
     * indicates this response is used during the portlet's beginPage method
     */
    private boolean beginPageResponse = false;

    static
    {
        initHeaderSettings();
    }

    public PortletResponseImpl( PortletInstanceEntry portletInstance,
                                PortletRequestImpl request,
                                HttpServletResponse response,
                                boolean beginPageResponse )
    {
        this.portletInstance = portletInstance;
        this.portletRequest = request;
        this.servletResponse = response;
        this.wpsWebAppServletResponse = response;
        this.beginPageResponse = beginPageResponse;
        
        initFilter();
    }

    public HttpServletResponse getProxiedHttpServletResponse()
    {
        return servletResponse;
    }

    public void setProxiedResponse(HttpServletResponse response)
    {
        this.servletResponse = response;
        responseFilter.setServletResponse(servletResponse);
    }

    public PrintWriter getWriter () throws IOException
    {
        return servletResponse.getWriter(); 
    }

    public String getContentType()
    {
        return portletRequest.getProvider().getContentType();
    }

    public String getCharacterEncoding()
    {
        return portletRequest.getProvider().getCharacterSet();
    }

    public String getCharacterSet()
    {
        return getCharacterEncoding();
    }

    public PortletURI createURI ()
    {
        return new PortletURIImpl( portletInstance,
                                   null,
                                   null,
                                   portletRequest, 
                                   servletResponse );
    }

    public PortletURI createURI (PortletWindow.State state)
    {
        return new PortletURIImpl( portletInstance,
                                   null,
                                   state, 
                                   portletRequest, 
                                   servletResponse );
    }

    public PortletURI createReturnURI ()
    {
        if (portletRequest.getProvider().getReturnPortletURI( portletInstance, null ) != null)
        {
            return new PortletURIImpl( portletInstance,
                                       portletRequest, 
                                       servletResponse );
        }
        else return createURI();
    }

    public String encodeURL(String _path)
    {
        String path = _path;

        if (path.toLowerCase().startsWith("http://"))
        {
            // do nothing
        }
        else if (!path.startsWith("/WEB-INF"))
        { 
            WebModuleInformationProvider provider = 
                (WebModuleInformationProvider)portletRequest.getProvider();
            String portalContextRoot = provider.getPortalContextRoot();
            String webModuleContextRoot = provider.getWebModuleContextRoot();
            if ((path.startsWith(portalContextRoot)) || (path.startsWith(webModuleContextRoot)))
            {
               // do nothing
            }
            else {
                // bring the URL in the correct Namespace

                String baseDir = provider.getWebAppBaseDir();
                if ((baseDir.endsWith("/")) && (path.startsWith("/")))
                    baseDir = baseDir.substring(0,baseDir.length()-1);
                if ((!baseDir.endsWith("/")) && (!path.startsWith("/")))
                    path = "/" + path;
                int idx = baseDir.indexOf("://");
                idx = baseDir.indexOf("/",idx+4);
                if (idx==-1) idx = baseDir.length();
                String contextRoot = baseDir.substring(idx);
                if (path.startsWith(contextRoot))
                {
                    path = baseDir.substring(0,idx) + path;
                }
                else {
                    path = baseDir + path;
                }

            }
        }

        path = servletResponse.encodeURL(path);

        return path;
    }

    public String encodeURI (String path)
    {
        return encodeURL(path);
    }

    public String encodeNamespace (String aValue)
    {
        return PortletNamespaceMapper.encode(portletInstance.getPiid(),aValue); 
    }

    public void addCookie(javax.servlet.http.Cookie cookie)
    {
        if (!beginPageResponse)
            throw new IllegalStateException();

        if (!PortletNamespaceMapper.isInNamespace(portletInstance.getPiid(),cookie.getName()))
        {
            cookie = CookieConverter.encode(portletInstance.getPiid(), 
                                            cookie);
        }
        getWPSServletResponse().addCookie(cookie);
    }

    public boolean containsHeader(String name)
    {
        return servletResponse.containsHeader(name);
    }

    public void addDateHeader(String name, long date)
    {
        if ((!beginPageResponse) || (!isHeaderAllowed(name)))
            throw new IllegalStateException();

        getWPSServletResponse().addDateHeader(name, date);
    }

    public void setDateHeader(String name, long date)
    {
        if ((!beginPageResponse) || (!isHeaderAllowed(name)))
            throw new IllegalStateException();

        getWPSServletResponse().setDateHeader(name, date);
    }

    public void setHeader(String name, String value)
    {
        if ((!beginPageResponse) || (!isHeaderAllowed(name)))
            throw new IllegalStateException();

        getWPSServletResponse().setHeader(name, value);
    }

    public void addHeader(String name, String value)
    {
        if ((!beginPageResponse) || (!isHeaderAllowed(name)))
            throw new IllegalStateException();

        getWPSServletResponse().addHeader(name, value);
    }

    public void setIntHeader(String name, int value)
    {
        if ((!beginPageResponse) || (!isHeaderAllowed(name)))
            throw new IllegalStateException();

        getWPSServletResponse().setIntHeader(name, value);
    }
    
    public void addIntHeader(String name, int value)
    {
        if ((!beginPageResponse) || (!isHeaderAllowed(name)))
            throw new IllegalStateException();

        getWPSServletResponse().addIntHeader(name, value);
    }

    // servlet methods, which cannot be changed by the filter, because they are deprecated

    public void setStatus(int sc, String sm)
    {
        // deprectated
    }

    public void setStatus(int sc)
    {
        // deprectated
    }

    public String encodeUrl(String url)
    {
        // deprectated
        return encodeURL(url);
    }

    public String encodeRedirectUrl(String url)
    {
        // deprectated
        return url;
    }

    // servlet methods, which can be changed by the filter

    public String encodeRedirectURL(String url)
    {
        return responseFilter.encodeRedirectURL(url);
    }

    public void sendRedirect(String location) throws IOException
    {
        responseFilter.sendRedirect(location);
    }

    public void sendError(int sc, String msg) throws IOException
    {
        responseFilter.sendError(sc, msg);
    }

    public void sendError(int sc) throws IOException
    {
        responseFilter.sendError(sc);
    }

    public void setContentLength(int len)
    {
        responseFilter.setContentLength(len);
    }

    public ServletOutputStream getOutputStream() throws IOException
    {
        return responseFilter.getOutputStream();
    }

    public void setBufferSize(int size)
    {
        responseFilter.setBufferSize(size);
    }

    public void flushBuffer() throws IOException
    {
        responseFilter.flushBuffer();
    }

    public void setContentType(String type)
    {
        responseFilter.setContentType(type);
    }

    public void reset()
    {
        responseFilter.reset();
    }

    public int getBufferSize()
    {
        return responseFilter.getBufferSize();
    }

    public Locale getLocale()
    {
        return responseFilter.getLocale();
    }

    public boolean isCommitted()
    {
        return responseFilter.isCommitted();
    }

    public void setLocale(Locale loc)
    {
        responseFilter.setLocale(loc);
    }

    // additional methods

    public void setServletResponse(HttpServletResponse response)
    {
        this.servletResponse = response;
        responseFilter.setServletResponse(servletResponse);
    }

    public HttpServletResponse getWPSServletResponse()
    {
        return wpsWebAppServletResponse;
    }

    public void setWPSServletResponse(HttpServletResponse response)
    {
        this.wpsWebAppServletResponse = response;
    }

    public PortalInformationProvider getProvider()
    {
        return portletRequest.getProvider();
    }

    public void setProvider(PortalInformationProvider provider)
    {
        portletRequest.setProvider(provider);
    }

    public PortletRequestImpl getPortletRequestImpl()
    {
        return portletRequest;
    }

    private void initFilter()
    {
        String filterClassName = TurbineResources.getString("portletcontainer.filter.response");
        if (filterClassName==null) {
            responseFilter = new PortletResponseFilter(servletResponse,
                                                     this);
        }
        else {
            Class[] parameterClasses = {HttpServletResponse.class, PortletResponse.class};
            Constructor constructor = null;
            try {
                constructor = Class.forName(filterClassName).getConstructor(parameterClasses);
            }
            catch (NoSuchMethodException e) {
                Log.error("Invalid PortletResponseFilter defined in resources key portletcontainer.filter.response.",e);
                throw new IllegalArgumentException("Invalid PortletResponseFilter defined in resources key portletcontainer.filter.response.");
            }
            catch (ClassNotFoundException e) {
                Log.error("Invalid PortletResponseFilter defined in resources key portletcontainer.filter.response.",e);
                throw new IllegalArgumentException("Invalid PortletResponseFilter defined in resources key portletcontainer.filter.response.");
            }
            Object[] parameters = {servletResponse, this };
            try {
                Object filter = constructor.newInstance(parameters);
                if (!(filter instanceof PortletResponseFilter)) {
                    Log.error("PortletResponseFilter defined in resources key portletcontainer.filter.response does not derive from PortletResponseFilter.");
                    throw new IllegalArgumentException("PortletResponseFilter defined in resources key portletcontainer.filter.response does not derive from PortletResponseFilter.");
                }
                responseFilter = (PortletResponseFilter)filter;
            }
            catch (InstantiationException e) {
                Log.error("PortletResponseFilter defined in resources key portletcontainer.filter.response could not be instantiated.",e);
                throw new IllegalArgumentException("PortletResponseFilter defined in resources key portletcontainer.filter.response could not be instantiated.");
            }
            catch (IllegalAccessException e) {
                Log.error("PortletResponseFilter defined in resources key portletcontainer.filter.response could not be instantiated.",e);
                throw new IllegalArgumentException("PortletResponseFilter defined in resources key portletcontainer.filter.response could not be instantiated.");
            }
            catch (InvocationTargetException e) {
                Log.error("PortletResponseFilter defined in resources key portletcontainer.filter.response could not be instantiated.",e);
                throw new IllegalArgumentException("PortletResponseFilter defined in resources key portletcontainer.filter.response could not be instantiated.");
            }
        }
    }

    private static void initHeaderSettings()
    {
        // transform all string in the HashSet "notAllowedHeaders" to lower case, because
        // headers are case-insensitive
        notAllowedHeaders.add("Accept-Ranges".toLowerCase());
        notAllowedHeaders.add("Location".toLowerCase());
        notAllowedHeaders.add("Proxy-Authenticate".toLowerCase());
        notAllowedHeaders.add("Server".toLowerCase());
        notAllowedHeaders.add("Vary".toLowerCase());
        notAllowedHeaders.add("WWW-Authenticate".toLowerCase());
        notAllowedHeaders.add("Allow".toLowerCase());
        notAllowedHeaders.add("Content-Encoding".toLowerCase());
        notAllowedHeaders.add("Content-Language".toLowerCase());
        notAllowedHeaders.add("Content-Length".toLowerCase());
        notAllowedHeaders.add("Content-Location".toLowerCase());
        notAllowedHeaders.add("Content-MD5".toLowerCase());
        notAllowedHeaders.add("Content-Range".toLowerCase());
        notAllowedHeaders.add("Content-Type".toLowerCase());
        notAllowedHeaders.add("Expires".toLowerCase());
        notAllowedHeaders.add("Last-Modified".toLowerCase());

        String[] additionallyNotAllowed = 
            TurbineResources.getStringArray("portletcontainer.response.headers.additionallyNotAllowed");
        if (additionallyNotAllowed!=null)
        {
            for (int i=0; i<additionallyNotAllowed.length; i++)
            {
                notAllowedHeaders.add(additionallyNotAllowed[i].toLowerCase());
            }
        }

        String[] forceAllowed = 
            TurbineResources.getStringArray("portletcontainer.response.headers.forceAllowed");
        if (forceAllowed!=null)
        {
            for (int i=0; i<forceAllowed.length; i++)
            {
                notAllowedHeaders.remove(forceAllowed[i].toLowerCase());
            }
        }

    }

    private boolean isHeaderAllowed(String name)
    {
        return (!PortletResponseImpl.notAllowedHeaders.contains(name.toLowerCase()));
    }

}


