/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.portletcontainer.service;

import org.apache.jetspeed.portlet.PortletRequest;
import org.apache.jetspeed.portlet.PortletResponse;
import org.apache.jetspeed.portlet.PortletConfig;
import org.apache.jetspeed.portlet.service.*;
import org.apache.jetspeed.portlet.service.spi.*;
import org.apache.jetspeed.portletcontainer.*;
//import org.apache.jetspeed.cache.disk.*;
import org.apache.turbine.services.resources.TurbineResources;

// import org.apache.turbine.util.Log;

import com.ibm.logging.*;
import com.ibm.logging.mgr.*;


import java.util.*;
import java.net.*;
import java.io.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.*;

import com.ibm.net.ssl.*;
import java.security.*;
import javax.net.ssl.*;

public class ContentAccessServiceImpl implements ContentAccessService, PortletServiceProvider
{

    private String         proxyName          = null;
    private String         httpsProxyName     = null;
    private String[]       noProxyURLs        = null;
    private int            proxyPort          = 80;  
    private int            httpsProxyPort     = 443;  
    private int            maxFollowRedirects = 10; 
    private int            bufferSize         = 4096; 
    private ServletConfig  servletConfig      = null;
    private boolean        doUseDiskCache     = false;

    // key store and trust store to be used
    private URL            ksURL              = null;
    private String         ksPwd              = null;
    private String         ksFMT              = "JKS";
    private URL            tsURL              = null;
    private String         tsPwd              = null;
    private String         tsFMT              = "JKS";

    public final static String USE_DISK_CACHE       = "use.disk.cache";
    public final static String BUFFER_SIZE          = "buffer.size";
    public final static String PROXY_HTTP_HOST      = "proxy.http.host";
    public final static String PROXY_HTTPS_HOST     = "proxy.https.host";
    public final static String PROXY_HTTP_PORT      = "proxy.http.port";
    public final static String PROXY_HTTPS_PORT     = "proxy.https.port";
    public final static String MAX_FOLLOW_REDIRECTS = "max.follow.redirects";
    public final static String NO_PROXY_FOR         = "no.proxy.for";
    public final static String PROTOCOL_HANDLERS    = "protocol.handlers";
    public final static String KEY_STORE_URL        = "key.store.url";
    public final static String KEY_STORE_PSWD       = "key.store.pswd";
    public final static String KEY_STORE_FORMAT     = "key.store.format";
    public final static String TRUST_STORE_URL      = "trust.store.url";
    public final static String TRUST_STORE_PSWD     = "trust.store.pswd";
    public final static String TRUST_STORE_FORMAT   = "trust.store.format";


    private LogManager logMgr = LogManager.getManager();
    private ILogger    msgLog = logMgr.getMessageLogger("TurbineMessageLogger");
    private ILogger    trcLog = logMgr.getMessageLogger("TurbineTraceLogger");

    public void destroy()
    {
        logMgr.returnObject(msgLog);
        logMgr.returnObject(trcLog);
    }

    public void init(PortletServiceConfig config)
    {
        // Write trace info of method entry, type is public method
        if (trcLog.isLogging()) trcLog.entry(IRecordType.TYPE_PUBLIC, this, "init()");

        // read the parms from the service properties - or take defaults

        // TODO: PFISHER - remove turbine resources stuff

        servletConfig = config.getServletConfig();

        String bufS = config.getInitParameter(BUFFER_SIZE);
        if (bufS == null)
        {
            bufferSize = TurbineResources.getInt("services.JspService.buffer.size", 4096);
        } else
        {
            bufferSize = Integer.parseInt(bufS);
        }

        proxyName = config.getInitParameter(PROXY_HTTP_HOST);
        if (proxyName == null)
        {
            proxyName = TurbineResources.getString("services.URLManager.proxy.http.host", null);
        }

        String port = config.getInitParameter(PROXY_HTTP_PORT);
        if (port == null)
        {
            proxyPort = TurbineResources.getInt("services.URLManager.proxy.http.port", -1);
        } else
        {
            proxyPort = Integer.parseInt(port);
        }

        httpsProxyName = config.getInitParameter(PROXY_HTTPS_HOST);
        if (httpsProxyName == null)
        {
            httpsProxyName = TurbineResources.getString("services.URLManager.proxy.https.host", null);
        }

        String httpsPort = config.getInitParameter(PROXY_HTTPS_PORT);
        if (httpsPort == null)
        {
            httpsProxyPort = TurbineResources.getInt("services.URLManager.proxy.https.port", -1);
        } else
        {
            httpsProxyPort = Integer.parseInt(port);
        }

        String max = config.getInitParameter(MAX_FOLLOW_REDIRECTS);
        if ((max != null) && !max.equals("") )
        {
            try
            {
                maxFollowRedirects = Integer.parseInt(max);
            } catch (Exception e)
            {
                maxFollowRedirects = 10;
            }
        }

        String noProxyFor = config.getInitParameter(NO_PROXY_FOR);
        if (noProxyFor != null)
        {
            StringTokenizer st = new StringTokenizer(noProxyFor, ";");
            noProxyURLs = new String[st.countTokens()];
            int i= 0;
            while (st.hasMoreElements())
            {
                noProxyURLs[i++] = ((String)st.nextElement()).toLowerCase();
            }
        } else
        {
            noProxyURLs = new String[0];
        }

        // get the protocol handler
        String configProtoStr   = config.getInitParameter(PROTOCOL_HANDLERS);
        String jetspeedProtoStr = TurbineResources.getString("services.URLManager.protocol.handlers", null);

        // if a jetspeed handler was specified
        if ((jetspeedProtoStr!=null) && (!jetspeedProtoStr.equals(""))) {

            // see if the handler has already been added
            String handlers = System.getProperty("java.protocol.handler.pkgs");
            if (handlers == null) {

                // nothing specified yet
                System.setProperty("java.protocol.handler.pkgs", jetspeedProtoStr);

                if (trcLog.isLogging()) trcLog.text(IRecordType.TYPE_PUBLIC, this, "init()", "protocol handler set to {0} from jetspeed", jetspeedProtoStr);

            } else {

                // something already there, if new handler not already in string
                if (handlers.indexOf(jetspeedProtoStr) == -1) {
                    // put new handler in front
                    System.setProperty("java.protocol.handler.pkgs", jetspeedProtoStr+"|"+handlers);

                    if (trcLog.isLogging()) trcLog.text(IRecordType.TYPE_PUBLIC, this, "init()", "protocol handler set to {0} from jetspeed", jetspeedProtoStr+"|"+handlers);

                }
            }
        }

        // if a handler was specified as a context parameter
        if ((configProtoStr!=null) && (!configProtoStr.equals(""))) {

            // see if the handler has already been added
            String handlers = System.getProperty("java.protocol.handler.pkgs");
            if (handlers == null) {

                // nothing specified yet
                System.setProperty("java.protocol.handler.pkgs", configProtoStr);

                if (trcLog.isLogging()) trcLog.text(IRecordType.TYPE_PUBLIC, this, "init()", "protocol handler set to {0} from config", configProtoStr);

            } else {

                // something already there, if new handler not already in string
                if (handlers.indexOf(configProtoStr) == -1) {

                    // put new handler in front
                    System.setProperty("java.protocol.handler.pkgs", configProtoStr+"|"+handlers);

                    if (trcLog.isLogging()) trcLog.text(IRecordType.TYPE_PUBLIC, this, "init()", "protocol handler set to {0} from config", configProtoStr+"|"+handlers);

                }
            }
        }

        // get the keystore and password
        String ksURLStr = config.getInitParameter(KEY_STORE_URL);
        String ksPwdStr = config.getInitParameter(KEY_STORE_PSWD);

        // if url is non-null
        if ((ksURLStr != null) && (!ksURLStr.equals(""))) {

            // the password must also be non-null
            if ((ksPwdStr != null) && (!ksPwdStr.equals(""))) {

                try {
                    ksURL = toURLOrFile(ksURLStr);
                    ksPwd = ksPwdStr;

                    if (trcLog.isLogging()) trcLog.text(IRecordType.TYPE_PUBLIC, this, "init()", "key store URL set to {0}", ksURL.toString());

                }
                catch (MalformedURLException mue)
                {
                    // log an error
                    if (msgLog.isLogging()) 
                    {
                        msgLog.text(IRecordType.TYPE_ERROR, this, "init()", "key store URL {0} could not be recognized as a readable file or URL", ksURLStr);
                    }
                }
            }
            else {
                // keystore URL is non-null, but password is null
                if (msgLog.isLogging()) msgLog.text(IRecordType.TYPE_ERROR, this, "init()", "key store {0} was found without a password", ksURLStr);
            }
        }
        else {

            // keystore URL is null.  if password is not null, log an error
            if ((ksPwdStr != null) && (!ksPwdStr.equals(""))) {
                if (msgLog.isLogging()) msgLog.text(IRecordType.TYPE_ERROR, this, "init()", "key store password was found without a key store");
            }
        }
        
        // get the keystore format
        String ksFMTStr = config.getInitParameter(KEY_STORE_FORMAT);

        if ((ksFMTStr != null) && (!ksFMTStr.equals(""))) {
            ksFMT = ksFMTStr;

            if (trcLog.isLogging()) trcLog.text(IRecordType.TYPE_PUBLIC, this, "init()", "key store format set to {0}", ksFMT);

        }
        
        // get the truststore
        String tsURLStr = config.getInitParameter(TRUST_STORE_URL);

        // if url is non-null
        if ((tsURLStr != null) && (!tsURLStr.equals(""))) {

            try {
                tsURL = toURLOrFile(tsURLStr);

                if (trcLog.isLogging()) trcLog.text(IRecordType.TYPE_PUBLIC, this, "init()", "trust store URL set to {0}", tsURL.toString());

            }
            catch (MalformedURLException mue) {
                // log an error
                if (msgLog.isLogging()) 
                {
                    msgLog.text(IRecordType.TYPE_ERROR, this, "init()", "trust store URL {0} could not be recognized as a readable file or URL", tsURLStr);
                }
            }
        }
        
        // get the truststore password (if specified)
        String tsPwdStr = config.getInitParameter(TRUST_STORE_PSWD);

        if ((tsPwdStr != null) && (!tsPwdStr.equals(""))) {
            tsPwd = tsPwdStr;

            if (trcLog.isLogging()) trcLog.text(IRecordType.TYPE_PUBLIC, this, "init()", "trust store password found");

        }
        
        // get the truststore format
        String tsFMTStr = config.getInitParameter(TRUST_STORE_FORMAT);

        if ((tsFMTStr != null) && (!tsFMTStr.equals(""))) {
            tsFMT = tsFMTStr;

            if (trcLog.isLogging()) trcLog.text(IRecordType.TYPE_PUBLIC, this, "init()", "trust store format set to {0}", tsFMT);

        }
        
        // Write trace info of method entry, type is public method
        if (trcLog.isLogging()) trcLog.exit(IRecordType.TYPE_PUBLIC, this, "init()");

    }

    // convert the string into a network URL (no file URLs)
    private URL toURLOnly(String toChange) throws MalformedURLException
    {

        URL res = null;

        // check for null URL string
        if ( (toChange == null ) || toChange.equals("") )
        {
            throw new MalformedURLException("No URL given");
        }

        res = new URL(toChange);

        return res;
    }

    // convert the string into a network or file url
    private URL  toURLOrFile(String toChange) throws MalformedURLException
    {

        URL res = null;

        // first see if it is a file
        try
        {   
            File f = new File(toChange);

            if (f.isFile() && f.canRead()) {

                return f.toURL();
            }
            else
            {
                return toURLOnly(toChange);
            }
        }
        catch (NullPointerException npe)
        {
            throw new MalformedURLException("No URL given");
        }
        catch (SecurityException se)
        {
            throw new MalformedURLException("File "+toChange+" could not be read");
        }
    }

    private boolean isProxyNeeded(URL url)
    {

        boolean res = true;

        // if no proxies have been defined for this protocol
        if ((url.getProtocol().equals("http") 
             && ((proxyName == null) || proxyName.equals(""))) 
            || (url.getProtocol().equals("https")
                && ((httpsProxyName == null) || httpsProxyName.equals("")))) 
        {
            res = false;
        }
        else
        {
            // proxy has been defined, is this URL already pointing to the proxy?
            if (url.getHost().toLowerCase().equals(proxyName)) {
                // URL already points to proxy, so return false
                res = false;
            }
            else
            {

                String urlHost = url.getHost();

                // run through the list of files not needing a proxy
                for (int i=0; i < noProxyURLs.length; i++)
                {
                    // if a match is found
                    if (urlHost.startsWith(noProxyURLs[i]))
                    {
                        // no proxy is needed
                        res = false;
                        break;
                    }
                }
            }
        }

        return res;
    }

    // is this url for the local host?
    private boolean isLocal(URL url, String hostName)
    {

        boolean res = false;

        String  urlProtocol      = url.getProtocol();
        String  urlHostName      = url.getHost();
        int     dotIndx          = 0;

        // if there is no host name (i.e. a file: url
        if ((urlHostName == null) || (urlHostName.equals(""))) {

            // assume the host is local
            res = true;
        }
        else 
        {
            // check to see if the url points to this server
            res = urlHostName.equals(hostName) 
                  || urlHostName.equals("127.0.0.1")
                  || urlHostName.equals("localhost");
        }
                
        return res;

    }

    // keep accessing the http url until you don't get a redirect 
    private URL followRedirects(URL url, HttpServletRequest hsr) throws MalformedURLException, PortletServiceException
    {

        // if redirects are not allowed
        if (maxFollowRedirects <= 0) {
            // return the original url
            return url;
        }

        int               count                = 0;
        int               response             = 0;
        String            HTTP_LOCATION_HEADER = "Location";       //taken from the HTTP RFC
        HttpURLConnection conn                 = null;
        Enumeration       headerNames          = hsr.getHeaderNames();
        URL               tmpURL               = url;
        String            urlProtocol          = url.getProtocol();

        do
        {
            // if this URL is only accessible through a proxy
            if (isProxyNeeded(tmpURL))
            {
                // prepend the URL with the name of the correct proxy
                if (urlProtocol.equals("http")) {
                    tmpURL = new URL("http", proxyName, proxyPort, " "+tmpURL.toString());
                }
                else if (urlProtocol.equals("https")) {
                    tmpURL = new URL("https", httpsProxyName, httpsProxyPort, " "+tmpURL.toString());
                }
            }

            try
            {
                // create the connection
                conn = (HttpURLConnection) tmpURL.openConnection();

                // set the SSLSocketFactory if necessary
                setSSLSocketFactory(conn);

                conn.disconnect();

                // loop through the header names in the original request and copy select ones to the new URL request
                while (headerNames.hasMoreElements())
                {
                    String headerName = (String) headerNames.nextElement();

                    if (
                       !headerName.equalsIgnoreCase("Connection")    &&
                       !headerName.equalsIgnoreCase("Age")           &&
                       !headerName.equalsIgnoreCase("Cache-Control") &&
                       !headerName.equalsIgnoreCase("Expires")       &&
                       !headerName.toLowerCase().startsWith("if")    &&
                       !headerName.equalsIgnoreCase("Last-Modified") &&
                       !headerName.equalsIgnoreCase("Pragma")        &&
                       !headerName.equalsIgnoreCase("TE")            &&
                       !headerName.equalsIgnoreCase("Upgrade")       &&
                       !headerName.equalsIgnoreCase("content-length")&&  
                       !headerName.equalsIgnoreCase("content-type")  &&  
                       !headerName.equalsIgnoreCase("accept")        &&  
                       !headerName.equalsIgnoreCase("Vary")
                       )
                    {
                        conn.setRequestProperty(headerName, hsr.getHeader(headerName));
                    }
                }   

                conn.setRequestProperty("accept", "*/*");

                conn.setFollowRedirects(false);

                conn.connect();

                response = conn.getResponseCode();

            } 
            catch (IOException ioe)
            {

                if (msgLog.isLogging()) 
                {
                    msgLog.exception(IRecordType.TYPE_ERROR, this, "followRedirects()", ioe);
                }

                throw new PortletServiceUnavailableException ();
            }

            // if the response indicates this is a redirect, and we have not exceeded the max redirects
            if ( (response>=300) && (response<400) && (count<maxFollowRedirects) )
            {

                String tmpUrlString = conn.getHeaderField(HTTP_LOCATION_HEADER);

                if (tmpUrlString == null)
                {
                    throw new PortletServiceException("Redirection failed - dead end");
                }
                else
                {
                    tmpURL = new URL(tmpUrlString);
                }

                count++;
            }

            if (count > maxFollowRedirects)
            {
                throw new PortletServiceException("Redirection failed - too many redirects - see properties file for portlet services");
            }

        }
        while ( (response>=300) && (response<400) );

        return tmpURL;
    }

    // return a URL based on the string and the proxy configuration
    public URL getURL(String urlString,
                      PortletRequest req,
                      PortletResponse resp) throws PortletServiceException, MalformedURLException
    {

        // Write trace info of method entry, type is public method
        if (trcLog.isLogging()) trcLog.entry(IRecordType.TYPE_PUBLIC, this, "getURL()", urlString);

        // see if URL can parse the urlString
        URL retURL = null;

        retURL = toURLOnly(urlString);

        // if this is http or https, check for redirects
        String urlProtocol = retURL.getProtocol();

        if ((urlProtocol.equals("http")) || (urlProtocol.equals("https")))
        {
            retURL = followRedirects(retURL, req);
        }

        // if a proxy is needed
        if (isProxyNeeded(retURL)) {

            // create a url prepended with the proxy
            String retURLProtocol = retURL.getProtocol();

            if (retURLProtocol.equals("http")) {
                retURL = new URL("http", proxyName, proxyPort, " "+retURL.toString());
            }
            else if (retURLProtocol.equals("https")) {
                retURL = new URL("https", httpsProxyName, httpsProxyPort, " "+retURL.toString());
            }
        }


        // Write trace info of method entry, type is public method
        if (trcLog.isLogging()) trcLog.exit(IRecordType.TYPE_PUBLIC, this, "getURL()", retURL);

        return retURL;

    }

    // set the SSLSocketFactory in the connection (if one is needed)
    private void setSSLSocketFactory(URLConnection conn)
    {

        // for now, only works on https connections
        if (!(conn instanceof HttpsURLConnection)) {
            return;
        }

        KeyManager[]   km = null;
        TrustManager[] tm = null;

        // if a keystore has been specified, define its key managers
        if (ksURL != null) {

            try
            {
                KeyStore ks = KeyStore.getInstance(ksFMT);
                ks.load(ksURL.openStream(), null);
                KeyManagerFactory kmf = KeyManagerFactory.getInstance("IbmX509");
                kmf.init(ks, ksPwd.toCharArray());
                km = kmf.getKeyManagers();

                // if there is not a trust store url, create a default one
                if (tsURL == null) {
                    TrustManagerFactory tmf = TrustManagerFactory.getInstance("IbmX509");
                    tmf.init(null);
                    tm = tmf.getTrustManagers();    
                }
            } 
            catch (java.security.NoSuchAlgorithmException nsae)
            {
                if (msgLog.isLogging()) 
                {
                    msgLog.text(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", "GeneralSecurityException while creating key store {0}", ksURL.toString());
                    msgLog.exception(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", nsae);
                }
            }
            catch (java.security.GeneralSecurityException gse)
            {
                if (msgLog.isLogging()) 
                {
                    msgLog.text(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", "GeneralSecurityException while creating key store {0}", ksURL.toString());
                    msgLog.exception(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", gse);
                }
            }
            catch (java.io.IOException ioe)
            {
                if (msgLog.isLogging()) 
                {
                    msgLog.text(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", "IOException while creating key store {0}", ksURL.toString());
                    msgLog.exception(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", ioe);
                }
            }
        }

        // if a truststore has been specified, define its key manager
        if (tsURL != null) {

            try
            {
                KeyStore ts = KeyStore.getInstance(tsFMT);
                ts.load(tsURL.openStream(), null);
                TrustManagerFactory tmf = TrustManagerFactory.getInstance("IbmX509");
                tmf.init(ts);
                tm = tmf.getTrustManagers();

                // if there is not a key store url, create a default one
                if (ksURL == null) {
                    KeyManagerFactory kmf = KeyManagerFactory.getInstance("IbmX509");
                    kmf.init(null,null);
                    km = kmf.getKeyManagers();    
                }
            } 
            catch (java.security.GeneralSecurityException gse)
            {
                if (msgLog.isLogging()) 
                {
                    msgLog.text(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", "GeneralSecurityException while creating trust store {0}", tsURL.toString());
                    msgLog.exception(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", gse);
                }
            }
            catch (java.io.IOException ioe)
            {
                if (msgLog.isLogging()) 
                {
                    msgLog.text(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", "IOException while creating trust store {0}", tsURL.toString());
                    msgLog.exception(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", ioe);
                }
            }
        }                

        // if either a key manager or a trust manager was defined
        if ((km != null) || (tm != null)) {

            SSLSocketFactory factory = null;

            try 
            {
                SSLContext sslContext = SSLContext.getInstance("SSL");
                sslContext.init(km, tm, null);
                factory = sslContext.getSocketFactory();
            }
            catch (java.security.GeneralSecurityException gse)
            {
                if (msgLog.isLogging()) 
                {
                    msgLog.text(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", "GeneralSecurityException while creating socket factory from key store {0} and trust store {1}", ksURL.toString(), tsURL.toString());
                    msgLog.exception(IRecordType.TYPE_ERROR, this, "setSSLSocketFactory()", gse);
                }
            }

            ((HttpsURLConnection)conn).setSSLSocketFactory(factory);
        }

    }

    public InputStream getInputStream(String urlString,
                                      PortletRequest req,
                                      PortletResponse resp) throws PortletServiceException, MalformedURLException

    {

        // Write trace info of method entry, type is public method
        if (trcLog.isLogging()) trcLog.entry(IRecordType.TYPE_PUBLIC, this, "getInputStream()", urlString);

        InputStream stream = null;

        try
        {
            URLConnection conn = null;
            URL           url  = getURL (urlString, req, resp);

            conn = (URLConnection) url.openConnection();

            if (conn == null)
            {
                throw new PortletServiceException ("Connection null");
            }
            else
            {
                setSSLSocketFactory(conn);
            }
            
            stream = conn.getInputStream();

        } catch (IOException ioe)
        {
            throw new PortletServiceException (ioe.getMessage());
        }

        // Write trace info of method entry, type is public method
        if (trcLog.isLogging()) trcLog.exit(IRecordType.TYPE_PUBLIC, this, "getInputStream()", stream);

        return stream;

    }



    public String getMarkup(String urlString,
                            PortletRequest req,
                            PortletResponse resp) throws PortletServiceException, MalformedURLException

    {

        // Write trace info of method entry, type is public method
        if (trcLog.isLogging()) trcLog.entry(IRecordType.TYPE_PUBLIC, this, "getMarkup()", urlString);

        try
        {
            InputStream inStream = getInputStream(urlString, req, resp);

            if (inStream == null)
            {
                throw new PortletServiceException ("InputStream null");
            }


            //((PortletResponseImpl)resp).getServletResponse().flushBuffer();

            StringWriter writer = new StringWriter();

            //writer.flush();

            int rc = 0;
            byte [] buffer = new byte[bufferSize];

            while (rc >= 0)
            {
                writer.write(new String(buffer).substring(0,rc));

                rc = inStream.read(buffer);
            }

            //writer.flush();


            // Write trace info of method entry, type is public method
            if (trcLog.isLogging()) trcLog.exit(IRecordType.TYPE_PUBLIC, this, "getMarkup()");

            return writer.getBuffer().toString();

        } catch (IOException ioe)
        {
            throw new PortletServiceException (ioe.toString());
        }
    }

    public void include(String urlString,
                        PortletRequest req,
                        PortletResponse resp) throws PortletServiceException, MalformedURLException

    {        

        // Write trace info of method entry, type is public method
        if (trcLog.isLogging()) trcLog.entry(IRecordType.TYPE_PUBLIC, this, "include()", urlString);

        
        String  loweredUrl = urlString.toLowerCase();
        
        if (!loweredUrl.startsWith("http://") && !loweredUrl.startsWith("https://") && (servletConfig != null))
        {

            ServletContext servletContext = servletConfig.getServletContext();

            RequestDispatcher dispatcher = servletContext.getRequestDispatcher(urlString);
            if (dispatcher != null)
            {

                try
                {

                    dispatcher.include(req, resp);

                    // Write trace info of method entry, type is public method
                    if (trcLog.isLogging()) trcLog.exit(IRecordType.TYPE_PUBLIC, this, "include()");

                    return;

                } catch (Throwable t)
                {
                    // Do nothing - we just could not invoke url via RequestDispatcher - now try it otherwise
                }
            }

        }

        try
        {
            PrintWriter writer = resp.getWriter();
            writer.print( getMarkup(urlString, req, resp));

        } catch (IOException ioe)
        {
            throw new PortletServiceException (ioe.toString());
        }

        // Write trace info of method entry, type is public method
        if (trcLog.isLogging()) trcLog.exit(IRecordType.TYPE_PUBLIC, this, "include()");

    }

}
