/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.portletcontainer.services.portletserviceregistry;

import org.apache.jetspeed.portlet.*;
import org.apache.jetspeed.portlet.service.*;
import org.apache.jetspeed.portlet.service.spi.*;
import org.apache.jetspeed.portletcontainer.service.*;

//turbine stuff
import org.apache.turbine.util.Log;
import org.apache.turbine.services.*;
import org.apache.turbine.services.resources.ResourceService;

import java.util.Properties;
import java.io.*;

import javax.servlet.ServletConfig;

/**
 * the PortletServiceRegistry manages the configured
 * PortletServices
 * 
 * @author Peter Fischer (peter.fischer@de.ibm.com)
 * @author <a href="mailto:stephan.hesmer@de.ibm.com">Stephan Hesmer</a>
 */
public final class FilePortletServiceRegistry extends TurbineBaseService 
                                              implements PortletServiceRegistryService {

    // default keys
    public final static String DEFAULT_CONFIGFILE       = "WEB-INF/conf/PortletServices.properties";
    // configuration keys
    private final static String CONFIG_CONFIGFILE       = ".configfile";

    private ServletConfig servletConfig = null;


    /**
     * contains the available portlet services 
     * including service configuration properties
     */
    private static Properties serviceProperties = new Properties();

/**
 * This is the early initialization method called by the
 * Turbine <code>Service</code> framework
 * @param conf The <code>ServletConfig</code>
 * @exception throws a <code>InitializationException</code> if the service
 * fails to initialize
 */
    public void init(ServletConfig conf) throws InitializationException
    {
        org.apache.turbine.util.Log.note ( "FilePortletServiceRegistryService early init()....starting!");

        servletConfig = conf;

        Properties props = TurbineServices.getInstance()
                                          .getProperties(PortletServiceRegistryService.SERVICE_NAME);
        
        String configfile = props.getProperty(CONFIG_CONFIGFILE,DEFAULT_CONFIGFILE);

        try {

            File  f = new File(configfile);
            if (!f.isAbsolute()) {
                String fileName = conf.getServletContext().getRealPath(configfile);
                f = new File(fileName); 
            }

            if ( f.exists() && (f.length() > 0) ) {
                FileInputStream in = new FileInputStream(f);
                serviceProperties.load(in);
            }
        
        } catch (IOException e) {
            Log.error("Failed to load PortletServiceEntries",e);
            throw new InitializationException("Failed to load PortletServiceEntries",e);
        }


        org.apache.turbine.util.Log.note ( "FilePortletServiceRegistryService early init()....finished!");
        setInit(true);
    }

    /**
     * This is the shutdown method called by the
     * Turbine <code>Service</code> framework
     */
    public void shutdown()
    {
        
    }




    /**
     * getPortletService finds the specified service
     * 
     * @param clazz  the name of the PortletService
     * @return the PortletService object
     * 
     */
    public PortletService getPortletService(Class service) throws PortletServiceNotFoundException,
                                                                  PortletServiceUnavailableException
    {

        PortletServiceFactory psf = null;

        // first look if there is an impl given for this service
        String impl = (String) serviceProperties.get(service.getName());

        // ok - is there some impl?
        if ((impl == null) || impl.equals("")) {
            org.apache.turbine.util.Log.
                warn("FilePortletServiceRegistry.getPortletService(Class): Service not registered.");
            throw new PortletServiceNotFoundException();
        }

        // and is the impl available?
        Class implClass;
        try {
            implClass = Class.forName(impl);

        } catch (ClassNotFoundException cnfe) {
            org.apache.turbine.util.Log.
                warn("FilePortletServiceRegistry.getPortletService(Class): No valid service implementation registered: " + cnfe);
            throw new PortletServiceNotFoundException();
        }

        // now we need a factory for that impl
        Object factory = serviceProperties.get(impl + ".factory");

        // no factory given? - try default factory
        if (factory == null) {
            factory = serviceProperties.get("org.apache.jetspeed.portlet.service.default.factory");
        }

        String errMsg = "";

        // serviceFactory is not instanciated yet if it is a String
        if (factory instanceof String) {

            try {
                Class factoryClass = Class.forName((String) factory);
                psf = (PortletServiceFactory) factoryClass.newInstance();
            } catch (ClassNotFoundException cnfe) {
                errMsg = cnfe.toString();
            } catch (java.lang.InstantiationException ie) {
                errMsg = ie.toString();
            } catch (IllegalAccessException iae) {
                errMsg = iae.toString();
            }

            if (psf != null) {
                serviceProperties.put(impl + ".factory", psf);
            }
        } else {
            // otherwise we allready hold an instance
            psf = (PortletServiceFactory) factory;
        }

        if (psf == null) {
            org.apache.turbine.util.Log.
                warn("FilePortletServiceRegistry.getPortletService(Class): Factory could not be instanciated: " + errMsg);
            throw new PortletServiceUnavailableException();
        }

        // create the service now and return it
        return (PortletService) psf.createPortletService(implClass, serviceProperties, servletConfig);
    }

}
