/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
 
package org.apache.jetspeed.portletcontainer.tags;

import java.lang.reflect.*;
import java.util.Hashtable;
import java.util.StringTokenizer;
import javax.servlet.jsp.*;
import javax.servlet.jsp.tagext.TagSupport;
import org.apache.jetspeed.portlet.*;
import org.apache.jetspeed.portletcontainer.Constants;

/**
 ** Supporting class for the <CODE>if</CODE> tag.
 ** Allows to check for different states used in the Portlet API. 
 ** Specifying more than one parameter results in a logical AND of these conditions. 
 ** That is only if all conditions evaluate to true the body of the tag will be executed. 
 ** Tag parameter:
 ** <UL>
 ** <LI><CODE>mode</CODE> (optional) - checks if the current portlet mode matches to the given one
 ** <LI><CODE>previousMode</CODE> (optional) - checks if the previous portlet mode matches to the given one
 ** <LI><CODE>state</CODE> (optional) - checks if the current portlet window state is one of the possible values
 **     <UL>
 **     <LI><CODE>MAXIMIZED</CODE>, <CODE>MAX</CODE>
 **     <LI><CODE>MINIMIZED</CODE>, <CODE>MIN</CODE>
 **     <LI><CODE>DETACHED</CODE>
 **     </UL>
 ** <LI><CODE>markup</CODE> (optional) - checks if the current markup matches to the given one
 ** <LI><CODE>mimetype</CODE> (optional) - checks if the current mimetype matches to the given one
 ** <LI><CODE>capability</CODE> (optional) - checks if the current client is capable of the given one
 ** </UL>
 ** @see org.apache.jetspeed.portlet.Portlet.Mode
 ** @see org.apache.jetspeed.portlet.PortletWindow.State
 ** @see org.apache.jetspeed.portlet.Capability
 **
 ** @author   <a href="mailto:h.schwarz@de.ibm.com">Hagen Schwarz</a>
 **/
public class IfTag extends TagSupport
{
    private static Hashtable capabilities = new Hashtable();

    static {
        Field[] fields;

        //Create hashtables with fieldnames
        try
        {
            //For parameter iCapability
            fields = Capability.class.getFields();
            for (int i = 0; i < fields.length; i++)
            {
                String name = fields[i].getName();
                capabilities.put(name,fields[i].get(null));
            }
        }
        catch (IllegalAccessException exc)
        {
            System.err.println("IfTag: IfTag(): Error with Field.get(null): " + exc.toString());
        }
    }
    
    /**
     * Processes the <CODE>if</CODE> start tag.
     * Evaluates the conditions.
     * @return <CODE>EVAL_BODY_INCLUDE</CODE> - if all conditions are true
     *         <BR><CODE>SKIP_BODY</CODE> - else
     *
     */
    public int doStartTag ()
    {
        PortletRequest portletRequest = (PortletRequest)pageContext.getRequest().getAttribute(Constants.PORTLET_REQUEST);

        if (iMarkup != null)
        {
            Client client = portletRequest.getClient();

            if (client != null &&
                ! included (iMarkup, client.getMarkupName()))
            {
                return (SKIP_BODY);
            }
        }

        if (iMimetype != null)
        {
            Client client = portletRequest.getClient();

            if (client != null &&
                ! included (iMimetype, client.getMimeType()))
            {
                return (SKIP_BODY);
            }
        }

        if (iMode != null)
        {
            if (! included (iMode, portletRequest.getMode() ))
            {
                return (SKIP_BODY);
            }

        }
        
        if (iPreviousMode != null)
        {
            if (! included (iPreviousMode, portletRequest.getPreviousMode() ))
            {
                return (SKIP_BODY);
            }
        }
        
        if (iState != null)
        {
            PortletWindow.State state = portletRequest.getWindow().getWindowState();

            if (state != null &&
                ! included (iState, state ))
            {
                return (SKIP_BODY);
            }
        }
        
        if (iCapability != null)
        {
            Client client = portletRequest.getClient();

            if (client != null &&
                ! includedCapabilities (iCapability, client ))
            {
                return (SKIP_BODY);
            }
        }

        return EVAL_BODY_INCLUDE;
    }
    
    /**
     * Processes the <CODE>if</CODE> end tag.
     * @return <CODE>EVAL_PAGE</CODE>
     */
    public final int doEndTag ()
    {
        resetCustomAttributes ();

        return EVAL_PAGE;
    }

    /**
    * The setter for the <CODE>mode</CODE> parameter.
    * @param aMode java.lang.String
    */
    public void setMode(String aMode)
    {
        iMode = aMode;
    }

    /**
    * The setter for the <CODE>previousMode</CODE> parameter.
    * @param aPreviousMode java.lang.String
    */
    public void setPreviousMode(String aPreviousMode)
    {
        iPreviousMode = aPreviousMode;
    }

    /**
    * The setter for the <CODE>state</CODE> parameter.
    * @param aState java.lang.String
    */
    public void setState(String aState)
    {
        iState = aState;
    }

    /**
    * The setter for the <CODE>markup</CODE> parameter.
    * @param aMarkup java.lang.String
    */
    public void setMarkup(String aMarkup)
    {
        iMarkup = aMarkup;
    }

    /**
    * The setter for the <CODE>mimetype</CODE> parameter.
    * @param aMimetype java.lang.String
    */
    public void setMimetype(String aMimetype)
    {
        iMimetype = aMimetype;
    }

    /**
    * The setter for the <CODE>capability</CODE> parameter.
    * @param aCapability java.lang.String
    */
    public void setCapability(String aCapability)
    {
        iCapability = aCapability;
    }


    private String iMode = null;
    private String iPreviousMode = null;
    private String iState = null;
    private String iMarkup = null;
    private String iMimetype = null;
    private String iCapability = null;

    private void resetCustomAttributes ()
    {
        iMode = null;
        iPreviousMode = null;
        iState = null;
        iMarkup = null;
        iMimetype = null;
        iCapability = null;
    }

    private boolean included (String aList, Object aItem)
    {
        boolean result = false;

        if (aList != null && aItem != null)
        {
            StringTokenizer tokenizer = new StringTokenizer (aList, ",;");
            String aItemString = aItem.toString();

            while (tokenizer.hasMoreTokens ())
            {
                String token = tokenizer.nextToken ();

                if (token.equalsIgnoreCase (aItemString))
                {
                    result = true;
                    break;
                }
            }
        }

        return(result);
    }

    private boolean includedCapabilities (String aList, Client aClient)
    {
        boolean result = false;

        if (aList != null && aClient != null)
        {
            StringTokenizer tokenizer = new StringTokenizer (aList, ",;");

            while (tokenizer.hasMoreTokens ())
            {
                String token = tokenizer.nextToken ();

                if (aClient.isCapableOf((Capability)capabilities.get(token.toUpperCase())))
                {
                    result = true;
                    break;
                }
            }
        }

        return(result);
    }

}
