/*
* Licensed to the Apache Software Foundation (ASF) under one or more
* contributor license agreements.  See the NOTICE file distributed with
* this work for additional information regarding copyright ownership.
* The ASF licenses this file to You under the Apache License, Version 2.0
* (the "License"); you may not use this file except in compliance with
* the License.  You may obtain a copy of the License at
*
*     http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/
package org.apache.jetspeed.aggregator;

import org.apache.jetspeed.cache.ContentCacheKey;
import org.apache.jetspeed.portlet.HeadElement;
import org.apache.jetspeed.util.KeyValue;

import javax.portlet.PortletMode;
import javax.portlet.WindowState;
import java.io.PrintWriter;
import java.util.List;

/**
 * Portlet content is a container holding the generated content of a fragment (portlet). It is used by the aggregation
 * engine to abstract access to rendered content per portlet. Content is represented as strings. This interface
 * also supports multi-threaded aggregation, and can be checked to determine if the generation of content has completed.
 * Content can also be cached. Expiration values are represented in this interface.
 *
 * @author <a href="mailto:weaver@apache.org">Scott T. Weaver</a>
 * @author <a href="mailto:taylor@apache.org">David S. Taylor</a>  
 * @version $Id: PortletContent.java 1510080 2013-08-03 23:35:19Z taylor $
 *
 */
public interface PortletContent
{
    /**
     * Retrieve the actual content of a portlet as a string
     * 
     * @return the content of a given fragment (portlet)
     */
    String getContent();
    
    /** 
     * Has the renderer completed rendering the content? Used by the multi-threaded (parallel) rendering engine.
     * 
     * @return <tt>true</tt> if the rendering of this fragment's content has completed.
     */
    boolean isComplete();
    
    /**
     * Notifies that this content is completed.
     *
     */
    void complete();
    
    /**
     * Get a writer to the content to stream content into this object
     *
     * @return a writer into this content's stream
     */
    PrintWriter getWriter();
    
    /**
     * Get the expiration setting for this content if it is cached.
     *
     * @return the expiration value in seconds
     */
    int getExpiration();
    
    /**
     * Sets the expiration setting for this content in seconds
     * 
     * @param expiration the expiration value in seconds
     */
    void setExpiration(int expiration);
    
    /**
     * Get the cache key used to cache this content
     *
     * @since 2.1.2 
     * @return the name of the cache key associated with this content
     */
    ContentCacheKey getCacheKey();
    
    /**
     * Get the cached title of the portlet
     * 
     * @return the title of the portlet
     */
    String getTitle();
    
    /**
     * Set the title of this portlet, used during caching
     *
     * @param title the title of this portlet
     */
    void setTitle(String title);
    
    /**
     * Gets the MIME content type of this portlet content.
     * 
     * @return the MIME content type
     */
    String getContentType();
    
    /**
     * Sets the MIME content type of this portlet content.
     * 
     * @param contentType the MIME content type
     */
    void setContentType(String contentType);
    
    /**
     * Gets the portlet mode of the portlet instance associated with this portlet content.
     * 
     * @return the portlet mode
     */
    PortletMode getPortletMode();
    
    /**
     * Gets the window state of the portlet instance associated with this portlet content.
     * 
     * @return the window state
     */
    WindowState getWindowState();
    
    /**
     * Adds an header element to be contributed to the page.
     * If keyHint argument is provided and if a header element 
     * with the provided key hint already exists, then 
     * the element will be overwritten to the item.
     * If the element is null and keyHint is not null, 
     * the element with the keyHint is removed from the response.
     * If the element is not null and keyHint is null,
     * then the keyHint will be automatically generated by the element content. 
     * Meanwhile, the element should implement java.io.Serializable.
     * Otherwise it will throw a java.io.NotSerializableException.
     * 
     * @param element the header element to contribute to the page
     * @param keyHint the key hint to generate the header element
     */
    void addHeadElement(HeadElement element, String keyHint);
    
    /**
     * Retrieves header element key value pairs to be contributed to the page.
     * Because the insertion order might be important for web development, it should be list instead of map.
     * 
     * @return list of key value pairs of head element names to HeadElements
     */
    List<KeyValue<String, HeadElement>> getHeadElements();

    /**
     * Reset the buffer of this instance to empty. Also resets all state associated with this content.
     *
     */
    void reset();

    /**
     * Reset the buffer of this instance to empty. Does not reset state associated with this content.
     *
     */
    void resetBuffer();
    
    /**
     * Release the buffers used by the portlet content cache. Note the actual release may not occur until garbage collection. 
     *
     */
    void release();                    
}