/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.syncope.common.rest.api.service;

import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.headers.Header;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.security.SecurityRequirements;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.constraints.NotNull;
import jakarta.ws.rs.Consumes;
import jakarta.ws.rs.DELETE;
import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.PUT;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import java.util.List;
import org.apache.syncope.common.lib.policy.PolicyTO;
import org.apache.syncope.common.lib.types.PolicyType;
import org.apache.syncope.common.rest.api.RESTHeaders;

/**
 * REST operations for policies.
 */
@Tag(name = "Policies")
@SecurityRequirements({
    @SecurityRequirement(name = "BasicAuthentication"),
    @SecurityRequirement(name = "Bearer") })
@Path("policies")
public interface PolicyService extends JAXRSService {

    /**
     * Returns the policy matching the given key.
     *
     * @param type policy type
     * @param key key of requested policy
     * @param <T> response type (extending PolicyTO)
     * @return policy with matching id
     */
    @GET
    @Path("{type}/{key}")
    @Produces({ MediaType.APPLICATION_JSON })
    <T extends PolicyTO> T read(@NotNull @PathParam("type") PolicyType type, @NotNull @PathParam("key") String key);

    /**
     * Returns a list of policies of the matching type.
     *
     * @param type Type selector for requested policies
     * @param <T> response type (extending PolicyTO)
     * @return list of policies with matching type
     */
    @GET
    @Path("{type}")
    @Produces({ MediaType.APPLICATION_JSON })
    <T extends PolicyTO> List<T> list(@NotNull @PathParam("type") PolicyType type);

    /**
     * Create a new policy.
     *
     * @param type policy type
     * @param policyTO Policy to be created (needs to match type)
     * @return Response object featuring Location header of created policy
     */
    @ApiResponses(
            @ApiResponse(responseCode = "201",
                    description = "Policy successfully created", headers = {
                @Header(name = RESTHeaders.RESOURCE_KEY, schema =
                        @Schema(type = "string"),
                        description = "UUID generated for the entity created"),
                @Header(name = HttpHeaders.LOCATION, schema =
                        @Schema(type = "string"),
                        description = "URL of the entity created") }))
    @POST
    @Path("{type}")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Produces({ MediaType.APPLICATION_JSON })
    Response create(@NotNull @PathParam("type") PolicyType type, @NotNull PolicyTO policyTO);

    /**
     * Updates policy matching the given key.
     *
     * @param type policy type
     * @param policyTO Policy to replace existing policy
     */
    @Parameter(name = "key", description = "Policy's key", in = ParameterIn.PATH, schema =
            @Schema(type = "string"))
    @ApiResponses(
            @ApiResponse(responseCode = "204", description = "Operation was successful"))
    @PUT
    @Path("{type}/{key}")
    @Consumes({ MediaType.APPLICATION_JSON })
    @Produces({ MediaType.APPLICATION_JSON })
    void update(@NotNull @PathParam("type") PolicyType type, @NotNull PolicyTO policyTO);

    /**
     * Delete policy matching the given key.
     *
     * @param type policy type
     * @param key key of policy to be deleted
     */
    @ApiResponses(
            @ApiResponse(responseCode = "204", description = "Operation was successful"))
    @DELETE
    @Path("{type}/{key}")
    @Produces({ MediaType.APPLICATION_JSON })
    void delete(@NotNull @PathParam("type") PolicyType type, @NotNull @PathParam("key") String key);

}
