/*
 * Decompiled with CFR 0.152.
 */
package org.apache.unomi.persistence.spi.conditions.datemath;

import java.time.DateTimeException;
import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.TemporalQueries;
import java.util.function.Function;
import java.util.function.LongSupplier;
import org.apache.unomi.persistence.spi.conditions.datemath.DateMathParseException;
import org.apache.unomi.persistence.spi.conditions.datemath.JavaDateFormatter;

public class DateMathParser {
    private final JavaDateFormatter formatter;
    private final DateTimeFormatter roundUpFormatter;

    public static boolean isNullOrEmpty(CharSequence cs) {
        return cs == null || cs.length() == 0;
    }

    public DateMathParser(JavaDateFormatter formatter, DateTimeFormatter roundUpFormatter) {
        this.formatter = formatter;
        this.roundUpFormatter = roundUpFormatter;
    }

    private String normalizeDateMathInput(String input) {
        input = input.replaceAll("(?<=\\d{4}-\\d{2}-\\d{2})t", "T");
        input = input.replaceAll("z$", "Z");
        input = input.replaceAll("(?<=[:\\d])z", "Z");
        return input;
    }

    public Instant parse(String text, LongSupplier now, boolean roundUpProperty, ZoneId timeZone) {
        String mathString;
        Instant time;
        if ((text = text.trim()).startsWith("now")) {
            try {
                time = Instant.ofEpochMilli(now.getAsLong());
            }
            catch (Exception e) {
                throw new DateMathParseException("could not read the current timestamp", e);
            }
            mathString = text.substring("now".length());
        } else {
            int index = text.indexOf("||");
            if (index == -1) {
                text = this.normalizeDateMathInput(text);
                return this.parseDateTime(text, timeZone, roundUpProperty);
            }
            time = this.parseDateTime(this.normalizeDateMathInput(text.substring(0, index).trim()), timeZone, false);
            mathString = text.substring(index + 2).trim();
        }
        return this.parseMath(mathString, time, roundUpProperty, timeZone);
    }

    private Instant parseMath(String mathString, Instant time, boolean roundUpProperty, ZoneId timeZone) throws DateMathParseException {
        if (timeZone == null) {
            timeZone = ZoneOffset.UTC;
        }
        ZonedDateTime dateTime = ZonedDateTime.ofInstant(time, timeZone);
        int i = 0;
        while (i < mathString.length()) {
            int num;
            int sign;
            boolean round;
            char c;
            if ((c = mathString.charAt(i++)) == '/') {
                round = true;
                sign = 1;
            } else {
                round = false;
                if (c == '+') {
                    sign = 1;
                } else if (c == '-') {
                    sign = -1;
                } else {
                    throw new DateMathParseException("operator not supported for date math [%s]", mathString);
                }
            }
            if (i >= mathString.length()) {
                throw new DateMathParseException("truncated date math [%s]", mathString);
            }
            int numStart = i;
            if (!Character.isDigit(mathString.charAt(i))) {
                num = 1;
            } else {
                while (i < mathString.length() && Character.isDigit(mathString.charAt(i))) {
                    ++i;
                }
                if (i >= mathString.length()) {
                    throw new DateMathParseException("truncated date math [%s]", mathString);
                }
                num = Integer.parseInt(mathString.substring(numStart, i));
            }
            if (round && num != 1) {
                throw new DateMathParseException("rounding `/` can only be used on single unit types [%s]", mathString);
            }
            char unit = mathString.charAt(i++);
            switch (unit) {
                case 'y': {
                    if (round) {
                        dateTime = dateTime.withDayOfYear(1).with(LocalTime.MIN);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusYears(1L);
                        break;
                    }
                    dateTime = dateTime.plusYears(sign * num);
                    break;
                }
                case 'M': {
                    if (round) {
                        dateTime = dateTime.withDayOfMonth(1).with(LocalTime.MIN);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusMonths(1L);
                        break;
                    }
                    dateTime = dateTime.plusMonths(sign * num);
                    break;
                }
                case 'w': {
                    if (round) {
                        dateTime = dateTime.with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY)).with(LocalTime.MIN);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusWeeks(1L);
                        break;
                    }
                    dateTime = dateTime.plusWeeks(sign * num);
                    break;
                }
                case 'd': {
                    if (round) {
                        dateTime = dateTime.with(LocalTime.MIN);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusDays(1L);
                        break;
                    }
                    dateTime = dateTime.plusDays(sign * num);
                    break;
                }
                case 'H': 
                case 'h': {
                    if (round) {
                        dateTime = dateTime.withMinute(0).withSecond(0).withNano(0);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusHours(1L);
                        break;
                    }
                    dateTime = dateTime.plusHours(sign * num);
                    break;
                }
                case 'm': {
                    if (round) {
                        dateTime = dateTime.withSecond(0).withNano(0);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusMinutes(1L);
                        break;
                    }
                    dateTime = dateTime.plusMinutes(sign * num);
                    break;
                }
                case 's': {
                    if (round) {
                        dateTime = dateTime.withNano(0);
                        if (!roundUpProperty) break;
                        dateTime = dateTime.plusSeconds(1L);
                        break;
                    }
                    dateTime = dateTime.plusSeconds(sign * num);
                    break;
                }
                default: {
                    String unitString = mathString.substring(numStart, numStart + Integer.toString(num).length()) + unit;
                    throw new DateMathParseException("unit [%s] not supported for date math [%s]", Character.valueOf(unit), unitString);
                }
            }
            if (!round || !roundUpProperty) continue;
            dateTime = dateTime.minus(1L, ChronoField.MILLI_OF_SECOND.getBaseUnit());
        }
        return dateTime.toInstant();
    }

    private Instant parseDateTime(String value, ZoneId timeZone, boolean roundUpIfNoTime) {
        if (DateMathParser.isNullOrEmpty(value)) {
            throw new DateMathParseException("cannot parse empty date");
        }
        Function<String, TemporalAccessor> parser = roundUpIfNoTime ? this.roundUpFormatter::parse : this.formatter::parse;
        try {
            TemporalAccessor accessor = parser.apply(value);
            ZonedDateTime zdt = DateFormatters.from(accessor);
            if (timeZone != null) {
                zdt = zdt.withZoneSameInstant(timeZone);
            }
            return zdt.toInstant();
        }
        catch (Throwable t) {
            throw new DateMathParseException("failed to parse date field [%s] with format [%s]: [%s]", value, this.formatter, t.getMessage());
        }
    }

    public static class DateFormatters {
        public static ZonedDateTime from(TemporalAccessor accessor) {
            ZoneId zone = accessor.query(TemporalQueries.zone());
            if (zone == null) {
                ZoneOffset offset = accessor.query(TemporalQueries.offset());
                ZoneId zoneId = zone = offset != null ? offset : ZoneOffset.UTC;
            }
            if (accessor.isSupported(ChronoField.INSTANT_SECONDS)) {
                return ZonedDateTime.ofInstant(Instant.from(accessor), zone);
            }
            LocalDate date = accessor.query(TemporalQueries.localDate());
            LocalTime time = accessor.query(TemporalQueries.localTime());
            if (date == null && time == null) {
                throw new DateTimeException("Cannot extract LocalDate or LocalTime from TemporalAccessor");
            }
            if (date == null) {
                date = LocalDate.ofEpochDay(0L);
            }
            if (time == null) {
                time = LocalTime.MIDNIGHT;
            }
            return ZonedDateTime.of(date, time, zone);
        }
    }
}

