use strict;
use warnings;
package JSON::Schema::Modern::Utilities;
# vim: set ts=8 sts=2 sw=2 tw=100 et :
# ABSTRACT: Internal utilities for JSON::Schema::Modern

our $VERSION = '0.631';

use 5.020;
use strictures 2;
use stable 0.031 'postderef';
use experimental 'signatures';
no autovivification warn => qw(fetch store exists delete);
use if "$]" >= 5.022, experimental => 're_strict';
no if "$]" >= 5.031009, feature => 'indirect';
no if "$]" >= 5.033001, feature => 'multidimensional';
no if "$]" >= 5.033006, feature => 'bareword_filehandles';
no if "$]" >= 5.041009, feature => 'smartmatch';
no feature 'switch';
use B;
use Carp 'croak';
use builtin::compat qw(blessed created_as_number);
use Scalar::Util 'looks_like_number';
use Storable 'dclone';
use Feature::Compat::Try;
use Mojo::JSON ();
use JSON::PP ();
use Types::Standard qw(Str InstanceOf);
use Mojo::File 'path';
use namespace::clean;

use Exporter 'import';

our @EXPORT_OK = qw(
  is_type
  get_type
  is_bool
  is_schema
  is_bignum
  is_equal
  is_elements_unique
  jsonp
  unjsonp
  jsonp_set
  local_annotations
  canonical_uri
  E
  A
  abort
  assert_keyword_exists
  assert_keyword_type
  assert_pattern
  assert_uri_reference
  assert_uri
  annotate_self
  sprintf_num
  true
  false
  json_pointer_type
  canonical_uri_type
  register_schema
  load_cached_document
);

use constant HAVE_BUILTIN => "$]" >= 5.035010;
use if HAVE_BUILTIN, experimental => 'builtin';

use constant _BUILTIN_BOOLS => 0;
use constant {
  _BUILTIN_BOOLS && HAVE_BUILTIN && eval { +require Storable; Storable->VERSION(3.27); 1 }
      && Mojo::JSON::JSON_XS && eval { Cpanel::JSON::XS->VERSION(4.38); 1 }
    ? (true => builtin::true, false => builtin::false)
    : (true => JSON::PP::true, false => JSON::PP::false)
};

# supports the six core types, plus integer (which is also a number)
# we do NOT check stringy_numbers here -- you must do that in the caller
# note that sometimes a value may return true for more than one type, e.g. integer+number,
# or number+string, depending on its internal flags.
# pass { legacy_ints => 1 } in $config to use draft4 integer behaviour
# behaviour is consistent with get_type() (where integers are also numbers).
sub is_type ($type, $value, $config = {}) {
  if ($type eq 'null') {
    return !(defined $value);
  }
  if ($type eq 'boolean') {
    return is_bool($value);
  }
  if ($type eq 'object') {
    return ref $value eq 'HASH';
  }
  if ($type eq 'array') {
    return ref $value eq 'ARRAY';
  }

  if ($type eq 'string' or $type eq 'number' or $type eq 'integer') {
    return 0 if not defined $value;
    my $flags = B::svref_2object(\$value)->FLAGS;

    # dualvars with the same string and (stringified) numeric value could be either a string or a
    # number, and before 5.36 we can't tell the difference, so we will answer yes for both.
    # in 5.36+, stringified numbers still get a PV but don't have POK set, whereas
    # numified strings do have POK set, so we can tell which one came first.

    if ($type eq 'string') {
      # like created_as_string, but rejects dualvars with stringwise-unequal string and numeric parts
      return !length ref($value)
        && !(HAVE_BUILTIN && builtin::is_bool($value))
        && $flags & B::SVf_POK
        && (!($flags & (B::SVf_IOK | B::SVf_NOK))
            || do { no warnings 'numeric'; 0+$value eq $value });
    }

    if ($type eq 'number') {
      # floats in json will always be parsed into Math::BigFloat, when allow_bignum is enabled
      return is_bignum($value) || created_as_number($value);
    }

    if ($type eq 'integer') {
      if ($config->{legacy_ints}) {
        # in draft4, an integer is "A JSON number without a fraction or exponent part.",
        # therefore 2.0 is NOT an integer
        return ref($value) eq 'Math::BigInt'
          || ($flags & B::SVf_IOK) && !($flags & B::SVf_NOK) && created_as_number($value);
      }
      else {
        # note: values that are larger than $Config{ivsize} will be represented as an NV, not IV,
        # therefore they will fail this check -- which is why use of Math::BigInt is recommended
        # if the exact type is important, or loss of any accuracy is unacceptable
        return is_bignum($value) && $value->is_int
          # if dualvar, PV and stringified NV/IV must be identical
          || created_as_number($value) && int($value) == $value;
      }
    }
  }

  if ($type =~ /^reference to (.+)\z/) {
    return !blessed($value) && ref($value) eq $1;
  }

  return ref($value) eq $type;
}

# returns one of the six core types, plus integer
# we do NOT check stringy_numbers here -- you must do that in the caller
# pass { legacy_ints => 1 } in $config to use draft4 integer behaviour
# behaviour is consistent with is_type().
sub get_type ($value, $config = {}) {
  return 'object' if ref $value eq 'HASH';
  return 'boolean' if is_bool($value);
  return 'null' if not defined $value;
  return 'array' if ref $value eq 'ARRAY';

  # floats in json will always be parsed into Math::BigFloat, when allow_bignum is enabled
  if (length ref $value) {
    my $ref = ref $value;
    return $ref eq 'Math::BigInt' ? 'integer'
      : $ref eq 'Math::BigFloat' ? (!$config->{legacy_ints} && $value->is_int ? 'integer' : 'number')
      : (defined blessed($value) ? '' : 'reference to ').$ref;
  }

  my $flags = B::svref_2object(\$value)->FLAGS;

  # dualvars with the same string and (stringified) numeric value could be either a string or a
  # number, and before 5.36 we can't tell the difference, so we choose number because it has been
  # evaluated as a number already.
  # in 5.36+, stringified numbers still get a PV but don't have POK set, whereas
  # numified strings do have POK set, so we can tell which one came first.

  # like created_as_string, but rejects dualvars with stringwise-unequal string and numeric parts
  return 'string'
    if $flags & B::SVf_POK
      && (!($flags & (B::SVf_IOK | B::SVf_NOK))
        || do { no warnings 'numeric'; 0+$value eq $value });

  if ($config->{legacy_ints}) {
    # in draft4, an integer is "A JSON number without a fraction or exponent part.",
    # therefore 2.0 is NOT an integer
    return ($flags & B::SVf_IOK) && !($flags & B::SVf_NOK) ? 'integer' : 'number'
      if created_as_number($value);
  }
  else {
    # note: values that are larger than $Config{ivsize} will be represented as an NV, not IV,
    # therefore they will fail this check -- which is why use of Math::BigInt is recommended
    # if the exact type is important, or loss of any accuracy is unacceptable
    return int($value) == $value ? 'integer' : 'number' if created_as_number($value);
  }

  # this might be a scalar with POK|IOK or POK|NOK set
  return 'ambiguous type';
}

# lifted from JSON::MaybeXS
# note: unlike builtin::compat::is_bool on older perls, we do not accept
# dualvar(0,"") or dualvar(1,"1") because JSON::PP and Cpanel::JSON::XS
# do not encode these as booleans.
sub is_bool ($value) {
  HAVE_BUILTIN and builtin::is_bool($value)
  or
  !!blessed($value)
    and ($value->isa('JSON::PP::Boolean')
      or $value->isa('Cpanel::JSON::XS::Boolean')
      or $value->isa('JSON::XS::Boolean'));
}

sub is_schema ($value) {
  ref $value eq 'HASH' || is_bool($value);
}

sub is_bignum ($value) {
  ref($value) =~ /^Math::Big(?:Int|Float)\z/;
}

# compares two arbitrary data payloads for equality, as per
# https://json-schema.org/draft/2020-12/json-schema-core.html#rfc.section.4.2.2
# $state hashref supports the following fields:
# - scalarref_booleans (input): treats \0 and \1 as boolean values
# - stringy_numbers (input): strings will also be compared numerically
# - path (output): location of the first difference
# - error (output): description of the first difference
sub is_equal ($x, $y, $state = {}) {
  $state->{path} //= '';

  my @types = map get_type($_), $x, $y;

  $state->{error} = 'ambiguous type encountered', return 0
    if grep $types[$_] eq 'ambiguous type', 0..1;

  if ($state->{scalarref_booleans}) {
    ($x, $types[0]) = (0+!!$$x, 'boolean') if $types[0] eq 'reference to SCALAR';
    ($y, $types[1]) = (0+!!$$y, 'boolean') if $types[1] eq 'reference to SCALAR';
  }

  if ($state->{stringy_numbers}) {
    ($x, $types[0]) = (0+$x, int(0+$x) == $x ? 'integer' : 'number')
      if $types[0] eq 'string' and looks_like_number($x);

    ($y, $types[1]) = (0+$y, int(0+$y) == $y ? 'integer' : 'number')
      if $types[1] eq 'string' and looks_like_number($y);
  }

  $state->{error} = "wrong type: $types[0] vs $types[1]", return 0 if $types[0] ne $types[1];
  return 1 if $types[0] eq 'null';
  ($x eq $y and return 1), $state->{error} = 'strings not equal', return 0
    if $types[0] eq 'string';
  ($x == $y and return 1), $state->{error} = "$types[0]s not equal", return 0
    if grep $types[0] eq $_, qw(boolean number integer);

  my $path = $state->{path};
  if ($types[0] eq 'object') {
    $state->{error} = 'property count differs: '.keys(%$x).' vs '.keys(%$y), return 0
      if keys %$x != keys %$y;

    if (not is_equal(my $arr_x = [ sort keys %$x ], my $arr_y = [ sort keys %$y ], my $s={})) {
      my $pos = substr($s->{path}, 1);
      $state->{error} = 'property names differ starting at position '.$pos.' ("'.$arr_x->[$pos].'" vs "'.$arr_y->[$pos].'")';
      return 0;
    }

    foreach my $property (sort keys %$x) {
      $state->{path} = jsonp($path, $property);
      return 0 if not is_equal($x->{$property}, $y->{$property}, $state);
    }

    return 1;
  }

  if ($types[0] eq 'array') {
    $state->{error} = 'element count differs: '.@$x.' vs '.@$y, return 0 if @$x != @$y;
    foreach my $idx (0 .. $x->$#*) {
      $state->{path} = $path.'/'.$idx;
      return 0 if not is_equal($x->[$idx], $y->[$idx], $state);
    }
    return 1;
  }

  $state->{error} = 'uh oh', return 0; # should never get here
}

# checks array elements for uniqueness. short-circuits on first pair of matching elements
# $state hashref supports the following fields:
# - scalarref_booleans (input): treats \0 and \1 as boolean values
# - stringy_numbers (input): strings will also be compared numerically
# - path (output): location of the first difference
# - error (output): description of the first difference
# - equal_indices (output): the indices of identical items
sub is_elements_unique ($array, $state = {}) {
  foreach my $idx0 (0 .. $array->$#*-1) {
    foreach my $idx1 ($idx0+1 .. $array->$#*) {
      if (is_equal($array->[$idx0], $array->[$idx1], $state)) {
        push $state->{equal_indices}->@*, $idx0, $idx1 if exists $state->{equal_indices};
        return 0;
      }
    }
  }
  return 1;
}

# shorthand for creating and appending json pointers
# the first argument is an already-encoded json pointer; remaining arguments are path segments to be
# encoded and appended
sub jsonp {
  warn q{first argument to jsonp should be '' or start with '/'} if length($_[0]) and substr($_[0],0,1) ne '/';
  return join('/', shift, map s!~!~0!gr =~ s!/!~1!gr, grep defined, @_);
}

# splits a json pointer apart into its path segments
sub unjsonp {
  warn q{argument to unjsonp should be '' or start with '/'} if length($_[0]) and substr($_[0],0,1) ne '/';
  return map s!~0!~!gr =~ s!~1!/!gr, split m!/!, $_[0];
}

# assigns a value to a data structure at a specific json pointer location
# operates destructively, in place, unless the root data or type is being modified
sub jsonp_set ($data, $pointer, $value) {
  if (not grep ref $data eq $_, qw(HASH ARRAY)) {
    return $value if defined wantarray;
    croak 'cannot write into non-reference in void context';
  }

  # assigning to the root overwrites existing data
  if (not length $pointer) {
    if (ref $data eq 'HASH' and ref $value ne 'HASH'
        or ref $data eq 'ARRAY' and ref $value ne 'ARRAY') {
      return $value if defined wantarray;
      croak 'cannot write into reference of different type in void context';
    }

    $data->%* = $value->%* if ref $data eq 'HASH';
    $data->@* = $value->@* if ref $data eq 'ARRAY';
    return $data;
  }

  my @keys = map +(s!~0!~!gr =~ s!~1!/!gr),
    (length $pointer ? (split /\//, $pointer, -1) : ($pointer));

  croak 'cannot write hashref into a reference to an array in void context'
    if @keys >= 2 and $keys[1] !~ /^\d+\z/a and ref $data eq 'ARRAY' and not defined wantarray;

  shift @keys;  # always '', indicating the root
  my $curp = \$data;

  foreach my $key (@keys) {
    # if needed, first remove the existing data so we can replace with a new hash key or array index
    undef $curp->$*
      if not ref $curp->$*
        or ref $curp->$* eq 'ARRAY' and $key !~ /^\d+\z/a;

    # use this existing hash key or array index location, or create new position
    use autovivification 'store';
    $curp = \(
      ref $curp->$* eq 'HASH' || $key !~ /^\d+\z/a
        ? $curp->$*->{$key}
        : $curp->$*->[$key]);
  }

  $curp->$* = $value;
  return $data;
}

# returns a reusable Types::Standard type for json pointers
# TODO: move this off into its own distribution, see JSON::Schema::Types
sub json_pointer_type () { Str->where('!length || m{^/} && !m{~(?![01])}'); }

# a URI without a fragment, or with a json pointer fragment
sub canonical_uri_type () {
  (InstanceOf['Mojo::URL'])->where(q{!defined($_->fragment) || $_->fragment =~ m{^/} && $_->fragment !~ m{~(?![01])}});
}

# simple runtime-wide cache of $ids to schema document objects that are sourced from disk
{
  my $document_cache = {};

  # Fetches a document from the cache (reading it from disk and creating the document if necessary),
  # and add it to the evaluator.
  # Normally this will just be a cache of schemas that are bundled with this distribution or a related
  # distribution (such as OpenAPI-Modern), as duplicate identifiers are not checked for, unlike for
  # normal schema additions.
  # Only JSON-encoded files are supported at this time.
  sub load_cached_document ($evaluator, $uri) {
    $uri =~ s/#\z//; # older draft $ids use an empty fragment

    # see if it already exists as a document in the cache
    my $document = $document_cache->{$uri};

    # otherwise, load it from disk using our filename cache and create the document
    if (not $document and my $filename = get_schema_filename($uri)) {
      my $file = path($filename);
      die "uri $uri maps to file $file which does not exist" if not -f $file;
      my $schema = $evaluator->_json_decoder->decode($file->slurp);

      # avoid calling add_schema, which checksums the file to look for duplicates
      $document = JSON::Schema::Modern::Document->new(schema => $schema, evaluator => $evaluator);

      # avoid calling add_document, which checks for duplicate identifiers (and would result in an
      # infinite loop)
      die JSON::Schema::Modern::Result->new(
        output_format => $evaluator->output_format,
        valid => 0,
        errors => [ $document->errors ],
        exception => 1,
      ) if $document->has_errors;

      $document_cache->{$uri} = $document;
    }

    return if not $document;

    # bypass the normal collision checks, to avoid an infinite loop: these documents are presumed safe
    $evaluator->_add_resources_unsafe(
      map +($_->[0] => +{ $_->[1]->%*, document => $document }),
        $document->resource_pairs
    );

    return $document;
  }
}

######## NO PUBLIC INTERFACES FOLLOW THIS POINT ########

# get all annotations produced for the current instance data location (that are visible to this
# schema location) - remember these are hashrefs, not Annotation objects
sub local_annotations ($state) {
  grep $_->{instance_location} eq $state->{data_path}, $state->{annotations}->@*;
}

# shorthand for finding the current uri of the present schema location
# ensure that this code is kept consistent with the absolute_keyword_location builder in ResultNode
# Note that this may not be canonical if keyword_path has not yet been reset via the processing of a
# local identifier keyword (e.g. '$id').
sub canonical_uri ($state, @extra_path) {
  return $state->{initial_schema_uri} if not @extra_path and not length($state->{keyword_path});
  my $uri = $state->{initial_schema_uri}->clone;
  my $fragment = ($uri->fragment//'').(@extra_path ? jsonp($state->{keyword_path}, @extra_path) : $state->{keyword_path});
  undef $fragment if not length($fragment);
  $uri->fragment($fragment);
  $uri;
}

# shorthand for creating error objects
# uses these keys from $state:
# - initial_schema_uri
# - keyword (optional)
# - data_path
# - traversed_keyword_path
# - keyword_path
# - _keyword_path_suffix (optional)
# - errors
# - exception (optional; set by abort())
# - recommended_response (optional)
# - depth
# - traverse (boolean, used for mode)
# returns defined-false, so callers can use 'return;' to differentiate between
# failed-with-no-error from failed-with-error.
sub E ($state, $error_string, @args) {
  croak 'E called in void context' if not defined wantarray;

  # sometimes the keyword shouldn't be at the very end of the schema path
  my $sps = delete $state->{_keyword_path_suffix};
  my @keyword_path_suffix = defined $sps && ref $sps eq 'ARRAY' ? $sps->@* : $sps//();

  # we store the absolute uri in unresolved form until needed,
  # and perform the rest of the calculations later.
  my $uri = [ $state->@{qw(initial_schema_uri keyword_path)}, $state->{keyword}//(), @keyword_path_suffix ];

  my $keyword_location = $state->{traversed_keyword_path}
    .jsonp($state->@{qw(keyword_path keyword)}, @keyword_path_suffix);

  require JSON::Schema::Modern::Error;
  push $state->{errors}->@*, JSON::Schema::Modern::Error->new(
    depth => $state->{depth} // 0,
    keyword => $state->{keyword},
    $state->{traverse} ? () : (instance_location => $state->{data_path}),
    keyword_location => $keyword_location,
    # we calculate absolute_keyword_location when instantiating the Error object for Result
    _uri => $uri,
    error => @args ? sprintf($error_string, @args) : $error_string,
    exception => $state->{exception},
    ($state->%{recommended_response})x!!$state->{recommended_response},
    mode => $state->{traverse} ? 'traverse' : 'evaluate',
  );

  return 0;
}

# shorthand for creating annotations
# uses these keys from $state:
# - initial_schema_uri
# - keyword (mandatory)
# - data_path
# - traversed_keyword_path
# - keyword_path
# - annotations
# - collect_annotations
# - _unknown (boolean)
# - depth
sub A ($state, $annotation) {
  # even if the user requested annotations, we only collect them for later drafts
  # ..but we always collect them if the lowest bit is set, indicating the presence of unevaluated*
  # keywords necessary for accurate validation
  return 1 if not ($state->{collect_annotations}
    & ($state->{specification_version} =~ /^draft[467]\z/ ? ~(1<<8) : ~0));

  # we store the absolute uri in unresolved form until needed,
  # and perform the rest of the calculations later.
  my $uri = [ $state->@{qw(initial_schema_uri keyword_path keyword)} ];

  my $keyword_location = $state->{traversed_keyword_path}.jsonp($state->@{qw(keyword_path keyword)});

  push $state->{annotations}->@*, {
    depth => $state->{depth} // 0,
    keyword => $state->{keyword},
    instance_location => $state->{data_path},
    keyword_location => $keyword_location,
    # we calculate absolute_keyword_location when instantiating the Annotation object for Result
    _uri => $uri,
    annotation => $annotation,
    $state->{_unknown} ? (unknown => 1) : (),
  };

  return 1;
}

# creates an error object, but also aborts evaluation immediately
# only this error is returned, because other errors on the stack might not actually be "real"
# errors (consider if we were in the middle of evaluating a "not" or "if").
# Therefore this is only appropriate during the evaluation phase, not the traverse phase.
sub abort ($state, $error_string, @args) {
  ()= E({ %$state, exception => 1 }, $error_string, @args);
  croak 'abort() called during traverse' if $state->{traverse};
  die pop $state->{errors}->@*;
}

sub assert_keyword_exists ($state, $schema) {
  croak 'assert_keyword_exists called in void context' if not defined wantarray;
  return E($state, '%s keyword is required', $state->{keyword}) if not exists $schema->{$state->{keyword}};
  return 1;
}

sub assert_keyword_type ($state, $schema, $type) {
  croak 'assert_keyword_type called in void context' if not defined wantarray;
  return 1 if is_type($type, $schema->{$state->{keyword}});
  E($state, '%s value is not a%s %s', $state->{keyword}, ($type =~ /^[aeiou]/ ? 'n' : ''), $type);
}

sub assert_pattern ($state, $pattern) {
  croak 'assert_pattern called in void context' if not defined wantarray;
  try {
    local $SIG{__WARN__} = sub { die @_ };
    qr/$pattern/;
  }
  catch ($e) { return E($state, $e); };
  return 1;
}

# this is only suitable for checking URIs within schemas themselves
# note that we cannot use $state->{specification_version} to more tightly constrain the plain-name
# fragment syntax, as we could be checking a $ref to a schema using a different version
sub assert_uri_reference ($state, $schema) {
  croak 'assert_uri_reference called in void context' if not defined wantarray;

  my $string = $schema->{$state->{keyword}};
  return E($state, '%s value is not a valid URI-reference', $state->{keyword})
    # see also uri-reference format sub
    if fc(Mojo::URL->new($string)->to_unsafe_string) ne fc($string)
      or $string =~ /[^[:ascii:]]/            # ascii characters only
      or $string =~ /#/                       # no fragment, except...
        and $string !~ m{#\z}                          # allow empty fragment
        and $string !~ m{#[A-Za-z_][A-Za-z0-9_:.-]*\z} # allow plain-name fragment, superset of all drafts
        and $string !~ m{#/(?:[^~]|~[01])*\z};         # allow json pointer fragment

  return 1;
}

# this is only suitable for checking URIs within schemas themselves,
# which have fragments consisting of plain names (anchors) or json pointers
sub assert_uri ($state, $schema, $override = undef) {
  croak 'assert_uri called in void context' if not defined wantarray;

  my $string = $override // $schema->{$state->{keyword}};
  my $uri = Mojo::URL->new($string);

  return E($state, '"%s" is not a valid URI', $string)
    # see also uri format sub
    if fc($uri->to_unsafe_string) ne fc($string)
      or $string =~ /[^[:ascii:]]/            # ascii characters only
      or not $uri->is_abs                     # must have a scheme
      or $string =~ /#/                       # no fragment, except...
        and $string !~ m{#\z}                          # empty fragment
        and $string !~ m{#[A-Za-z][A-Za-z0-9_:.-]*\z}  # plain-name fragment
        and $string !~ m{#/(?:[^~]|~[01])*\z};         # json pointer fragment

  return 1;
}

# produces an annotation whose value is the same as that of the current schema keyword
# makes a copy as this is passed back to the user, who cannot be trusted to not mutate it
sub annotate_self ($state, $schema) {
  A($state, ref $schema->{$state->{keyword}} ? dclone($schema->{$state->{keyword}})
    : $schema->{$state->{keyword}});
}

# use original value as stored in the NV, without losing precision
sub sprintf_num ($value) {
  is_bignum($value) ? $value->bstr : sprintf('%s', $value);
}

{
  # simple runtime-wide cache of $ids to filenames that are sourced from disk
  my $schema_filename_cache = {};

  # adds a mapping from a URI to an absolute filename in the global runtime
  # (available to all instances of the evaluator running in the same process).
  sub register_schema ($uri, $filename) {
    $schema_filename_cache->{$uri} = $filename;
  }

  sub get_schema_filename ($uri) {
    $schema_filename_cache->{$uri};
  }
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

JSON::Schema::Modern::Utilities - Internal utilities for JSON::Schema::Modern

=head1 VERSION

version 0.631

=head1 SYNOPSIS

  use JSON::Schema::Modern::Utilities qw(func1 func2..);

=head1 DESCRIPTION

This class contains internal utilities to be used by L<JSON::Schema::Modern>, and other useful helpers.

=for Pod::Coverage is_bignum local_annotations
canonical_uri E A abort assert_keyword_exists assert_keyword_type assert_pattern assert_uri_reference assert_uri
annotate_self sprintf_num HAVE_BUILTIN true false
register_schema get_schema_filename

=head1 FUNCTIONS

=for stopwords schema metaschema dualvar jsonp unjsonp

=head2 is_type

  if (is_type('string', $value)) { ... }

Returns a boolean indicating whether the provided value is of the specified core type (C<null>,
C<boolean>, C<string>, C<number>, C<object>, C<array>) or C<integer>. Also optionally takes a hashref
C<{ legacy_ints => 1 }> indicating that draft4 number semantics should apply (where unlike later
drafts, C<2.0> is B<not> an integer).

=head2 get_type

  my $type = get_type($value);

Returns one of the core types (C<null>, C<boolean>, C<string>, C<number>, C<object>, C<array>) or
C<integer>. Also optionally takes a hashref C<{ legacy_ints => 1 }> indicating that draft4 number
semantics should apply. Behaviour is consistent with L</is_type>.

=head2 is_bool

  if (is_bool($value)) { ... }

Equivalent to C<is_type('boolean', $value)>.
Accepts JSON booleans and L<builtin> booleans, but not dualvars (because JSON encoders do not
recognize these as booleans).

=head2 is_schema

  if (is_schema($value)) { ... }

Equivalent to C<is_type('object') || is_type('boolean')>.

=head2 is_equal

  if (not is_equal($x, $y, my $state = {})) {
    say "values differ starting at $state->{path}: $state->{error}";
  }

Compares two arbitrary data payloads for equality, as per
L<Instance Equality in the JSON Schema draft2020-12 specification|https://json-schema.org/draft/2020-12/json-schema-core.html#rfc.section.4.2.2>.

The optional third argument hashref supports the following fields:

=over 4

=item *

C<scalarref_booleans> (provided by caller input): as in L<JSON::Schema::Modern/scalarref_booleans>

=item *

C<stringy_numbers> (provided by caller input): when set, strings will also be compared numerically, as in L<JSON::Schema::Modern/stringy_numbers>

=item *

C<path> (populated by function): if result is false, the json pointer location of the first difference

=item *

C<error> (populated by function): if result is false, an error description of the first difference

=back

=head2 is_elements_unique

  if (not is_elements_unique($arrayref, my $state = {}) {
    say "lists differ starting at $state->{path}: $state->{error}";
  }

Compares all elements of an arrayref for uniqueness.

The optional second argument hashref supports the same options as L</is_equal>, plus:

=over 4

=item *

C<equal_indices> (populated by function): if result is false, the list of indices of the (first set of) equal items found.

=back

=head2 jsonp

  # '/paths/~1foo~1{foo_id}/get/responses'
  my $jsonp = jsonp(qw(/paths /foo/{foo_id} get responses));

Constructs a json pointer string from a list of path components, with correct escaping; the first
argument must be C<''> or an already-escaped json pointer, to which the rest of the path components
are appended.

=head2 unjsonp

  # ('', 'paths', '/foo/{foo_id}', 'get', 'responses')
  my @components = unjsonp('/paths/~1foo~1{foo_id}/get/responses');

Splits a json pointer string into its path components, with correct unescaping.

=head2 jsonp_set

  my $data = { a => 1, b => { c => 3, d => 4 } };
  my $defaults = {
    '/b/d' => 5,
    '/b/e' => 6,
    '/f' => 7,
    '/g/h/i/1' => [ 10 ],
  };
  jsonp_set($data, $_, $defaults->{$_}) foreach keys %$defaults;

  # data is now:
  # { a => 1, b => { c => 3, d => 5, e => 6 }, f => 7, g => { h => { i => [ undef, [ 10 ] ] } } }

Given an arbitrary data structure, a json pointer string, and an arbitrary value, assigns that value
to the given position in the data structure. This is a destructive operation, overwriting whatever
data was there before if needed (even if an incompatible type: e.g. a hash key will overwrite an
existing arrayref). Intermediary keys or indexes will spring into existence as needed.

=head2 json_pointer_type

A L<Type::Tiny> type representing a json pointer string.

=head2 canonical_uri_type

A L<Type::Tiny> type representing a canonical URI: a L<Mojo::URL> with either no fragment, or with a
json pointer fragment.

=head2 load_cached_document

  my $evaluator = JSON::Schema::Modern->new;
  my $uri = 'https://json-schema.org/draft-07/schema#';
  my $document = load_cached_document($evaluator, $uri);

  my $result = $evaluator->evaluate($data, $uri);

Loads a document object from global cache, loading data from disk if needed. This should only be
used for officially-published schemas and metaschemas that are bundled with this distribution or
another related one.

=head1 GIVING THANKS

=for stopwords MetaCPAN GitHub

If you found this module to be useful, please show your appreciation by
adding a +1 in L<MetaCPAN|https://metacpan.org/dist/JSON-Schema-Modern>
and a star in L<GitHub|https://github.com/karenetheridge/JSON-Schema-Modern>.

=head1 SUPPORT

Bugs may be submitted through L<https://github.com/karenetheridge/JSON-Schema-Modern/issues>.

I am also usually active on irc, as 'ether' at C<irc.perl.org> and C<irc.libera.chat>.

=for stopwords OpenAPI

You can also find me on the L<JSON Schema Slack server|https://json-schema.slack.com> and L<OpenAPI Slack
server|https://open-api.slack.com>, which are also great resources for finding help.

=head1 AUTHOR

Karen Etheridge <ether@cpan.org>

=head1 COPYRIGHT AND LICENCE

This software is copyright (c) 2020 by Karen Etheridge.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

Some schema files have their own licence, in share/LICENSE.

=cut
