#!perl
use strict;
use warnings;

use Test::More;

BEGIN {
  use_ok('Time::Moment');
  use_ok('Time::Moment::Adjusters', qw[ NextDayOfWeek 
                                        PreviousDayOfWeek 
                                        NextOrSameDayOfWeek 
                                        PreviousOrSameDayOfWeek
                                        NearestDayOfWeek
                                        FirstDayOfWeekInMonth
                                        LastDayOfWeekInMonth
                                        NthDayOfWeekInMonth
                                        WesternEasterSunday
                                        NearestMinuteInterval
                                      ]);
}

my $Sunday = Time::Moment->from_string('2018-W02-7T00Z');

{
    my @M = (
    # M T W T F S S
      7,1,2,3,4,5,6, # M
      6,7,1,2,3,4,5, # T
      5,6,7,1,2,3,4, # W
      4,5,6,7,1,2,3, # T
      3,4,5,6,7,1,2, # F
      2,3,4,5,6,7,1, # S
      1,2,3,4,5,6,7, # S
    );

    foreach my $d1 (1..7) {
        my $tm = $Sunday->plus_days($d1);
        foreach my $d2 (1..7) {
            my $got = $tm->with(NextDayOfWeek($d2));
            my $exp = $tm->plus_days($M[7 * $d1 + $d2 - 8]);
            is($got, $exp, "$tm->with(NextDayOfWeek($d2))");
        }
    }

    foreach my $d1 (1..7) {
        my $tm = $Sunday->plus_days($d1);
        foreach my $d2 (1..7) {
            my $got = $tm->with(PreviousDayOfWeek($d2));
            my $exp = $tm->minus_days($M[7 * $d2 + $d1 - 8]);
            is($got, $exp, "$tm->with(PreviousDayOfWeek($d2))");
        }
    }
}

{
    my @M = (
    # M T W T F S S
      0,1,2,3,4,5,6, # M
      6,0,1,2,3,4,5, # T
      5,6,0,1,2,3,4, # W
      4,5,6,0,1,2,3, # T
      3,4,5,6,0,1,2, # F
      2,3,4,5,6,0,1, # S
      1,2,3,4,5,6,0, # S
    );

    foreach my $d1 (1..7) {
        my $tm = $Sunday->plus_days($d1);
        foreach my $d2 (1..7) {
            my $got = $tm->with(NextOrSameDayOfWeek($d2));
            my $exp = $tm->plus_days($M[7 * $d1 + $d2 - 8]);
            is($got, $exp, "$tm->with(NextOrSameDayOfWeek($d2))");
        }
    }

    foreach my $d1 (1..7) {
        my $tm = $Sunday->plus_days($d1);
        foreach my $d2 (1..7) {
            my $got = $tm->with(PreviousOrSameDayOfWeek($d2));
            my $exp = $tm->minus_days($M[7 * $d2 + $d1 - 8]);
            is($got, $exp, "$tm->with(PreviousOrSameDayOfWeek($d2))");
        }
    }
}

my $Monday = Time::Moment->from_string('1996-01-01T00Z');

{
    my @M = (undef, 12, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11);

    for (my $tm = $Monday; $tm->year == $Monday->year; $tm = $tm->plus_days(1)) {
        foreach my $d1 (1..7) {
            my $got = $tm->with(FirstDayOfWeekInMonth($d1));
            is($got->day_of_week, $d1, "$tm->with(FirstDayOfWeekInMonth($d1))->day_of_week == $d1");
            is($got->month, $tm->month, "$tm->with(FirstDayOfWeekInMonth($d1))->month == $tm->month");

            my $got2 = $got->minus_days(7);
            is($got2->month, $M[$got->month], "$tm->with(FirstDayOfWeekInMonth($d1))->minus_days(7)->month == $M[$got->month]");
        }
    }
}

{
    my @M = (undef, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 1);

    for (my $tm = $Monday; $tm->year == $Monday->year; $tm = $tm->plus_days(1)) {
        foreach my $d1 (1..7) {
            my $got = $tm->with(LastDayOfWeekInMonth($d1));
            is($got->day_of_week, $d1, "$tm->with(LastDayOfWeekInMonth($d1))->day_of_week == $d1");
            is($got->month, $tm->month, "$tm->with(LastDayOfWeekInMonth($d1))->month == $tm->month");

            my $got2 = $got->plus_days(7);
            is($got2->month, $M[$got->month], "$tm->with(LastDayOfWeekInMonth($d1))->plus_days(7)->month == $M[$got->month]");
        }
    }

}

{
    for my $m1 (1..12) {
        my $tm = $Monday->with_month($m1);
        for my $d1 (1..7) {
            for my $o1 (1..4, -4..-1) {
                my $got = $tm->with(NthDayOfWeekInMonth($o1, $d1));
                is($got->day_of_week, $d1, "$tm->with(NthDayOfWeekInMonth($o1, $d1))->day_of_week == $d1");
                is($got->month, $tm->month, "$tm->with(NthDayOfWeekInMonth($o1, $d1))->month == $tm->month");

                if ($o1 > 0) {
                    my $n = $o1 - 1;
                    for my $o2 (1..$n) {
                        my $got2 = $got->minus_days(7 * $o2);
                        is($got2->month, $tm->month, "$got->minus_days(7 * $o2)->month == $tm->month");
                    }
                } else {
                    my $n = abs($o1) - 1;
                    for my $o2 (1..$n) {
                        my $got2 = $got->plus_days(7 * $o2);
                        is($got2->month, $tm->month, "$got->plus_days(7 * $o2)->month == $tm->month")
                            or done_testing, exit;
                    }
                }
            }

        }
    }
}

{
    my @WesternEasterSundays = qw(
       1700-04-11   1725-04-01   1750-03-29   1775-04-16
       1701-03-27   1726-04-21   1751-04-11   1776-04-07
       1702-04-16   1727-04-13   1752-04-02   1777-03-30
       1703-04-08   1728-03-28   1753-04-22   1778-04-19
       1704-03-23   1729-04-17   1754-04-14   1779-04-04
       1705-04-12   1730-04-09   1755-03-30   1780-03-26
       1706-04-04   1731-03-25   1756-04-18   1781-04-15
       1707-04-24   1732-04-13   1757-04-10   1782-03-31
       1708-04-08   1733-04-05   1758-03-26   1783-04-20
       1709-03-31   1734-04-25   1759-04-15   1784-04-11
       1710-04-20   1735-04-10   1760-04-06   1785-03-27
       1711-04-05   1736-04-01   1761-03-22   1786-04-16
       1712-03-27   1737-04-21   1762-04-11   1787-04-08
       1713-04-16   1738-04-06   1763-04-03   1788-03-23
       1714-04-01   1739-03-29   1764-04-22   1789-04-12
       1715-04-21   1740-04-17   1765-04-07   1790-04-04
       1716-04-12   1741-04-02   1766-03-30   1791-04-24
       1717-03-28   1742-03-25   1767-04-19   1792-04-08
       1718-04-17   1743-04-14   1768-04-03   1793-03-31
       1719-04-09   1744-04-05   1769-03-26   1794-04-20
       1720-03-31   1745-04-18   1770-04-15   1795-04-05
       1721-04-13   1746-04-10   1771-03-31   1796-03-27
       1722-04-05   1747-04-02   1772-04-19   1797-04-16
       1723-03-28   1748-04-14   1773-04-11   1798-04-08
       1724-04-16   1749-04-06   1774-04-03   1799-03-24

       1800-04-13   1825-04-03   1850-03-31   1875-03-28
       1801-04-05   1826-03-26   1851-04-20   1876-04-16
       1802-04-18   1827-04-15   1852-04-11   1877-04-01
       1803-04-10   1828-04-06   1853-03-27   1878-04-21
       1804-04-01   1829-04-19   1854-04-16   1879-04-13
       1805-04-14   1830-04-11   1855-04-08   1880-03-28
       1806-04-06   1831-04-03   1856-03-23   1881-04-17
       1807-03-29   1832-04-22   1857-04-12   1882-04-09
       1808-04-17   1833-04-07   1858-04-04   1883-03-25
       1809-04-02   1834-03-30   1859-04-24   1884-04-13
       1810-04-22   1835-04-19   1860-04-08   1885-04-05
       1811-04-14   1836-04-03   1861-03-31   1886-04-25
       1812-03-29   1837-03-26   1862-04-20   1887-04-10
       1813-04-18   1838-04-15   1863-04-05   1888-04-01
       1814-04-10   1839-03-31   1864-03-27   1889-04-21
       1815-03-26   1840-04-19   1865-04-16   1890-04-06
       1816-04-14   1841-04-11   1866-04-01   1891-03-29
       1817-04-06   1842-03-27   1867-04-21   1892-04-17
       1818-03-22   1843-04-16   1868-04-12   1893-04-02
       1819-04-11   1844-04-07   1869-03-28   1894-03-25
       1820-04-02   1845-03-23   1870-04-17   1895-04-14
       1821-04-22   1846-04-12   1871-04-09   1896-04-05
       1822-04-07   1847-04-04   1872-03-31   1897-04-18
       1823-03-30   1848-04-23   1873-04-13   1898-04-10
       1824-04-18   1849-04-08   1874-04-05   1899-04-02

       1900-04-15   1925-04-12   1950-04-09   1975-03-30
       1901-04-07   1926-04-04   1951-03-25   1976-04-18
       1902-03-30   1927-04-17   1952-04-13   1977-04-10
       1903-04-12   1928-04-08   1953-04-05   1978-03-26
       1904-04-03   1929-03-31   1954-04-18   1979-04-15
       1905-04-23   1930-04-20   1955-04-10   1980-04-06
       1906-04-15   1931-04-05   1956-04-01   1981-04-19
       1907-03-31   1932-03-27   1957-04-21   1982-04-11
       1908-04-19   1933-04-16   1958-04-06   1983-04-03
       1909-04-11   1934-04-01   1959-03-29   1984-04-22
       1910-03-27   1935-04-21   1960-04-17   1985-04-07
       1911-04-16   1936-04-12   1961-04-02   1986-03-30
       1912-04-07   1937-03-28   1962-04-22   1987-04-19
       1913-03-23   1938-04-17   1963-04-14   1988-04-03
       1914-04-12   1939-04-09   1964-03-29   1989-03-26
       1915-04-04   1940-03-24   1965-04-18   1990-04-15
       1916-04-23   1941-04-13   1966-04-10   1991-03-31
       1917-04-08   1942-04-05   1967-03-26   1992-04-19
       1918-03-31   1943-04-25   1968-04-14   1993-04-11
       1919-04-20   1944-04-09   1969-04-06   1994-04-03
       1920-04-04   1945-04-01   1970-03-29   1995-04-16
       1921-03-27   1946-04-21   1971-04-11   1996-04-07
       1922-04-16   1947-04-06   1972-04-02   1997-03-30
       1923-04-01   1948-03-28   1973-04-22   1998-04-12
       1924-04-20   1949-04-17   1974-04-14   1999-04-04

       2000-04-23   2025-04-20   2050-04-10   2075-04-07
       2001-04-15   2026-04-05   2051-04-02   2076-04-19
       2002-03-31   2027-03-28   2052-04-21   2077-04-11
       2003-04-20   2028-04-16   2053-04-06   2078-04-03
       2004-04-11   2029-04-01   2054-03-29   2079-04-23
       2005-03-27   2030-04-21   2055-04-18   2080-04-07
       2006-04-16   2031-04-13   2056-04-02   2081-03-30
       2007-04-08   2032-03-28   2057-04-22   2082-04-19
       2008-03-23   2033-04-17   2058-04-14   2083-04-04
       2009-04-12   2034-04-09   2059-03-30   2084-03-26
       2010-04-04   2035-03-25   2060-04-18   2085-04-15
       2011-04-24   2036-04-13   2061-04-10   2086-03-31
       2012-04-08   2037-04-05   2062-03-26   2087-04-20
       2013-03-31   2038-04-25   2063-04-15   2088-04-11
       2014-04-20   2039-04-10   2064-04-06   2089-04-03
       2015-04-05   2040-04-01   2065-03-29   2090-04-16
       2016-03-27   2041-04-21   2066-04-11   2091-04-08
       2017-04-16   2042-04-06   2067-04-03   2092-03-30
       2018-04-01   2043-03-29   2068-04-22   2093-04-12
       2019-04-21   2044-04-17   2069-04-14   2094-04-04
       2020-04-12   2045-04-09   2070-03-30   2095-04-24
       2021-04-04   2046-03-25   2071-04-19   2096-04-15
       2022-04-17   2047-04-14   2072-04-10   2097-03-31
       2023-04-09   2048-04-05   2073-03-26   2098-04-20
       2024-03-31   2049-04-18   2074-04-15   2099-04-12

       2100-03-28   2125-04-22   2150-04-12   2175-04-09
       2101-04-17   2126-04-14   2151-04-04   2176-03-31
       2102-04-09   2127-03-30   2152-04-23   2177-04-20
       2103-03-25   2128-04-18   2153-04-15   2178-04-05
       2104-04-13   2129-04-10   2154-03-31   2179-03-28
       2105-04-05   2130-03-26   2155-04-20   2180-04-16
       2106-04-18   2131-04-15   2156-04-11   2181-04-01
       2107-04-10   2132-04-06   2157-03-27   2182-04-21
       2108-04-01   2133-04-19   2158-04-16   2183-04-13
       2109-04-21   2134-04-11   2159-04-08   2184-03-28
       2110-04-06   2135-04-03   2160-03-23   2185-04-17
       2111-03-29   2136-04-22   2161-04-12   2186-04-09
       2112-04-17   2137-04-07   2162-04-04   2187-03-25
       2113-04-02   2138-03-30   2163-04-24   2188-04-13
       2114-04-22   2139-04-19   2164-04-08   2189-04-05
       2115-04-14   2140-04-03   2165-03-31   2190-04-25
       2116-03-29   2141-03-26   2166-04-20   2191-04-10
       2117-04-18   2142-04-15   2167-04-05   2192-04-01
       2118-04-10   2143-03-31   2168-03-27   2193-04-21
       2119-03-26   2144-04-19   2169-04-16   2194-04-06
       2120-04-14   2145-04-11   2170-04-01   2195-03-29
       2121-04-06   2146-04-03   2171-04-21   2196-04-17
       2122-03-29   2147-04-16   2172-04-12   2197-04-09
       2123-04-11   2148-04-07   2173-04-04   2198-03-25
       2124-04-02   2149-03-30   2174-04-17   2199-04-14
    );

    for my $date (@WesternEasterSundays) {
        my $exp = Time::Moment->from_string($date . "T00Z");
        my $tm = Time::Moment->new(year => $exp->year);
        my $got = $tm->with(WesternEasterSunday);
        is($got, $exp, "$tm->with(WesternEasterSunday) == $got");
    }
}

{
    for my $plus_unit (qw(plus_nanoseconds plus_microseconds plus_seconds)) {
        my $tm = $Sunday->$plus_unit(1);
        my $got = $tm->with(NearestMinuteInterval(1));
        my $exp = $Sunday;
        is($got, $exp, "$tm->with(NearestMinuteInterval(1)) == $exp");
    }

    my %T = (
        '2018-01-01T10:14:59Z' => '2018-01-01T10:00:00Z',
        '2018-01-01T10:15:00Z' => '2018-01-01T10:30:00Z',
        '2018-01-01T10:29:59Z' => '2018-01-01T10:30:00Z',
        '2018-01-01T23:55:00Z' => '2018-01-02T00:00:00Z',
    );

    while (my ($from, $to) = each %T) {
        my $tm = Time::Moment->from_string($from);
        my $exp = Time::Moment->from_string($to);
        my $got = $tm->with(NearestMinuteInterval(30));
        is($got, $exp, "$tm->with(NearestMinuteInterval(30)) == $exp");
    }
}

{
    my $tm = Time::Moment->from_string("2012-12-24T00Z");
    # 2012-12-24 is Monday (day_of_week == 1)

    for my $day (1..7) {
        my $got = $tm->with(NearestDayOfWeek(($day)));
        my $prefix = "$tm->NearestDayOfWeek($day)";
        is($got->day_of_week, $day, "$prefix->day_of_week");
    }
}

{
    my $tm = Time::Moment->from_string("2012-12-26T00Z");
    # 2012-12-26 is Wednesday (day_of_week == 3)

    # Nearest Monday (1) from Wednesday (3) is the previous Monday (2012-12-24)
    {
        my $got = $tm->with(NearestDayOfWeek(1));
        my $prefix = "NearestDayOfWeek(1)->($tm)";
        is($got->to_string, "2012-12-24T00:00:00Z", "$prefix->to_string");
    }

    # Nearest Friday (5) from Wednesday (3) is the coming Friday (2012-12-28)
    {
        my $got = $tm->with(NearestDayOfWeek(5));
        my $prefix = "NearestDayOfWeek(5)->($tm)";
        is($got->to_string, "2012-12-28T00:00:00Z", "$prefix->to_string");
    }
}



done_testing();
