/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jgit.dircache;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.text.MessageFormat;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import org.eclipse.jgit.dircache.DirCacheBuilder;
import org.eclipse.jgit.dircache.DirCacheEditor;
import org.eclipse.jgit.dircache.DirCacheEntry;
import org.eclipse.jgit.dircache.DirCacheIterator;
import org.eclipse.jgit.dircache.DirCacheTree;
import org.eclipse.jgit.errors.CorruptObjectException;
import org.eclipse.jgit.errors.IndexReadException;
import org.eclipse.jgit.errors.LockFailedException;
import org.eclipse.jgit.errors.UnmergedPathException;
import org.eclipse.jgit.events.IndexChangedEvent;
import org.eclipse.jgit.events.IndexChangedListener;
import org.eclipse.jgit.internal.JGitText;
import org.eclipse.jgit.internal.storage.file.FileSnapshot;
import org.eclipse.jgit.internal.storage.file.LockFile;
import org.eclipse.jgit.internal.storage.io.NullMessageDigest;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.Constants;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.ObjectInserter;
import org.eclipse.jgit.lib.ObjectReader;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.treewalk.FileTreeIterator;
import org.eclipse.jgit.treewalk.TreeWalk;
import org.eclipse.jgit.treewalk.filter.PathFilterGroup;
import org.eclipse.jgit.util.FS;
import org.eclipse.jgit.util.IO;
import org.eclipse.jgit.util.MutableInteger;
import org.eclipse.jgit.util.NB;
import org.eclipse.jgit.util.TemporaryBuffer;
import org.eclipse.jgit.util.io.SilentFileInputStream;

public class DirCache {
    private static final byte[] SIG_DIRC = new byte[]{68, 73, 82, 67};
    private static final int EXT_TREE = 0x54524545;
    private static final DirCacheEntry[] NO_ENTRIES = new DirCacheEntry[0];
    private static final byte[] NO_CHECKSUM = new byte[0];
    static final Comparator<DirCacheEntry> ENT_CMP = (o1, o2) -> {
        int cr = DirCache.cmp(o1, o2);
        if (cr != 0) {
            return cr;
        }
        return o1.getStage() - o2.getStage();
    };
    private final File liveFile;
    private DirCacheEntry[] sortedEntries;
    private int entryCnt;
    private DirCacheTree tree;
    private LockFile myLock;
    private FileSnapshot snapshot;
    private byte[] readIndexChecksum;
    private byte[] writeIndexChecksum;
    private IndexChangedListener indexChangedListener;
    private Repository repository;
    private DirCacheVersion version;
    private boolean skipHash;

    static int cmp(DirCacheEntry a, DirCacheEntry b) {
        return DirCache.cmp(a.path, a.path.length, b);
    }

    static int cmp(byte[] aPath, int aLen, DirCacheEntry b) {
        return DirCache.cmp(aPath, aLen, b.path, b.path.length);
    }

    static int cmp(byte[] aPath, int aLen, byte[] bPath, int bLen) {
        int cPos = 0;
        while (cPos < aLen && cPos < bLen) {
            int cmp = (aPath[cPos] & 0xFF) - (bPath[cPos] & 0xFF);
            if (cmp != 0) {
                return cmp;
            }
            ++cPos;
        }
        return aLen - bLen;
    }

    public static DirCache newInCore() {
        return new DirCache(null, null);
    }

    public static DirCache read(ObjectReader reader, AnyObjectId treeId) throws IOException {
        DirCache d = DirCache.newInCore();
        DirCacheBuilder b = d.builder();
        b.addTree(null, 0, reader, treeId);
        b.finish();
        return d;
    }

    public static DirCache read(Repository repository) throws CorruptObjectException, IOException {
        DirCache c = DirCache.read(repository.getIndexFile(), repository.getFS());
        c.repository = repository;
        return c;
    }

    public static DirCache read(File indexLocation, FS fs) throws CorruptObjectException, IOException {
        DirCache c = new DirCache(indexLocation, fs);
        c.read();
        return c;
    }

    public static DirCache lock(File indexLocation, FS fs) throws CorruptObjectException, IOException {
        DirCache c = new DirCache(indexLocation, fs);
        if (!c.lock()) {
            throw new LockFailedException(indexLocation);
        }
        try {
            c.read();
        }
        catch (IOException | Error | RuntimeException e) {
            c.unlock();
            throw e;
        }
        return c;
    }

    public static DirCache lock(Repository repository, IndexChangedListener indexChangedListener) throws CorruptObjectException, IOException {
        DirCache c = DirCache.lock(repository.getIndexFile(), repository.getFS(), indexChangedListener);
        c.repository = repository;
        return c;
    }

    public static DirCache lock(File indexLocation, FS fs, IndexChangedListener indexChangedListener) throws CorruptObjectException, IOException {
        DirCache c = DirCache.lock(indexLocation, fs);
        c.registerIndexChangedListener(indexChangedListener);
        return c;
    }

    public DirCache(File indexLocation, FS fs) {
        this.liveFile = indexLocation;
        this.clear();
    }

    public DirCacheBuilder builder() {
        return new DirCacheBuilder(this, this.entryCnt + 16);
    }

    public DirCacheEditor editor() {
        return new DirCacheEditor(this, this.entryCnt + 16);
    }

    DirCacheVersion getVersion() {
        return this.version;
    }

    void replace(DirCacheEntry[] e, int cnt) {
        this.sortedEntries = e;
        this.entryCnt = cnt;
        this.tree = null;
    }

    public void read() throws IOException, CorruptObjectException {
        if (this.liveFile == null) {
            throw new IOException(JGitText.get().dirCacheDoesNotHaveABackingFile);
        }
        if (!this.liveFile.exists()) {
            this.clear();
        } else if (this.snapshot == null || this.snapshot.isModified(this.liveFile)) {
            try {
                Throwable throwable = null;
                Object var2_4 = null;
                try (SilentFileInputStream inStream = new SilentFileInputStream(this.liveFile);){
                    this.clear();
                    this.readFrom(inStream);
                }
                catch (Throwable throwable2) {
                    if (throwable == null) {
                        throwable = throwable2;
                    } else if (throwable != throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
            }
            catch (FileNotFoundException fnfe) {
                if (this.liveFile.exists()) {
                    throw new IndexReadException(MessageFormat.format(JGitText.get().cannotReadIndex, this.liveFile.getAbsolutePath(), fnfe));
                }
                this.clear();
            }
            this.snapshot = FileSnapshot.save(this.liveFile);
        }
    }

    public boolean isOutdated() throws IOException {
        if (this.liveFile == null || !this.liveFile.exists()) {
            return false;
        }
        return this.snapshot == null || this.snapshot.isModified(this.liveFile);
    }

    public void clear() {
        this.snapshot = null;
        this.sortedEntries = NO_ENTRIES;
        this.entryCnt = 0;
        this.tree = null;
        this.readIndexChecksum = NO_CHECKSUM;
    }

    private void readFrom(InputStream inStream) throws IOException, CorruptObjectException {
        byte[] hdr;
        MessageDigest md;
        block14: {
            BufferedInputStream in = new BufferedInputStream(inStream);
            this.readConfig();
            md = this.newMessageDigest();
            hdr = new byte[20];
            IO.readFully(in, hdr, 0, 12);
            md.update(hdr, 0, 12);
            if (!DirCache.is_DIRC(hdr)) {
                throw new CorruptObjectException(JGitText.get().notADIRCFile);
            }
            int versionCode = NB.decodeInt32(hdr, 4);
            DirCacheVersion ver = DirCacheVersion.fromInt(versionCode);
            if (ver == null) {
                throw new CorruptObjectException(MessageFormat.format(JGitText.get().unknownDIRCVersion, versionCode));
            }
            boolean extended = false;
            switch (ver) {
                case DIRC_VERSION_MINIMUM: {
                    break;
                }
                case DIRC_VERSION_EXTENDED: 
                case DIRC_VERSION_PATHCOMPRESS: {
                    extended = true;
                    break;
                }
                default: {
                    throw new CorruptObjectException(MessageFormat.format(JGitText.get().unknownDIRCVersion, ver));
                }
            }
            this.version = ver;
            this.entryCnt = NB.decodeInt32(hdr, 8);
            if (this.entryCnt < 0) {
                throw new CorruptObjectException(JGitText.get().DIRCHasTooManyEntries);
            }
            this.snapshot = FileSnapshot.save(this.liveFile);
            Instant smudge = this.snapshot.lastModifiedInstant();
            int infoLength = DirCacheEntry.getMaximumInfoLength(extended);
            byte[] infos = new byte[infoLength * this.entryCnt];
            this.sortedEntries = new DirCacheEntry[this.entryCnt];
            MutableInteger infoAt = new MutableInteger();
            int i2 = 0;
            while (i2 < this.entryCnt) {
                this.sortedEntries[i2] = new DirCacheEntry(infos, infoAt, in, md, smudge, this.version, i2 == 0 ? null : this.sortedEntries[i2 - 1]);
                ++i2;
            }
            block8: while (true) {
                in.mark(21);
                IO.readFully(in, hdr, 0, 20);
                if (in.read() < 0) break block14;
                in.reset();
                md.update(hdr, 0, 8);
                IO.skipFully(in, 8L);
                long sz = NB.decodeUInt32(hdr, 4);
                switch (NB.decodeInt32(hdr, 0)) {
                    case 0x54524545: {
                        if (Integer.MAX_VALUE < sz) {
                            throw new CorruptObjectException(MessageFormat.format(JGitText.get().DIRCExtensionIsTooLargeAt, DirCache.formatExtensionName(hdr), sz));
                        }
                        byte[] raw = new byte[(int)sz];
                        IO.readFully(in, raw, 0, raw.length);
                        md.update(raw, 0, raw.length);
                        this.tree = new DirCacheTree(raw, new MutableInteger(), null);
                        continue block8;
                    }
                }
                if (hdr[0] < 65 || hdr[0] > 90) break;
                this.skipOptionalExtension(in, md, hdr, sz);
            }
            throw new CorruptObjectException(MessageFormat.format(JGitText.get().DIRCExtensionNotSupportedByThisVersion, DirCache.formatExtensionName(hdr)));
        }
        this.readIndexChecksum = md.digest();
        if (!(this.skipHash || Arrays.equals(this.readIndexChecksum, hdr) || Arrays.equals(NullMessageDigest.getInstance().digest(), hdr))) {
            throw new CorruptObjectException(JGitText.get().DIRCChecksumMismatch);
        }
    }

    private void skipOptionalExtension(InputStream in, MessageDigest md, byte[] hdr, long sz) throws IOException {
        byte[] b = new byte[4096];
        while (0L < sz) {
            int n = in.read(b, 0, (int)Math.min((long)b.length, sz));
            if (n < 0) {
                throw new EOFException(MessageFormat.format(JGitText.get().shortReadOfOptionalDIRCExtensionExpectedAnotherBytes, DirCache.formatExtensionName(hdr), sz));
            }
            md.update(b, 0, n);
            sz -= (long)n;
        }
    }

    private static String formatExtensionName(byte[] hdr) {
        return "'" + new String(hdr, 0, 4, StandardCharsets.ISO_8859_1) + "'";
    }

    private static boolean is_DIRC(byte[] hdr) {
        if (hdr.length < SIG_DIRC.length) {
            return false;
        }
        int i2 = 0;
        while (i2 < SIG_DIRC.length) {
            if (hdr[i2] != SIG_DIRC[i2]) {
                return false;
            }
            ++i2;
        }
        return true;
    }

    public boolean lock() throws IOException {
        if (this.liveFile == null) {
            throw new IOException(JGitText.get().dirCacheDoesNotHaveABackingFile);
        }
        LockFile tmp = new LockFile(this.liveFile);
        if (tmp.lock()) {
            tmp.setNeedStatInformation(true);
            this.myLock = tmp;
            return true;
        }
        return false;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void write() throws IOException {
        LockFile tmp = this.myLock;
        this.requireLocked(tmp);
        try {
            Throwable throwable = null;
            Object var3_5 = null;
            try {
                OutputStream o = tmp.getOutputStream();
                try {
                    try (BufferedOutputStream bo = new BufferedOutputStream(o);){
                        this.writeTo(this.liveFile.getParentFile(), bo);
                    }
                    if (o == null) return;
                }
                catch (Throwable throwable2) {
                    if (throwable == null) {
                        throwable = throwable2;
                    } else if (throwable != throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    if (o == null) throw throwable;
                    o.close();
                    throw throwable;
                }
                o.close();
                return;
            }
            catch (Throwable throwable3) {
                if (throwable == null) {
                    throwable = throwable3;
                    throw throwable;
                } else {
                    if (throwable == throwable3) throw throwable;
                    throwable.addSuppressed(throwable3);
                }
                throw throwable;
            }
        }
        catch (IOException | Error | RuntimeException err) {
            tmp.unlock();
            throw err;
        }
    }

    void writeTo(File dir, OutputStream os) throws IOException {
        boolean writeTree;
        Instant smudge;
        this.readConfig();
        MessageDigest foot = this.newMessageDigest();
        DigestOutputStream dos = new DigestOutputStream(os, foot);
        if (this.version == null || this.version == DirCacheVersion.DIRC_VERSION_MINIMUM) {
            this.version = DirCacheVersion.DIRC_VERSION_MINIMUM;
            int i2 = 0;
            while (i2 < this.entryCnt) {
                if (this.sortedEntries[i2].isExtended()) {
                    this.version = DirCacheVersion.DIRC_VERSION_EXTENDED;
                    break;
                }
                ++i2;
            }
        }
        byte[] tmp = new byte[128];
        System.arraycopy(SIG_DIRC, 0, tmp, 0, SIG_DIRC.length);
        NB.encodeInt32(tmp, 4, this.version.getVersionCode());
        NB.encodeInt32(tmp, 8, this.entryCnt);
        dos.write(tmp, 0, 12);
        if (this.myLock != null) {
            this.myLock.createCommitSnapshot();
            this.snapshot = this.myLock.getCommitSnapshot();
            smudge = this.snapshot.lastModifiedInstant();
        } else {
            smudge = Instant.EPOCH;
        }
        boolean bl = writeTree = this.tree != null;
        if (this.repository != null && this.entryCnt > 0) {
            this.updateSmudgedEntries();
        }
        int i3 = 0;
        while (i3 < this.entryCnt) {
            DirCacheEntry e = this.sortedEntries[i3];
            if (e.mightBeRacilyClean(smudge)) {
                e.smudgeRacilyClean();
            }
            e.write(dos, this.version, i3 == 0 ? null : this.sortedEntries[i3 - 1]);
            ++i3;
        }
        if (writeTree) {
            TemporaryBuffer.LocalFile bb = new TemporaryBuffer.LocalFile(dir, 0x500000);
            try {
                this.tree.write(tmp, bb);
                bb.close();
                NB.encodeInt32(tmp, 0, 0x54524545);
                NB.encodeInt32(tmp, 4, (int)((TemporaryBuffer)bb).length());
                dos.write(tmp, 0, 8);
                ((TemporaryBuffer)bb).writeTo(dos, null);
            }
            finally {
                ((TemporaryBuffer)bb).destroy();
            }
        }
        this.writeIndexChecksum = foot.digest();
        os.write(this.writeIndexChecksum);
        os.close();
    }

    private void readConfig() {
        if (this.version == null && this.repository != null) {
            DirCacheConfig config = this.repository.getConfig().get(DirCacheConfig::new);
            this.version = config.getIndexVersion();
            this.skipHash = config.isSkipHash();
        }
    }

    private MessageDigest newMessageDigest() {
        if (this.skipHash) {
            return NullMessageDigest.getInstance();
        }
        return Constants.newMessageDigest();
    }

    public boolean commit() {
        LockFile tmp = this.myLock;
        this.requireLocked(tmp);
        this.myLock = null;
        if (!tmp.commit()) {
            return false;
        }
        this.snapshot = tmp.getCommitSnapshot();
        if (this.indexChangedListener != null && !Arrays.equals(this.readIndexChecksum, this.writeIndexChecksum)) {
            this.indexChangedListener.onIndexChanged(new IndexChangedEvent(true));
        }
        return true;
    }

    private void requireLocked(LockFile tmp) {
        if (this.liveFile == null) {
            throw new IllegalStateException(JGitText.get().dirCacheIsNotLocked);
        }
        if (tmp == null) {
            throw new IllegalStateException(MessageFormat.format(JGitText.get().dirCacheFileIsNotLocked, this.liveFile.getAbsolutePath()));
        }
    }

    public void unlock() {
        LockFile tmp = this.myLock;
        if (tmp != null) {
            this.myLock = null;
            tmp.unlock();
        }
    }

    public int findEntry(String path) {
        byte[] p = Constants.encode(path);
        return this.findEntry(p, p.length);
    }

    public int findEntry(byte[] p, int pLen) {
        return this.findEntry(0, p, pLen);
    }

    int findEntry(int low, byte[] p, int pLen) {
        int high = this.entryCnt;
        while (low < high) {
            int mid = low + high >>> 1;
            int cmp = DirCache.cmp(p, pLen, this.sortedEntries[mid]);
            if (cmp < 0) {
                high = mid;
                continue;
            }
            if (cmp == 0) {
                while (mid > 0 && DirCache.cmp(p, pLen, this.sortedEntries[mid - 1]) == 0) {
                    --mid;
                }
                return mid;
            }
            low = mid + 1;
        }
        return -(low + 1);
    }

    public int nextEntry(int position) {
        DirCacheEntry last = this.sortedEntries[position];
        int nextIdx = position + 1;
        while (nextIdx < this.entryCnt) {
            DirCacheEntry next = this.sortedEntries[nextIdx];
            if (DirCache.cmp(last, next) != 0) break;
            last = next;
            ++nextIdx;
        }
        return nextIdx;
    }

    int nextEntry(byte[] p, int pLen, int nextIdx) {
        while (nextIdx < this.entryCnt) {
            DirCacheEntry next = this.sortedEntries[nextIdx];
            if (!DirCacheTree.peq(p, next.path, pLen)) break;
            ++nextIdx;
        }
        return nextIdx;
    }

    public int getEntryCount() {
        return this.entryCnt;
    }

    public DirCacheEntry getEntry(int i2) {
        return this.sortedEntries[i2];
    }

    public DirCacheEntry getEntry(String path) {
        int i2 = this.findEntry(path);
        return i2 < 0 ? null : this.sortedEntries[i2];
    }

    public DirCacheEntry[] getEntriesWithin(String path) {
        int pLen;
        byte[] p;
        int eIdx;
        if (((String)path).length() == 0) {
            DirCacheEntry[] r = new DirCacheEntry[this.entryCnt];
            System.arraycopy(this.sortedEntries, 0, r, 0, this.entryCnt);
            return r;
        }
        if (!((String)path).endsWith("/")) {
            path = (String)path + "/";
        }
        if ((eIdx = this.findEntry(p = Constants.encode((String)path), pLen = p.length)) < 0) {
            eIdx = -(eIdx + 1);
        }
        int lastIdx = this.nextEntry(p, pLen, eIdx);
        DirCacheEntry[] r = new DirCacheEntry[lastIdx - eIdx];
        System.arraycopy(this.sortedEntries, eIdx, r, 0, r.length);
        return r;
    }

    void toArray(int i2, DirCacheEntry[] dst, int off, int cnt) {
        System.arraycopy(this.sortedEntries, i2, dst, off, cnt);
    }

    public DirCacheTree getCacheTree(boolean build) {
        if (build) {
            if (this.tree == null) {
                this.tree = new DirCacheTree();
            }
            this.tree.validate(this.sortedEntries, this.entryCnt, 0, 0);
        }
        return this.tree;
    }

    public ObjectId writeTree(ObjectInserter ow) throws UnmergedPathException, IOException {
        return this.getCacheTree(true).writeTree(this.sortedEntries, 0, 0, ow);
    }

    public boolean hasUnmergedPaths() {
        int i2 = 0;
        while (i2 < this.entryCnt) {
            if (this.sortedEntries[i2].getStage() > 0) {
                return true;
            }
            ++i2;
        }
        return false;
    }

    private void registerIndexChangedListener(IndexChangedListener listener) {
        this.indexChangedListener = listener;
    }

    private void updateSmudgedEntries() throws IOException {
        ArrayList<String> paths = new ArrayList<String>(128);
        Throwable throwable = null;
        Object var3_4 = null;
        try (TreeWalk walk = new TreeWalk(this.repository);){
            walk.setOperationType(TreeWalk.OperationType.CHECKIN_OP);
            int i2 = 0;
            while (i2 < this.entryCnt) {
                if (this.sortedEntries[i2].isSmudged()) {
                    paths.add(this.sortedEntries[i2].getPathString());
                }
                ++i2;
            }
            if (paths.isEmpty()) {
                return;
            }
            walk.setFilter(PathFilterGroup.createFromStrings(paths));
            DirCacheIterator iIter = new DirCacheIterator(this);
            FileTreeIterator fIter = new FileTreeIterator(this.repository);
            walk.addTree(iIter);
            walk.addTree(fIter);
            fIter.setDirCacheIterator(walk, 0);
            walk.setRecursive(true);
            while (walk.next()) {
                DirCacheEntry entry;
                iIter = walk.getTree(0, DirCacheIterator.class);
                if (iIter == null || (fIter = walk.getTree(1, FileTreeIterator.class)) == null || !(entry = iIter.getDirCacheEntry()).isSmudged() || !iIter.idEqual(fIter)) continue;
                entry.setLength(fIter.getEntryLength());
                entry.setLastModified(fIter.getEntryLastModifiedInstant());
            }
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    private static class DirCacheConfig {
        private final DirCacheVersion indexVersion;
        private final boolean skipHash;

        public DirCacheConfig(Config cfg) {
            boolean manyFiles = cfg.getBoolean("feature", "manyFiles", false);
            this.indexVersion = (DirCacheVersion)cfg.getEnum(DirCacheVersion.values(), "index", null, "version", manyFiles ? DirCacheVersion.DIRC_VERSION_PATHCOMPRESS : DirCacheVersion.DIRC_VERSION_EXTENDED);
            this.skipHash = cfg.getBoolean("index", "skiphash", false);
        }

        public DirCacheVersion getIndexVersion() {
            return this.indexVersion;
        }

        public boolean isSkipHash() {
            return this.skipHash;
        }
    }

    static enum DirCacheVersion implements Config.ConfigEnum
    {
        DIRC_VERSION_MINIMUM(2),
        DIRC_VERSION_EXTENDED(3),
        DIRC_VERSION_PATHCOMPRESS(4);

        private final int version;

        private DirCacheVersion(int versionCode) {
            this.version = versionCode;
        }

        public int getVersionCode() {
            return this.version;
        }

        @Override
        public String toConfigValue() {
            return Integer.toString(this.version);
        }

        @Override
        public boolean matchConfigValue(String in) {
            try {
                return this.version == Integer.parseInt(in);
            }
            catch (NumberFormatException e) {
                return false;
            }
        }

        public static DirCacheVersion fromInt(int val) {
            DirCacheVersion[] dirCacheVersionArray = DirCacheVersion.values();
            int n = dirCacheVersionArray.length;
            int n2 = 0;
            while (n2 < n) {
                DirCacheVersion v = dirCacheVersionArray[n2];
                if (val == v.getVersionCode()) {
                    return v;
                }
                ++n2;
            }
            return null;
        }
    }
}

