/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.fluss.client.table.scanner.log;

import org.apache.fluss.metadata.TableBucket;
import org.apache.fluss.record.FileLogRecords;
import org.apache.fluss.record.LogRecordReadContext;
import org.apache.fluss.remote.RemoteLogSegment;

/**
 * {@link RemotePendingFetch} is a {@link PendingFetch} that represents a pending fetch that waiting
 * for the remote log file fetched to local disk.
 */
class RemotePendingFetch implements PendingFetch {

    final RemoteLogSegment remoteLogSegment;
    private final RemoteLogDownloadFuture downloadFuture;

    private final int posInLogSegment;
    private final long fetchOffset;
    private final long highWatermark;
    private final LogRecordReadContext readContext;
    private final LogScannerStatus logScannerStatus;
    private final boolean isCheckCrc;

    RemotePendingFetch(
            RemoteLogSegment remoteLogSegment,
            RemoteLogDownloadFuture downloadFuture,
            int posInLogSegment,
            long fetchOffset,
            long highWatermark,
            LogRecordReadContext readContext,
            LogScannerStatus logScannerStatus,
            boolean isCheckCrc) {
        this.remoteLogSegment = remoteLogSegment;
        this.downloadFuture = downloadFuture;
        this.posInLogSegment = posInLogSegment;
        this.fetchOffset = fetchOffset;
        this.highWatermark = highWatermark;
        this.readContext = readContext;
        this.logScannerStatus = logScannerStatus;
        this.isCheckCrc = isCheckCrc;
    }

    @Override
    public TableBucket tableBucket() {
        return remoteLogSegment.tableBucket();
    }

    @Override
    public boolean isCompleted() {
        return downloadFuture.isDone();
    }

    @Override
    public CompletedFetch toCompletedFetch() {
        FileLogRecords fileLogRecords = downloadFuture.getFileLogRecords(posInLogSegment);
        return new RemoteCompletedFetch(
                remoteLogSegment.tableBucket(),
                fileLogRecords,
                highWatermark,
                readContext,
                logScannerStatus,
                isCheckCrc,
                fetchOffset,
                downloadFuture.getRecycleCallback());
    }

    @Override
    public String toString() {
        return "RemotePendingFetch{"
                + "remoteLogSegment="
                + remoteLogSegment
                + ", fetchOffset="
                + fetchOffset
                + ", posInLogSegment="
                + posInLogSegment
                + ", highWatermark="
                + highWatermark
                + '}';
    }
}
